<?php
/*
 * web-vision GmbH
 * Do not edit or add to this file if you wish to upgrade Magento to newer
 * versions in the future. If you wish to customize Magento for your
 * needs please refer to https://www.web-vision.de for more information.
 * @category    Extendware
 * @copyright   Copyright (c) 2001-2024 web-vision GmbH (https://www.web-vision.de)
 * @author      Mahesh Makwana <mmakwana@web-vision.de>
 */
namespace Extendware\AdvanceUrl\Helper;

use Extendware\AdvanceUrl\Logger\Logger;
use Extendware\AdvanceUrl\Model\AdvanceUrlFactory as AdvanceUrlFactory;
use Magento\Framework\App\Helper\AbstractHelper;
use Magento\Framework\App\Helper\Context;
use Magento\Store\Model\ScopeInterface;
use Magento\Store\Model\StoreManagerInterface;
use Magento\UrlRewrite\Model\UrlRewriteFactory;
use Magento\Framework\Serialize\Serializer\Json;
use Magento\Framework\App\ResourceConnection;

class Data extends AbstractHelper
{
    public const MODULE_STATUS = 'system/advanced_url/enable';

    public const DEBUG_STATUS = 'system/advanced_url/enable_log';

    public const URL_PRE_FIX = '.html';

    public const PRODUCT_ENTITY = 'product';

    public const CUSTOM_ENTITY = 'custom';

    public const PRODUCT_CATLOG_URL = 'catalog/product/view/id/';

    public const REDIRECT_TYPE_301 = '301';

    public const REDIRECT_TYPE_0 = '0';

    public const REDIRECT_DESC = 'The product URL has been updated by the advanced URL cron.';

    public const ADVANCE_URL_STATUS_ATTR = 'advance_url_status';

    public const URL_CLOSE = 0;

    public const URL_PENDING = 1;

    public const URL_UPDATED = 2;

    public const URL_FAILED = 3;

    public const URL_QUEUE = 4;

    public const STATUS_CODE = 'status';

    public const MESSAGE_CODE = 'message';

    public const DEFAULT_DATETIME_FORMATE = 'Y-m-d H:i:s';

    public const NEW_DATETIME_FORMATE = 'l, F j, Y \a\t g:i:s A';

    public const URL_REWRITE_TABLE = 'url_rewrite';

    public const EXTENDWARE_ADVANCE_URL_HISTORY_TABLE = 'extendware_advance_url_history';

    public const EXTENDWARE_ADVANCE_URL_MAIN_TABLE = 'extendware_advance_url_main';

    /**
     * @var Logger
     */
    protected $logger;

    /**
     * @var StoreManagerInterface
     */
    protected $storeManager;

    /**
     * @var ScopeInterface
     */
    protected $scopeConfig;

    /**
     * @var AdvanceUrlFactory
     */
    protected $advanceUrlFactory;

    /**
     * @var Json
     */
    protected $json;

    /**
     * @var ResourceConnection
     */
    protected $resource;

    /**
     * @var ResourceConnection
     */
    protected $connection;

    /**
     * @var UrlRewriteFactory
     */
    protected $urlRewriteFactory;

    /**
     * Construct
     *
     * @param Context $context
     * @param Logger $logger
     * @param StoreManagerInterface $storeManager
     * @param AdvanceUrlFactory $advanceUrlFactory
     * @param UrlRewriteFactory $urlRewriteFactory
     * @param Json $json
     * @param ResourceConnection $resource
     */
    public function __construct(
        Context $context,
        Logger $logger,
        StoreManagerInterface $storeManager,
        AdvanceUrlFactory $advanceUrlFactory,
        UrlRewriteFactory $urlRewriteFactory,
        Json $json,
        ResourceConnection $resource
    ) {
        $this->logger = $logger;
        $this->storeManager = $storeManager;
        $this->scopeConfig = $context->getScopeConfig();
        $this->advanceUrlFactory = $advanceUrlFactory;
        $this->urlRewriteFactory = $urlRewriteFactory;
        $this->json = $json;
        $this->resource = $resource;
        $this->connection = $resource->getConnection(ResourceConnection::DEFAULT_CONNECTION);
        parent::__construct($context);
    }

    /**
     * Get configuration value
     *
     * @param string $field
     * @param int|null $storeId
     *
     * @return mixed
     */
    public function getConfigValue($field, $storeId = null)
    {
        return $this->scopeConfig->getValue(
            $field,
            ScopeInterface::SCOPE_STORE,
            $storeId
        );
    }

    /**
     * Get debug logger status
     *
     * @return int
     */
    public function getDebugStatus()
    {
        return (int) $this->getConfigValue(self::DEBUG_STATUS);
    }

    /**
     * Get Module status
     *
     * @return int
     */
    public function getModuleStatus()
    {
        return (int) $this->getConfigValue(self::MODULE_STATUS);
    }

    /**
     * Get all stores.
     *
     * @return \Magento\Store\Api\Data\StoreInterface[]
     */
    public function getAllStore()
    {
        return $this->storeManager->getStores();
    }

    /**
     * Get advance URL data.
     *
     * @return \Extendware\AdvanceUrl\Model\AdvanceUrl
     */
    public function getAdvanceUrlData()
    {
        return $this->advanceUrlFactory->create();
    }

    /**
     * Update product URL.
     *
     * @param \Magento\Catalog\Model\Product $productData
     *
     * @return array
     */
    public function updateProductUrl($productData)
    {
        $productUrlStatus = $this->getUrlStatus(true, '');

        try {
            $mainId = $productData->getId();
            $productUrl = $this->getRealUrl($productData->getProductUrlkey());
            $productId = $productData->getProductId();
            $productRealUrl = $this->getRealProductUrl($productId);
            $storeId = $productData->getStoreId();
            $productSku = $productData->getProductSku();

            // Check product catalog URL
            $checkUrlRewrite = $this->checkProductUrl($storeId, $productRealUrl);
            $checkUrl = $checkUrlRewrite->getFirstItem();

            // If product url not exits then create new url
            if (count($checkUrlRewrite->getData()) == 0) {
                $productUrlStatus = $this->newProductUrlCreate($storeId, $productRealUrl, $productId, $productUrl, $productUrlStatus);
                if ($productUrlStatus[self::STATUS_CODE]) {
                    $message = 'The Product Url created: ' . $productUrl . ' for Store ID: ' . $storeId;
                    $productUrlStatus = $this->getUrlStatus($productUrlStatus[self::STATUS_CODE], $message, 'info');

                    $this->addUrlUpdateHistory($mainId, $productId, '', $productUrl, $storeId, $productSku, self::URL_CLOSE);
                } else {
                    $message = 'The Product Url not created: ' . $productUrl . ' for Store ID: ' . $storeId;
                    $productUrlStatus = $this->getUrlStatus($productUrlStatus[self::STATUS_CODE], $message, 'critical', $productUrlStatus[self::MESSAGE_CODE]);
                }

                return $productUrlStatus;
            }

            // If Product url is exist then check and update
            if ($checkUrl && $checkUrl->getRequestPath() !== $productUrl) {
                // URLs do not match, handle accordingly
                $oldUrl = $checkUrl->getRequestPath();
                // Check custom url
                $checkCustomUrlRewrite = $this->checkProductUrl($storeId, $productUrl, $oldUrl);

                // Update product URL
                $productUrlStatus = $this->updateUrlRewite($checkCustomUrlRewrite, $productUrl, $productRealUrl, $productId, $storeId);
                if ($productUrlStatus[self::STATUS_CODE]) {
                    // Add updated url history
                    $this->addUrlUpdateHistory($mainId, $productId, $oldUrl, $productUrl, $storeId, $productSku, self::URL_CLOSE);

                    $message = 'The Product Url updated: ' . $oldUrl . ' -> ' . $productUrl . ' for Store ID: ' . $storeId;
                    $productUrlStatus = $this->getUrlStatus($productUrlStatus[self::STATUS_CODE], $message, 'info', $productUrlStatus[self::MESSAGE_CODE]);
                } else {
                    $message = 'The Product Url not updated: ' . $oldUrl . ' -> ' . $productUrl . ' for Store ID: ' . $storeId;
                    $productUrlStatus = $this->getUrlStatus(false, $message, 'critical', $productUrlStatus[self::MESSAGE_CODE]);
                }
            } else {
                $message = 'The Product Url is proper : ' . $productUrl . ' for Store ID: ' . $storeId;
                $productUrlStatus = $this->getUrlStatus($productUrlStatus[self::STATUS_CODE], $message, 'info');
            }
        } catch (\Exception $e) {
            $message = 'updateProductUrl() Function Issue : ' . $e->getMessage();
            $productUrlStatus = $this->getUrlStatus(false, $message, 'critical', $productUrlStatus[self::MESSAGE_CODE]);
        }

        return $productUrlStatus;
    }

    public function addUrlUpdateHistory($mainId, $productId, $oldUrl, $newUrl, $storeId, $sku, $status)
    {
        try {
            $newData = [
                'main_id' => $mainId,
                'old_urlkey' => $oldUrl,
                'new_urlkey' => $newUrl,
                'store_id' => $storeId,
                'url_status' => $status,
                'product_sku' => $sku,
            ];
            $this->connection->insert(
                $this->connection->getTableName(self::EXTENDWARE_ADVANCE_URL_HISTORY_TABLE),
                $newData
            );
        } catch (\Exception $e) {
            $this->logger->critical('addUrlUpdateHistory() Function Issue : ' . $e->getMessage());
        }
    }

    /**
     * Update URL rewrite.
     *
     * @param \Magento\UrlRewrite\Model\ResourceModel\UrlRewriteCollection $urlRewrite
     * @param string $productUrl
     * @param string $productRealUrl
     * @param int $productId
     * @param int $storeId
     *
     * @return bool
     */
    public function updateUrlRewite($urlRewrite, $productUrl, $productRealUrl, $productId, $storeId)
    {
        $productUrlStatus = $this->getUrlStatus(true, '');
        foreach ($urlRewrite as $url) {
            try {
                // Convert main URL into custom URL
                if ($url->getTargetPath() == $productRealUrl) {
                    $updateProductUrl = [
                        'url_rewrite_id' => $url->getUrlRewriteId(),
                        'target_path' => $productUrl,
                        'entity_type' => self::CUSTOM_ENTITY,
                        'redirect_type' => self::REDIRECT_TYPE_301,
                        'description' => self::REDIRECT_DESC,
                        'is_autogenerated' => 0
                    ];

                    $this->connection->insertOnDuplicate(
                        $this->connection->getTableName(self::URL_REWRITE_TABLE),
                        $updateProductUrl
                    );
                    continue;
                }

                // Convert custom URL into main URL
                if ($url->getRequestPath() == $productUrl) {
                    $updateCustomUrl = [
                        'url_rewrite_id' => $url->getUrlRewriteId(),
                        'target_path' => $productRealUrl,
                        'entity_id' => $productId,
                        'entity_type' => self::PRODUCT_ENTITY,
                        'redirect_type' => self::REDIRECT_TYPE_0,
                        'description' => self::REDIRECT_DESC,
                        'is_autogenerated' => 1
                    ];

                    $this->connection->insertOnDuplicate(
                        $this->connection->getTableName(self::URL_REWRITE_TABLE),
                        $updateCustomUrl
                    );
                }
            } catch (\Exception $e) {
                $message = 'updateUrlRewite() Function ForEach Loop Issue : ' . $e->getMessage();
                $productUrlStatus = $this->getUrlStatus(false, $message, 'critical', $productUrlStatus[self::MESSAGE_CODE]);

                break;
            }
        }

        // Created new URL if not exist
        if ($productUrlStatus[self::STATUS_CODE]) {
            $productUrlStatus = $this->newProductUrlCreate($storeId, $productRealUrl, $productId, $productUrl, $productUrlStatus);
        }

        return $productUrlStatus;
    }

    /**
     * Create new product URL.
     *
     * @param int $storeId
     * @param string $productRealUrl
     * @param int $productId
     * @param string $productUrl
     * @param bool $productUrlStatus
     *
     * @return array
     */
    public function newProductUrlCreate($storeId, $productRealUrl, $productId, $productUrl, $productUrlStatus)
    {
        // If Main product url not exits then create new url
        try {
            $finalUrlCheck = $this->checkProductUrl($storeId, $productRealUrl);
            if (count($finalUrlCheck) == 0) {
                $newData = [
                    'entity_type' => self::PRODUCT_ENTITY,
                    'entity_id' => $productId,
                    'request_path' => $productUrl,
                    'target_path' => $productRealUrl,
                    'redirect_type' => self::REDIRECT_TYPE_0,
                    'store_id' => $storeId,
                    'description' => self::REDIRECT_DESC,
                    'is_autogenerated' => 1
                ];

                $this->connection->insert(
                    $this->connection->getTableName(self::URL_REWRITE_TABLE),
                    $newData
                );
            }
        } catch (\Exception $e) {
            $message = 'updateUrlRewite() Function New URL Create Issue : ' . $e->getMessage();
            $productUrlStatus = $this->getUrlStatus(false, $message, 'critical');
        }

        return $productUrlStatus;
    }

    /**
     * Check product URL.
     *
     * @param int $storeId
     * @param string $targetPath
     * @param string|bool $requestPath
     *
     * @return \Magento\UrlRewrite\Model\ResourceModel\UrlRewriteCollection|bool
     */
    public function checkProductUrl($storeId, $targetPath, $requestPath = false)
    {
        $urlRewrite = $this->getUrlFactory()->getCollection();

        try {
            $urlRewrite->addFieldToFilter('store_id', $storeId);
            if ($requestPath) {
                $urlRewrite->addFieldToFilter(
                    ['target_path', 'target_path', 'request_path', 'request_path'],
                    [
                        ['eq' => $targetPath],
                        ['eq' => $requestPath],
                        ['eq' => $targetPath],
                        ['eq' => $requestPath],
                    ]
                );
            } else {
                $urlRewrite->addFieldToFilter('target_path', ['eq' => $targetPath]);
            }
        } catch (\Exception $e) {
            $this->logger->critical('checkProductUrl() Function Issue : ' . $e->getMessage());
            $urlRewrite = false;
        }

        return $urlRewrite;
    }

    /**
     * Get URL factory.
     *
     * @return \Magento\UrlRewrite\Model\UrlRewrite
     */
    public function getUrlFactory()
    {
        return $this->urlRewriteFactory->create();
    }

    /**
     * Get real URL.
     *
     * @param string $url
     *
     * @return string
     */
    public function getRealUrl($url)
    {
        return $url . self::URL_PRE_FIX;
    }

    /**
     * Get real product URL.
     *
     * @param int $id
     *
     * @return string
     */
    public function getRealProductUrl($id)
    {
        return self::PRODUCT_CATLOG_URL . $id;
    }

    /**
     * Get URL status.
     *
     * @param int $status
     * @param string $message
     * @param mixed $logType
     * @param mixed $oldMessage
     *
     * @return array
     */
    public function getUrlStatus($status, $message, $logType = false, $oldMessage = '')
    {

        // Log messages if debug is enabled
        if ($this->getDebugStatus()) {
            $cleanMessage = str_replace(PHP_EOL, '', $message);
            switch ($logType) {
                case 'info':
                    $this->logger->info($cleanMessage);
                    break;
                case 'critical':
                    $this->logger->critical($cleanMessage);
                    break;
            }
        }

        // Append old message if present
        if (!empty($oldMessage)) {
            $message = $oldMessage . "\n" . $message;
        }

        return [
            self::STATUS_CODE => (int) $status,
            self::MESSAGE_CODE => $message,
        ];
    }

    /**
     * Formatted DateTime With current timezone
     *
     * @return string
     */
    public function formateDateWithoutTimeZone($format, $date)
    {
        $dt = \DateTime::createFromFormat(self::DEFAULT_DATETIME_FORMATE, $date);
        return $dt->format($format);
    }

    /**
     * Get Json Unserialize Data
     *
     * @return array
     */
    public function jsonUnserialize($data)
    {
        return $this->json->unserialize($data);
    }

    /**
     * Get Json serialize Data
     *
     * @return string
     */
    public function jsonSerialize($data)
    {
        return $this->json->serialize($data);
    }

    /**
     * Close Database Rersource Connection
     *
     * @return string
     */
    public function closeResourceConnection()
    {
        $this->connection->closeConnection();
    }
}
