<?php
/**
 * @category  Extendware
 *
 * @copyright 2020 Extendware
 */
declare(strict_types=1);
namespace Extendware\AskQuestion\Controller\Question;

use Extendware\AskQuestion\Helper\Data;
use Extendware\AskQuestion\Model\QuestionFactory;
use Extendware\AskQuestion\Model\ResourceModel\Question as QuestionResource;
use Magento\Framework\App\Action\Context;
use Magento\Framework\App\Action\HttpPostActionInterface;
use Magento\Framework\App\CsrfAwareActionInterface;
use Magento\Framework\App\Request\InvalidRequestException;
use Magento\Framework\App\RequestInterface;
use Magento\Framework\Controller\Result\JsonFactory;
use Magento\Framework\Controller\Result\Redirect;
use Magento\Framework\Controller\Result\RedirectFactory;
use Magento\Framework\Data\Form\FormKey\Validator as FormKeyValidator;
use Magento\Framework\Phrase;
use Psr\Log\LoggerInterface;

class Ask implements HttpPostActionInterface, CsrfAwareActionInterface
{
    /**
     * @var RequestInterface
     */
    protected $request;

    /**
     * @var FormKeyValidator
     */
    private $formKeyValidator;

    /**
     * @var QuestionFactory
     */
    protected $_questionFactory;

    /**
     * @var LoggerInterface
     */
    protected $logger;

    /**
     * @var Data
     */
    protected $helper;

    /**
     * @var JsonFactory
     */
    private JsonFactory $resultJsonFactory;

    /**
     * @var RedirectFactory
     */
    private RedirectFactory $resultRedirectFactory;
    /**
     * @var QuestionResource
     */
    private QuestionResource $questionResource;

    /**
     * Ask constructor.
     *
     * @param LoggerInterface $logger
     * @param Data $helper
     * @param QuestionFactory $questionFactory
     * @param JsonFactory $resultJsonFactory
     * @param FormKeyValidator $formKeyValidator
     * @param RedirectFactory $resultRedirectFactory
     * @param Context $context
     * @param QuestionResource $questionResource
     */
    public function __construct(
        LoggerInterface $logger,
        Data $helper,
        QuestionFactory $questionFactory,
        JsonFactory $resultJsonFactory,
        FormKeyValidator $formKeyValidator,
        RedirectFactory $resultRedirectFactory,
        Context $context,
        QuestionResource $questionResource
    ) {
        $this->logger = $logger;
        $this->helper = $helper;
        $this->_questionFactory = $questionFactory;
        $this->resultJsonFactory = $resultJsonFactory;
        $this->formKeyValidator = $formKeyValidator;
        $this->resultRedirectFactory = $resultRedirectFactory;
        $this->request = $context->getRequest();
        $this->questionResource = $questionResource;
    }

    public function execute()
    {
        $resultJson = $this->resultJsonFactory->create();
        $question = $this->_questionFactory->create();
        $data = $this->getRequest()->getParams();

        try {
            if (!empty($data['email']) && !empty($data['product_sku']) && !empty($data['question'])) {
                $question->setName($data['name']);
                $question->setProductSku($data['product_sku']);
                $question->setQuestion($data['question']);
                $question->setEmail($data['email']);
                $this->questionResource->save($question);
                $this->helper->sendEmailToAdmin(
                    $data['name'],
                    $data['product_sku'],
                    $data['question']
                );

                return $resultJson->setData([
                    'msg' => __('Thank you for submitting your message. We will contact you shortly.'),
                    'success' => true,
                ]);
            }

            return $resultJson->setData([
                    'msg' => __('Please enter all the required fields.'),
                    'success' => false,
                ]);
        } catch (\Exception  $e) {
            $this->logger->critical($e->getMessage());

            return $resultJson->setData([
                'msg' => __('There is some issue. Please contact us through contact page.'),
                'success' => false,
            ]);
        }
    }

    /**
     * @inheritDoc
     */
    public function createCsrfValidationException(RequestInterface $request): ?InvalidRequestException
    {
        /** @var Redirect $resultRedirect */
        $resultRedirect = $this->resultRedirectFactory->create();
        $resultRedirect->setPath('not-found');

        return new InvalidRequestException(
            $resultRedirect,
            [
                new Phrase('Invalid Form Key. Please refresh the page.'),
            ]
        );
    }

    /**
     * @inheritDoc
     */
    public function validateForCsrf(RequestInterface $request): ?bool
    {
        return $this->formKeyValidator->validate($request);
    }

    /**
     * @return RequestInterface
     */
    protected function getRequest(): RequestInterface
    {
        return $this->request;
    }
}
