<?php
/*
 *
 *  * web-vision GmbH
 *  * Do not edit or add to this file if you wish to upgrade Magento to newer
 *  * versions in the future. If you wish to customize Magento for your
 *  * needs please refer to https://www.web-vision.de for more information.
 *  * @category    Extendware
 *  * @copyright   Copyright (c) 2001-2023 web-vision GmbH (https://www.web-vision.de)
 *  * @author      Fenil Timbadiya
 *
 */
namespace Extendware\AttachmentPathSegment\ViewModel;

use Extendware\AttachmentPathSegment\Model\ResourceModel\AttachmentPathSegment\CollectionFactory as PathSegmentCollectionFactory;
use Extendware\EWProductAttachment\Model\FileIcons;
use Extendware\EWProductAttachment\Model\ResourceModel\ProductattachmentRules\CollectionFactory as AttachmentCollectionFactory;
use Magento\Customer\Model\Session;
use Magento\Framework\App\Config\ScopeConfigInterface;
use Magento\Framework\App\ResourceConnection;
use Magento\Framework\Filesystem\DirectoryList;
use Magento\Framework\UrlInterface;
use Magento\Framework\View\Element\Block\ArgumentInterface;
use Magento\Store\Model\ScopeInterface;
use Magento\Store\Model\StoreManagerInterface;

class Attchement implements ArgumentInterface
{
    const EXTENDWARE_ATTACHMENTRULES_DOWNLOAD_ALLOWED = 'extendware_attachmentrules/general/download_allowed';
    const EXTENDWARE_ATTACHMENTRULES_FILE_SORTING_BY_ORDER = 'extendware_attachmentrules/general/file_sorting_by_order';
    const EXTENDWARE_ATTACHMENTRULES_FRONTEND_VIEW = 'extendware_attachmentrules/general/frontend_view';

    /**
     * @var AttachmentCollectionFactory
     */
    private $attachmentCollectionFactory;

    /**
     * @var PathSegmentCollectionFactory
     */
    private $pathSegmentCollectionFactory;

    /**
     * @var StoreManagerInterface
     */
    protected $storeManager;

    /**
     * @var DirectoryList $directoryList
     */
    protected $directoryList;

    /**
     * @var ResourceConnection
     */
    protected $resource;

    /**
     * @var FileIcons
     */
    protected $fileicons;

    /**
     * @var Session
     */
    protected $_session;

    /**
     * @var ScopeConfigInterface $scopeConfig
     */
    protected $_scopeConfig;

    /**
     * @param AttachmentCollectionFactory $attachmentCollectionFactory
     * @param PathSegmentCollectionFactory $attachmentPathSegmentCollectionFactory
     * @param StoreManagerInterface $storeManager
     * @param DirectoryList $directoryList
     * @param ResourceConnection $resource
     * @param FileIcons $fileicons
     * @param Session $session
     * @param copeConfigInterface $scopeConfig
     */
    public function __construct(
        AttachmentCollectionFactory $attachmentCollectionFactory,
        PathSegmentCollectionFactory $pathSegmentCollectionFactory,
        StoreManagerInterface $storeManager,
        DirectoryList $directoryList,
        ResourceConnection $resource,
        FileIcons $fileicons,
        Session $session,
        ScopeConfigInterface $scopeConfig
    ) {
        $this->attachmentCollectionFactory = $attachmentCollectionFactory;
        $this->pathSegmentCollectionFactory = $pathSegmentCollectionFactory;
        $this->storeManager = $storeManager;
        $this->directoryList = $directoryList;
        $this->resource = $resource;
        $this->fileicons = $fileicons;
        $this->_session = $session;
        $this->_scopeConfig = $scopeConfig;
    }

    /**
     * ProductAttchemnetDetails function
     *
     * @param integer $entity_id
     * @return void
     */
    public function productAttchemnetDetails($entity_id)
    {
        $now = new \DateTime();
        $currentStoreId = $this->storeManager->getStore()->getId();
        /** @var \Extendware\EWProductAttachment\Model\ResourceModel\ProductattachmentRules\Collection $ProductattachmentCollectionFactory */
        $productattachmenCollection = $this->attachmentCollectionFactory->create();
        $productattachmenCollection->addFilter('entity_id', $entity_id);
        $productattachmenCollection->getSelect()->joinLeft(
            ['ear' => $this->getTable('extendware_attachmentrules')],
            'main_table.rule_id = ear.rule_id',
            ['name', 'description', 'from_date', 'to_date', 'is_active', 'sort_order', 'created_at AS rule_created_at']
        );
        $productattachmenCollection->addFieldToFilter('ear.is_active', 1);
        $productattachmenCollection->addFieldToFilter('entity_id', $entity_id);
        $productattachmenCollection->addFieldToFilter('ear.from_date', ['lteq' => $now->format('Y-m-d')]);
        $productattachmenCollection->addFieldToFilter(
            'ear.to_date',
            [
                [
                    'gteq' => $now->format('Y-m-d')
                ],
                [
                    'null' => true
                ]
            ]
        );
        $sorting_on = $this->getConfig(self::EXTENDWARE_ATTACHMENTRULES_FILE_SORTING_BY_ORDER);
        if ($sorting_on == 'alphabetical') {
            $productattachmenCollection->setOrder('sort_order', 'ASC');
            $productattachmenCollection->setOrder('file_name', 'ASC');
        }
        if ($sorting_on == 'date') {
            $productattachmenCollection->setOrder('rule_created_at', 'ASC');
        }
        if ($sorting_on == 'rule_priority') {
            $productattachmenCollection->setOrder('sort_order', 'ASC');
        }
        $segmentCollection = $this->pathSegmentCollectionFactory->create();
        $segments = [];

        foreach ($segmentCollection->getData() as $segment) {
            $segments[$segment['term_name']][$segment['store_id']] = $segment['front_label'];
        }

        $rules_grouped_array = [];

        foreach ($productattachmenCollection->getData() as $attachment) {
            $termName = $this->getFileLastDir($attachment['file_path']);
            $frontLabel = $segments[$termName][$currentStoreId] ?? '';

            $rules_grouped_array[$termName]['front_label'] = $frontLabel;
            $rules_grouped_array[$termName]['product_attachment'][] = $attachment;
        }

        return $rules_grouped_array;
    }

    /**
     * GetTable function
     *
     * @param string $name
     * @return string
     */
    public function getTable(string $name)
    {
        return $this->resource->getTableName($name);
    }

    /**
     * GetFileIconUrl function
     *
     * @param string $url
     * @return string
     */
    public function getMediaUrl($url)
    {
        $mediaUrl = $this->storeManager->getStore()->getBaseUrl(\Magento\Framework\UrlInterface::URL_TYPE_MEDIA);
        $result = explode('media/', $url);
        return $mediaUrl . $result[1];
    }

    /**
     * GetFileIconUrl function
     *
     * @param string $url
     * @return string
     */
    public function getFileIconUrl($url)
    {
        //get the file type
        $iconUrl = '';
        $path_info = pathinfo($url);
        $mediaUrl = $this->storeManager->getStore()->getBaseUrl(\Magento\Framework\UrlInterface::URL_TYPE_MEDIA);
        $fileType = $path_info['extension'];
        $fileiconsCollection = $this->fileicons->getCollection()->addFieldToFilter('is_active', ['eq' => 1]);
        foreach ($fileiconsCollection as $icons) {
            if (strpos($fileType, $icons->getType()) !== false) {
                $iconUrl = $mediaUrl . 'extendware/ewproduct/icon/' . $icons->getImage();
            }
        }
        return $iconUrl;
    }

    /**
     * GetFileInfo function
     *
     * @param string $url
     * @return string
     */
    public function getFileInfo($url)
    {
        $image = ["jpg", "png", "jpeg", "svg", "PNG", "JPEG", "SVG", "JPG"];
        $path_info = pathinfo($url);
        if (in_array($path_info['extension'], $image)) {
            return "image";
        } elseif ($path_info['extension'] == 'pdf') {
            return "pdffile";
        } else {
            return 'file';
        }
    }

    /**
     * GetDownloadAccsee function
     *
     * @return void
     */
    public function getDownloadAccsee()
    {
        $dowwnload_access = $this->getConfig(self::EXTENDWARE_ATTACHMENTRULES_DOWNLOAD_ALLOWED);
        return !$dowwnload_access || $this->_session->isLoggedIn();
    }

    /**
     * GetFrontendView function
     *
     * @return void
     */
    public function getFrontendView()
    {
        $fronend_view = $this->getConfig(self::EXTENDWARE_ATTACHMENTRULES_FRONTEND_VIEW);
        return $fronend_view ? $fronend_view : 'grid';
    }

    /**
     * GetConfig function
     *
     * @param string $value
     * @return void
     */
    public function getConfig($value)
    {
        return $this->_scopeConfig->getValue($value, ScopeInterface::SCOPE_STORE);
    }

    public function getAbsoluteUrl($filePath)
    {
        $mediaDir = $this->directoryList->getPath(UrlInterface::URL_TYPE_MEDIA);
        $relativePath = explode('media/', $filePath);

        return $mediaDir . '/' . $relativePath[1];
    }

    public function getFileName($filePath)
    {
        $absolutePath = $this->getAbsoluteUrl($filePath);

        return $this->getFileMetaDetails($absolutePath)
             ?: ucwords(str_replace(['_', '-'], ' ', pathinfo($filePath, PATHINFO_FILENAME)));
    }

    public function getFileMetaDetails($absolutePath)
    {
        $fileInfo = pathinfo($absolutePath);

        try {
            if (strtolower($fileInfo['extension']) === 'pdf') {
                $pdf = \Zend_Pdf::load($absolutePath);

                return mb_convert_encoding($pdf->properties['Title'], 'UTF-8', 'ISO-8859-1') ?: null;
            }
        } catch (\Exception $e) {
            return null;
        }

        return null;
    }

    public function getFileLastDir($filePath)
    {
        $pathParts = explode('/', rtrim(str_replace('\\', '/', $this->getAbsoluteUrl($filePath)), '/'));

        return count($pathParts) > 1 ? $pathParts[count($pathParts) - 2] : null;
    }
}
