<?php
/*
 *
 *  *
 *  *  * web-vision GmbH
 *  *  * Do not edit or add to this file if you wish to upgrade Magento to newer
 *  *  * versions in the future. If you wish to customize Magento for your
 *  *  * needs please refer to https://www.web-vision.de for more information.
 *  *  * @copyright   Copyright (c) 2001-2025 web-vision GmbH (https://www.web-vision.de)
 *  *  * @author      Parth Trivedi
 *  *
 *
 */
namespace Extendware\AutoRelatedProducts\Controller\Ajax;

use Magento\Catalog\Helper\Image as ImageHelper;
use Magento\Catalog\Model\ResourceModel\Product\Attribute\CollectionFactory as AttributeCollectionFactory;
use Magento\Catalog\Model\ResourceModel\Product\CollectionFactory;
use Magento\Framework\App\Action\Action;
use Magento\Framework\App\Action\Context;
use Magento\Framework\App\CsrfAwareActionInterface;
use Magento\Framework\App\Request\InvalidRequestException;
use Magento\Framework\App\RequestInterface;
use Magento\Framework\Controller\Result\JsonFactory;
use Magento\Framework\Pricing\PriceCurrencyInterface;
use Magento\Store\Model\StoreManagerInterface;

class Products extends Action implements CsrfAwareActionInterface
{
    /**
     * @var JsonFactory
     */
    protected $resultJsonFactory;
    /**
     * @var CollectionFactory
     */
    protected $productCollectionFactory;
    /**
     * @var ImageHelper
     */
    protected $imageHelper;
    /**
     * @var StoreManagerInterface
     */
    protected $storeManager;
    /**
     * @var RequestInterface
     */
    protected $request;
    /**
     * @var PriceCurrencyInterface
     */
    protected $priceCurrency;
    /**
     * @var AttributeCollectionFactory
     */
    protected $attributeCollectionFactory;

    /**
     * @param Context $context
     * @param JsonFactory $resultJsonFactory
     * @param CollectionFactory $productCollectionFactory
     * @param ImageHelper $imageHelper
     * @param StoreManagerInterface $storeManager
     * @param RequestInterface $request
     * @param PriceCurrencyInterface $priceCurrency
     * @param AttributeCollectionFactory $attributeCollectionFactory
     */
    public function __construct(
        Context $context,
        JsonFactory $resultJsonFactory,
        CollectionFactory $productCollectionFactory,
        ImageHelper $imageHelper,
        StoreManagerInterface $storeManager,
        RequestInterface $request,
        PriceCurrencyInterface $priceCurrency,
        AttributeCollectionFactory $attributeCollectionFactory
    ) {
        $this->resultJsonFactory = $resultJsonFactory;
        $this->productCollectionFactory = $productCollectionFactory;
        $this->imageHelper = $imageHelper;
        $this->storeManager = $storeManager;
        $this->request = $request;
        $this->priceCurrency = $priceCurrency;
        $this->attributeCollectionFactory = $attributeCollectionFactory;
        parent::__construct($context);
    }

    // === CSRF bypass ===

    /**
     * @param RequestInterface $request
     * @return InvalidRequestException|null
     */
    public function createCsrfValidationException(RequestInterface $request): ?InvalidRequestException
    {
        return null;
    }

    /**
     * @param RequestInterface $request
     * @return bool|null
     */
    public function validateForCsrf(RequestInterface $request): ?bool
    {
        return true;
    }

    public function execute()
    {
        // Set noindex, nofollow for AJAX endpoints
        $this->getResponse()->setHeader('X-Robots-Tag', 'noindex, nofollow');
        
        $result = $this->resultJsonFactory->create();
        $params = json_decode($this->request->getContent(), true);
        $productIds = $params['product_ids'] ?? [];
        $page = isset($params['page']) ? (int)$params['page'] : 1;
        $pageSize = isset($params['per_page']) ? (int)$params['per_page'] : 4;
        $sortBy = isset($params['sort_by']) ? $params['sort_by'] : 'name';
        $sortDirection = isset($params['sort_direction']) ? strtoupper($params['sort_direction']) : 'ASC';

        if (empty($productIds)) {
            return $result->setData([
                'products' => [],
                'total_pages' => 0,
            ]);
        }

        $collection = $this->productCollectionFactory->create();
        $collection->addAttributeToSelect(['name', 'image', 'price', 'created_at'])
            ->addAttributeToFilter('entity_id', ['in' => $productIds])
            ->addAttributeToFilter('visibility', ['neq' => \Magento\Catalog\Model\Product\Visibility::VISIBILITY_NOT_VISIBLE])
            ->addAttributeToFilter('status', \Magento\Catalog\Model\Product\Attribute\Source\Status::STATUS_ENABLED);

        $sortableAttributes = $this->attributeCollectionFactory->create()
            ->addFieldToFilter('used_for_sort_by', 1)
            ->addFieldToFilter('is_visible', 1);

        $validSortAttributes = [];
        foreach ($sortableAttributes as $attribute) {
            $validSortAttributes[] = $attribute->getAttributeCode();
        }

        if (in_array($sortBy, $validSortAttributes)) {
            $collection->setOrder($sortBy, $sortDirection);
        } else {
            $collection->setOrder('name', 'ASC');
        }

        $collection->setPageSize($pageSize)
            ->setCurPage($page);

        $products = [];
        foreach ($collection as $product) {
            $products[] = [
                'id' => $product->getId(),
                'name' => $product->getName(),
                'image_url' => $this->imageHelper->init($product, 'product_page_image_small')->getUrl(),
                'url' => $product->getProductUrl(),
                'price' => $this->priceCurrency->format($product->getPrice(), false),
                'created_at' => $product->getCreatedAt()
            ];
        }

        return $result->setData([
            'products' => $products,
            'total_pages' => ceil(count($productIds) / $pageSize),
        ]);
    }
}
