<?php
/*
 web-vision GmbH
 Do not edit or add to this file if you wish to upgrade Magento to newer
 versions in the future. If you wish to customize Magento for your
 needs please refer to https://www.web-vision.de for more information.
 @category    Extendware
 @copyright   Copyright (c) 2001-2023 web-vision GmbH (https://www.web-vision.de)
 @author      Bushra Asif <ba@web-vision.de>
*/

namespace Extendware\Bfcache\Helper;

use Extendware\Core\Api\Data\ExtendwareModuleInterface;
use Magento\Store\Model\ScopeInterface;
use Magento\Store\Model\StoreManagerInterface;
use Magento\Framework\App\Config\ScopeConfigInterface;
use Extendware\Bfcache\Model\ConstantConfig;
use Extendware\Core\Model\ExtendwareModule;

class Data
{
    /**
     * @var StoreManagerInterface
     */
    protected $scopeConfig;

    /**
     * @var ScopeConfigInterface
     */
    protected $storeManager;

    /**
     * @var ExtendwareModule
     */
    protected $extendwareModule;
    /**
     * @param StoreManagerInterface $storeManager
     * @param ScopeConfigInterface $scopeConfig
     * @param ExtendwareModule $extendwareModule
     */
    public function __construct(
        StoreManagerInterface $storeManager,
        ScopeConfigInterface  $scopeConfig,
        ExtendwareModule $extendwareModule
    )
    {
        $this->storeManager = $storeManager;
        $this->scopeConfig = $scopeConfig;
        $this->extendwareModule = $extendwareModule;
    }


    /**
     * Check if the module is enabled in extendware core
     *
     * @return bool
     */
    public function isModuleRegistered()
    {
        $module = $this->extendwareModule->load(ConstantConfig::MODULE_NAME, ExtendwareModuleInterface::MODULE_NAME);

        return $module->getModuleActive();
    }

    /**
     * Check if the back-forward cache (BFCache) is enabled.
     *
     * @return null|bool
     */
    public function isBfCacheEnabled()
    {
        return $this->scopeConfig->getValue(ConstantConfig::MODULE_ENABLED, ScopeInterface::SCOPE_STORE);
    }

    /**
     * Check if the mini cart should be updated after interaction.
     *
     * @return null|bool
     */
    public function miniCartUpdateAfterInteraction()
    {
        return $this->scopeConfig->getValue(ConstantConfig::MINICART_UPDATE_FIELD, ScopeInterface::SCOPE_STORE);
    }

    /**
     * Retrieve the list of excluded routes from configuration.
     *
     * @return array
     */
    public function excludedRoutes()
    {
        $list = $this->scopeConfig->getValue(ConstantConfig::EXCLUDED_ROUTES, ScopeInterface::SCOPE_STORE);

        if (empty($list)) {
            return [];
        }

        return array_map('trim', explode(',', $list));
    }

    /**
     * Check if page preload is enabled.
     *
     * @return null|bool
     */
    public function isPagePreloadEnabled()
    {
        return $this->scopeConfig->getValue(ConstantConfig::PAGE_PRELOAD_ENABLED, ScopeInterface::SCOPE_STORE);
    }


    /**
     * Get the link preload setting.
     *
     * @return null|string
     */
    public function getPreloadIntensity()
    {
        return $this->scopeConfig->getValue(ConstantConfig::PAGE_PRELOAD_SETTNG, ScopeInterface::SCOPE_STORE);
    }

    /**
     * Get the custom delay for page preload.
     *
     * @return null|string
     */
    public function getCustomDelay()
    {
        return $this->scopeConfig->getValue(ConstantConfig::PAGE_PRELOAD_CUSTOM_DELAY, ScopeInterface::SCOPE_STORE);
    }

    /**
     * Get the maximum age for cache control header.
     *
     * @return null|string
     */
    public function getHeaderMaxAge()
    {
        return $this->scopeConfig->getValue(ConstantConfig::PAGE_PRELOAD_CACHE_CONTROL, ScopeInterface::SCOPE_STORE);
    }

    /**
     * Retrieve the URLs in the preload blacklist.
     *
     * @return null|array
     */
    public function getBlacklistUrl()
    {
        $urls = $this->scopeConfig->getValue(ConstantConfig::PAGE_PRELOAD_BLACKLIST_URL, ScopeInterface::SCOPE_STORE);

        if (empty($urls)) {
            return [];
        }

        return array_map('trim', explode(',', $urls));
    }
}
