<?php
/*
 web-vision GmbH
 Do not edit or add to this file if you wish to upgrade Magento to newer
 versions in the future. If you wish to customize Magento for your
 needs please refer to https://www.web-vision.de for more information.
 @category    Extendware
 @copyright   Copyright (c) 2001-2023 web-vision GmbH (https://www.web-vision.de)
 @author      Bushra Asif <ba@web-vision.de>
*/

declare(strict_types=1);

namespace Extendware\ContactUsTracker\Plugin\Model;

use Exception;
use Extendware\ContactUsTracker\Model\Config;
use Extendware\ContactUsTracker\Model\Repository;
use Magento\Contact\Model\Mail as Subject;
use Magento\Framework\DataObject;
use Magento\Store\Model\StoreManagerInterface;
use Psr\Log\LoggerInterface;
use Magento\Framework\Mail\Template\TransportBuilder;

class Mail
{
    private const DATA_TO_UNSET = ['name', 'lastname', 'email', 'telephone', 'comment', 'form_key', 'hideit'];

    /**
     * @var TransportBuilder
     */
    protected $transportBuilder;
    /**
     * @var Repository
     */
    protected $contactUsRepository;
    /**
     * @var StoreManagerInterface
     */
    protected $storeManager;
    /**
     * @var Config
     */
    protected $config;
    /**
     * @var LoggerInterface
     */
    protected $logger;

    /**
     * @param Repository $contactUsRepository
     * @param StoreManagerInterface $storeManager
     * @param Config $config
     * @param LoggerInterface $logger
     * @param TransportBuilder $transportBuilder
     */
    public function __construct(
        Repository $contactUsRepository,
        StoreManagerInterface $storeManager,
        Config $config,
        LoggerInterface $logger,
        TransportBuilder $transportBuilder
    ) {
        $this->contactUsRepository = $contactUsRepository;
        $this->storeManager = $storeManager;
        $this->config = $config;
        $this->logger = $logger;
        $this->transportBuilder = $transportBuilder;
    }

    /**
     * Creates a contact us entity before sending an email.
     *
     * The contact us entity will be saved in the database even if email sending fails.
     */
    public function beforeSend(Subject $subject, string|array $replyTo, array $variables): array
    {
        if (!$this->config->isEnabled()) {
            return [$replyTo, $variables];
        }

        // Prepare email transport to ensure the email subject is captured
        $templateVars = $variables;
        if (!is_array($templateVars)) {
            $templateVars = ['data' => new DataObject()];
        }

        $this->transportBuilder
            ->setTemplateIdentifier('contact_email_email_template') // Use the actual template identifier
            ->setTemplateOptions([
                'area' => \Magento\Framework\App\Area::AREA_FRONTEND,
                'store' => $this->storeManager->getStore()->getId(),
            ])
            ->setTemplateVars($templateVars)
            ->setFrom('general') // Use the actual sender configuration
            ->addTo($replyTo);

        // Get the email subject from the transport
        $transport = $this->transportBuilder->getTransport();
        $emailSubject = $transport->getMessage()->getSubject();

        $contactUs = $this->contactUsRepository->createEmptyModel();
        $data = clone $variables['data'];
        $additionalInfo = [];

        $contactUs->setName($data->getData('name') ?? '');
        $contactUs->setLastName($data->getData('lastname') ?? '');
        $contactUs->setCompany($data->getData('company') ?? '');
        $contactUs->setEmail($data->getData('email') ?? '');
        $contactUs->setPhone($data->getData('telephone') ?? '');
        $contactUs->setSubject($emailSubject); // Set the email subject here
        $contactUs->setComment($data->getData('comment') ?? '');

        $this->unsetExtraData($data);

        foreach ($data->getData() as $key => $value) {
            $additionalInfo[$key] = $value;
        }

        if ($additionalInfo) {
            $contactUs->setAdditionalInfo($additionalInfo);
        }

        try {
            $contactUs->setStoreId($this->storeManager->getStore()->getId());
            $this->contactUsRepository->save($contactUs);

        } catch (Exception $e) {
            $this->logger->error($e->getMessage());
        }

        return [$replyTo, $variables];
    }

    /**
     * Unsets extra data from the data object.
     */
    private function unsetExtraData(DataObject $data): void
    {
        $data->unsetData(self::DATA_TO_UNSET);
    }
}
