<?php
/**
 * Extendware by web-vision GmbH
 *
 * NOTICE OF LICENSE
 *
 * <!--LICENSETEXT-->
 *
 * DISCLAIMER
 *
 * Do not edit or add to this file if you wish to upgrade Magento to newer
 * versions in the future. If you wish to customize Magento for youre
 * needs please refer to https://www.extendware.com/magento-custom-development.html for more information.
 *
 * @category ExtendWare_EWBotBlocker
 *
 * @copyright (c) 2001-2021 Extendware by web-vision GmbH (https://extendware.com/)
 * @license https://support.extendware.com/
 * @affilate https://www.extendware.com/affiliate
 *
 * @author Vaibhav Ahalpara <vaibhav@web-vision.de>
 */

declare(strict_types=1);
namespace Extendware\EWBotBlocker\Helper;

use Extendware\EWBotBlocker\Helper\Config as configHelper;
use Magento\Backend\Setup\ConfigOptionsList as BackendConfigOptionsList;
use Magento\Framework\App\Area;
use Magento\Framework\App\DeploymentConfig;
use Magento\Framework\App\Helper\AbstractHelper;
use Magento\Framework\App\Request\Http;
use Magento\Framework\App\RequestInterface;
use Magento\Framework\App\State;
use Magento\Framework\HTTP\PhpEnvironment\Request;
use Magento\Framework\Stdlib\Cookie\CookieMetadataFactory;
use Magento\Framework\Stdlib\CookieManagerInterface;
use Magento\Framework\UrlInterface;


class Data extends AbstractHelper
{
    /**
     * @var Magento\Framework\UrlInterface
     */
    protected $_urlInterface;
    /**
     * @var Magento\Framework\HTTP\PhpEnvironment\Request
     */
    protected $_serverrequest;
    /**
     * @var Magento\Framework\App\RequestInterface
     */
    protected $_request;
    /**
     * @var Magento\Framework\Stdlib\CookieManagerInterface
     */
    protected $_cookieManager;
    /**
     * @var Magento\Framework\Stdlib\Cookie\CookieMetadataFactory
     */
    protected $_cookieMetadataFactory;
    /**
     * @var Magento\Framework\App\DeploymentConfig
     */
    protected $_deploymentConfig;
    /**
     * @var Extendware\EWBotBlocker\Helper\Config
     */
    protected $_configHelper;
    protected $httpRequest;
    /**
     * @var Magento\Framework\App\State
     */
    protected $state;
    
    /**
     * @param UrlInterface $urlInterface
     * @param Request $serverrequest
     * @param RequestInterface $request
     * @param CookieManagerInterface $cookieManager
     * @param CookieMetadataFactory $cookieMetadataFactory
     * @param DeploymentConfig $deploymentConfig
     * @param Config $configHelper
     * @param Http $httpRequest
     * @param State $state
     */
    public function __construct(
        UrlInterface $urlInterface,
        Request $serverrequest,
        RequestInterface $request,
        CookieManagerInterface $cookieManager,
        CookieMetadataFactory $cookieMetadataFactory,
        DeploymentConfig $deploymentConfig,
        configHelper $configHelper,
        Http $httpRequest,
        State $state
    ) {
        $this->_urlInterface = $urlInterface;
        $this->_serverrequest = $serverrequest;
        $this->_request = $request;
        $this->_cookieManager = $cookieManager;
        $this->_cookieMetadataFactory = $cookieMetadataFactory;
        $this->_deploymentConfig = $deploymentConfig;
        $this->_configHelper = $configHelper;
        $this->httpRequest = $httpRequest;
        $this->state = $state;
    }
    /**
     * Get Honeypot Url
     *
     * @param  string $honeypotKey
     * @param mixed $honeypotPrefix
     *
     * @return Magento\Framework\UrlInterface
     */
    public function getHoneypotUrl($honeypotPrefix, $honeypotKey)
    {
        return $this->_urlInterface->getUrl(
            '',
            [
            '_direct' => $honeypotPrefix . $honeypotKey,
            '_secure' => $this->_request->isSecure(), ]
        );
    }

    /**
     * Get Honeypot Url
     *
     * @param  string $salt
     *
     * @return string
     */
    public function getHoneypotKey($salt = null)
    {
        $suffixes = ['', '.html', '.php'];
        $suffix = $suffixes[array_rand($suffixes)];
        $prefix = substr(hash('sha256', date('W') . $salt . $suffix), 0, 4);
        $hash = substr(hash('sha256', $prefix . $this->_configHelper->getSalt()), 0, 6);

        return $prefix . $hash . $suffix;
    }
    /**
     * Get Url Path Prefix
     *
     * @return string
     */
    public function getPathPrefix()
    {
        $prefix = '';
        if ($this->_configHelper->getPathPrefix()) {
            $prefix = trim($this->_configHelper->getPathPrefix(), '/\\');
            if ($prefix) {
                $prefix .= '/';
            }
        }

        return $prefix;
    }
    /**
     * Check If Path Match With Honeypot Urls
     *
     * @param  string  $path
     *
     * @return bool
     */
    public function isHoneypotPath($path)
    {
        $area_code = $this->getAreaCode();
        if ($area_code == Area::AREA_ADMINHTML) {
            return false;
        }
        $path = ltrim($path, '/');
        $path = str_replace($this->getPathPrefix(), '', $path);
        $prefix = substr($path, 0, 4);
        $hash = substr($path, 4, 6);
        $isHoneypot = (bool) (substr(hash('sha256', $prefix . $this->_configHelper->getSalt()), 0, 6) == $hash);
        if ($isHoneypot === false) {
            $regexps = $this->_configHelper->getHoneypotPaths();
            if ($regexps && count($regexps)) {
                foreach ($regexps as $regexp) {
                    if (isset($_SERVER['REQUEST_URI']) and preg_match($regexp['string'], $_SERVER['REQUEST_URI'])) {
                        $isHoneypot = true;

                        break;
                    }
                }
            }
        }

        return $isHoneypot;
    }
    /**
     * Get Area Code
     *
     * @return string
     */
    public function getAreaCode()
    {
        return $this->state->getAreaCode();
    }
    /**
     * Get User's Agent
     *
     * @return string
     */
    public function getUserAgent()
    {
        return isset($_SERVER['HTTP_USER_AGENT']) ? $_SERVER['HTTP_USER_AGENT'] : 'UNKNOWN';
    }
    /**
     * Set cookies
     *
     * @return $this
     */
    public function sendVerificationCookie()
    {
        if ($this->_configHelper->isVerificationCookieEnabled()) {
            $time = time();
            $hash = substr(hash('sha256', $time . 'ewbb:verification' . $this->getIpAddress() . $this->_configHelper->getSalt()), 0, 10);
            $metadata = $this->_cookieMetadataFactory
             ->createPublicCookieMetadata()
             ->setDuration('86400');
            $this->_cookieManager->setPublicCookie(
                'ewbb',
                $time . ':' . $hash,
                $metadata
            );
        }

        return $this;
    }
    /**
     * Delete Cookies
     *
     * @return $this
     */
    public function deleteVerificationCookie()
    {
        $this->_cookieManager->deleteCookie('ewbb');

        return $this;
    }
    /**
     * Check if cookies valid
     *
     * @return bool
     */
    public function isVerificationCookieValid()
    {
        if (!$this->_configHelper->isVerificationCookieEnabled()) {
            return false;
        }
        $ewbb = $this->_cookieManager->getCookie('ewbb');
        if (!$ewbb) {
            return false;
        }
        if ($this->httpRequest->getParam('__ewbb_skip_verification')) {
            $this->deleteVerificationCookie();

            return false;
        }
        $parts = explode(':', $ewbb);
        if ((time() - $parts[0]) >= 30 * 60) {
            return false;
        } // check every 30 minutes;
        $hash = substr(hash('sha256', $parts[0] . 'ewbb:verification' . $this->getIpAddress() . $this->_configHelper->getSalt()), 0, 10);

        return $hash == $parts[1];
    }
    /**
     * Check if url qualifies for honeypot
     *
     * @return bool
     */
    public function uriMatchesDisqualifiers()
    {
        static $bool = null;
        if ($bool === null) {
            $bool = false;

            $list = $this->_configHelper->getRunnableUrlFilters();

            $key = (string)$this->_deploymentConfig->get(BackendConfigOptionsList::CONFIG_PATH_BACKEND_FRONTNAME);
            $list[]['string'] = '/' . preg_quote($key, '/') . '/';
            if (empty($list) === true) {
                $list[]['string'] = '/' . preg_quote('admin', '/') . '/';
            }
            $list[]['string'] = '/' . preg_quote('dashboard/index', '/') . '/';
            $list[]['string'] = '/' . preg_quote('extendware_', '/') . '/';
            foreach ($list as $regexp) {
                if (isset($_SERVER['REQUEST_URI']) and preg_match($regexp['string'], $_SERVER['REQUEST_URI'])) {
                    $bool = true;

                    break;
                }
            }
        }

        return $bool;
    }
    /**
     * Get User's IP Address
     *
     * @return string
     */
    public function getIpAddress()
    {
        $ip = null;
        $regExp = '/(\d+\.\d+\.\d+\.\d+)/';
        if (getenv('RW_DEV_SERVER_PASSPHRASE') == 'HyL6hNOs3lxN7Xo') {
            return '108.174.50.179';
        }
        if ($this->_configHelper->getIpMode() == 'remote_addr') {
            if (isset($_SERVER['HTTP_CF_CONNECTING_IP']) && preg_match($regExp, $_SERVER['HTTP_CF_CONNECTING_IP'])) {
                $ip = $_SERVER['HTTP_CF_CONNECTING_IP'];
            } elseif (isset($_SERVER['REMOTE_ADDR']) && preg_match($regExp, $_SERVER['REMOTE_ADDR'])) {
                $ip = $_SERVER['REMOTE_ADDR'];
            }
        } else {
            if (isset($_SERVER['HTTP_CLIENT_IP']) && preg_match($regExp, $_SERVER['HTTP_CLIENT_IP'])) {
                $ip = $_SERVER['HTTP_CLIENT_IP'];
            } elseif (isset($_SERVER['HTTP_X_FORWARDED_FOR']) && preg_match($regExp, $_SERVER['HTTP_X_FORWARDED_FOR'])) {
                $ip = $_SERVER['HTTP_X_FORWARDED_FOR'];
            } elseif (isset($_SERVER['HTTP_X_FORWARDED']) && preg_match($regExp, $_SERVER['HTTP_X_FORWARDED'])) {
                $ip = $_SERVER['HTTP_X_FORWARDED'];
            } elseif (isset($_SERVER['HTTP_FORWARDED_FOR']) && preg_match($regExp, $_SERVER['HTTP_FORWARDED_FOR'])) {
                $ip = $_SERVER['HTTP_FORWARDED_FOR'];
            } elseif (isset($_SERVER['HTTP_FORWARDED']) && preg_match($regExp, $_SERVER['HTTP_FORWARDED'])) {
                $ip = $_SERVER['HTTP_FORWARDED'];
            } elseif (isset($_SERVER['REMOTE_ADDR']) && preg_match($regExp, $_SERVER['REMOTE_ADDR'])) {
                $ip = $_SERVER['REMOTE_ADDR'];
            }
        }
        if (preg_match($regExp, $ip, $match)) {
            return $match[0];
        }

        return 'UNKNOWN';
    }
    /**
     * Check if url contains dis quallifiers keywords
     *
     * @return bool
     */
    public function doesUrlContainsDisquallifiers()
    {
        $isItTrue = false;
        $urlFilters = $this->_configHelper->getRunnableUrlFilters();
        if ($urlFilters && count($urlFilters)) {
            foreach ($urlFilters as $regexp) {
                if (null !== $this->_serverrequest->getRequestUri() and preg_match($regexp['string'], $this->_serverrequest->getRequestUri())) {
                    $isItTrue = true;

                    break;
                }
            }
        }

        return $isItTrue;
    }

    /**
     * Get Stop Words for forms
     *
     * @param mixed $identifier
     *
     * @return []
     */
    public function getStopWordForForm($identifier)
    {
        $enable = $this->_configHelper->getStopWordForForms();
        $useSameForAll = $this->_configHelper->getUseSameStopWordForAllTheForm();
        $stopWordForAll = $this->_configHelper->getStopWordForAllForms();
        $stopWordForSelected = $this->_configHelper->getStopWordForSelectedForm();
        if ($enable) {
            if ($useSameForAll && $stopWordForAll) {
                return explode(',', $stopWordForAll);
            } elseif ($stopWordForSelected && count($stopWordForSelected)) {
                $stopWordArray = [];
                foreach ($stopWordForSelected as $item) {
                    if (isset($item['stop_word']) && isset($item['form_action']) && $item['form_action'] && $item['stop_word'] && $item['form_action'] == $identifier) {
                        return explode(',', $item['stop_word']);
                    }
                }
            }
        }

        return [];
    }

    /**
     * Check if the given IP address or user agent is whitelisted.
     *
     * @param string $ipAddress
     * @param string $userAgent
     * @return bool
     */
    public function isWhitelisted($ipAddress, $userAgent)
    {
        $whitelistedIps = $this->_configHelper->getWhitelistedIps();
        $whitelistedUserAgents = $this->_configHelper->getWhitelistedUserAgents();

        foreach ($whitelistedIps as $whitelistedIp) {
            if ($this->isIpInRange($ipAddress, trim($whitelistedIp))) {
                return true;
            }
        }

        foreach ($whitelistedUserAgents as $whitelistUserAgent) {
            if (strpos($userAgent, trim($whitelistUserAgent)) !== false) {
                return true;
            }
        }

        return false;
    }

    /**
     * Check if an IP address is within a given range.
     *
     * @param string $ip
     * @param string $range
     * @return bool
     */
    protected function isIpInRange($ip, $range)
    {
        if (strpos($range, '/') !== false) {
            list($range, $netmask) = explode('/', $range, 2);
            $rangeDecimal = ip2long($range);
            $ipDecimal = ip2long($ip);
            $wildcardDecimal = pow(2, (32 - $netmask)) - 1;
            $netmaskDecimal = ~$wildcardDecimal;
            return (($ipDecimal & $netmaskDecimal) == ($rangeDecimal & $netmaskDecimal));
        } else {
            return $ip == $range;
        }
    }
}
