<?php
/*
 * web-vision GmbH
 * Do not edit or add to this file if you wish to upgrade Magento to newer
 * versions in the future. If you wish to customize Magento for your
 * needs please refer to https://www.web-vision.de for more information.
 * @category    Extendware
 * @copyright   Copyright (c) 2001-2022 web-vision GmbH (https://www.web-vision.de)
 * @author      Mahesh Makwana <mmakwana@web-vision.de>
 */
namespace Extendware\ImportDataUI\Cron;

use Extendware\ImportDataUI\Model\Config\Source\CronStatus;
use Magento\Framework\App\Filesystem\DirectoryList;
use WebVision\CustomDataSearch\Api\Data\CustomDataInterface;
ini_set('max_execution_time', 0);

class RunProfiler
{
    public const HISTORY_FILE_PATH = 'ImportDataUI';

    public const HISTORY_FILE_NAME = 'History';

    /**
     * @var \Extendware\ImportDataUI\Logger\Logger
     */
    protected $logger;

    /**
     * @var \Magento\Framework\HTTP\Client\Curl
     */
    protected $_curl;

    /**
     * @var \Magento\Framework\App\Filesystem\DirectoryList
     */
    protected $directoryList;

    /**
     * @var \Magento\Framework\Filesystem\Io\File
     */
    protected $file;

    /**
     * @var \Extendware\ImportDataUI\Model\ImportHistoryFactory
     */
    protected $importHistory;

    /**
     * @var \Magento\Store\Model\StoreManagerInterface
     */
    protected $_storeManager;

    /**
     * @var \Extendware\ImportDataUI\Helper\Data
     */
    protected $mhsiHelper;

    /**
     * @var \Magento\Framework\Stdlib\DateTime\TimezoneInterface
     */
    protected $timezoneInterface;

    /**
     * @var \Extendware\ImportDataUI\Logger\Logger
     */
    protected $logDirectory;

    /**
     * @var CleanUrlsProcess
     */
    protected $cleanUrlsProcess;

    /**
     * @var CleanUrlsProcess
     */
    protected $customDataSearchModel;

    /**
     * Constructor
     *
     * @param \Extendware\ImportDataUI\Logger\Logger $logger
     * @param \Magento\Framework\HTTP\Client\Curl $curl
     * @param \Magento\Framework\App\Filesystem\DirectoryList $directoryList
     * @param \Magento\Framework\Filesystem\Io\File $file
     * @param \Magento\Framework\Filesystem $filesystem
     * @param \Magento\Framework\Stdlib\DateTime\TimezoneInterface $timezoneInterface
     * @param \Extendware\ImportDataUI\Model\ImportHistoryFactory $importHistory
     * @param \Magento\Store\Model\StoreManagerInterface $storeManager
     */
    public function __construct(
        \Extendware\ImportDataUI\Logger\Logger $logger,
        \Magento\Framework\HTTP\Client\Curl $curl,
        \Magento\Framework\App\Filesystem\DirectoryList $directoryList,
        \Magento\Framework\Filesystem\Io\File $file,
        \Magento\Framework\Filesystem $filesystem,
        \Magento\Framework\Stdlib\DateTime\TimezoneInterface $timezoneInterface,
        \Extendware\ImportDataUI\Model\ImportHistoryFactory $importHistory,
        \Magento\Store\Model\StoreManagerInterface $storeManager,
        \Extendware\ImportDataUI\Helper\Data $mhsiHelper,
        \Extendware\AdvanceUrl\Model\CleanUrlsProcess $cleanUrlsProcess,
        \WebVision\CustomDataSearch\Model\CustomDataSearchFactory $customDataSearchModel
    ) {
        $this->logger = $logger;
        $this->_curl = $curl;
        $this->directoryList = $directoryList;
        $this->file = $file;
        $this->logDirectory = $filesystem->getDirectoryWrite(DirectoryList::MEDIA);
        $this->timezoneInterface = $timezoneInterface;
        $this->importHistory = $importHistory;
        $this->_storeManager = $storeManager;
        $this->mhsiHelper = $mhsiHelper;
        $this->cleanUrlsProcess = $cleanUrlsProcess;
        $this->customDataSearchModel = $customDataSearchModel;
    }

    /**
     * Execute the cron
     *
     * @return void
     */
    public function execute()
    {
        try {
            $importdata = $this->importHistory->create()->getCollection();
            $importdata->addFieldToFilter('cron_status', ['eq' => CronStatus::CRON_PROCESS]);

            if ($importdata->count() == 0) {
                $newImportdata = $this->importHistory->create()->getCollection();
                $newImportdata->addFieldToFilter('cron_status', ['eq' => CronStatus::CRON_PENDING]);
                $newImportdata->setOrder('importhistory_id', 'ASC');
                if ($newImportdata->count()) {
                    $newImport = $newImportdata->getFirstItem();

                    try {
                        set_time_limit(0);
                        $this->logger->info($newImport->getProfileName() . ' profile is running..');
                        $newImport->setCronStatus(CronStatus::CRON_PROCESS);
                        $newImport->setStartDate($this->timezoneInterface->date()->format('Y-m-d H:i:s'));
                        $newImport->save();

                        $curlUrl = $this->getProfilerUrl($newImport->getProfileName());

                        $dateTime = $this->timezoneInterface->date()->format('Y-m-d-H-i-s');
                        $historyFileName = self::HISTORY_FILE_NAME . '-' . $newImport->getImporthistoryId() . '-' . $dateTime . '.html';

                        $filePath = $this->directoryList->getPath('media') . '/' . self::HISTORY_FILE_PATH;
                        $ioAdapter = $this->file;
                        if (!$this->logDirectory->isDirectory($filePath)) {
                            $ioAdapter->mkdir($filePath, 0775);
                        }

                        $this->_curl->addHeader('Content-Type', 'application/json');
                        $this->_curl->setOption(CURLOPT_TIMEOUT, 0);
                        $this->_curl->setOption(CURLOPT_RETURNTRANSFER, true);

                        // Check Auth Login
                        if ($this->mhsiHelper->getMhsiAuthStatus()) {
                            $authPass = $this->mhsiHelper->getUserName() . ':' . $this->mhsiHelper->getPassword();
                            $this->_curl->setOption(CURLOPT_HTTPAUTH, CURLAUTH_ANY);
                            $this->_curl->setOption(CURLOPT_USERPWD, $authPass);
                        }

                        $this->_curl->setTimeout(0);
                        $this->_curl->get($curlUrl);
                        $response = $this->_curl->getBody();

                        $filePath = $filePath . '/' . $historyFileName;
                        $ioAdapter->write($filePath, $response, 0666);
                        $ioAdapter->close();

                        // Profile Name
                        $profileName = "None";

                        // Update Import History table status for success
                        $updateRunImport = $this->importHistory->create()->getCollection();
                        $updateRunImport->addFieldToFilter('cron_status', ['eq' => CronStatus::CRON_PROCESS]);
                        if($updateRunImport->count()){
                            $updateImportData = $updateRunImport->getFirstItem();
                            $updateImportData->setFileName($historyFileName);
                            $updateImportData->setMessage('Import history is added');
                            $updateImportData->save();
                            $this->logger->info($newImport->getProfileName() . ' profile history has been added successfully.');
                            $profileName  = $updateImportData->getProfileNameShow();
                        }

                        // Cleaning URLs
                        $this->cleanUrlsProcess->processProductUrls($profileName);

                        // Update Import History table status for success
                        $updateRunImport = $this->importHistory->create()->getCollection();
                        $updateRunImport->addFieldToFilter('cron_status', ['eq' => CronStatus::CRON_PROCESS]);
                        if($updateRunImport->count()){
                            $updateImportData = $updateRunImport->getFirstItem();
                            $updateImportData->setCronStatus(CronStatus::CRON_SUCCESS);
                            $updateImportData->setMessage('Import process is done');
                            $updateImportData->setEndDate($this->timezoneInterface->date()->format('Y-m-d H:i:s'));
                            $updateImportData->save();
                            $this->logger->info($newImport->getProfileName() . ' profile has been imported successfully.');
                            // Sync Custom Search Data after Import
                            $this->syncManufacturereCategory($updateImportData->getAffectedCategory());
                        }

                        ini_restore('max_execution_time');
                    } catch (Exception $e) {
                        $this->logger->error($newImport->getProfileName());
                        $this->logger->error($e->getMessage());
                        $newImport->setCronStatus(CronStatus::CRON_FAILED);
                        $newImport->setMessage($e->getMessage());
                        $newImport->save();
                        ini_restore('max_execution_time');
                    }
                } else {
                    $this->logger->warning('No New CronJob');
                }
            } else {
                $this->logger->warning('Skip New CronJob');
            }
        } catch (Exception $e) {
            $this->logger->warning($e->getMessage());
        }
    }

    /**
     * Sync Manufacturere Categories
     *
     * @param null|string $categoryIds
     *
     * @return void
     */
    public function syncManufacturereCategory($categoryIds)
    {
        // Ensure categoryIds is always a string before exploding
        $categoryIds = is_string($categoryIds) ? $categoryIds : '';
        $categoryIds = array_filter(explode(',', $categoryIds ?? []));
        if(!empty($categoryIds)){
            $customSearchCollection = $this->customDataSearchModel->create()->getCollection();
            $customSearchCollection->addFieldToSelect(['id','cat_id']);
            $customSearchCollection->addFieldToFilter('cat_id',['in' => $categoryIds]);
            foreach($customSearchCollection as $customSearch){
                $customSearch->setStatus(CustomDataInterface::PROCESS_STATUS);
                $customSearch->save();
            }
        }
    }

    /**
     * Execute.
     *
     * @param string $profileName
     *
     * @return string
     */
    public function getProfilerUrl($profileName)
    {
        $storeUrl = $this->_storeManager->getStore()->getBaseUrl(\Magento\Framework\UrlInterface::URL_TYPE_WEB);
        $storeUrl .= 'mhsi/index.php?config=' . urlencode($profileName);

        return $storeUrl;
    }
}
