<?php
/*
 * web-vision GmbH
 * Do not edit or add to this file if you wish to upgrade Magento to newer
 * versions in the future. If you wish to customize Magento for your
 * needs please refer to https://www.web-vision.de for more information.
 * @category    Extendware
 * @copyright   Copyright (c) 2001-2022 web-vision GmbH (https://www.web-vision.de)
 * @author      Mahesh Makwana <mmakwana@web-vision.de>
 */
declare(strict_types=1);
namespace Extendware\ImportDataUI\Model;

use Extendware\ImportDataUI\Api\Data\ImportHistoryInterface;
use Extendware\ImportDataUI\Api\Data\ImportHistoryInterfaceFactory;
use Extendware\ImportDataUI\Api\Data\ImportHistorySearchResultsInterfaceFactory;
use Extendware\ImportDataUI\Api\ImportHistoryRepositoryInterface;
use Extendware\ImportDataUI\Model\ResourceModel\ImportHistory as ResourceImportHistory;
use Extendware\ImportDataUI\Model\ResourceModel\ImportHistory\CollectionFactory as ImportHistoryCollectionFactory;
use Magento\Framework\Api\SearchCriteria\CollectionProcessorInterface;
use Magento\Framework\Exception\CouldNotDeleteException;
use Magento\Framework\Exception\CouldNotSaveException;
use Magento\Framework\Exception\NoSuchEntityException;

class ImportHistoryRepository implements ImportHistoryRepositoryInterface
{
    /**
     * @var ImportHistoryCollectionFactory
     */
    protected $importHistoryCollectionFactory;

    /**
     * @var ResourceImportHistory
     */
    protected $resource;

    /**
     * @var CollectionProcessorInterface
     */
    protected $collectionProcessor;

    /**
     * @var ImportHistoryInterfaceFactory
     */
    protected $importHistoryFactory;

    /**
     * @var ImportHistory
     */
    protected $searchResultsFactory;

    /**
     * @param ResourceImportHistory $resource
     * @param ImportHistoryInterfaceFactory $importHistoryFactory
     * @param ImportHistoryCollectionFactory $importHistoryCollectionFactory
     * @param ImportHistorySearchResultsInterfaceFactory $searchResultsFactory
     * @param CollectionProcessorInterface $collectionProcessor
     */
    public function __construct(
        ResourceImportHistory $resource,
        ImportHistoryInterfaceFactory $importHistoryFactory,
        ImportHistoryCollectionFactory $importHistoryCollectionFactory,
        ImportHistorySearchResultsInterfaceFactory $searchResultsFactory,
        CollectionProcessorInterface $collectionProcessor
    ) {
        $this->resource = $resource;
        $this->importHistoryFactory = $importHistoryFactory;
        $this->importHistoryCollectionFactory = $importHistoryCollectionFactory;
        $this->searchResultsFactory = $searchResultsFactory;
        $this->collectionProcessor = $collectionProcessor;
    }

    /**
     * @inheritDoc
     */
    public function save(ImportHistoryInterface $importHistory)
    {
        try {
            $this->resource->save($importHistory);
        } catch (\Exception $exception) {
            throw new CouldNotSaveException(__(
                'Could not save the importHistory: %1',
                $exception->getMessage()
            ));
        }

        return $importHistory;
    }

    /**
     * @inheritDoc
     */
    public function get($importHistoryId)
    {
        $importHistory = $this->importHistoryFactory->create();
        $this->resource->load($importHistory, $importHistoryId);
        if (!$importHistory->getId()) {
            throw new NoSuchEntityException(__('ImportHistory with id "%1" does not exist.', $importHistoryId));
        }

        return $importHistory;
    }

    /**
     * @inheritDoc
     */
    public function getList(
        \Magento\Framework\Api\SearchCriteriaInterface $criteria
    ) {
        $collection = $this->importHistoryCollectionFactory->create();

        $this->collectionProcessor->process($criteria, $collection);

        $searchResults = $this->searchResultsFactory->create();
        $searchResults->setSearchCriteria($criteria);

        $items = [];
        foreach ($collection as $model) {
            $items[] = $model;
        }

        $searchResults->setItems($items);
        $searchResults->setTotalCount($collection->getSize());

        return $searchResults;
    }

    /**
     * @inheritDoc
     */
    public function delete(ImportHistoryInterface $importHistory)
    {
        try {
            $importHistoryModel = $this->importHistoryFactory->create();
            $this->resource->load($importHistoryModel, $importHistory->getImporthistoryId());
            $this->resource->delete($importHistoryModel);
        } catch (\Exception $exception) {
            throw new CouldNotDeleteException(__(
                'Could not delete the ImportHistory: %1',
                $exception->getMessage()
            ));
        }

        return true;
    }

    /**
     * @inheritDoc
     */
    public function deleteById($importHistoryId)
    {
        return $this->delete($this->get($importHistoryId));
    }
}
