<?php
/*
 web-vision GmbH
 Do not edit or add to this file if you wish to upgrade Magento to newer
 versions in the future. If you wish to customize Magento for your
 needs please refer to https://www.web-vision.de for more information.
 @category    Extendware
 @copyright   Copyright (c) 2001-2023 web-vision GmbH (https://www.web-vision.de)
 @author      Fenil Timbadiya <fenil@web-vision.de>
*/

namespace Extendware\Quickorder\Controller\Order;

use Extendware\Quickorder\Helper\Data;
use Extendware\Quickorder\Model\Quickorder;
use Magento\CatalogInventory\Model\Stock\StockItemRepository;
use Magento\Customer\Model\Session;
use Magento\Framework\App\Action\Context;
use Magento\Framework\App\Http\Context as httpContext;
use Magento\Framework\View\Result\PageFactory;
use Magento\Framework\Controller\ResultFactory;
use Psr\Log\LoggerInterface;

class Place extends \Magento\Framework\App\Action\Action
{
    /**
     * @var PageFactory $resultPageFactory
     */
    protected $resultPageFactory;

    /**
     * @var ResultFactory $resultFactory
     */
    protected $resultFactory;

    /**
     * @var Quickorder $quickOrder
     */
    protected $quickOrder;

    /**
     * @var Session $customerSession
     */
    protected $customerSession;

    /**
     * @var Data $helper
     */
    protected $helper;

    /**
     * @var httpContext
     */
    protected $httpContext;

    /**
     * @var StockItemRepository $stockItemRepository
     */
    protected $stockItemRepository;

    /**
     * @var LoggerInterface $logger
     */
    protected $logger;

    /**
     * __construct function
     *
     * @param Context $context
     * @param PageFactory $resultPageFactory
     * @param ResultFactory $resultFactory
     * @param Session $customerSession
     * @param Quickorder $quickOrder
     * @param Data $helper
     * @param httpContext $httpContext
     * @param StockItemRepository $stockItemRepository
     * @param LoggerInterface $logger
     */
    public function __construct(
        Context $context,
        PageFactory $resultPageFactory,
        ResultFactory $resultFactory,
        Session $customerSession,
        Quickorder $quickOrder,
        Data $helper,
        httpContext $httpContext,
        StockItemRepository $stockItemRepository,
        LoggerInterface $logger
    ) {
        $this->resultPageFactory = $resultPageFactory;
        $this->resultFactory = $resultFactory;
        $this->customerSession = $customerSession;
        $this->quickOrder = $quickOrder;
        $this->helper = $helper;
        $this->httpContext = $httpContext;
        $this->stockItemRepository = $stockItemRepository;
        $this->logger = $logger;
        parent::__construct($context);
    }

    /**
     * Execute
     *
     * @return void
     */
    public function execute()
    {
        $resultRedirect = $this->resultFactory->create(ResultFactory::TYPE_REDIRECT);
        try {
            $preparedData = $this->getPreparedData();

            if ($this->helper->getOrderMethod() == 'order') {
                // Create order
                if (!$preparedData || count($preparedData) == 0) {
                    $this->helper->throwErrorMessage(__("Something went wrong. Can't not create order"));
                } else {
                    $resultmsg = $this->quickOrder->placeToOrder($preparedData);
                    $this->helper->throwSuccessMessage(__('Order # %1 was successfully created.', $resultmsg['msg']));
                }
            } elseif ($this->helper->getOrderMethod() == 'cart') {
                // Add products to cart
                if (!$preparedData || count($preparedData) == 0) {
                    $this->helper->throwErrorMessage(__("Something went wrong. Can't add products to cart"));
                } else {
                    $resultmsg = $this->quickOrder->placeToCart($preparedData);
                    $this->helper->throwSuccessMessage(__($resultmsg['msg']));
                    $resultRedirect->setPath('checkout/cart/index');
                    return $resultRedirect;
                }
            } else {
                // error message
                $this->helper->throwWarningMessage(__('Configuration is wrong. Please check a configuration.'));
            }
        } catch (\Exception $e) {
            $this->logger->debug($e->getMessage());
        }
        $this->_redirect('quickorder/index/view');
    }

    /**
     * CreateOrderData function
     *
     * @return array
     */
    private function getPreparedData()
    {
        $orderData = [];
        $customer = $this->customerSession->getCustomer();

        if (!$customer) {
            $this->helper->throwErrorMessage(__('Customer is not logged in. Please log in.'));

            return [];
        }

        if ($customer) {
            // set Shipping Address to order data
            if ($this->helper->getOrderMethod() != 'cart') {
                $shippingAddress = $customer->getDefaultShippingAddress();
                $billingAddress = $customer->getDefaultBillingAddress();
                if (!$shippingAddress && !$billingAddress) {
                    $this->helper->throwErrorMessage(
                        __('Customer has no addresses. They are required to create order. Please update addresses.')
                    );

                    return [];
                }
                // set addresses
                if ($shippingAddress) {
                    $orderData['shipping_address'] = $shippingAddress->toArray();
                    if (!$billingAddress) {
                        $orderData['billing_address'] = $shippingAddress->toArray();
                    }
                }

                if ($billingAddress) {
                    $orderData['billing_address'] = $billingAddress->toArray();
                    if (!$shippingAddress) {
                        $orderData['shipping_address'] = $billingAddress->toArray();
                    }
                }
            }

            //set email to order data
            $orderData['email'] = $customer->getEmail();
            // $orderData['email'] = $this->httpContext->getValue('customer_email');

            //set currency code to order data
            $orderData['currency'] = $this->helper->getCurrency();

            //set shipment method
            if (!$this->helper->getShipmentMethodCode()) {
                $this->helper->throwErrorMessage(
                    __("Shipment method is required to create order. Something went wrong. Can't not create order")
                );

                return [];
            }
            $orderData['shipping_method_code'] = $this->helper->getShipmentMethodCode();

            //set payment method
            if (!$this->helper->getPaymentMethodCode()) {
                $this->helper->throwErrorMessage(
                    __("Payment method is required to create order. Something went wrong. Can't not create order")
                );

                return [];
            }
            $orderData['payment_method_code'] = $this->helper->getPaymentMethodCode();
            //get product id's array to create quote items
            $items = $this->getItems();

            if (!$items || count($items) == 0) {
                $this->helper->throwErrorMessage(__('Sorry, no products found. Please try again.'));

                return [];
            }

            $orderData['items'] = $items;
            $orderData['form_key'] = $this->getRequest()->getParam('form_key');

            return $orderData;
        }
    }

    /**
     * Create product items array from request
     *
     * @param void
     *
     * @return array
     */
    protected function getItems()
    {
        $requestItems = $this->getRequest()->getParam('item');

        //create items array from request form
        $items = [];
        $failed_items = [];
        if (count($requestItems) > 0) {
            foreach ($requestItems as $item) {
                if ($item['sku'] == '') {
                    continue;
                }
                $product_id = $this->helper->validateProduct($item['sku']);
                $stock_item = $this->getStockItem($product_id);
                if ($product_id && $item['qty']) {
                    $items[] = ['product_id' => $product_id, 'qty' => (int) $item['qty']];
                } else {
                    if (!$product_id) {
                        $failed_items[] = $item['sku'];
                    }
                }
            }
        }

        // Add error message if some products are not found
        if (count($failed_items) > 0) {
            if ($this->helper->getUseSku()) {
                $errorMsg = __('There are no products with SKU\'s: %1', implode(',', $failed_items));
            } else {
                $errorMsg = __('There are no products with Item Number: %1', implode(',', $failed_items));
            }
            $this->helper->throwErrorMessage($errorMsg);
        }

        return $items;
    }

    /**
     * GetStockItem function
     *
     * @param mixed $productId
     * @return void
     */
    public function getStockItem($productId)
    {
        try {
            $stockItem = $this->stockItemRepository->get($productId)->getIsInStock();
            return true;
        } catch (\Exception $e) {
            $this->logger->debug($e->getMessage());
        }
    }
}
