<?php
/*
 web-vision GmbH
 Do not edit or add to this file if you wish to upgrade Magento to newer
 versions in the future. If you wish to customize Magento for your
 needs please refer to https://www.web-vision.de for more information.
 @category    Extendware
 @copyright   Copyright (c) 2001-2023 web-vision GmbH (https://www.web-vision.de)
 @author      Fenil Timbadiya <fenil@web-vision.de>
*/

namespace Extendware\Quickorder\Helper;

use Extendware\Core\Model\ExtendwareModule;
use Magento\Catalog\Model\ResourceModel\Product\CollectionFactory;
use Magento\Framework\App\Config\ScopeConfigInterface;
use Magento\Framework\App\Helper\Context;
use Magento\Framework\Message\ManagerInterface;
use Magento\Store\Model\StoreManagerInterface;

class Data extends \Magento\Framework\App\Helper\AbstractHelper
{
    public const MODULE_NAME = 'Extendware_Quickorder';

    public const XML_PATH_EXTENDWARE_QUICKORDER= 'extendware_quickorder/quickorder/';

    /**
     * @var StoreManagerInterface $storeManager
     */
    protected $storeManager;

    /**
     * @var CollectionFactory $collectionFactory
     */
    protected $collectionFactory;

    /**
     * @var ManagerInterface $messageManager
     */
    protected $messageManager;

    /**
     * @var ScopeConfigInterface $scopeConfig
     */
    protected $scopeConfig;

    /**
     * @var ExtendwareModule
     */
    protected $_extendwareModule;

    /**
     * @param Context $context
     * @param StoreManagerInterface $storeManager
     * @param ManagerInterface $messageManager
     * @param CollectionFactory $collectionFactory
     * @param ScopeConfigInterface $scopeConfig
     * @param ExtendwareModule $extendwareModule
     */
    public function __construct(
        Context $context,
        StoreManagerInterface $storeManager,
        ManagerInterface $messageManager,
        CollectionFactory $collectionFactory,
        ScopeConfigInterface $scopeConfig,
        ExtendwareModule $extendwareModule,
    ) {
        $this->messageManager = $messageManager;
        $this->storeManager = $storeManager;
        $this->collectionFactory = $collectionFactory;
        $this->scopeConfig = $scopeConfig;
        $this->_extendwareModule = $extendwareModule;
        parent::__construct($context);
    }

    /**
     * IsModuleEnabled function
     *
     * @return boolean
     */
    public function isModuleEnabled()
    {
        $module = $this->_extendwareModule->load(
            self::MODULE_NAME,
            \Extendware\Core\Api\Data\ExtendwareModuleInterface::MODULE_NAME
        );

        return $module->getModuleActive();
    }

    /**
     * GetConfigValue function
     *
     * @param string $value
     * @return void
     */
    public function getConfigValue($value)
    {
        $storeScope = \Magento\Store\Model\ScopeInterface::SCOPE_STORE;

        return $this->scopeConfig->getValue(self::XML_PATH_EXTENDWARE_QUICKORDER . $value, $storeScope);
    }

    /**
     * GetIsEnabled
     *
     * @return void
     */
    public function getIsEnabled()
    {
        return (bool) $this->getConfigValue('is_enabled');
    }

    /**
     * GetUseSku
     *
     * @return void
     */
    public function getUseSku()
    {
        return (bool) $this->getConfigValue('use_sku');
    }

    /**
     * GetShipmentMethodCode
     *
     * @return void
     */
    public function getShipmentMethodCode()
    {
        return $this->getConfigValue('shipment_method_code');
    }

    /**
     * GetPaymentMethodCode
     *
     * @return void
     */
    public function getPaymentMethodCode()
    {
        return $this->getConfigValue('payment_method_code');
    }

    /**
     * GetConfigAttributeCode
     *
     * @return void
     */
    public function getConfigAttributeCode()
    {
        $attribute_code = $this->getConfigValue('attribute_code');
        return $attribute_code;
    }

    /**
     * GetOrderMethod
     *
     * @return void
     */
    public function getOrderMethod()
    {
        return $this->getConfigValue('order_method');
    }

    /**
     * GetGridRowField
     *
     * @return void
     */
    public function getGridRowField()
    {
        return $this->getConfigValue('grid_row_field');
    }

    /**
     * GetCurrency
     *
     * @return void
     */
    public function getCurrency()
    {
        return $this->storeManager->getStore()->getCurrentCurrency()->getCode();
    }

    /**
     * Throw error message on frontend
     *
     * @param string $messageText
     * @return void
     */
    public function throwErrorMessage($messageText)
    {
        if ($messageText && $messageText != '') {
            $this->messageManager->addError(__($messageText));
        }
    }

    /**
     * Throw warning message on frontend
     *
     * @param string $messageText
     * @return void
     */
    public function throwWarningMessage($messageText)
    {
        if ($messageText && $messageText != '') {
            $this->messageManager->addWarning(__($messageText));
        }
    }

    /**
     * Throw success message on frontend
     *
     * @param string $messageText
     * @return void
     */
    public function throwSuccessMessage($messageText)
    {
        if ($messageText && $messageText != '') {
            $this->messageManager->addWarning(__($messageText));
        }
    }

    /**
     * Validate product by SKU or attribute_code
     *
     * Depends on configuration
     *
     * @param type $sku
     * @return mixed $entity_id or false
     */
    public function validateProduct($sku)
    {

        $productCollection = $this->collectionFactory->create();

        if ($this->getUseSku()) {
            $productCollection->addAttributeToFilter('sku', $sku);
        } elseif ($this->getConfigAttributeCode() != '') {
            $productCollection->addAttributeToFilter($this->getConfigAttributeCode(), $sku);
        }

        $productCollection->load();
        $product = $productCollection->getFirstItem();
        return $product->getId();
    }
}
