<?php
/*
 web-vision GmbH
 Do not edit or add to this file if you wish to upgrade Magento to newer
 versions in the future. If you wish to customize Magento for your
 needs please refer to https://www.web-vision.de for more information.
 @category    Extendware
 @copyright   Copyright (c) 2001-2023 web-vision GmbH (https://www.web-vision.de)
 @author      Fenil Timbadiya <fenil@web-vision.de>
*/

namespace Extendware\Quickorder\Model;

use Magento\Catalog\Api\ProductRepositoryInterface;
use Magento\Checkout\Model\Cart;
use Magento\Checkout\Model\Session;
use Magento\Customer\Api\CustomerRepositoryInterface;
use Magento\Customer\Model\CustomerFactory;
use Magento\Framework\DataObject\Factory;
use Magento\Quote\Model\Quote\Address\Rate;
use Magento\Quote\Model\Quote\ItemFactory;
use Magento\Quote\Model\QuoteFactory;
use Magento\Quote\Model\QuoteManagement;
use Magento\Store\Model\StoreManagerInterface;

class Quickorder
{

    /**
     * @var StoreManagerInterface $storeManager
     */
    protected $storeManager;

    /**
     * @var Product $product
     */
    protected $product;

    /**
     * @var FormKey $formkey
     */
    protected $formkey;

    /**
     * @var QuoteFactory $quote
     */
    protected $quote;

    /**
     * @var QuoteFactory $quote
     */
    protected $quoteManagement;

    /**
     * @var CustomerFactory $customerFactory
     */
    protected $customerFactory;

    /**
     * @var CustomerRepositoryInterface $customerRepository
     */
    protected $customerRepository;

    /**
     * @var OrderService $orderService
     */
    protected $orderService;

    /**
     * @var ItemFactory $quoteItemFactory
     */
    protected $quoteItemFactory;

    /**
     * @var Rate $shippingRate
     */
    protected $shippingRate;

    /**
     * @var Cart $cart
     */
    protected $cart;

    /**
     * @var Session $checkoutSession
     */
    protected $checkoutSession;

    /**
     * @var Processor $quoteItemProcessor
     */
    protected $quoteItemProcessor;

    /**
     * @var ProductRepositoryInterface $productRepository
     */
    protected $productRepository;

    /**
     * @var Factory $objectFactory
     */
    protected $objectFactory;

    /**
     * @param StoreManagerInterface $storeManager
     * @param ProductRepositoryInterface $productRepository
     * @param QuoteFactory $quote
     * @param QuoteManagement $quoteManagement
     * @param ItemFactory $quoteItemFactory
     * @param CustomerFactory $customerFactory
     * @param CustomerRepositoryInterface $customerRepository
     * @param Rate $shippingRate
     * @param Cart $cart
     * @param Session $checkoutSession
     * @param Factory $objectFactory
     */
    public function __construct(
        StoreManagerInterface $storeManager,
        ProductRepositoryInterface $productRepository,
        QuoteFactory $quote,
        QuoteManagement $quoteManagement,
        ItemFactory $quoteItemFactory,
        CustomerFactory $customerFactory,
        CustomerRepositoryInterface $customerRepository,
        Rate $shippingRate,
        Cart $cart,
        Session $checkoutSession,
        Factory $objectFactory
    ) {
        $this->storeManager = $storeManager;
        $this->productRepository = $productRepository;
        $this->quote = $quote;
        $this->quoteManagement = $quoteManagement;
        $this->quoteItemFactory = $quoteItemFactory;
        $this->customerFactory = $customerFactory;
        $this->customerRepository = $customerRepository;
        $this->shippingRate = $shippingRate;
        $this->cart = $cart;
        $this->checkoutSession = $checkoutSession;
        $this->objectFactory = $objectFactory;
    }

    /**
     * PlaceToOrder function
     *
     * @param array $orderData
     * @return void
     */
    public function placeToOrder($orderData)
    {

        $store = $this->storeManager->getStore();
        $websiteId = $this->storeManager->getStore()->getWebsiteId();
        $customer = $this->customerFactory->create();
        $customer->setWebsiteId($websiteId);

        // Load customer by email address
        $customer->loadByEmail($orderData['email']);

        if (!$customer->getEntityId()) {

            // If not avilable then create this customer
            $customer->setWebsiteId($websiteId)
                ->setStore($store)
                ->setFirstname($orderData['shipping_address']['firstname'])
                ->setLastname($orderData['shipping_address']['lastname'])
                ->setEmail($orderData['email'])
                ->setPassword($orderData['email']);
            $customer->save();
        }

        // Create quote object
        $quote = $this->quote->create();

        // Set store for quote
        $quote->setStore($store);

        // Load customer data
        $customer = $this->customerRepository->getById($customer->getEntityId());
        $quote->setCurrency();

        // Assign Customer to quote
        $quote->assignCustomer($customer);

        $quote->save();

        // Add items in quote
        foreach ($orderData['items'] as $item) {
            $_product = $this->productRepository->getById($item['product_id']);

            $quoteItem = $quote->getItemByProduct($_product);
            if (!$quoteItem) {
                $quoteItem = $this->quoteItemFactory->create();
                $quoteItem->setQuote($quote);
                $quoteItem->setStoreId($this->storeManager->getStore()->getId());

                $quoteItem->setOptions($_product->getCustomOptions())->setProduct($_product);
                $quoteItem->setQty($item['qty']);
                $quoteItem->setPrice($_product->getPrice());

                // Add only item that is not in quote already
                $quote->addItem($quoteItem);
            } else {
                // Add product to existing item or create new
                $quote->addProduct($_product, $item['qty']);
            }
        }

        try {
            $quote->save();
        } catch (Exception $ex) {
            $result = ['error' => 1,  'msg' => "Something went wrong. Can't not create order"];
        }

        // Set Address to quote
        $quote->getBillingAddress()->addData($orderData['billing_address']);
        $quote->getShippingAddress()->addData($orderData['shipping_address']);

        // Collect Rates and Set Shipping & Payment Method
        $shippingAddress = $quote->getShippingAddress();

        $shippingAddress->setCollectShippingRates(true)
            ->collectShippingRates()
            ->setShippingMethod($orderData['shipping_method_code']);

        // Set shipping method
        $_shippingRate = $shippingAddress->getShippingRateByCode($orderData['shipping_method_code']);
        if ($_shippingRate) {
            $this->shippingRate = $_shippingRate;
        } else {
            $this->shippingRate
                ->setCode('flatrate_flatrate')
                ->setPrice(1);
        }

        // Add payment method,
        $quote->setPaymentMethod($orderData['payment_method_code']);
        $quote->setInventoryProcessed();
        $quote->save();

        // Set Sales Order Payment
        $quote->getPayment()->importData(['method' => $orderData['payment_method_code']]);

        // Collect Totals & Save Quote
        $quote->collectTotals()->save();

        // Create Order From Quote
        $order = $this->quoteManagement->submit($quote);

        // Do not sent email to customer
        $order->setEmailSent(0);
        $increment_id = $order->getRealOrderId();
        if ($order->getEntityId()) {
            $result['order_id'] = $order->getRealOrderId();
            $result = ['msg' => 'All products were successfully added.'];
        } else {
            $result = ['error' => 1, 'msg' => 'Could not to dispatch an order.'];
        }

        // Return RealOrderId OR ErrorMessage
        return $result;
    }

    /**
     * Adds products to Cart
     *
     * @param array $orderData
     * @return string
     */
    public function placeToCart($orderData)
    {
        $store = $this->storeManager->getStore();
        $websiteId = $this->storeManager->getStore()->getWebsiteId();

        if ($orderData['email']) {
            // Load customer by email address
            $customer = $this->customerFactory->create();
            $customer->setWebsiteId($websiteId);
            $customer->loadByEmail($orderData['email']);

            // Loading customer data
            $customer = $this->customerRepository->getById($customer->getEntityId());
        }

        // Add items to cart/quote
        $quote = $this->cart->getQuote();

        foreach ($orderData['items'] as $item) {
            $_product = $this->productRepository->getById($item['product_id']);

            $params = $this->objectFactory->create(['qty' => $item['qty']]);

            $quoteItem = $quote->getItemByProduct($_product);
            if (!$quoteItem) {
                $quoteItem = $this->quoteItemFactory->create();
                $quoteItem->setQuote($quote);
                $quoteItem->setStoreId($this->storeManager->getStore()->getId());

                $quoteItem->setOptions($_product->getCustomOptions())->setProduct($_product);
                $quoteItem->setQty($item['qty']);
                $quoteItem->setPrice($_product->getPrice());

                // Add only item that is not in quote already
                $quote->addItem($quoteItem);
            } else {
                $quote->addProduct($_product, $item['qty']);
            }
        }

        try {
            // Save changes to quote/cart
            $quote->save();
            $this->cart->save();
            $this->checkoutSession->setCartWasUpdated(true);
        } catch (Exception $ex) {
            $resultmsg = ['error' => 1,  'msg' => 'Error! Could not update cart! Please try again.'];
        }

        $resultmsg = ['msg' => 'All products were successfully added.'];
        return $resultmsg;
    }
}
