<?php
declare(strict_types=1);
namespace Mzat\AutoRelatedProducts\Controller\Ajax;

use Magento\Catalog\Api\ProductRepositoryInterface;
use Magento\Checkout\Controller\Cart as MagentoCart;
use Magento\Checkout\Model\Cart as CustomerCart;
use Magento\Checkout\Model\Cart\RequestQuantityProcessor;
use Magento\Checkout\Model\Session;
use Magento\Framework\App\Action\HttpPostActionInterface;
use Magento\Framework\App\Config\ScopeConfigInterface;
use Magento\Framework\Controller\Result\JsonFactory;
use Magento\Framework\Data\Form\FormKey\Validator as FormKeyValidator;
use Magento\Framework\Exception\LocalizedException;
use Magento\Framework\Filter\LocalizedToNormalized;
use Magento\Framework\Locale\ResolverInterface;
use Magento\Store\Model\StoreManagerInterface;
use Psr\Log\LoggerInterface;

class Cart extends MagentoCart implements HttpPostActionInterface
{
    private ProductRepositoryInterface $productRepository;
    private RequestQuantityProcessor $quantityProcessor;
    private ResolverInterface $localeResolver;
    private JsonFactory $jsonFactory;
    private LoggerInterface $logger;

    public function __construct(
        \Magento\Framework\App\Action\Context $context,
        ScopeConfigInterface $scopeConfig,
        Session $checkoutSession,
        StoreManagerInterface $storeManager,
        FormKeyValidator $formKeyValidator,
        CustomerCart $cart,
        ProductRepositoryInterface $productRepository,
        RequestQuantityProcessor $quantityProcessor,
        ResolverInterface $localeResolver,
        JsonFactory $jsonFactory,
        LoggerInterface $logger
    ) {
        parent::__construct(
            $context,
            $scopeConfig,
            $checkoutSession,
            $storeManager,
            $formKeyValidator,
            $cart
        );
        $this->productRepository = $productRepository;
        $this->quantityProcessor = $quantityProcessor;
        $this->localeResolver = $localeResolver;
        $this->jsonFactory = $jsonFactory;
        $this->logger = $logger;
    }

    /**
     * View page action
     *
     * @return \Magento\Framework\Controller\ResultInterface
     */
    public function execute()
    {
        // Set noindex, nofollow for AJAX endpoints
        $this->getResponse()->setHeader('X-Robots-Tag', 'noindex, nofollow');

        $result = $this->jsonFactory->create();

        if (!$this->_formKeyValidator->validate($this->getRequest())) {
            return $result->setData([
                'success' => false,
                'message' => __('Your session has expired. Please refresh and try again.'),
            ]);
        }

        $params = $this->getRequest()->getParams();

        try {
            if (isset($params['qty'])) {
                $params['qty'] = $this->prepareQuantity((string)$params['qty']);
            }

            $product = $this->initProduct((int)($params['product'] ?? 0));
            if (!$product || !$product->isSalable()) {
                throw new LocalizedException(__('This product is not available.'));
            }

            $this->cart->addProduct($product, $params);

            if (!empty($params['related_product'])) {
                $this->cart->addProductsByIds(explode(',', $params['related_product']));
            }

            $this->cart->save();

            $this->_eventManager->dispatch(
                'checkout_cart_add_product_complete',
                [
                    'product' => $product,
                    'request' => $this->getRequest(),
                    'response' => $this->getResponse(),
                ]
            );

            return $result->setData([
                'success' => true,
                'message' => __('You added %1 to your shopping cart.', $product->getName()),
                'product' => [
                    'id' => $product->getId(),
                    'name' => $product->getName(),
                    'sku' => $product->getSku(),
                ],
                'cart_url' => $this->_url->getUrl('checkout/cart', ['_secure' => true]),
            ]);
        } catch (LocalizedException $e) {
            return $result->setData([
                'success' => false,
                'message' => $e->getMessage(),
            ]);
        } catch (\Exception $e) {
            $this->logger->critical($e);

            return $result->setData([
                'success' => false,
                'message' => __('We can\'t add this item to your shopping cart right now.'),
            ]);
        }
    }

    /**
     * Prepare Quantity function
     *
     * @param string $qty
     *
     * @return float
     */
    private function prepareQuantity(string $qty): float
    {
        $locale = $this->localeResolver->getLocale();
        $filter = new LocalizedToNormalized(['locale' => $locale]);

        return (float) $this->quantityProcessor->prepareQuantity($filter->filter($qty));
    }

    /**
     * Init Product function
     *
     * @param int $productId
     *
     * @return void
     */
    private function initProduct(int $productId)
    {
        if (!$productId) {
            return false;
        }

        try {
            $storeId = $this->_storeManager->getStore()->getId();

            return $this->productRepository->getById($productId, false, $storeId);
        } catch (LocalizedException $e) {
            return false;
        }
    }
}
