<?php
namespace Mzat\AutoRelatedProducts\Controller\Ajax;

use Magento\Catalog\Api\ProductRepositoryInterface;
use Magento\Catalog\Helper\Product\Compare as CompareHelper;
use Magento\Catalog\Model\Product\Compare\ListCompare;
use Magento\Catalog\ViewModel\Product\Checker\AddToCompareAvailability;
use Magento\Framework\App\Action\Action;
use Magento\Framework\App\Action\Context;
use Magento\Framework\Controller\Result\JsonFactory;
use Magento\Framework\Data\Form\FormKey\Validator;
use Magento\Framework\Escaper;
use Magento\Framework\Event\ManagerInterface;
use Magento\Framework\Exception\NoSuchEntityException;

class Compare extends Action
{
    protected $jsonFactory;
    protected $formKeyValidator;
    protected $compareList;
    protected $productRepository;
    protected $eventManager;
    protected $compareHelper;
    protected $compareAvailability;
    protected $escaper;

    public function __construct(
        Context $context,
        JsonFactory $jsonFactory,
        Validator $formKeyValidator,
        ListCompare $compareList,
        ProductRepositoryInterface $productRepository,
        ManagerInterface $eventManager,
        CompareHelper $compareHelper,
        AddToCompareAvailability $compareAvailability,
        Escaper $escaper
    ) {
        parent::__construct($context);
        $this->jsonFactory = $jsonFactory;
        $this->formKeyValidator = $formKeyValidator;
        $this->compareList = $compareList;
        $this->productRepository = $productRepository;
        $this->eventManager = $eventManager;
        $this->compareHelper = $compareHelper;
        $this->compareAvailability = $compareAvailability;
        $this->escaper = $escaper;
    }

    public function execute()
    {
        // Set noindex, nofollow for AJAX endpoints
        $this->getResponse()->setHeader('X-Robots-Tag', 'noindex, nofollow');

        $result = $this->jsonFactory->create();

        if (!$this->formKeyValidator->validate($this->getRequest())) {
            return $result->setData([
                'success' => false,
                'message' => __('Invalid form key. Please refresh the page.'),
            ]);
        }

        $productId = (int)$this->getRequest()->getParam('product');
        if (!$productId) {
            return $result->setData([
                'success' => false,
                'message' => __('Product ID is missing.'),
            ]);
        }

        try {
            $product = $this->productRepository->getById($productId);
        } catch (NoSuchEntityException $e) {
            return $result->setData([
                'success' => false,
                'message' => __('Product not found.'),
            ]);
        }

        if (!$this->compareAvailability->isAvailableForCompare($product)) {
            return $result->setData([
                'success' => false,
                'message' => __('This product cannot be added to the compare list.'),
            ]);
        }

        $this->compareList->addProduct($product);

        // Dispatch Magento's event
        $this->eventManager->dispatch('catalog_product_compare_add_product', ['product' => $product]);

        // Recalculate compare hash
        $this->compareHelper->calculate();

        return $result->setData([
            'success' => true,
            'message' => __('You added %1 to your compare list.', $this->escaper->escapeHtml($product->getName())),
        ]);
    }
}
