<?php
/*
 *
 *  * web-vision GmbH
 *  * Do not edit or add to this file if you wish to upgrade Magento to newer
 *  * versions in the future. If you wish to customize Magento for your
 *  * needs please refer to https://www.web-vision.de for more information.
 *  * @copyright   Copyright (c) 2001-2023 web-vision GmbH (https://www.web-vision.de)
 *  * @author      Fenil Timbadiya
 *
 */
namespace Mzat\AutoRelatedProducts\Controller\Ajax;

use Extendware\ManufacturerLogo\Helper\Data;
use Magento\Catalog\Helper\Image as ImageHelper;
use Magento\Catalog\Model\CategoryRepository;
use Magento\Catalog\Model\ResourceModel\Product\Attribute\CollectionFactory as AttributeCollectionFactory;
use Magento\Catalog\Model\ResourceModel\Product\CollectionFactory;
use Magento\Framework\App\Action\Action;
use Magento\Framework\App\Action\Context;
use Magento\Framework\App\CsrfAwareActionInterface;
use Magento\Framework\App\Request\InvalidRequestException;
use Magento\Framework\App\RequestInterface;
use Magento\Framework\Controller\Result\JsonFactory;
use Magento\Framework\Pricing\PriceCurrencyInterface;
use Magento\Framework\Pricing\Render;
use Magento\Framework\View\LayoutInterface;
use Magento\Store\Model\StoreManagerInterface;
use Psr\Log\LoggerInterface;

class Products extends Action implements CsrfAwareActionInterface
{
    /**
     * @var JsonFactory
     */
    protected $resultJsonFactory;
    /**
     * @var CollectionFactory
     */
    protected $productCollectionFactory;
    /**
     * @var ImageHelper
     */
    protected $imageHelper;
    /**
     * @var StoreManagerInterface
     */
    protected $storeManager;
    /**
     * @var RequestInterface
     */
    protected $request;
    /**
     * @var PriceCurrencyInterface
     */
    protected $priceCurrency;
    /**
     * @var AttributeCollectionFactory
     */
    protected $attributeCollectionFactory;

    /**
     * @var Data
     */
    protected $exAttLogoHelper;

    /**
     * @var CategoryRepository
     */
    protected $categoryRepository;

    /**
     * @var LoggerInterface
     */
    protected $logger;

    /**
     * @var \Magento\Framework\Pricing\Render
     */
    private $_priceRenderer = null;

    /**
     * @var LayoutInterface
     */
    protected LayoutInterface $layout;

    /**
     * @param Context $context
     * @param JsonFactory $resultJsonFactory
     * @param CollectionFactory $productCollectionFactory
     * @param ImageHelper $imageHelper
     * @param StoreManagerInterface $storeManager
     * @param RequestInterface $request
     * @param PriceCurrencyInterface $priceCurrency
     * @param AttributeCollectionFactory $attributeCollectionFactory
     * @param Data $exAttLogoHelper
     * @param CategoryRepository $categoryRepository
     * @param LoggerInterface $logger
     * @param LayoutInterface $layout;
     */
    public function __construct(
        Context $context,
        JsonFactory $resultJsonFactory,
        CollectionFactory $productCollectionFactory,
        ImageHelper $imageHelper,
        StoreManagerInterface $storeManager,
        RequestInterface $request,
        PriceCurrencyInterface $priceCurrency,
        AttributeCollectionFactory $attributeCollectionFactory,
        Data $exAttLogoHelper,
        CategoryRepository $categoryRepository,
        LoggerInterface $logger,
        LayoutInterface $layout,
    ) {
        $this->resultJsonFactory = $resultJsonFactory;
        $this->productCollectionFactory = $productCollectionFactory;
        $this->imageHelper = $imageHelper;
        $this->storeManager = $storeManager;
        $this->request = $request;
        $this->priceCurrency = $priceCurrency;
        $this->attributeCollectionFactory = $attributeCollectionFactory;
        $this->categoryRepository = $categoryRepository;
        $this->exAttLogoHelper = $exAttLogoHelper;
        $this->logger = $logger;
        $this->layout = $layout;
        parent::__construct($context);
    }

    // === CSRF bypass ===

    /**
     * @param RequestInterface $request
     *
     * @return InvalidRequestException|null
     */
    public function createCsrfValidationException(RequestInterface $request): ?InvalidRequestException
    {
        return null;
    }

    /**
     * @param RequestInterface $request
     *
     * @return bool|null
     */
    public function validateForCsrf(RequestInterface $request): ?bool
    {
        return true;
    }

    public function execute()
    {
        // Set noindex, nofollow for AJAX endpoints
        $this->getResponse()->setHeader('X-Robots-Tag', 'noindex, nofollow');

        $result = $this->resultJsonFactory->create();
        $params = json_decode($this->request->getContent(), true);
        $productIds = $params['product_ids'] ?? [];
        $page = isset($params['page']) ? (int)$params['page'] : 1;
        $pageSize = isset($params['per_page']) ? (int)$params['per_page'] : 4;
        $sortBy = isset($params['sort_by']) ? $params['sort_by'] : 'name';
        $sortDirection = isset($params['sort_direction']) ? strtoupper($params['sort_direction']) : 'ASC';

        if (empty($productIds)) {
            return $result->setData([
                'products' => [],
                'total_pages' => 0,
            ]);
        }

        $collection = $this->productCollectionFactory->create();
        $collection->addAttributeToSelect(['name', 'sku', 'image', 'price', 'created_at', 'price_on_request', Data::MANUFACTURER_LOGO_ATTRIBUTE])
            ->addAttributeToFilter('entity_id', ['in' => $productIds])
            ->addAttributeToFilter('visibility', ['neq' => \Magento\Catalog\Model\Product\Visibility::VISIBILITY_NOT_VISIBLE])
            ->addAttributeToFilter('status', \Magento\Catalog\Model\Product\Attribute\Source\Status::STATUS_ENABLED);

        $sortableAttributes = $this->attributeCollectionFactory->create()
            ->addFieldToFilter('used_for_sort_by', 1)
            ->addFieldToFilter('is_visible', 1);

        $validSortAttributes = [];
        foreach ($sortableAttributes as $attribute) {
            $validSortAttributes[] = $attribute->getAttributeCode();
        }

        if (in_array($sortBy, $validSortAttributes)) {
            $collection->setOrder($sortBy, $sortDirection);
        } else {
            $collection->setOrder('name', 'ASC');
        }

        $collection->setPageSize($pageSize)
            ->setCurPage($page);

        $products = [];
        foreach ($collection as $product) {
            $products[] = [
                'id' => $product->getId(),
                'name' => $product->getName(),
                'sku' => $product->getSku(),
                'image_url' => $this->imageHelper->init($product, 'product_base_image')->getUrl(),
                'url' => $product->getProductUrl(),
                'created_at' => $product->getCreatedAt(),
                'logo_data' => $this->getLogoData($product),
                'price_on_request' => $product->getData('price_on_request'),
                'price_html' => $this->renderProductPrice($product, \Magento\Catalog\Pricing\Price\FinalPrice::PRICE_CODE),
            ];
        }

        $totalItems = count($productIds);
        $totalPages = ceil($totalItems / $pageSize);
        $from = $totalPages > 1 ? ($page - 1) * $pageSize + 1 : 1;
        $to = min($page * $pageSize, count($productIds));

        return $result->setData([
            'products' => $products,
            'total_pages' => $totalPages,
            'total_items' => $totalItems,
            'from' => $from,
            'to' => $to,
        ]);
    }

    /**
     * Get logo data
     *
     * @param mixed $productdata
     *
     * @return array
     */
    public function getLogoData($productdata)
    {
        $showLogo = [];

        try {

            if ($categoryId = (int) $productdata->getData(Data::MANUFACTURER_LOGO_ATTRIBUTE)) {
                $categoryData = $this->categoryRepository->get($categoryId, $this->exAttLogoHelper->getStoreId());
                $showLogo['logo_url'] = $this->exAttLogoHelper->getLogoMediaUrl($categoryData->getData(Data::MANUFACTURER_LOGO_ATTRIBUTE));
                $showLogo['cat_link'] = $categoryData->getUrl();
                $showLogo['cat_name'] = $categoryData->getName();
                $showLogo['logo_height'] = $this->exAttLogoHelper->getLogoHeight();
                $showLogo['logo_width'] = $this->exAttLogoHelper->getLogoWidth();
                $showLogo['status'] = $this->exAttLogoHelper->getModuleStatus();
            }
        } catch (\Exception $e) {
            $this->logger->debug($e->getMessage());
        }

        return $showLogo;
    }

    /**
     * Renders product price.
     *
     * @param \Magento\Catalog\Model\Product $product   The product
     * @param string                         $priceCode The Price Code to render
     *
     * @return string
     */
    private function renderProductPrice(\Magento\Catalog\Model\Product $product, $priceCode)
    {
        $priceRender = $this->getPriceRenderer();

        $price = $product->getData($priceCode);

        if ($priceRender) {
            $price = $priceRender->render(
                $priceCode,
                $product,
                [
                    'include_container' => false,
                    'display_minimal_price' => true,
                    'zone' => Render::ZONE_ITEM_LIST,
                    'list_category_page' => true,
                ]
            );
        }

        return $price;
    }

    /**
     * Retrieve Price Renderer Block
     *
     * @return bool|\Magento\Framework\View\Element\BlockInterface
     */
    private function getPriceRenderer()
    {
        if ($this->_priceRenderer === null) {
            $this->layout->getUpdate()->addHandle('default');
            $priceRenderer = $this->layout->getBlock('product.price.render.default');

            if (!$priceRenderer) {
                $priceRenderer = $this->layout->createBlock(
                    'Magento\Framework\Pricing\Render',
                    'product.price.render.default',
                    ['data' => ['price_render_handle' => 'catalog_product_prices']]
                );
            }

            $this->_priceRenderer = $priceRenderer;
        }

        return $this->_priceRenderer;
    }

}
