<?php
/*
 * web-vision GmbH
 * Do not edit or add to this file if you wish to upgrade Magento to newer
 * versions in the future. If you wish to customize Magento for your
 * needs please refer to https://www.web-vision.de for more information.
 * @category    Mzat
 * @copyright   Copyright (c) 2001-2022 web-vision GmbH (https://www.web-vision.de)
 * @author      Mahesh Makwana <mmakwana@web-vision.de>
 */
namespace Mzat\Catalog\Console;

use Magento\Eav\Model\Config;
use Magento\Framework\App\ResourceConnection;
use Symfony\Component\Console\Command\Command;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Input\InputOption;
use Symfony\Component\Console\Output\OutputInterface;

class MigrationProductAttributeValues extends Command
{
    public const OLD_ATTR = 'old-attr';

    public const NEW_ATTR = 'new-attr';

    /**
     * @var Config
     */
    protected $eavConfig;

    /**
     * @var ResourceConnection
     */
    protected $resourceConnection;

    /**
     * Constructor
     *
     * @param ResourceConnection $resourceConnection
     * @param Config $eavConfig
     */
    public function __construct(
        ResourceConnection $resourceConnection,
        Config $eavConfig
    ) {
        $this->resourceConnection = $resourceConnection;
        $this->eavConfig = $eavConfig;
        parent::__construct();
    }

    /**
     * {@inheritdoc}
     */
    protected function configure()
    {
        $this->setName('migration:product:attribute:values');
        $this->setDescription('Migrate product attribute values from one attribute to other attribute.');

        $this->addOption(
            self::OLD_ATTR,
            null,
            InputOption::VALUE_REQUIRED,
            'Old Attribute code'
        );

        $this->addOption(
            self::NEW_ATTR,
            null,
            InputOption::VALUE_REQUIRED,
            'NEW Attribute code'
        );

        parent::configure();
    }

    /**
     * {@inheritdoc}
     */
    protected function execute(InputInterface $input, OutputInterface $output)
    {
        try {
            $oldAttr = $input->getOption(self::OLD_ATTR);
            $newAttr = $input->getOption(self::NEW_ATTR);

            // Checking blank values
            if (empty($oldAttr) || empty($newAttr)) {
                $output->writeln('<error>Both --old-attr and --new-attr are required.</error>');

                return \Magento\Framework\Console\Cli::RETURN_FAILURE;
            }

            $output->writeln("<info>Migrating from '$oldAttr' to '$newAttr' product attribute...</info>");

            // Retrieve attribute IDs dynamically using the provided attribute codes
            $attributeOld = $this->eavConfig->getAttribute('catalog_product', $oldAttr);
            $attributeNew = $this->eavConfig->getAttribute('catalog_product', $newAttr);

            // Check if old attributes exist
            if (!$attributeOld->getId()) {
                $output->writeln("<error>Attribute with code '$oldAttr' does not exist.</error>");

                return \Magento\Framework\Console\Cli::RETURN_FAILURE;
            }

            // Check if new attributes exist
            if (!$attributeNew->getId()) {
                $output->writeln("<error>Attribute with code '$newAttr' does not exist..</error>");

                return \Magento\Framework\Console\Cli::RETURN_FAILURE;
            }

            $attributeOldId = (int) $attributeOld->getId();
            $attributeNewId = (int) $attributeNew->getId();

            $oldBackendTable = 'catalog_product_entity_' . $attributeOld->getBackendType();
            $newBackendTable = 'catalog_product_entity_' . $attributeNew->getBackendType();

            // Set batch size for processing
            $batchSize = 200;
            $offset = 0;
            $recordCount = 0;

            // Get DB connection
            $connection = $this->resourceConnection->getConnection();

            // Estimate total rows
            $totalCountSql = $connection->fetchOne(
                "SELECT COUNT(*) FROM {$oldBackendTable} WHERE attribute_id = :attribute_id",
                ['attribute_id' => $attributeOldId]
            );
            $totalCount = (int)$totalCountSql;

            for ($offset = 0; $offset < $totalCount; $offset += $batchSize) {
                $selectSql = sprintf(
                    'SELECT * FROM %s WHERE attribute_id = %d LIMIT %d, %d',
                    $oldBackendTable,
                    $attributeOldId,
                    $offset,
                    $batchSize
                );

                $products = $connection->fetchAll($selectSql);
                if (!empty($products)) {
                    try {
                        foreach ($products as $product) {
                            $product['value'] = $attributeNew->getBackendType() == 'int' ? (int) $product['value'] : (string) $product['value'];

                            // Check Attribute values is exist or not
                            $select = $connection->select()
                                            ->from($newBackendTable) // New attribute table
                                            ->where('entity_id = ?', $product['entity_id'])
                                            ->where('attribute_id = ?', $attributeNewId)
                                            ->where('store_id = ?', $product['store_id']);

                            $existingRecord = $connection->fetchRow($select);
                            if (empty($existingRecord)) {
                                // If doesn't exist, insert new record
                                $insertData = [
                                    'attribute_id' => $attributeNewId,
                                    'store_id' => $product['store_id'],
                                    'entity_id' => $product['entity_id'],
                                    'value' => $product['value'],
                                ];
                                $connection->insert($newBackendTable, $insertData);
                                $output->writeln("<info>Inserted new value : {$product['value']} for product ID {$product['entity_id']} and store ID {$product['store_id']}</info>");
                                $recordCount++;
                            } else {
                                // If exists, update the value
                                $where = [
                                    'entity_id = ?' => $product['entity_id'],
                                    'attribute_id = ?' => $attributeNewId,
                                    'store_id = ?' => $product['store_id'],
                                    'value_id' => $existingRecord['value_id'],
                                ];
                                $connection->update($newBackendTable, ['value' => $product['value']], $where);
                                $output->writeln("<info>Updated new value : {$product['value']} for product ID {$product['entity_id']} and store ID {$product['store_id']}</info>");
                                $recordCount++;
                            }
                        }
                        $output->writeln('*************************************************************************************************');
                    } catch (\Exception $e) {
                        $output->writeln('<error>An error encountered: {$e->getMessage()}</error>');
                    }
                }
            }
            $output->writeln("<info>Migration from '{$oldAttr}' to '{$newAttr}' attribute is finished, Total Migrated Record count : {$recordCount}</info>");
        } catch (\Exception $e) {
            $output->writeln('<error>An error encountered: {$e->getMessage()}</error>');

            return \Magento\Framework\Console\Cli::RETURN_FAILURE;
        }

        return \Magento\Framework\Console\Cli::RETURN_SUCCESS;
    }
}
