<?php
namespace Mzat\Catalog\ViewModel;

use Magento\Catalog\Api\Data\ProductInterface;
use Magento\Framework\View\Element\Block\ArgumentInterface;

class ProductAttributes implements ArgumentInterface
{
    /**
     * Get visible product attributes sorted by position
     *
     * @param ProductInterface $product
     *
     * @return array
     */
    public function getSortedVisibleAttributes(ProductInterface $product): array
    {
        $attributes = $product->getAttributes();
        $primaryAttributes = [];
        $secondaryAttributes = [];

        foreach ($attributes as $attributeCode => $attribute) {
            if ($attribute->getIsVisibleOnFront()) {
                $value = $attribute->getFrontend()->getValue($product);

                if (!empty($value) && $value !== __('N/A')) {

                    if ($attribute->getIsAttributePriority() === 'secondary') {
                        $secondaryAttributes[] = $this->buildAttributeData($attributeCode, $attribute, $value);
                    } else {
                        $primaryAttributes[] = $this->buildAttributeData($attributeCode, $attribute, $value);
                    }
                }
            }
        }

        return [
            'primary' => $this->sortedByPosition($primaryAttributes),
            'secondary' => $this->sortedByPosition($secondaryAttributes),
        ];
    }

    /**
     * Get visible product attributes for PDF, chunked into 'first' and 'rest'
     *
     * @param ProductInterface $product
     *
     * @return array
     */
    public function getSortedVisibleAttributesForPDF(ProductInterface $product): array
    {
        $avoidedAttributes = [
            'manufacturer_logo',
            'description',
            'long_description',
            'description_2',
            'description_3',
            'description_4',
        ];

        $attributes = $this->getSortedVisibleAttributes($product);

        // Filter out avoided attributes
        $primaryAttributes = array_filter(
            $attributes['primary'],
            fn ($attr) => !in_array($attr['code'], $avoidedAttributes, true)
        );

        $secondaryAttributes = array_filter(
            $attributes['secondary'],
            fn ($attr) => !in_array($attr['code'], $avoidedAttributes, true)
        );

        // Chunk into first 10 and rest
        $primaryChunks = [
            'first' => array_slice($primaryAttributes, 0, 12),
            'rest' => array_slice($primaryAttributes, 12),
        ];

        $secondaryChunks = [
            'first' => array_slice($secondaryAttributes, 0, 12),
            'rest' => array_slice($secondaryAttributes, 12),
        ];

        return [
            'primary' => $primaryChunks,
            'secondary' => $secondaryChunks,
        ];
    }

    /**
     * Build Attribute Data function
     *
     * @param string $attributeCode
     * @param $attribute
     * @param $value
     *
     * @return array
     */
    public function buildAttributeData(string $attributeCode, $attribute, $value): array
    {
        return [
            'code' => $attributeCode,
            'label' => $attribute->getStoreLabel(),
            'value' => $value,
            'position' => $attribute->getPosition(),
        ];
    }


    /**
     * Sorted By Position function
     *
     * @param $attribute
     *
     * @return array
     */
    public function sortedByPosition($attribute): array
    {
        usort($attribute, static function ($a, $b) {
            return $a['position'] <=> $b['position'];
        });

        return $attribute;
    }
}
