<?php
namespace Mzat\CompareProductPrint\Controller\Compare;

use Magento\Backend\App\Action\Context;
use Magento\Catalog\Model\Config as CatalogConfig;
use Magento\Catalog\Model\Product\Compare\ListCompare;
use Magento\Catalog\Model\Product\Visibility;
use Magento\Catalog\Model\ResourceModel\Product\Compare\Item\CollectionFactory;
use Magento\Customer\Model\Session as CustomerSession;
use Magento\Customer\Model\Visitor as CustomerVisitor;
use Magento\Framework\App\Filesystem\DirectoryList;
use Magento\Framework\App\Http\Context as HttpContext;
use Magento\Framework\Controller\Result\JsonFactory;
use Magento\Framework\Filesystem;
use Magento\Framework\Stdlib\DateTime\TimezoneInterface;
use Magento\Framework\View\Result\PageFactory;
use Magento\Store\Model\StoreManagerInterface;
use Zend_Loader;

class PrintComparePdf extends \Magento\Framework\App\Action\Action
{
    protected ?\Magento\Catalog\Model\ResourceModel\Product\Compare\Item\Collection $_items = null;
    protected ?array $_attributes = null;
    protected ?int $_customerId = null;

    public function __construct(
        Context $context,
        protected PageFactory $resultPageFactory,
        protected DirectoryList $dir,
        protected JsonFactory $resultJsonFactory,
        protected StoreManagerInterface $storeManager,
        protected TimezoneInterface $timezone,
        private CustomerSession $currentCustomer,
        private CustomerVisitor $customerVisitor,
        private ListCompare $compareProduct,
        private CollectionFactory $itemCollectionFactory,
        private CatalogConfig $catalogConfig,
        private Visibility $catalogProductVisibility,
        private HttpContext $httpContext,
        private Filesystem $filesystem,
    ) {
        parent::__construct($context);
        $this->initializePDFLibrary();
    }

    public function execute()
    {
        $resultJson = $this->resultJsonFactory->create();

        try {
            $mediaDirectory = $this->filesystem->getDirectoryWrite(
                \Magento\Framework\App\Filesystem\DirectoryList::MEDIA
            );

            $relativePath = 'printpdf/compare';
            if (!$mediaDirectory->isExist($relativePath)) {
                $mediaDirectory->create($relativePath);
            }

            $pdfPath = $this->generatePdf();
            $mediaUrl = $this->storeManager->getStore()
                ->getBaseUrl(\Magento\Framework\UrlInterface::URL_TYPE_MEDIA);

            return $resultJson->setData([
                'msg' => $mediaUrl . 'printpdf/compare/' . basename($pdfPath),
                'success' => true,
            ]);
        } catch (\Exception $e) {
            return $this->jsonError(__('PDF generation failed'));
        }
    }

    protected function generatePdf(): string
    {
        $html = $this->getBodyHtml();

        $mpdf = $this->createMpdfInstance();
        $mpdf->SetTopMargin(50);

        $this->configurePdfHeader($mpdf);
        $this->configurePdfFooter($mpdf);

        $mpdf->SetAutoPageBreak(true, 18);
        $mpdf->WriteHTML(html_entity_decode($html));

        $pdfName = 'tmp-' . time() . '-comparelistprint.pdf';
        $pdfFilePath = $this->dir->getPath(DirectoryList::MEDIA) . '/printpdf/compare/' . $pdfName;
        $mpdf->Output($pdfFilePath, 'F');

        return $pdfFilePath;
    }

    protected function getBodyHtml(): string
    {
        // Get IDs explicitly
        $customerId = (int)$this->currentCustomer->getCustomerId();
        $visitorId = (int)$this->customerVisitor->getId();
        $onlyDifferences = (bool)$this->getRequest()->getParam('onlyDifferences', false);

        return $this->resultPageFactory->create()->getLayout()
            ->createBlock(\Magento\Catalog\Block\Product\Compare\ListCompare::class)
            ->setTemplate('Mzat_CompareProductPrint::compareprint/print.phtml')
            ->setData('items', $this->getItems($customerId, $visitorId))
            ->setData('attributes', $this->getAttributes())
            ->setData('onlyDifferences', $onlyDifferences)
            ->toHtml();
    }

    /**
     * Retrieve Product Compare items collection for a specific customer or visitor.
     *
     * @param int|null $customerId
     * @param int|null $visitorId
     *
     * @return \Magento\Catalog\Model\ResourceModel\Product\Compare\Item\Collection
     */
    protected function getItems(?int $customerId = null, ?int $visitorId = null)
    {
        if ($this->_items === null) {
            $this->compareProduct->setAllowUsedFlat(false);

            $collection = $this->itemCollectionFactory->create();
            $collection->useProductItem()->setStoreId($this->storeManager->getStore()->getId());

            // Determine which customer/visitor to use
            if ($customerId) {
                $collection->setCustomerId($customerId);
            } elseif ($visitorId) {
                $collection->setVisitorId($visitorId);
            }

            $collection->addAttributeToSelect(
                $this->catalogConfig->getProductAttributes()
            )->loadComparableAttributes()
            ->addMinimalPrice()
            ->addTaxPercents()
            ->setVisibility(
                $this->catalogProductVisibility->getVisibleInSiteIds()
            );

            $collection->setPageSize(4)->setCurPage(1);

            $this->_items = $collection;
        }

        return $this->_items;
    }

    /**
     * Retrieve Product Compare Attributes
     *
     * @return array
     */
    protected function getAttributes(): array
    {
        if ($this->_attributes === null) {
            $this->_attributes = $this->getItems()->getComparableAttributes();
        }

        return $this->_attributes;
    }

    protected function configurePdfHeader(\Mpdf\Mpdf $mpdf): void
    {
        $headerHtml = $this->getHeaderBlock();
        $mpdf->SetHTMLHeader($headerHtml);
    }

    protected function getHeaderBlock(): string
    {
        return $this->resultPageFactory->create()->getLayout()
            ->createBlock(\Magento\Framework\View\Element\Template::class)
            ->setTemplate('Mzat_CompareProductPrint::compareprint/header.phtml')
            ->setData('logo', $this->getStaticLogoPath())
            ->toHtml();
    }

    protected function getStaticLogoPath(): string
    {
        return $this->dir->getPath(DirectoryList::STATIC_VIEW)
            . '/frontend/Mz-at/default/de_DE/images/logo.png';
    }

    protected function configurePdfFooter(\Mpdf\Mpdf $mpdf, $productId = null): void
    {
        $formattedDate = $this->timezone->formatDateTime(
            new \DateTime(),
            \IntlDateFormatter::SHORT,
            \IntlDateFormatter::NONE,
            null,
            null,
            'dd.MM.yyyy'
        );

        $compareUrl = $this->storeManager->getStore()->getBaseUrl() . 'catalog/product_compare/';

        $footerHtml = $this->resultPageFactory->create()->getLayout()
            ->createBlock(\Magento\Framework\View\Element\Template::class)
            ->setTemplate('Mzat_CompareProductPrint::compareprint/footer.phtml')
            ->setData('compare_url', $compareUrl)
            ->setData('formatted_date', $formattedDate)
            ->toHtml();

        $mpdf->SetHTMLFooter($footerHtml);
    }

    protected function jsonError($message): \Magento\Framework\Controller\Result\Json
    {
        return $this->resultJsonFactory->create()->setData([
            'msg' => $message,
            'success' => false,
        ]);
    }

    protected function createMpdfInstance(): \Mpdf\Mpdf
    {
        $config = [
            'mode' => 'utf-8',
            'format' => 'A4',
            'default_font_size' => 14,
            'default_font' => 'Nunito',
            'margin_top' => 20,
            'margin_bottom' => 10,
            'allow_charset_conversion' => true,
            'orientation' => 'P',
            'showBarcodeNumbers' => false,
            'tempDir' => $this->dir->getPath('var'),
        ];

        return new \Mpdf\Mpdf($config);
    }

    protected function initializePDFLibrary(): void
    {
        error_reporting(0);
        $base = $this->dir->getRoot() . '/vendor/mpdf/mpdf/src/';
        $files = [
            'Mpdf.php',
            'Config/ConfigVariables.php',
            'Ucdn.php',
            'Css/DefaultCss.php',
            'SizeConverter.php',
            'Color/ColorConverter.php',
            'Gradient.php',
            'TableOfContents.php',
            'Cache.php',
            'Fonts/FontCache.php',
            'Fonts/FontFileFinder.php',
            'CssManager.php',
            'Otl.php',
            'Form.php',
            'Hyphenator.php',
            'Tag.php',
            'Color/NamedColors.php',
            'PageFormat.php',
            'Config/FontVariables.php',
            'Css/TextVars.php',
            'Css/Border.php',
            'Log/Context.php',
            'TTFontFile.php',
            'Output/Destination.php',
            'MpdfException.php',
            'Fonts/MetricsGenerator.php',
            'Fonts/GlyphOperator.php',
            'Barcode.php',
        ];

        foreach ($files as $file) {
            Zend_Loader::loadFile($base . $file, null, true);
        }
    }
}
