<?php
/*
 *
 *  * web-vision GmbH
 *  * Do not edit or add to this file if you wish to upgrade Magento to newer
 *  * versions in the future. If you wish to customize Magento for your
 *  * needs please refer to https://www.web-vision.de for more information.
 *  * @copyright   Copyright (c) 2001-2023 web-vision GmbH (https://www.web-vision.de)
 *  * @author      Fenil Timbadiya
 *
 */
namespace Mzat\CompareProductPrint\ViewModel;

use Hyva\Theme\ViewModel\ProductAttributes;
use Magento\Eav\Model\Entity\Attribute\AbstractAttribute;
use Magento\Framework\View\Element\Block\ArgumentInterface;

class AttributesViewModel implements ArgumentInterface
{
    /**
     * Attributes to skip from both primary and secondary lists.
     *
     * @var string[]
     */
    private array $avoidedAttributes = [
        'manufacturer_logo',
        'long_description',
        'description_2',
        'description_3',
        'description_4',
    ];

    public function __construct(
        private readonly ProductAttributes $attributesViewModel
    ) {
    }

    /**
     * Filter attributes by type (primary or secondary).
     *
     * @param array  $attributes
     * @param string $type 'primary' or 'secondary'
     *
     * @return array
     */
    public function filterAttributes(array $attributes, string $type): array
    {
        return array_filter($attributes, function ($attr) use ($type) {
            if (in_array($attr->getAttributeCode(), $this->avoidedAttributes, true)) {

                return false;
            }

            $priority = $attr->getIsAttributePriority();

            return $type === 'secondary'
                ? $priority === 'secondary'
                : $priority !== 'secondary';
        });
    }

    /**
     * Get primary attributes.
     *
     * @param array $attributes
     *
     * @return array
     */
    public function getPrimaryAttributes(array $attributes): array
    {
        return $this->filterAttributes($attributes, 'primary');
    }

    /**
     * Get secondary attributes.
     *
     * @param array $attributes
     *
     * @return array
     */
    public function getSecondaryAttributes(array $attributes): array
    {
        return $this->filterAttributes($attributes, 'secondary');
    }

    /**
     * Check if any of the given products has a value for the attribute
     *
     * @param array|\Traversable $items    Array/collection of products
     * @param \Magento\Catalog\Model\ResourceModel\Eav\Attribute $attribute
     *
     * @return bool
     */
    public function hasAttributeValueForProducts($items, $attribute): bool
    {
        $code = $attribute->getAttributeCode();
        foreach ($items as $item) {
            if ($item->hasData($code) && $item->getData($code) !== '') {
                return true;
            }
        }

        return false;
    }

    /**
     * Check if an attribute has different values across a set of products.
     *
     * @param array|\Traversable $items    Array/collection of products
     * @param AbstractAttribute $attribute
     *
     * @return bool True if at least one product has a different value for this attribute.
     */
    public function attributeHasDifference($items, AbstractAttribute $attribute): bool
    {
        $values = [];
        foreach ($items as $product) {
            $data = $this->attributesViewModel->getAttributeData($attribute, $product);
            $values[] = trim((string) $data['value']);
        }

        return count(array_unique($values)) > 1;
    }

}
