<?php
/*
 * web-vision GmbH
 * @category    Mzat
 * @author      Parth Trivedi <parth@web-vision.de>
 */

declare(strict_types=1);
namespace Mzat\HtmlToPdf\Controller\Datasheet;

use Hyva\Theme\ViewModel\Logo\LogoPathResolver;
use Magento\Backend\App\Action\Context;
use Magento\Framework\App\Filesystem\DirectoryList;
use Magento\Framework\App\Request\Http;
use Magento\Framework\Controller\Result\JsonFactory;
use Magento\Framework\Filesystem;
use Magento\Framework\Serialize\Serializer\Json;
use Magento\Framework\Stdlib\DateTime\TimezoneInterface;
use Magento\Framework\View\Result\PageFactory;
use Magento\Store\Model\StoreManagerInterface;
use Zend_Loader;

class Index extends \Magento\Framework\App\Action\Action
{
    public function __construct(
        Context $context,
        protected PageFactory $resultPageFactory,
        protected Http $request,
        protected DirectoryList $dir,
        protected Json $json,
        protected JsonFactory $resultJsonFactory,
        protected Filesystem $filesystem,
        protected StoreManagerInterface $storeManager,
        protected LogoPathResolver $logoPathResolver,
        protected TimezoneInterface $timezone,
    ) {
        parent::__construct($context);
        $this->initializePDFLibrary();
    }

    public function execute()
    {
        $id = $this->request->getParam('id');
        $this->getResponse()->setHeader('X-Robots-Tag', 'noindex, nofollow', true);
        $post = $this->getPostedData();
        if (!$post) {
            return $this->jsonError(__('Content not available'));
        }
        $resultJson = $this->resultJsonFactory->create();

        if (!$id) {
            return $this->jsonError(__('Unable to download PDF'));
        }

        try {
            $pdfPath = $this->generatePdf($id, $post);
            $mediaUrl = $this->storeManager->getStore()->getBaseUrl(\Magento\Framework\UrlInterface::URL_TYPE_MEDIA);

            return $resultJson->setData([
                'msg' => $mediaUrl . basename($pdfPath),
                'success' => true,
            ]);
        } catch (\Exception $e) {
            return $this->jsonError(__('PDF generation failed'));
        }
    }

    protected function generatePdf($productId, array $post): string
    {
        $html = $this->getBodyHtml($productId, $post);
        $mpdf = $this->createMpdfInstance();

        $mpdf->SetTopMargin(50);

        $this->configurePdfHeader($mpdf);
        $this->configurePdfFooter($mpdf, $productId);
        $mpdf->SetAutoPageBreak(true, 18);
        $mpdf->WriteHTML(html_entity_decode($html));

        $pdfName = 'tmp-' . time() . '-datasheet.pdf';
        $pdfFilePath = $this->dir->getPath(DirectoryList::MEDIA) . '/' . $pdfName;
        $mpdf->Output($pdfFilePath, 'F');

        return $pdfFilePath;
    }

    protected function getBodyHtml($productId, array $post): string
    {
        return $this->resultPageFactory->create()->getLayout()
            ->createBlock(\Mzat\HtmlToPdf\Block\Datasheet\Attributes::class)
            ->setTemplate('Mzat_HtmlToPdf::datasheet/attributes.phtml')
            ->setChildDetails($post)
            ->setProductId($productId)
            ->toHtml();
    }

    protected function configurePdfHeader(\Mpdf\Mpdf $mpdf): void
    {
        $hederHtml = $this->getHeaderBlock();
        $mpdf->SetHTMLHeader(
            $hederHtml
        );
    }

    protected function getHeaderBlock(): string
    {
        return $this->resultPageFactory->create()->getLayout()
            ->createBlock(\Mzat\HtmlToPdf\Block\Datasheet\Attributes::class)
            ->setTemplate('Mzat_HtmlToPdf::datasheet/header.phtml')
            ->setLogo($this->getStaticLogoPath())
            ->toHtml();
    }

    protected function getStaticLogoPath(): string
    {
        return $this->dir->getPath(DirectoryList::STATIC_VIEW)
            . '/frontend/Mz-at/default/de_DE/images/logo.png';
    }

    protected function configurePdfFooter(\Mpdf\Mpdf $mpdf, $productId): void
    {
        $formattedDate = $this->timezone->formatDateTime(
            new \DateTime(),
            \IntlDateFormatter::SHORT, // date format
            \IntlDateFormatter::NONE,
            null,
            null,
            'dd.MM.yyyy'
        );

        $footerHtml = $this->resultPageFactory->create()->getLayout()
            ->createBlock(\Mzat\HtmlToPdf\Block\Datasheet\Attributes::class)
            ->setTemplate('Mzat_HtmlToPdf::datasheet/footer.phtml')
            ->setProductId($productId)
            ->setData('formatted_date', $formattedDate)
            ->toHtml();

        $mpdf->SetHTMLFooter($footerHtml);
    }

    protected function jsonError($message): \Magento\Framework\Controller\Result\Json
    {
        return $this->resultJsonFactory->create()->setData([
            'msg' => $message,
            'success' => false,
        ]);
    }

    protected function createMpdfInstance(): \Mpdf\Mpdf
    {
        $config = [
            'mode' => 'utf-8',
            'format' => 'A4',
            'default_font_size' => 10,
            'default_font' => 'Nunito',
            'margin_top' => 20,
            'margin_bottom' => 10,
            'allow_charset_conversion' => true,
            'orientation' => 'P',
            'showBarcodeNumbers' => false,
            'tempDir' => $this->dir->getPath('var'),
        ];

        return new \Mpdf\Mpdf($config);
    }

    protected function getPostedData(): array
    {
        if (!$this->request->isPost()) {
            return [];
        }

        $rawBody = (string) $this->request->getContent();
        if ($rawBody !== '') {
            try {
                $data = (array) $this->json->unserialize($rawBody);
                if (!empty($data)) {
                    return $data;
                }
            } catch (\InvalidArgumentException $e) {
            }
        }

        return (array) ($this->request->getPostValue() ?? []);
    }

    protected function initializePDFLibrary(): void
    {
        error_reporting(0);
        $base = $this->dir->getRoot() . '/vendor/mpdf/mpdf/src/';
        $files = [
            'Mpdf.php',
            'Config/ConfigVariables.php',
            'Ucdn.php',
            'Css/DefaultCss.php',
            'SizeConverter.php',
            'Color/ColorConverter.php',
            'Gradient.php',
            'TableOfContents.php',
            'Cache.php',
            'Fonts/FontCache.php',
            'Fonts/FontFileFinder.php',
            'CssManager.php',
            'Otl.php',
            'Form.php',
            'Hyphenator.php',
            'Tag.php',
            'Color/NamedColors.php',
            'PageFormat.php',
            'Config/FontVariables.php',
            'Css/TextVars.php',
            'Css/Border.php',
            'Log/Context.php',
            'TTFontFile.php',
            'Output/Destination.php',
            'MpdfException.php',
            'Fonts/MetricsGenerator.php',
            'Fonts/GlyphOperator.php',
            'Barcode.php',
        ];

        foreach ($files as $file) {
            Zend_Loader::loadFile($base . $file, null, true);
        }
    }
}
