<?php
/*
 * web-vision GmbH
 * Do not edit or add to this file if you wish to upgrade Magento to newer
 * versions in the future. If you wish to customize Magento for your
 * needs please refer to https://www.web-vision.de for more information.
 * @category    WebVision
 * @copyright   Copyright (c) 2001-2022 web-vision GmbH (https://www.web-vision.de)
 * @author      Fenil Timbadiya <fenil@web-vision.de>
 */
namespace WebVision\AutocompleteAddProductData\Model\Autocomplete\Product;

use Magento\Framework\ObjectManagerInterface;
use Magento\Catalog\Helper\Image as ImageHelper;
use Magento\Catalog\Api\Data\ProductInterface;
use Magento\Framework\Pricing\Render;
use Smile\ElasticsuiteCatalog\Model\Autocomplete\Product\AttributeConfig;
use Magento\Search\Model\Autocomplete\ItemFactory as CoreItemFactory;
/**
 * ItemFactory class
 */
class ItemFactory extends \Smile\ElasticsuiteCatalog\Model\Autocomplete\Product\ItemFactory
{

    /**
     * @var ImageHelper
     */
    private $imageHelper;

    /**
     * @var \Magento\Framework\Pricing\Render
     */
    private $priceRenderer = null;

    /**
     * @var ObjectManagerInterface
     */
    private $objectManager;

    /**
     * @var array
     */
    private $attributes;

    /**
     * Constructor.
     *
     * @param ObjectManagerInterface $objectManager   Object manager used to instantiate new item.
     * @param ImageHelper            $imageHelper     Catalog product image helper.
     * @param AttributeConfig        $attributeConfig Autocomplete attribute config.
     */
    public function __construct(
        ObjectManagerInterface $objectManager,
        ImageHelper $imageHelper,
        AttributeConfig $attributeConfig
    ) {
        $this->attributes    = $attributeConfig->getAdditionalSelectedAttributes();
        $this->imageHelper   = $imageHelper;
        $this->objectManager = $objectManager;
        parent::__construct($objectManager, $imageHelper, $attributeConfig);
    }
    /**
     * {@inheritDoc}
     */
    public function create(array $data)
    {
        $data = $this->_addProductData($data);
        unset($data['product']);

        return CoreItemFactory::create($data);
    }

    /**
     * Load product data and append them to the original data.
     *
     * @param array $data Autocomplete item data.
     *
     * @return array
     *
     */
    private function _addProductData($data)
    {
        $product = $data['product'];
        /* web-V: added sku */
        $productData = [
            'title' => $product->getName(),
            'sku' => $product->getSku(),
            'image' => $this->_getImageUrl($product),
            'url'   => $product->getProductUrl(),
            'price' => $this->_renderProductPrice($product, \Magento\Catalog\Pricing\Price\FinalPrice::PRICE_CODE),
        ];
        /* web-V: added sku end */

        foreach ($this->attributes as $attributeCode) {
            if ($product->hasData($attributeCode)) {
                $productData[$attributeCode] = $product->getData($attributeCode);
                if ($product->getResource()->getAttribute($attributeCode)->usesSource()) {
                    $productData[$attributeCode] = $product->getAttributeText($attributeCode);
                }
            }
        }

        $data = array_merge($data, $productData);

        return $data;
    }

     /**
     * Get resized image URL.
     *
     * @param ProductInterface $product Current product.
     *
     * @return string
     */
    private function _getImageUrl($product)
    {
        $this->imageHelper->init($product, ItemFactory::AUTOCOMPLETE_IMAGE_ID);

        return $this->imageHelper->getUrl();
    }


    /**
     * Renders product price.
     *
     * @param \Magento\Catalog\Model\Product $product   The product
     * @param string                         $priceCode The Price Code to render
     *
     * @return string
     */
    private function _renderProductPrice(\Magento\Catalog\Model\Product $product, $priceCode)
    {
        $priceRender = $this->_getPriceRenderer();

        $price = $product->getData($priceCode);

        if ($priceRender) {
            $price = $priceRender->render(
                $priceCode,
                $product,
                [
                    'include_container' => false,
                    'display_minimal_price' => true,
                    'zone' => Render::ZONE_ITEM_LIST,
                    'list_category_page' => true,
                ]
            );
        }

        return $price;
    }

    /**
     * Retrieve Price Renderer Block
     *
     * @return bool|\Magento\Framework\View\Element\BlockInterface
     */
    private function _getPriceRenderer()
    {
        if (null === $this->priceRenderer) {
            /** @var \Magento\Framework\View\LayoutInterface $layout */
            $layout = $this->objectManager->get('\Magento\Framework\View\LayoutInterface');
            $layout->getUpdate()->addHandle('default');
            $priceRenderer = $layout->getBlock('product.price.render.default');

            if (!$priceRenderer) {
                $priceRenderer = $layout->createBlock(
                    'Magento\Framework\Pricing\Render',
                    'product.price.render.default',
                    ['data' => ['price_render_handle' => 'catalog_product_prices']]
                );
            }

            $this->priceRenderer = $priceRenderer;
        }

        return $this->priceRenderer;
    }
}