<?php
/**
 * web-vision GmbH
 *
 * NOTICE OF LICENSE
 *
 * <!--LICENSETEXT-->
 *
 * DISCLAIMER
 *
 * Do not edit or add to this file if you wish to upgrade Magento to newer
 * versions in the future. If you wish to customize Magento for your
 * needs please refer to http://www.web-vision.de for more information.
 *
 * @category    WebVision
 *
 * @copyright   Copyright (c) 2001-2021 web-vision GmbH (https://www.web-vision.de)
 * @license     <!--LICENSEURL-->
 * @author      Parth Trivedi <parth@web-vision.de>
 */
namespace WebVision\BannerSlider\Model\ResourceModel;

use Magento\Framework\Event\ManagerInterface;
use Magento\Framework\Exception\LocalizedException;
use Magento\Framework\Model\AbstractModel;
use Magento\Framework\Model\ResourceModel\Db\AbstractDb;
use Magento\Framework\Model\ResourceModel\Db\Context;
use Magento\Framework\Stdlib\DateTime\DateTime;

/**
 * Class Banner
 */
class Banner extends AbstractDb
{
    /**
     * Date model
     *
     * @var DateTime
     */
    protected $date;

    /**
     * Slider relation model
     *
     * @var string
     */
    protected $bannerSliderTable;

    /**
     * Event Manager
     *
     * @var ManagerInterface
     */
    protected $eventManager;

    /**
     * constructor
     *
     * @param DateTime $date
     * @param ManagerInterface $eventManager
     * @param Context $context
     */
    public function __construct(
        DateTime $date,
        ManagerInterface $eventManager,
        Context $context
    ) {
        $this->date = $date;
        $this->eventManager = $eventManager;

        parent::__construct($context);
        $this->bannerSliderTable = $this->getTable('webvision_bannerslider_banner_slider');
    }

    /**
     * Initialize resource model
     *
     * @return void
     */
    protected function _construct()
    {
        $this->_init('webvision_bannerslider_banner', 'banner_id');
    }

    /**
     * @param $id
     *
     * @throws LocalizedException
     *
     * @return string
     */
    public function getBannerNameById($id)
    {
        $adapter = $this->getConnection();
        $select = $adapter->select()
            ->from($this->getMainTable(), 'name')
            ->where('banner_id = :banner_id');
        $binds = ['banner_id' => (int)$id];

        return $adapter->fetchOne($select, $binds);
    }

    /**
     * @param AbstractModel $object
     *
     * @return $this|AbstractDb
     */
    protected function _beforeSave(AbstractModel $object)
    {
        //set default Update At and Create At time post
        $object->setUpdatedAt($this->date->date());

        if ($object->isObjectNew()) {
            $object->setCreatedAt($this->date->date());
        }

        return $this;
    }

    /**
     * @param AbstractModel $object
     *
     * @return AbstractDb
     */
    protected function _afterSave(AbstractModel $object)
    {
        $this->saveSliderRelation($object);

        return parent::_afterSave($object);
    }

    /**
     * @param \WebVision\BannerSlider\Model\Banner $banner
     *
     * @return $this
     */
    protected function saveSliderRelation(\WebVision\BannerSlider\Model\Banner $banner)
    {
        $banner->setIsChangedSliderList(false);
        $id = $banner->getId();
        $sliders = $banner->getSlidersIds();

        if ($sliders === null) {
            return $this;
        }
        $oldSliders = $banner->getSliderIds();

        $insert = array_diff($sliders, $oldSliders);
        $delete = array_diff($oldSliders, $sliders);
        $adapter = $this->getConnection();

        if (!empty($delete)) {
            $condition = ['slider_id IN(?)' => $delete, 'banner_id=?' => $id];
            $adapter->delete($this->bannerSliderTable, $condition);
        }

        if (!empty($insert)) {
            $data = [];

            foreach ($insert as $tagId) {
                $data[] = [
                    'banner_id' => (int)$id,
                    'slider_id' => (int)$tagId,
                    'position' => 1,
                ];
            }
            $adapter->insertMultiple($this->bannerSliderTable, $data);
        }

        if (!empty($insert) || !empty($delete)) {
            $sliderIds = array_unique(array_merge(array_keys($insert), array_keys($delete)));
            $this->eventManager->dispatch(
                'webvision_bannerslider_banner_after_save_sliders',
                ['banner' => $banner, 'slider_ids' => $sliderIds]
            );

            $banner->setIsChangedSliderList(true);
            $sliderIds = array_keys($insert + $delete);
            $banner->setAffectedSliderIds($sliderIds);
        }

        return $this;
    }

    /**
     * @param \WebVision\BannerSlider\Model\Banner $banner
     *
     * @return array
     */
    public function getSliderIds(\WebVision\BannerSlider\Model\Banner $banner)
    {
        $adapter = $this->getConnection();
        $select = $adapter->select()
            ->from($this->bannerSliderTable, 'slider_id')
            ->where('banner_id = ?', (int)$banner->getId());

        return $adapter->fetchCol($select);
    }
}
