<?php
/*
 * web-vision GmbH
 * Do not edit or add to this file if you wish to upgrade Magento to newer
 * versions in the future. If you wish to customize Magento for your
 * needs please refer to https://www.web-vision.de for more information.
 * @category    WebVision
 * @copyright   Copyright (c) 2001-2022 web-vision GmbH (https://www.web-vision.de)
 * @author      Mahesh Makwana <mmakwana@web-vision.de>
 */
namespace WebVision\CustomDataSearch\Helper;

use Magento\CatalogUrlRewrite\Model\CategoryUrlPathGenerator;
use Magento\Store\Model\ScopeInterface;
use Smile\ElasticsuiteCore\Helper\AbstractConfiguration;
use WebVision\CustomDataSearch\Api\Data\CustomDataInterface as CustomData;
use Magento\Catalog\Model\CategoryRepository;
use WebVision\CustomDataSearch\Logger\Logger;

class Data extends AbstractConfiguration
{
    const SORTING_DATA = 'desc';

    const CUSTOM_DATA_SEARCH_STATUS = 'customdata_base_setting/enable_customdata_search';

    const CUSTOM_DATA_STORE_FILTER_STATUS = 'customdata_base_setting/enable_customdata_store_filter';

    const CUSTOM_DATA_STORE_ID_DATA = 'customdata_base_setting/customdata_table_store_filter_data';

    const CUSTOM_DATA_BASE_CATEGORY_ID = 'customdata_base_setting/customdata_base_category';

    const CUSTOM_DATA_SHOW_SEARCH_PAGE = 'smile_elasticsuite_catalogsearch_settings/catalogsearch/result_page_show_customdata_page';

    public const DEFAULT_DATETIME_FORMATE = 'Y-m-d H:i:s';

    public const NEW_DATETIME_FORMATE = 'l, F j, Y \a\t g:i:s A';

    /**
     * @var \Smile\ElasticsuiteCore\Client\Client
     */
    public $elasticsearchClient;

    /**
     * @var \Smile\ElasticsuiteCore\Search\Request\Builder
     */
    public $requestBuilder;

    /**
     * @var ContainerConfigurationFactory
     */
    public $containerConfigFactory;

    /**
     * @var \Magento\Framework\App\Config\ScopeConfigInterface
     */
    protected $scopeConfig;

    /**
     * @var \Magento\Framework\App\Config\ScopeConfigInterface
     */
    protected $json;

    /**
     * @var \Magento\UrlRewrite\Model\UrlRewriteFactory
     */
    protected $urlRewriteData;

    /**
     * @var \Magento\Framework\App\DeploymentConfig
     */
    private $deploymentConfig;

    /**
     * @var CategoryRepository
     */
    protected $categoryRepository;

    /**
     * @var Logger
     */
    protected $logger;

    /**
     * @param \Magento\Framework\App\Helper\Context $context
     * @param \Magento\Store\Model\StoreManagerInterface $storeManager
     * @param \Smile\ElasticsuiteCore\Search\Request\Builder $requestBuilder
     * @param \Smile\ElasticsuiteCore\Search\Request\ContainerConfigurationFactory $containerConfigFactory
     * @param \Smile\ElasticsuiteCore\Client\Client $elasticsearchClient
     * @param \Magento\Framework\App\Config\ScopeConfigInterface $scopeConfig
     * @param \Magento\Framework\Serialize\Serializer\Json $json
     * @param \Magento\UrlRewrite\Model\UrlRewriteFactory $urlRewriteData
     * @param \Magento\Framework\App\DeploymentConfig $deploymentConfig
     */
    public function __construct(
        \Magento\Framework\App\Helper\Context $context,
        \Magento\Store\Model\StoreManagerInterface $storeManager,
        \Smile\ElasticsuiteCore\Search\Request\Builder $requestBuilder,
        \Smile\ElasticsuiteCore\Search\Request\ContainerConfigurationFactory $containerConfigFactory,
        \Smile\ElasticsuiteCore\Client\Client $elasticsearchClient,
        \Magento\Framework\App\Config\ScopeConfigInterface $scopeConfig,
        \Magento\Framework\Serialize\Serializer\Json $json,
        \Magento\UrlRewrite\Model\UrlRewriteFactory $urlRewriteData,
        \Magento\Framework\App\DeploymentConfig $deploymentConfig,
        CategoryRepository $categoryRepository,
        Logger $logger
    ) {
        $this->elasticsearchClient = $elasticsearchClient;
        $this->containerConfigFactory = $containerConfigFactory;
        $this->requestBuilder = $requestBuilder;
        $this->scopeConfig = $scopeConfig;
        $this->storeManager = $storeManager;
        $this->json = $json;
        $this->urlRewriteData = $urlRewriteData;
        $this->deploymentConfig = $deploymentConfig;
        $this->categoryRepository = $categoryRepository;
        $this->logger = $logger;
        parent::__construct($context, $storeManager);
    }

    /**
     * Get customdata search status
     *
     * @return string
     */
    public function getCustomDataSearchStatus()
    {
        return $this->getElasticSuiteConfigParam(self::CUSTOM_DATA_SEARCH_STATUS);
    }

    /**
     * Get customdata search page status
     *
     * @return string
     */
    public function getCustomDataSearchPageStatus()
    {
        return $this->scopeConfig->getValue(self::CUSTOM_DATA_SHOW_SEARCH_PAGE, ScopeInterface::SCOPE_STORE);
    }

    /**
     * Get base category id
     *
     * @return string
     */
    public function getBaseCategoryId()
    {
        return $this->getElasticSuiteConfigParam(self::CUSTOM_DATA_BASE_CATEGORY_ID);
    }

    /**
     * Get manufacturere category path value
     *
     * @param string|null $storeId
     *
     * @return string|bool
     */
    public function getBaseCategoryPath($storeId, $categoryId = false)
    {
        // Ensure $categoryId is set to a valid value
        if ($categoryId === false) {
            $categoryId = $this->getBaseCategoryId();
        }

        try {
            return $this->getCustomDataSearchStatus()
            ? $this->categoryRepository->get($categoryId, $storeId)->getPath()
            : false;
        } catch (\Exception $e) {
            $this->logger->critical($e->getMessage());
        }

        return false;
    }

    /**
     * Get customdata store filter status
     *
     * @return string
     */
    public function getStoreFilterStatus()
    {
        return trim($this->getElasticSuiteConfigParam(self::CUSTOM_DATA_STORE_FILTER_STATUS));
    }

    /**
     * Get customdata store filter Data
     *
     * @param mixed $storeId
     *
     * @return string
     */
    public function getStoreFilterData($storeId = 0)
    {
        $storeFilterData = $this->json->unserialize(
            trim($this->getElasticSuiteConfigParam(self::CUSTOM_DATA_STORE_ID_DATA)) ?? ''
        );

        foreach ($storeFilterData as $filterKey => $filterVal) {
            if ($filterVal['attr_store'] == $storeId) {
                $storeId = $filterVal['attr_id'];

                break;
            }
        }

        return $storeId;
    }

    /**
     * Get category url suffix
     *
     * @return string
     */
    public function getCategoryUrlSuffix()
    {
        return $this->scopeConfig->getValue(CategoryUrlPathGenerator::XML_PATH_CATEGORY_URL_SUFFIX, ScopeInterface::SCOPE_STORE);
    }

    /**
     * Get Page Ids
     *
     * @param string $searchRequestName
     * @param int $storeId
     * @param int $size
     * @param int $from
     * @param string $searchTerms
     *
     * @return array
     */
    public function getElasticsuiteCustomData($searchRequestName, $storeId, $size, $from, $searchTerms)
    {
        try{
            $params = [];
            $containerConfig = $this->getRequestContainerConfiguration($storeId, $searchRequestName);
            $params['index'] = $containerConfig->getIndexName();
            $params['body'] = $this->getEsBody($size, $from, $searchTerms);
            $customdataEsData = $this->elasticsearchClient->search($params);

            return $this->getPageIds($customdataEsData);
        } catch (\Exception $e) {
            $this->logger->critical($e->getMessage());
        }

        return [];
    }

    /**
     * Get Page Ids
     *
     * @param array $customdataEsData
     *
     * @return array
     */
    public function getPageIds($customdataEsData)
    {
        $pageIds = [];
        $pageIds[] = 0;
        if (isset($customdataEsData['hits']['hits'])) {
            $esHits = $customdataEsData['hits']['hits'];
            $mappedPageIds = array_map([$this, 'callableMappingPageIds'], $esHits);
        }

        return count($mappedPageIds) ? $mappedPageIds : $pageIds;
    }

    /**
     * Callable Mapping Page Ids
     *
     * @param array $mapValue
     *
     * @return array
     */
    public function callableMappingPageIds($mapValue)
    {
        return isset($mapValue['_id']) ? $mapValue['_id'] : 0;
    }

    /**
     * Get customdata pages ES body.
     *
     * @param int $searchTerms
     * @param int $from
     * @param string $searchTerms
     * @param mixed $size
     *
     * @return array
     */
    public function getEsBody($size, $from, $searchTerms)
    {
        $pagesEsBody = [];
        $pagesEsBody['size'] = $size;
        $pagesEsBody['from'] = $from;
        $pagesEsBody['query']['bool']['should'] = $this->getCustomQuery($searchTerms);
        $pagesEsBody['query']['bool']['filter'] = ['term' => ['store_id' => $this->getStoreId()]];
        $pagesEsBody['sort'] = [['_score' => ['order' => 'desc']]];
        $pagesEsBody['min_score'] = 0.1;

        return $pagesEsBody;
    }

    /**
     * Get sorting
     *
     * @return array
     */
    public function getSorting()
    {
        return ['order' => self::SORTING_DATA];
    }

    /**
     * Get customdata query.
     *
     * @param string $searchTerms
     *
     * @return array
     */
    public function getCustomQuery($searchTerms)
    {
        return [
            [
                'match_phrase' => [
                    'title.standard' => [
                        'query' => $searchTerms,
                        'boost' => 15,
                    ],
                ],
            ],
            [
                'regexp' => [
                    'title.untouched' => [
                        'value' => '.*'.$searchTerms.'.*',
                        "case_insensitive" => true,
                        'boost' => 13,
                    ],
                ],
            ],
            [
                'match_phrase' => [
                    'content.standard' => [
                        'query' => $searchTerms,
                        'boost' => 10,
                    ],
                ],
            ],
            [
                'match_phrase' => [
                    'sub_categories.standard' => [
                        'query' => $searchTerms,
                        'boost' => 5,
                    ],
                ],
            ],
            [
                'regexp' => [
                    'sub_categories.untouched' => [
                        'value' => '.*'.$searchTerms.'.*',
                        "case_insensitive" => true,
                        'boost' => 3,
                    ],
                ],
            ],
        ];
    }

    /**
     * Get current store id
     *
     * @return int
     */
    public function getStoreId()
    {
        return $this->storeManager->getStore()->getId();
    }

    /**
     * Load the search request configuration (index, type, mapping, ...) using the search request container name.
     *
     *
     * @param int $storeId       Store id.
     * @param string  $containerName Search request container name.
     *
     * @throws \LogicException Thrown when the search container is not found into the configuration.
     *
     * @return ContainerConfigurationInterface
     */
    private function getRequestContainerConfiguration($storeId, $containerName)
    {
        if ($containerName === null) {
            throw new \LogicException('Request name is not set');
        }

        $config = $this->containerConfigFactory->create(
            ['containerName' => $containerName, 'storeId' => $storeId]
        );

        if ($config === null) {
            throw new \LogicException("No configuration exists for request {$containerName}");
        }

        return $config;
    }

    /**
     * Checked conflict typoe page url
     *
     * @param mixed $pageData
     * @param mixed $customdataUrl
     *
     * @return int|null
     */
    public function customdataMappingPageUrl($pageData, $customdataUrl)
    {
        if (isset($pageData['request_path'])) {
            $customdataUrl = $customdataUrl . '' . $this->getCategoryUrlSuffix();
            if ($customdataUrl == $pageData['request_path']) {
                return 1;
            }
        }
    }
}
