<?php
/*
 * web-vision GmbH
 * Do not edit or add to this file if you wish to upgrade Magento to newer
 * versions in the future. If you wish to customize Magento for your
 * needs please refer to https://www.web-vision.de for more information.
 * @category    WebVision
 * @copyright   Copyright (c) 2001-2022 web-vision GmbH (https://www.web-vision.de)
 * @author      Mahesh Makwana <mmakwana@web-vision.de>
 */
namespace WebVision\CustomDataSearch\Model\Autocomplete\Typo3;

use Magento\Search\Model\Autocomplete\DataProviderInterface;
use Magento\Search\Model\QueryFactory;
use Smile\ElasticsuiteCatalog\Helper\Autocomplete as ConfigurationHelper;
use Smile\ElasticsuiteCore\Model\Autocomplete\Terms\DataProvider as TermDataProvider;
use WebVision\CustomDataSearch\Helper\Data as CustomDataHelper;
use WebVision\CustomDataSearch\Model\ResourceModel\Typo3\Fulltext\CollectionFactory as Typo3CollectionFactory;
use WebVision\CustomDataSearch\Logger\Logger;

class DataProvider implements DataProviderInterface
{
    /**
     * Autocomplete type
     */
    const AUTOCOMPLETE_TYPE = 'customdata';

    /**
     * Autocomplete result item factory
     *
     * @var ItemFactory
     */
    protected $itemFactory;

    /**
     * Query factory
     *
     * @var QueryFactory
     */
    protected $queryFactory;

    /**
     * @var TermDataProvider
     */
    protected $termDataProvider;

    /**
     * @var Typo3CollectionFactory
     */
    protected $customdataCollectionFactory;

    /**
     * @var ConfigurationHelper
     */
    protected $configurationHelper;

    /**
     * @var CustomDataHelper
     */
    protected $customdataHelper;

    /**
     * @var Logger
     */
    protected $logger;

    /**
     * @var string Autocomplete result type
     */
    private $type;

    /**
     * Constructor.
     *
     * @param ItemFactory               $itemFactory               Suggest item factory.
     * @param QueryFactory              $queryFactory              Search query factory.
     * @param TermDataProvider          $termDataProvider          Search terms suggester.
     * @param Typo3CollectionFactory    $customdataCollectionFactory    Typo3 collection factory.
     * @param ConfigurationHelper       $configurationHelper       Autocomplete configuration helper.
     * @param CustomDataHelper          $customdataHelper
     * @param Logger                    $logger
     * @param string                    $type                      Autocomplete provider type.
     */
    public function __construct(
        ItemFactory $itemFactory,
        QueryFactory $queryFactory,
        TermDataProvider $termDataProvider,
        Typo3CollectionFactory $customdataCollectionFactory,
        ConfigurationHelper $configurationHelper,
        CustomDataHelper $customdataHelper,
        Logger $logger,
        $type = self::AUTOCOMPLETE_TYPE
    ) {
        $this->itemFactory = $itemFactory;
        $this->queryFactory = $queryFactory;
        $this->termDataProvider = $termDataProvider;
        $this->customdataCollectionFactory = $customdataCollectionFactory;
        $this->configurationHelper = $configurationHelper;
        $this->customdataHelper = $customdataHelper;
        $this->logger = $logger;
        $this->type = $type;
    }

    /**
     * @return string
     */
    public function getType()
    {
        return $this->type;
    }

    /**
     * {@inheritDoc}
     */
    public function getItems()
    {
        $result = [];
        try{
            $customdataStatus = $this->customdataHelper->getCustomDataSearchStatus();
            $autoSuggestionStatus = $this->configurationHelper->isEnabled($this->getType());

            if (($customdataStatus) && ($autoSuggestionStatus)) {
                $customdataCollection = $this->getCustomDataCollection();
                if ($customdataCollection) {
                    foreach ($customdataCollection as $customdata) {
                        $result[] = $this->itemFactory->create([
                            'customdata' => $customdata,
                            'type' => $this->getType(),
                        ]);
                    }
                }
            }
        } catch (\Exception $e) {
            $this->logger->critical($e->getMessage());
        }

        return $result;
    }

    /**
     * List of search terms suggested by the search terms data provider.
     *
     * @return array
     */
    private function getSuggestedTerms()
    {
        $terms = array_map(
            function (\Magento\Search\Model\Autocomplete\Item $termItem) {
                return $termItem->getTitle();
            },
            $this->termDataProvider->getItems()
        );

        return $terms;
    }

    /**
     * Suggested customdata page collection.
     * Returns null if no suggested search terms.
     *
     * @param bool $size
     *
     * @throws \Magento\Framework\Exception\LocalizedException
     *
     * @return \WebVision\CustomDataSearch\Model\ResourceModel\Typo3\Fulltext\Collection|null
     */
    public function getCustomDataCollection($size = true)
    {
        $customdataCollection = null;
        $terms = $this->queryFactory->get()->getQueryText();
        if ($terms) {
            $customdataCollection = $this->customdataCollectionFactory->create();
            if($size){
                $customdataCollection->setPageSize($this->getResultsPageSize());
            }
            $customdataCollection->setPageSearchTearm($this->queryFactory->get()->getQueryText());
            $customdataCollection->isSpellchecked(true);
        }

        return $customdataCollection;
    }

    /**
     * Retrieve number of customdata pages to display in autocomplete results
     *
     * @return int
     */
    private function getResultsPageSize()
    {
        return $this->configurationHelper->getMaxSize($this->getType());
    }
}
