<?php
/*
 * web-vision GmbH
 * Do not edit or add to this file if you wish to upgrade Magento to newer
 * versions in the future. If you wish to customize Magento for your
 * needs please refer to https://www.web-vision.de for more information.
 * @category    WebVision
 * @copyright   Copyright (c) 2001-2022 web-vision GmbH (https://www.web-vision.de)
 * @author      Mahesh Makwana <mmakwana@web-vision.de>
 */
namespace WebVision\CustomDataSearch\Model;

use WebVision\CustomDataSearch\Logger\Logger;
use WebVision\CustomDataSearch\Helper\Data;
use Magento\Store\Model\StoreManagerInterface;
use Magento\Catalog\Model\ResourceModel\Category\CollectionFactory as CategoryCollectionFactory;
use Magento\Catalog\Model\ResourceModel\Category\Collection as CategoryCollection;
use Magento\Framework\Json\Helper\Data as JsonHelper;
use WebVision\CustomDataSearch\Model\CustomDataSearchFactory as CustomDataSearchModel;
use WebVision\CustomDataSearch\Api\Data\CustomDataInterface;
use Magento\Framework\App\RequestInterface;
use Magento\Framework\Stdlib\DateTime\TimezoneInterface;
use Magento\Indexer\Model\IndexerFactory;
use Magento\Catalog\Model\CategoryRepository;
use Magento\Store\Model\App\Emulation;
use Magento\Framework\App\Area;

class CustomSearcManage
{
    /**
     * @var Logger
     */
    protected $logger;

    /**
     * @var Data
     */
    protected $customDataSearchHelper;

    /**
     * @var StoreManagerInterface
     */
    protected $_storeManager;

    /**
     * @var CategoryCollectionFactory
     */
    protected $categoriesCollFactory;

    /**
     * @var JsonHelper
     */
    protected $jsonHelper;

    /**
     * @var CustomDataSearchModel
     */
    protected $customDataSearchModel;

    /**
     * @var RequestInterface
     */
    protected $request;

    /**
     * @var TimezoneInterface
     */
    protected $timezoneInterface;

    /**
     * @var IndexerFactory
     */
    protected $indexerFactory;

    /**
     * @var CategoryRepository
     */
    protected $categoryRepository;

	/**
	 * @var Emulation
	 */
	protected $emulation;

     /**
      * Constructor
      *
      * @param Logger $logger
      * @param Data $customDataSearchHelper
      * @param StoreManagerInterface $storeManager
      * @param CategoryCollectionFactory $categoriesCollFactory
      * @param JsonHelper $jsonHelper
      * @param CustomDataSearchModel $customDataSearchModel
      * @param RequestInterface $request
      * @param TimezoneInterface $timezoneInterface
      * @param IndexerFactory $indexerFactory
      * @param CategoryRepository $categoryRepository
      * @param Emulation $emulation
      */
    public function __construct(
        Logger $logger,
        Data $customDataSearchHelper,
        StoreManagerInterface $storeManager,
        CategoryCollectionFactory $categoriesCollFactory,
        JsonHelper $jsonHelper,
        CustomDataSearchModel $customDataSearchModel,
        RequestInterface $request,
        TimezoneInterface $timezoneInterface,
        IndexerFactory $indexerFactory,
        CategoryRepository $categoryRepository,
        Emulation $emulation
    ) {
        $this->logger = $logger;
        $this->customDataSearchHelper = $customDataSearchHelper;
        $this->_storeManager = $storeManager;
        $this->categoriesCollFactory = $categoriesCollFactory;
        $this->jsonHelper = $jsonHelper;
        $this->customDataSearchModel = $customDataSearchModel;
        $this->request = $request;
        $this->timezoneInterface = $timezoneInterface;
        $this->indexerFactory = $indexerFactory;
        $this->categoryRepository = $categoryRepository;
        $this->emulation = $emulation;
    }

    /**
     * Process data
     *
     * @param int $nextProcessId
     *
     * @return void
     */
    public function process($nextProcessId = 0)
    {
        // Initial data load if zero record exit
        $this->isDataEmpty();

        // Check existing process is runnig or not
        if($this->isCronRunning()){
            return false;
        }

        // get next cron to process
        $nextCronData = $this->getNextQueuedItem($nextProcessId);
        if ($nextCronData && $nextCronData->getId()) {
            $cronId = $nextCronData->getId();
            $storeId = $nextCronData->getStoreId();
            $catId = $nextCronData->getCatId();

            // Update status to runnig cron
            $this->addStatus($cronId, CustomDataInterface::RUNNIG_STATUS);

            // Process category
            $this->processCategories($storeId, $catId);
        }
    }

    /**
     * Execute the clean urls process
     *
     * @return void|bool
     */
    public function processAllSku()
    {
        // Check existing process is runnig or not
        if($this->isCronRunning()){
            return false;
        }

        try {
            foreach($this->_storeManager->getStores() as $store) {
                if($store->getId()){
                    $this->processCategories($store->getId());
                }
            }
        } catch (\Exception $e) {
            $this->logger->error($e->getMessage());
        }
    }

    /**
     * Process Categories
     *
     * @param int $storeId
     * @param int $categoryId
     *
     * @return void
     */
    public function processCategories($storeId, $categoryId = 0)
    {
        try {
            $categoryCollection = $this->getCategory($storeId, $categoryId);
            if (!($categoryCollection instanceof CategoryCollection)) {
                $this->logger->error("The category collection is not CategoryCollection");
                return false;
            }

            foreach($categoryCollection as $category) {
                try {
                    // skip the base category
                    if ($category->getId() == $this->customDataSearchHelper->getBaseCategoryId()) {
                        continue;
                    }
                    $productData = $this->getProductData($category, $storeId);
                    $this->saveSearchData($category, $productData, $storeId);

                } catch (\Exception $e) {
                    $this->logger->error($e->getMessage());
                    break;
                }
            }
        } catch (\Exception $e) {
            $this->logger->error($e->getMessage());
        }
    }

    /**
     * Get Category data
     *
     * @param int $storeId
     * @param int $categoryId
     *
     * @return void
     */
    public function getCategory($storeId, $categoryId = 0)
    {
        try {
            $baseCatPath = $this->customDataSearchHelper->getBaseCategoryPath($storeId);
            $categories = $this->categoriesCollFactory->create()
                ->addAttributeToSelect('*')
                ->addAttributeToFilter('path', ['like' => $baseCatPath."/%"]);

            // Apply category filter
            if($categoryId != 0) {
                $categories->addAttributeToFilter('entity_id', ['eq' => $categoryId]);
            }

            // Apply store filter
            $categories->setStore($this->_storeManager->getStore($storeId));

            return $categories;
        } catch (\Exception $e) {
            $this->logger->error($e->getMessage());
        }

        return [];
    }

    /**
     * Get Product Data
     *
     * @param obj $category
     * @param int $storeId
     *
     * @return string
     */
    public function getProductData($category, $storeId)
    {
        try {
            $productData = [];
            foreach($category->getProductCollection() as $product){
                // Check product visibility
                $productVisibility = $product->getResource()->getAttributeRawValue($product->getId(), 'visibility', $storeId);
                if(in_array($productVisibility, [2,3,4])) {
                    $productData[] = $product->getSku();
                    $productData[] = $product->getResource()->getAttributeRawValue($product->getId(), 'name', $storeId);
                }
            }
        } catch (\Exception $e) {
            $this->logger->error($e->getMessage());
        }

        return $this->jsonHelper->jsonEncode($productData);
    }

    /**
     * Save Search Data
     *
     * @param object $category
     * @param string $productContent
     * @param string $storeId
     *
     * @return void
     */
    public function saveSearchData($category, $productContent, $storeId)
    {
        try {
            // Get Sub Categories Name
            $subCategories = $this->getChildCategoriesData($category);

            $categoryId = $category->getId();
            $searchRawData = $this->getCustomSearchModal();
            $searchData = $this->getCustomSearchModal()->getCollection();
            $searchData->addFieldToFilter('cat_id', $categoryId);
            $searchData->addFieldToFilter('store_id', $storeId);

            if(count($searchData->getData())) {
                // Load existing search data
                $searchRawData->load($searchData->getFirstItem()->getId());
            }

            $searchRawData->setCatId($categoryId);
            $searchRawData->setStoreId($storeId);
            $searchRawData->setContent($productContent);
            $searchRawData->setStatus(CustomDataInterface::SUCCESS_STATUS);
            $searchRawData->setTitle($category->getName());
            $searchRawData->setSubCategories($subCategories);
            $searchRawData->save();
        } catch (\Exception $e) {
            $this->logger->error($e->getMessage());
        }
    }

    /**
     * Get Child Categories Names
     *
     * @param Obj $category
     *
     * @return string
     */
    public function getChildCategoriesData($category)
    {
        $child = [];

        try {
            $childCategories = $category->getChildrenCategories();
            foreach ($childCategories as $childCategroy) {
                $child[] = $childCategroy->getName();
            }
        } catch (\Exception $e) {
            $this->logger->error($e->getMessage());
        }

        return $this->getJsonHelper()->jsonEncode($child);
    }

    /**
     * Get Custom Search Modal
     *
     * @return void
     */
    public function getCustomSearchModal()
    {
        return $this->customDataSearchModel->create();
    }

    /**
     * Get json object
     *
     * @return obj
     */
    public function getJsonHelper()
    {
        return $this->jsonHelper;
    }

    /**
     * Get param value
     *
     * @param null|string $paramKey
     *
     * @return void
     */
    public function getParam($paramKey = null)
    {
        if ($paramKey) {
            return $this->request->getParam($paramKey);
        } else {
            return $this->request->getParams();
        }
    }

    /**
     * Add status value to record
     *
     * @param int $id
     * @param int $status
     *
     * @return bool
     */
    public function addStatus($id, $status)
    {
        try {
            $syncData = $this->getCustomSearchModal()->load($id);
            $syncData->setStatus($status);
            $syncData->save();

            return true;
        } catch (\Exception $e) {
            $this->logger->error($e->getMessage());
        }

        return false;
    }

    /**
     * Get Next Queued Item from grid
     *
     * @param int $id
     * @return obj
     */
    public function getNextQueuedItem($id)
    {
        $syncData = false;
        try {
            if($id){
                $syncData = $this->getCustomSearchModal()->load($id);
            } else {
                $syncData = $this->getCustomSearchModal()->getCollection();
                $syncData->addFieldToFilter(CustomDataInterface::STATUS, CustomDataInterface::PROCESS_STATUS);
                $syncData->setOrder(CustomDataInterface::UPDATED_AT, 'ASC');
                $syncData = $syncData->getFirstItem();
            }

            return $syncData;
        } catch (\Exception $e) {
            $this->logger->error($e->getMessage());
        }

        return $syncData;
    }

    /**
     * Check is cron is runnign
     *
     * @return void
     */
    public function isCronRunning()
    {
        $runnigStatus = 0;
        try {
            $syncData = $this->getCustomSearchModal()->getCollection();
            $syncData->addFieldToFilter(CustomDataInterface::STATUS, CustomDataInterface::RUNNIG_STATUS);
            $runnigStatus = count($syncData->getData());

            return $runnigStatus;
        } catch (\Exception $e) {
            $this->logger->error($e->getMessage());
        }

        return false;
    }

    /**
     * Check is data empty
     *
     * @return void
     */
    public function isDataEmpty()
    {
        $syncData = $this->getCustomSearchModal()->getCollection();
        if(count($syncData->getData()) == 0){
            $this->processAllSku();
        }
    }

    /**
     * Get new category
     *
     * @return void
     */
    public function newCategorySync()
    {
        try {
            foreach($this->_storeManager->getStores() as $store) {
                if($storeId = $store->getId()){

                    $categoryCollection = $this->getCategory($storeId);
                    if (!($categoryCollection instanceof CategoryCollection)) {
                        $this->logger->error("The category collection is not CategoryCollection");
                        return false;
                    }

                    foreach($categoryCollection as $category) {
                        try {
                            $categoryId = $category->getId();
                            // skip the base category
                            if ($categoryId == $this->customDataSearchHelper->getBaseCategoryId()) {
                                continue;
                            }

                            // check new categories
                            $searchData = $this->getCustomSearchModal()->getCollection();
                            $searchData->addFieldToFilter('cat_id', $categoryId);
                            $searchData->addFieldToFilter('store_id', $storeId);

                            if(count($searchData->getData())) {
                                continue;
                            }

                            // Get Sub Categories Name
                            $subCategories = $this->getChildCategoriesData($category);

                            $searchRawData = $this->getCustomSearchModal();
                            $searchRawData->setCatId($categoryId);
                            $searchRawData->setStoreId($storeId);
                            $searchRawData->setStatus(CustomDataInterface::PROCESS_STATUS);
                            $searchRawData->setTitle($category->getName());
                            $searchRawData->setSubCategories($subCategories);
                            $searchRawData->save();

                        } catch (\Exception $e) {
                            $this->logger->error($e->getMessage());
                            break;
                        }
                    }
                }
            }
        } catch (\Exception $e) {
            $this->logger->error($e->getMessage());
        }
    }

    /**
     * Get current time
     *
     * @param string $format
     * @param string $date
     *
     * @return string
     */
    public function getCurrentTime($format = CustomDataInterface::DEFAULT_DATETIME_FORMATE, $date = null)
    {
        return $this->timezoneInterface->date($date)->format($format);
    }
    /**
     * Sync Custom Data via Reindex
     *
     * @param string $indexValues
     *
     * @return void
     */
    public function syncCustomData($indexValues)
    {
        foreach ($indexValues as $index) {
            try {
                $indexer = $this->indexerFactory->create();
                $indexer->load($index);
                if ($indexer->getId()) {
                    $indexer->invalidate();
                    $indexer->reindexAll($index);
                } else {
                    $this->logger->error("Indexer ".$index." not found.");
                }
            } catch (\Exception $e) {
                $this->logger->error($e->getMessage());
            }
        }
    }

    /**
     * Get Custom Data Search Status
     *
     * @return void
     */
    public function getCustomDataSearchStatus()
    {
        return $this->customDataSearchHelper->getCustomDataSearchStatus();
    }

    /**
     * Formatted DateTime With current timezone
     *
     * @return string
     */
    public function formateDateWithoutTimeZone($format, $date)
    {
        $dt = \DateTime::createFromFormat(CustomDataInterface::DEFAULT_DATETIME_FORMATE, $date);
        return $dt->format($format);
    }

    /**
     * Get category url
     *
     * @param int $categoryId
     *
     * @return string
     */
    public function getCategoryUrl($categoryId)
    {
        $category = $this->categoryRepository->get($categoryId);
        $categoryUrl = $category->getUrl();
        return $categoryUrl;
    }

    /**
     * Reset Running Processes
     *
     * @return int
     */
    public function resetRunningProcess()
    {
        $runningFlage = 0;
        try {
            $searchData = $this->getCustomSearchModal()->getCollection();
            $searchData->addFieldToSelect([CustomDataInterface::ID]);
            $searchData->addFieldToFilter(CustomDataInterface::STATUS, CustomDataInterface::RUNNIG_STATUS);
            foreach($searchData as $search){
                $search->setStatus(CustomDataInterface::PROCESS_STATUS);
                $search->save();
                $runningFlage++;
            }
        } catch (\Exception $e) {
            $this->logger->error($e->getMessage());
        }

        return $runningFlage;
    }
}
