<?php
/*
 * web-vision GmbH
 * Do not edit or add to this file if you wish to upgrade Magento to newer
 * versions in the future. If you wish to customize Magento for your
 * needs please refer to https://www.web-vision.de for more information.
 * @category    WebVision
 * @copyright   Copyright (c) 2001-2024 web-vision GmbH (https://www.web-vision.de)
 * @author      Mahesh Makwana <mmakwana@web-vision.de>
 */
namespace WebVision\CustomRedirection\Helper;

use Magento\Framework\App\Filesystem\DirectoryList;
use Magento\Framework\App\Helper\AbstractHelper;
use Magento\Framework\App\Helper\Context;
use Magento\Framework\App\ResourceConnection;
use Magento\Framework\Filesystem\Driver\File;
use Magento\Framework\Filesystem\Io\File as FileIo;
use Magento\Store\Model\ScopeInterface;
use WebVision\CustomRedirection\Api\Data\CustomRedirectionInterface;
use WebVision\CustomRedirection\Logger\Logger;

class Data extends AbstractHelper
{
    public const MODULE_STATUS = 'custom_redirection/general/enable';

    public const FILE_NAME = 'custom_redirection/general/file_name';

    public const BATCH_COUNT = 'custom_redirection/general/batch_count';

    public const REDIRECTION_FOLDER = 'redirection';

    public const URL_FILE_NAME = 'RedirectUrls.csv';

    public const CSV_SEPARATOR = ';';

    public const BATCH_NAME = 'batch_';

    public const MAIN_TABLE = 'custom_redirection_main';

    public const HISTORY_TABLE = 'custom_redirection_main_history';

    public const URL_REWRITE_TABLE = 'url_rewrite';

    /**
     * @var Logger
     */
    protected $logger;

    /**
     * @var ScopeInterface
     */
    protected $scopeConfig;

    /**
     * @var DirectoryList
     */
    protected $directoryList;

    /**
     * @var File
     */
    protected $fileDriver;

    /**
     * @var FileIo
     */
    protected $fileIo;

    /**
     * @var MessageQueue
     */
    protected $messageQueue;

    /**
     * @var ResourceConnection
     */
    protected $resource;

    /**
     * @var ResourceConnection
     */
    protected $connection;

    /**
     * Construct
     *
     * @param Context $context
     * @param Logger $logger
     * @param DirectoryList $directoryList
     * @param File $fileDriver
     * @param FileIo $fileIo
     * @param MessageQueue $messageQueue
     * @param ResourceConnection $resource
     */
    public function __construct(
        Context $context,
        Logger $logger,
        DirectoryList $directoryList,
        File $fileDriver,
        FileIo $fileIo,
        MessageQueue $messageQueue,
        ResourceConnection $resource
    ) {
        $this->logger = $logger;
        $this->scopeConfig = $context->getScopeConfig();
        $this->directoryList = $directoryList;
        $this->fileDriver = $fileDriver;
        $this->fileIo = $fileIo;
        $this->messageQueue = $messageQueue;
        $this->resource = $resource;
        $this->connection = $resource->getConnection(ResourceConnection::DEFAULT_CONNECTION);
        parent::__construct($context);
    }

    /**
     * Get configuration value
     *
     * @param string $field
     * @param int|null $storeId
     *
     * @return mixed
     */
    public function getConfigValue($field, $storeId = null)
    {
        return $this->scopeConfig->getValue(
            $field,
            ScopeInterface::SCOPE_STORE,
            $storeId
        );
    }

    /**
     * Get Module status
     *
     * @return int
     */
    public function getModuleStatus()
    {
        return (int) $this->getConfigValue(self::MODULE_STATUS);
    }

    /**
     * Get Module status
     *
     * @return int
     */
    public function getFileName()
    {
        return $this->getConfigValue(self::FILE_NAME);
    }

    /**
     * Get Module status
     *
     * @return int
     */
    public function getBatchCount()
    {
        return $this->getConfigValue(self::BATCH_COUNT);
    }

    /**
     * Get file path
     *
     * @param string $fileName
     *
     * @return string
     */
    public function getFilePath(string $fileName = self::URL_FILE_NAME, bool $batch = false)
    {
        return $this->getPath($batch) . ($this->getFileName() ?: self::URL_FILE_NAME);
    }

    /**
     * Get Import redirection and batch file path
     *
     * @param bool $batch
     *
     * @return string
     */
    public function getPath(bool $batch = false)
    {
        return $this->directoryList->getPath(DirectoryList::VAR_DIR) . '/import/' . self::REDIRECTION_FOLDER . '/' . ($batch ? 'batch/' : '');
    }

    /**
     * Get Media Path
     *
     * @param string $fileName
     *
     * @return string
     */
    public function getMediaPath($fileName)
    {
        $historyPath = $this->directoryList->getPath(DirectoryList::MEDIA) . '/CustomRedirection/';

        if (!$this->fileDriver->isDirectory($historyPath)) {
            $this->fileIo->mkdir($historyPath, 0775);
        }

        return $historyPath . $fileName;
    }

    /**
     * Process Batch File
     *
     * @param string $filePath
     * @param mixed $rawId
     *
     * @return array|false
     */
    public function processBatchFile(string $filePath, $rawId)
    {
        // Check module status
        if (!$this->getModuleStatus()) {
            return false;
        }

        if (!file_exists($filePath)) {
            return false;
        }

        try {
            // Process batch csv file
            if (($handle = fopen($filePath, 'r')) !== false) {
                $header = fgetcsv($handle, 0, self::CSV_SEPARATOR);

                if ($header !== false) {
                    while (($data = fgetcsv($handle, 0, self::CSV_SEPARATOR)) !== false) {
                        if (count($data)) {
                            $oldUrl = $data[0];
                            $newUrl = $data[1];
                            $storeId = $data[2];
                            $output = $this->processCustomUrl($oldUrl, $newUrl, $storeId);

                            $this->connection->insert(
                                $this->connection->getTableName(self::HISTORY_TABLE),
                                [
                                    'main_id' => $rawId,
                                    'status' => $output['status'],
                                    'store_id' => $storeId,
                                    'old_url' => $oldUrl,
                                    'new_url' => $newUrl,
                                    'message' => $output['msg'],
                                ]
                            );
                        }
                    }
                    $batchStatus = true;
                }

                fclose($handle);
            }
        } catch (\Exception $e) {
            $batchStatus = false;
            $this->logger->error('Error fetching URL rewrite: ' . $e->getMessage());
        }

        return $batchStatus;
    }

    /**
     * Process Custom URLs
     *
     * @param string $oldUrl
     * @param string $newUrl
     * @param int $storeId
     *
     * @return array
     */
    public function processCustomUrl($oldUrl, $newUrl, $storeId)
    {
        $output = ['msg' => 'Please try again. Something went wrong.', 'status' => false];

        try {
            $rewriteTable = $this->connection->getTableName(self::URL_REWRITE_TABLE);
            $select = $this->connection->select()
                        ->from($rewriteTable)
                        ->where('request_path = ?', $oldUrl)
                        ->where('store_id = ?', $storeId)
                        ->limit(1);
            $existingRedirect = $this->connection->fetchRow($select);

            if (!empty($existingRedirect)) {
                if ((strpos($existingRedirect['target_path'], 'catalog/product/view/id/') !== false)) {
                    return [
                        'msg' => "The product's main URL is already linked. Please verify the URL.",
                        'status' => false,
                    ];
                }
                // Update URL
                $this->connection->update(
                    $rewriteTable,
                    [
                        'entity_type' => 'custom',
                        'target_path' => $newUrl,
                        'is_autogenerated' => 0,
                        'redirect_type' => 301,
                        'description' => 'Custom Redirection : Updated New Custom URL',
                    ],
                    ['url_rewrite_id = ?' => $existingRedirect['url_rewrite_id']]
                );

                $output = [
                    'msg' => 'Custom Redirection : Updated New Custom URL',
                    'status' => true,
                ];
            } else {
                // New Entries
                $this->connection->insert(
                    $rewriteTable,
                    [
                        'entity_type' => 'custom',
                        'request_path' => $oldUrl,
                        'target_path' => $newUrl,
                        'redirect_type' => 301,
                        'store_id' => $storeId,
                        'description' => 'Custom Redirection : Added New Custom URL',
                        'is_autogenerated' => 0,
                    ]
                );

                $output = [
                    'msg' => 'Custom Redirection : Added New Custom URL',
                    'status' => true,
                ];
            }
        } catch (\Exception $e) {
            $this->logger->error('Error fetching URL rewrite: ' . $e->getMessage());
        }

        return $output;
    }

    /**
     * Create Lock File
     *
     * @return void
     */
    public function createLockFile()
    {
        try {
            $fileName = $this->getFilePath();
            $this->fileDriver->rename($fileName, $fileName . '.lock');
        } catch (\Exception $e) {
            $this->logger->error($e->getMessage());
        }
    }

    /**
     * Formatted DateTime With current timezone
     *
     * @param mixed $format
     * @param null|mixed $date
     *
     * @return string
     */
    public function formateDateWithoutTimeZone($format, $date = null)
    {
        $dt = $date ? \DateTime::createFromFormat(CustomRedirectionInterface::DEFAULT_DATETIME_FORMATE, $date) : new \DateTime();

        return $dt->format($format);
    }

    /**
     * Published message queue
     *
     * @param string $fileName
     * @param bool $id
     *
     * @return obj
     */
    public function publish($fileName, $id = false)
    {
        return $this->messageQueue->publish($fileName, $id);
    }

    /**
     * Get File driver Object
     *
     * @return File
     */
    public function getFileDriver()
    {
        return $this->fileDriver;
    }

    /**
     * Get File input and output object
     *
     * @return FileIo
     */
    public function getFileIo()
    {
        return $this->fileIo;
    }

    /**
     * Delete All lock file and batch file
     *
     * @return void
     */
    public function deleteAll()
    {
        try {
            $batchPath = $this->getPath(true);
            // Removed Batch file and folder
            if ($this->fileDriver->isExists($batchPath)) {
                $paths = $this->fileDriver->readDirectory($batchPath);
                foreach ($paths as $path) {
                    $this->fileDriver->deleteFile($path);
                }
                $this->fileDriver->deleteDirectory($batchPath);
            }

            // Removed Lock File
            $filePaths = glob($this->getPath() . '*.lock');
            foreach ($filePaths as $path) {
                $dateTime = $this->formateDateWithoutTimeZone(CustomRedirectionInterface::HISTORY_DATETIME_FORMATE);
                $lockFileName = basename($path);
                $errorFileName = "Error-{$dateTime}-{$lockFileName}";
                $errorFilePath = $this->getMediaPath($errorFileName);
                $this->fileDriver->rename($path, $errorFilePath);
            }
        } catch (\Exception $e) {
            $this->logger->critical($e->getMessage());
        }
    }

    /**
     * Close Database Rersource Connection
     *
     * @return string
     */
    public function closeResourceConnection()
    {
        $this->connection->closeConnection();
    }
}
