<?php
/*
 * web-vision GmbH
 * Do not edit or add to this file if you wish to upgrade Magento to newer
 * versions in the future. If you wish to customize Magento for your
 * needs please refer to https://www.web-vision.de for more information.
 * @category    WebVision
 * @copyright   Copyright (c) 2001-2022 web-vision GmbH (https://www.web-vision.de)
 * @author      Mahesh Makwana <mmakwana@web-vision.de>
 */
namespace WebVision\CustomRedirection\Helper;

use Magento\AsynchronousOperations\Api\Data\OperationInterface;
use Magento\AsynchronousOperations\Api\Data\OperationInterfaceFactory;
use Magento\Authorization\Model\UserContextInterface;
use Magento\Framework\App\Helper\AbstractHelper;
use Magento\Framework\App\Helper\Context;
use Magento\Framework\Bulk\BulkManagementInterface;
use Magento\Framework\DataObject\IdentityGeneratorInterface;
use Magento\Framework\Serialize\SerializerInterface;
use WebVision\CustomRedirection\Logger\Logger;

class MessageQueue extends AbstractHelper
{
    /**
     * @var UserContextInterface
     */
    protected $_userContext;

    /**
     * @var IdentityGeneratorInterface
     */
    protected $_identityService;

    /**
     * @var BulkManagementInterface
     */
    protected $_bulkManagement;

    /**
     * @var SerializerInterface
     */
    protected $_serializer;

    /**
     * @var OperationInterfaceFactory
     */
    protected $_operationFactory;

    /**
     * @var Logger
     */
    protected $logger;

    /**
     * @param Context $context
     * @param IdentityGeneratorInterface $identityService
     * @param BulkManagementInterface $bulkManagement
     * @param SerializerInterface $serializer
     * @param OperationInterfaceFactory $operationFactory
     * @param UserContextInterface $userContext
     */
    public function __construct(
        Context $context,
        IdentityGeneratorInterface $identityService,
        BulkManagementInterface $bulkManagement,
        SerializerInterface $serializer,
        OperationInterfaceFactory $operationFactory,
        UserContextInterface $userContext,
        Logger $logger
    ) {
        $this->_identityService = $identityService;
        $this->_bulkManagement = $bulkManagement;
        $this->_serializer = $serializer;
        $this->_operationFactory = $operationFactory;
        $this->_userContext = $userContext;
        $this->logger = $logger;
        parent::__construct($context);
    }

    /**
     * @param $processFileName
     * @param null|mixed $rawId
     *
     * @return void
     */
    public function publish($processFileName, $rawId = null): void
    {
        try {
            $bulkUuid = $this->_identityService->generateId();
            $fileName = basename($processFileName);
            $bulkDescription = __('Custom Redirection Processing File: ' . $fileName);

            $operations = [];
            $operations[] = $this->makeOperation(
                'Custom Redirection Processing File: ' . $fileName,
                'webvision_custom_redirection.process',
                $processFileName,
                $rawId,
                $bulkUuid,
            );

            if (!empty($operations)) {
                $this->_bulkManagement->scheduleBulk(
                    $bulkUuid,
                    $operations,
                    $bulkDescription,
                    $this->_userContext->getUserId()
                );
            }
        } catch (\Exception $e) {
            $this->logger->critical($e->getMessage());
        }
    }

    /**
     * @param $meta
     * @param $queue
     * @param mixed $processFileName
     * @param mixed $rawId
     * @param $bulkUuid
     *
     * @return OperationInterface
     */
    public function makeOperation(
        $meta,
        $queue,
        $processFileName,
        $rawId,
        $bulkUuid,
    ): OperationInterface {
        $dataToEncode = [
            'meta_information' => $meta,
            'filename' => $processFileName,
            'raw_id' => $rawId,
        ];

        $data = [
            'data' => [
                'bulk_uuid' => $bulkUuid,
                'topic_name' => $queue,
                'serialized_data' => $this->_serializer->serialize($dataToEncode),
                'status' => \Magento\Framework\Bulk\OperationInterface::STATUS_TYPE_OPEN,
            ],
        ];

        return $this->_operationFactory->create($data);
    }
}
