<?php
/*
 * web-vision GmbH
 * Do not edit or add to this file if you wish to upgrade Magento to newer
 * versions in the future. If you wish to customize Magento for your
 * needs please refer to https://www.web-vision.de for more information.
 * @category    WebVision
 * @copyright   Copyright (c) 2001-2022 web-vision GmbH (https://www.web-vision.de)
 * @author      Mahesh Makwana <mmakwana@web-vision.de>
 */
namespace WebVision\CustomRedirection\Model\Queue;

use Magento\AsynchronousOperations\Api\Data\OperationInterface;
use Magento\Framework\Bulk\OperationManagementInterface;
use Magento\Framework\EntityManager\EntityManager;
use Magento\Framework\MessageQueue\ConsumerConfiguration;
use Magento\Framework\Serialize\SerializerInterface;
use WebVision\CustomRedirection\Helper\Data as RedirectionHelper;
use WebVision\CustomRedirection\Logger\Logger;
use WebVision\CustomRedirection\Model\RedirectionManager;

class Consumer extends ConsumerConfiguration
{
    /**
     * @var Logger
     */
    protected $_logger;

    /**
     * @var SerializerInterface
     */
    protected $_serializer;

    /**
     * @var OperationManagementInterface
     */
    protected $_operationManagement;

    /**
     * @var EntityManager
     */
    protected $_entityManager;

    /**
     * @var RedirectionHelper
     */
    protected $redirectionHelper;

    /**
     * @var RedirectionManager
     */
    protected $redirectionManager;

    /**
     * Construct
     *
     * @param OperationManagementInterface $operationManagement
     * @param Logger $logger
     * @param SerializerInterface $serializer
     * @param EntityManager $entityManager
     * @param RedirectionHelper $redirectionHelper
     * @param RedirectionManager $redirectionManager
     */
    public function __construct(
        OperationManagementInterface $operationManagement,
        Logger $logger,
        SerializerInterface $serializer,
        EntityManager $entityManager,
        RedirectionHelper $redirectionHelper,
        RedirectionManager $redirectionManager
    ) {
        $this->_logger = $logger;
        $this->_serializer = $serializer;
        $this->_operationManagement = $operationManagement;
        $this->_entityManager = $entityManager;
        $this->redirectionHelper = $redirectionHelper;
        $this->redirectionManager = $redirectionManager;
    }

    /**
     * @param \Magento\AsynchronousOperations\Api\Data\OperationInterface $operation
     *
     * @throws \Exception
     *
     * @return void
     */
    public function process(OperationInterface $operation)
    {
        $serializedData = $operation->getSerializedData();
        $message = '';
        $errorCode = null;
        $status = OperationInterface::STATUS_TYPE_COMPLETE;
        $data = $this->_serializer->unserialize($serializedData);
        $fileDiver = $this->redirectionHelper->getFileDriver();

        // Check module status
        if (!$this->redirectionHelper->getModuleStatus()) {
            $operation->setStatus(OperationInterface::STATUS_TYPE_NOT_RETRIABLY_FAILED)
                ->setErrorCode($errorCode)
                ->setResultMessage($message);
            $this->_entityManager->save($operation);
        }

        try {
            $filename = basename($data['filename']);
            $rawId = $data['raw_id'];

            $mainFileName = $this->redirectionHelper->getFilePath();

            // Process main File and Create batches
            if (($filename == basename($mainFileName)) && (!$fileDiver->isExists($mainFileName . '.lock'))) {
                $this->redirectionManager->createBatches($rawId);
                $message = __($filename . ' File Process finished successfully.');
                $this->redirectionManager->updateProcessStatus($rawId, false);
            }

            // Process batch File
            if (str_starts_with($filename, 'batch_')) {
                $batchFile = $this->redirectionHelper->getPath(true) . $filename;
                if ($this->redirectionHelper->processBatchFile($batchFile, $rawId)) {
                    $message = __($filename . ' batch File Process finished successfully.');
                    // Delete file after finish processing
                    $fileDiver->deleteFile($batchFile);
                    $this->redirectionManager->updateProcessStatus($rawId, true);
                    // Close Database connection
                    $this->redirectionHelper->closeResourceConnection();
                } else {
                    $errorCode = 500;
                    $message = __($filename . ' Batch file processing has an issue..');
                }
            }
        } catch (\Exception $e) {
            $status = OperationInterface::STATUS_TYPE_NOT_RETRIABLY_FAILED;
            $errorCode = $e->getCode();
            $message = $e->getMessage();
            $this->_logger->critical($e->getMessage());
        }

        $operation->setStatus($status)
            ->setErrorCode($errorCode)
            ->setResultMessage($message);
        $this->_entityManager->save($operation);
    }
}
