<?php
/*
 * web-vision GmbH
 * Do not edit or add to this file if you wish to upgrade Magento to newer
 * versions in the future. If you wish to customize Magento for your
 * needs please refer to https://www.web-vision.de for more information.
 * @category    WebVision
 * @copyright   Copyright (c) 2001-2022 web-vision GmbH (https://www.web-vision.de)
 * @author      Mahesh Makwana <mmakwana@web-vision.de>
 */
namespace WebVision\CustomRedirection\Model;

use WebVision\CustomRedirection\Api\Data\CustomRedirectionInterface;
use WebVision\CustomRedirection\Helper\Data as RedirectionData;
use WebVision\CustomRedirection\Logger\Logger;

class RedirectionManager
{
    /**
     * @var Logger
     */
    protected $logger;

    /**
     * @var RedirectionData
     */
    protected $redirectionHelper;

    /**
     * @var CustomRedirectionFactory
     */
    protected $customRedirectionModel;

    /**
     * Constructor
     *
     * @param Logger $logger
     */
    public function __construct(
        Logger $logger,
        RedirectionData $redirectionHelper,
        CustomRedirectionFactory $customRedirectionModel
    ) {
        $this->logger = $logger;
        $this->redirectionHelper = $redirectionHelper;
        $this->customRedirectionModel = $customRedirectionModel;
    }

    /**
     * Process data
     *
     * @return void
     */
    public function process()
    {
        // Check module status
        if (!$this->redirectionHelper->getModuleStatus()) {
            return false;
        }

        // Create Batches
        $this->createBatches();
    }

    public function createBatches($rawId)
    {
        // Check module status
        if (!$this->redirectionHelper->getModuleStatus()) {
            return false;
        }

        try {
            $filePath = $this->redirectionHelper->getFilePath();
            $lockFileName = $filePath . '.lock';
            $totalBatches = $this->redirectionHelper->getBatchCount();
            // get File Operation objects
            $fileDriver = $this->redirectionHelper->getFileDriver();
            $fileIo = $this->redirectionHelper->getFileIo();

            // Check Process file is exist or not
            if ($fileDriver->isExists($lockFileName)) {
                $this->logger->info('Lock File Exists: ' . $lockFileName);

                return false;
            }

            // Check Process file is exist or not
            if (!$fileDriver->isExists($filePath)) {
                $this->logger->error('File Not Exists: ' . $filePath);

                return false;
            }

            // Create folder if not exit
            $batchFilePath = $this->redirectionHelper->getPath(true);
            if (!$fileDriver->isDirectory($batchFilePath)) {
                $fileIo->mkdir($batchFilePath, 0775);
            }

            // Open the main CSV file
            if (($mainFile = fopen($filePath, 'r')) === false) {
                $this->logger->error('Failed to open the file: ' . $filePath);

                return false;
            }

            // Read the header
            $header = fgetcsv($mainFile, 0, RedirectionData::CSV_SEPARATOR);

            // Count total rows
            $totalRows = 0;
            while (($row = fgetcsv($mainFile, 0, RedirectionData::CSV_SEPARATOR)) !== false) {
                $totalRows++;
            }

            // Count batch size
            $rowsPerBatch = ceil($totalRows / $totalBatches);

            // Reset the file pointer to the beginning of the file
            rewind($mainFile);

            // Skip the header
            fgetcsv($mainFile, 0, RedirectionData::CSV_SEPARATOR);

            // Create batches
            for ($batchIndex = 0; $batchIndex < $rowsPerBatch; $batchIndex++) {
                $batchFile = $batchFilePath . RedirectionData::BATCH_NAME . ($batchIndex + 1) . '.csv';

                $batchRows = [];
                // Prepare the batch data
                for ($i = 0; $i < $totalBatches; $i++) {
                    if (($row = fgetcsv($mainFile, 0, RedirectionData::CSV_SEPARATOR)) !== false) {
                        $batchRows[] = $row;
                    }
                }

                // Write to new CSV file
                if (($newBatchFile = fopen($batchFile, 'w')) !== false) {
                    fputcsv($newBatchFile, $header, RedirectionData::CSV_SEPARATOR); // Write the header to each batch file
                    foreach ($batchRows as $batchRow) {
                        fputcsv($newBatchFile, $batchRow, RedirectionData::CSV_SEPARATOR); // Keep the same delimiter (;)
                    }
                    fclose($newBatchFile);

                    // Publish batch File Message queue
                    $this->redirectionHelper->publish($batchFile, $rawId);
                } else {
                    $this->logger->error('Failed to create batch file: ' . $batchFile);
                }
            }

            fclose($mainFile);

            // create lock file
            $this->redirectionHelper->createLockFile();
        } catch (\Exception $e) {
            $this->logProcessError($rawId, $e->getMessage());
            $this->logger->critical($e->getMessage());
        }

        return true;
    }

    public function updateProcessStatus($id, $batch = false)
    {
        try {
            // Check Record ID
            $redirectionModel = $this->customRedirectionModel->create()->load($id);
            if (!$redirectionModel->getId()) {
                $message = 'Record ID : ' . $id . ' does not exist in the system. Please verify the records.';
                $this->logProcessError($id, $message);
                $this->logger->critical($message);

                return false;
            }

            // Check status is already set false
            if ($redirectionModel->getStatus() == CustomRedirectionInterface::PROCESS_FAIELD) {
                $this->logger->critical($message);

                return false;
            }

            $fileDriver = $this->redirectionHelper->getFileDriver();
            // Update Status
            if ($batch) {
                // Update status for Batch File
                $batchPath = $this->redirectionHelper->getPath(true);
                $paths = $fileDriver->readDirectory($batchPath);

                // Check batch files
                if (empty($paths)) {
                    // delete batch folder
                    $fileDriver->deleteDirectory($batchPath);

                    // Update record status
                    $redirectionModel->setStatus(CustomRedirectionInterface::PROCESS_COMPLETED);

                    // Move file to new location
                    $mainFile = $this->redirectionHelper->getFilePath();
                    $historyFilePath = $this->redirectionHelper->getMediaPath($id . '-' . basename($mainFile));
                    $fileDriver->rename($mainFile . '.lock', $historyFilePath);
                }
            } else {
                // Update status for main File
                $redirectionModel->setStatus(CustomRedirectionInterface::PROCESS_RUNNING);
            }
            $redirectionModel->save();

            return true;
        } catch (\Exception $e) {
            $this->logProcessError($id, $e->getMessage());
            $this->logger->critical($e->getMessage());
        }

        return false;
    }

    public function logProcessError($id, $message)
    {
        try {
            $redirectionModel = $this->customRedirectionModel->create()->load($id);
            if (!$redirectionModel->getId()) {
                $this->logger->critical('Record ID : ' . $id . ' does not exist in the system. Please verify the records.');

                return false;
            }

            $redirectionModel->setStatus(CustomRedirectionInterface::PROCESS_FAIELD);
            $redirectionModel->setMessage($message);
            $redirectionModel->save();
            $this->redirectionHelper->deleteAll();
        } catch (\Exception $e) {
            $this->logger->critical($e->getMessage());
        }
    }
}
