<?php
/*
 * web-vision GmbH
 * Do not edit or add to this file if you wish to upgrade Magento to newer
 * versions in the future. If you wish to customize Magento for your
 * needs please refer to https://www.web-vision.de for more information.
 * @category    Abeco
 * @copyright   Copyright (c) 2001-2023 web-vision GmbH (https://www.web-vision.de)
 * @author      Parth Trivedi <parth@web-vision.de>
 */
namespace WebVision\Redirection\Console\Command;

use Magento\Framework\App\Filesystem\DirectoryList;
use Magento\Framework\App\State;
use Magento\Framework\Console\Cli;
use Magento\Framework\File\Csv;
use Magento\UrlRewrite\Model\UrlRewriteFactory;
use Symfony\Component\Console\Command\Command;
use Symfony\Component\Console\Input\InputArgument;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Input\InputOption;
use Symfony\Component\Console\Output\OutputInterface;

class AddUrlRewrite extends Command
{
    const NAME_ARGUMENT = 'name';

    const NAME_OPTION = 'option';

    protected const CSV_FILE_PATH = '/var/Redirections.csv';

    /**
     * @var UrlRewriteFactory
     */
    private $urlRewriteFactory;
    /**
     * @var Csv
     */
    private $csvProcessor;
    /**
     * @var DirectoryList
     */
    private $directoryList;
    /**
     * @var State
     */
    private $appState;

    /**
     * @param UrlRewriteFactory $urlRewriteFactory
     * @param Csv $csvProcessor
     * @param DirectoryList $directoryList
     * @param State $appState
     * @param string|null $name
     */
    public function __construct(
        UrlRewriteFactory $urlRewriteFactory,
        Csv $csvProcessor,
        DirectoryList $directoryList,
        State $appState,
        string $name = null
    ) {
        parent::__construct($name);
        $this->urlRewriteFactory = $urlRewriteFactory;
        $this->csvProcessor = $csvProcessor;
        $this->directoryList = $directoryList;
        $this->appState = $appState;
    }

    /**
     * @return void
     */
    protected function configure()
    {
        $this->setName('webvision:redirection');
        $this->setDescription('redirection add');
        $this->setDefinition([
            new InputArgument(self::NAME_ARGUMENT, InputArgument::OPTIONAL, 'Name'),
            new InputOption(self::NAME_OPTION, '-a', InputOption::VALUE_NONE, 'Option functionality'),
        ]);
        parent::configure();
    }

    /**
     * @param InputInterface $input
     * @param OutputInterface $output
     *
     * @throws \Magento\Framework\Exception\LocalizedException
     *
     * @return int
     */
    protected function execute(InputInterface $input, OutputInterface $output)
    {
        try {
            $this->appState->setAreaCode(\Magento\Framework\App\Area::AREA_GLOBAL);
        } catch (\Magento\Framework\Exception\LocalizedException $e) {
            $this->logger->info('Area code is already set. Proceeding without resetting.');
        }

        $name = $input->getArgument(self::NAME_ARGUMENT);
        $option = $input->getOption(self::NAME_OPTION);

        $filePath = $this->directoryList->getRoot() . self::CSV_FILE_PATH;

        if (!file_exists($filePath)) {
            $output->writeln('<error>CSV file not found at ' . $filePath . '</error>');

            return Cli::RETURN_FAILURE;
        }

        $rows = $this->csvProcessor->getData($filePath);
        $rows = array_slice($rows, 1); // Skip the header row

        $errors = [];
        foreach ($rows as $row) {
            if (count($row) < 2) {
                $errors[] = 'Invalid row: ' . implode(';', $row);

                continue;
            }

            list($requestPath, $targetPath, $storeId) = $row;

            try {
                $urlRewrite = $this->urlRewriteFactory->create();
                $urlRewrite->setRequestPath(parse_url($requestPath, PHP_URL_PATH))
                    ->setTargetPath(parse_url($targetPath, PHP_URL_PATH))
                    ->setRedirectType(301)
                    ->setStoreId($storeId)
                    ->setIsSystem(0)
                    ->setDescription('Custom URL rewrite created by CLI command')
                    ->save();
            } catch (\Exception $e) {
                $errors[] = 'Failed to add URL rewrite for ' . $requestPath . ' to ' . $targetPath . ': ' . $e->getMessage();
            }
        }

        if (empty($errors)) {
            $output->writeln('<info>URL rewrites added successfully.</info>');

            return Cli::RETURN_SUCCESS;
        }
        foreach ($errors as $error) {
            $output->writeln('<error>' . $error . '</error>');
        }

        return Cli::RETURN_FAILURE;

    }
}
