<?php
/*
 * web-vision GmbH
 * Do not edit or add to this file if you wish to upgrade Magento to newer
 * versions in the future. If you wish to customize Magento for your
 * needs please refer to https://www.web-vision.de for more information.
 * @category    WebVision
 * @copyright   Copyright (c) 2001-2022 web-vision GmbH (https://www.web-vision.de)
 * @author      Mahesh Makwana <mmakwana@web-vision.de>
 */
namespace WebVision\UnityElasticSearch\Helper;

use Magento\CatalogUrlRewrite\Model\CategoryUrlPathGenerator;
use Magento\Store\Model\ScopeInterface;
use Smile\ElasticsuiteCore\Helper\AbstractConfiguration;

class Data extends AbstractConfiguration
{
    const SORTING_DATA = 'desc';

    const CUSTOM_TRIGGER_POST_FIX = 'magento';

    const DEFAULT_CONNECTION_NAME = 'default';

    const TYPO3_SEARCH_STATUS = 'typo3_base_setting/enable_typo3_search';

    const TYPO3_CONNECTION_NAME = 'typo3_base_setting/db_connection_name';

    const TYPO3_TABLE_NAME = 'typo3_base_setting/typo3_table_name';

    const TYPO3_PRIMARY_KEY = 'typo3_base_setting/typo3_table_primary_key';

    const TYPO3_TITLE_COLUMN = 'typo3_base_setting/typo3_table_title_column';

    const TYPO3_URL_COLUMN = 'typo3_base_setting/typo3_table_url_column';

    const TYPO3_STORE_FILTER_STATUS = 'typo3_base_setting/enable_typo3_store_filter';

    const TYPO3_STORE_ID_COLUMN = 'typo3_base_setting/typo3_table_store_filter_column';

    const TYPO3_STORE_ID_DATA = 'typo3_base_setting/typo3_table_store_filter_data';

    const TYPO3_FILTER_DATA = 'typo3_base_setting/filter_data';

    const TYPO3_SHOW_SEARCH_PAGE = 'smile_elasticsuite_catalogsearch_settings/catalogsearch/result_page_show_typo3_page';

    const SHOW_SEACHABLE_CATEGORIES = 'smile_elasticsuite_catalogsearch_settings/catalogsearch/result_page_show_category';

    const ENV_DEFAULT_DB = 'db/connection/default/dbname';

    /**
     * @var \Smile\ElasticsuiteCore\Client\Client
     */
    public $elasticsearchClient;

    /**
     * @var \Smile\ElasticsuiteCore\Search\Request\Builder
     */
    public $requestBuilder;

    /**
     * @var ContainerConfigurationFactory
     */
    public $containerConfigFactory;

    /**
     * @var \Magento\Framework\App\Config\ScopeConfigInterface
     */
    protected $scopeConfig;

    /**
     * @var \Magento\Framework\App\Config\ScopeConfigInterface
     */
    protected $json;

    /**
     * @var \Magento\UrlRewrite\Model\UrlRewriteFactory
     */
    protected $urlRewriteData;

    /**
     * @var \Magento\Framework\App\DeploymentConfig
     */
    protected $deploymentConfig;

    /**
     * @var \Psr\Log\LoggerInterface
     */
    protected $logger;

    /**
     * @param \Magento\Framework\App\Helper\Context $context
     * @param \Magento\Store\Model\StoreManagerInterface $storeManager
     * @param \Smile\ElasticsuiteCore\Search\Request\Builder $requestBuilder
     * @param \Smile\ElasticsuiteCore\Search\Request\ContainerConfigurationFactory $containerConfigFactory
     * @param \Smile\ElasticsuiteCore\Client\Client $elasticsearchClient
     * @param \Magento\Framework\App\Config\ScopeConfigInterface $scopeConfig
     * @param \Magento\Framework\Serialize\Serializer\Json $json
     * @param \Magento\UrlRewrite\Model\UrlRewriteFactory $urlRewriteData
     * @param \Magento\Framework\App\DeploymentConfig $deploymentConfig
     * @param \Psr\Log\LoggerInterface $logger
     */
    public function __construct(
        \Magento\Framework\App\Helper\Context $context,
        \Magento\Store\Model\StoreManagerInterface $storeManager,
        \Smile\ElasticsuiteCore\Search\Request\Builder $requestBuilder,
        \Smile\ElasticsuiteCore\Search\Request\ContainerConfigurationFactory $containerConfigFactory,
        \Smile\ElasticsuiteCore\Client\Client $elasticsearchClient,
        \Magento\Framework\App\Config\ScopeConfigInterface $scopeConfig,
        \Magento\Framework\Serialize\Serializer\Json $json,
        \Magento\UrlRewrite\Model\UrlRewriteFactory $urlRewriteData,
        \Magento\Framework\App\DeploymentConfig $deploymentConfig,
        \Psr\Log\LoggerInterface $logger
    ) {
        $this->elasticsearchClient = $elasticsearchClient;
        $this->containerConfigFactory = $containerConfigFactory;
        $this->requestBuilder = $requestBuilder;
        $this->scopeConfig = $scopeConfig;
        $this->storeManager = $storeManager;
        $this->json = $json;
        $this->urlRewriteData = $urlRewriteData;
        $this->deploymentConfig = $deploymentConfig;
        $this->logger = $logger;
        parent::__construct($context, $storeManager);
    }

    /**
     * Get typo3 search status
     *
     * @return string
     */
    public function getTypo3SearchStatus()
    {
        return $this->getElasticSuiteConfigParam(self::TYPO3_SEARCH_STATUS);
    }

    /**
     * Get typo3 connection name
     *
     * @return string
     */
    public function getTypo3ConnectionName()
    {
        return trim($this->getElasticSuiteConfigParam(self::TYPO3_CONNECTION_NAME));
    }

    /**
     * Get typo3 Primary key name
     *
     * @return string
     */
    public function getTypo3PrimaryKey()
    {
        return trim($this->getElasticSuiteConfigParam(self::TYPO3_PRIMARY_KEY));
    }

    /**
     * Get typo3 Table name
     *
     * @return string
     */
    public function getTypo3TableName()
    {
        return trim($this->getElasticSuiteConfigParam(self::TYPO3_TABLE_NAME));
    }

    /**
     * Get typo3 Title column name
     *
     * @return string
     */
    public function getTypo3TitleColumn()
    {
        return trim($this->getElasticSuiteConfigParam(self::TYPO3_TITLE_COLUMN));
    }

    /**
     * Get typo3 URL column name
     *
     * @return string
     */
    public function getTypo3UrlColumn()
    {
        return trim($this->getElasticSuiteConfigParam(self::TYPO3_URL_COLUMN));
    }

    /**
     * Get typo3 search page status
     *
     * @return string
     */
    public function getTypo3SearchPageStatus()
    {
        return $this->scopeConfig->getValue(self::TYPO3_SHOW_SEARCH_PAGE, ScopeInterface::SCOPE_STORE);
    }

    /**
     * Get searchable category status
     *
     * @return string
     */
    public function getSeachableCategoryStatus()
    {
        return $this->scopeConfig->getValue(self::SHOW_SEACHABLE_CATEGORIES, ScopeInterface::SCOPE_STORE);
    }

    /**
     * Get typo3 store filter status
     *
     * @return string
     */
    public function getTypo3StoreFilterStatus()
    {
        return trim($this->getElasticSuiteConfigParam(self::TYPO3_STORE_FILTER_STATUS));
    }

    /**
     * Get typo3 store filter column
     *
     * @return string
     */
    public function getTypo3StoreFilterColumn()
    {
        return trim($this->getElasticSuiteConfigParam(self::TYPO3_STORE_ID_COLUMN));
    }

    /**
     * Get typo3 store filter Data
     *
     * @param mixed $storeId
     *
     * @return string
     */
    public function getTypo3StoreFilterData($storeId = 0)
    {
        $storeFilterData = $this->json->unserialize(
            trim($this->getElasticSuiteConfigParam(self::TYPO3_STORE_ID_DATA)) ?? ''
        );

        foreach ($storeFilterData as $filterKey => $filterVal) {
            if ($filterVal['attr_store'] == $storeId) {
                $storeId = $filterVal['attr_id'];

                break;
            }
        }

        return $storeId;
    }

    /**
     * Get category url suffix
     *
     * @return string
     */
    public function getCategoryUrlSuffix()
    {
        return $this->scopeConfig->getValue(CategoryUrlPathGenerator::XML_PATH_CATEGORY_URL_SUFFIX, ScopeInterface::SCOPE_STORE);
    }

    /**
     * Get typo3 filter data
     *
     * @return array
     */
    public function getTypo3FilterData()
    {
        $filterColl = [];
        $filterData = $this->json->unserialize(
            $this->getElasticSuiteConfigParam(self::TYPO3_FILTER_DATA) ?? ''
        );

        foreach ($filterData as $filterKey => $filterVal) {
            if (str_contains($filterVal['attr_val'], ',')) {
                $filterValue = array_unique(explode(',', $filterVal['attr_val']));
            } else {
                $filterValue = $filterVal['attr_val'];
            }
            $filterColl[] = [
                'attr_code' => $filterVal['attr_code'],
                'attr_val' => $filterValue,
                'attr_condition' => $filterVal['attr_condition'],
            ];
        }

        return $filterColl;
    }

    /**
     * Get Page Ids
     *
     * @param string $searchRequestName
     * @param int $storeId
     * @param int $size
     * @param int $from
     * @param string $searchTerms
     *
     * @return array
     */
    public function getTypo3ElasticsuiteData($searchRequestName, $storeId, $size, $from, $searchTerms)
    {
        try{
            $params = [];
            $containerConfig = $this->getRequestContainerConfiguration($storeId, $searchRequestName);
            $params['index'] = $containerConfig->getIndexName();
            $params['body'] = $this->getTypo3EsBody($size, $from, $searchTerms);
            $typo3EsData = $this->elasticsearchClient->search($params);

            return $this->getPageIds($typo3EsData);
        } catch (\Exception $e) {
            $this->logger->critical($e->getMessage());
        }

        return [];
    }

    /**
     * Get Page Ids
     *
     * @param array $typo3EsData
     *
     * @return array
     */
    public function getPageIds($typo3EsData)
    {
        $pageIds = [];
        $pageIds[] = 0;
        if (isset($typo3EsData['hits']['hits'])) {
            $esHits = $typo3EsData['hits']['hits'];
            $mappedPageIds = array_map([$this, 'callableMappingPageIds'], $esHits);
        }

        return count($mappedPageIds) ? $mappedPageIds : $pageIds;
    }

    /**
     * Callable Mapping Page Ids
     *
     * @param array $mapValue
     *
     * @return array
     */
    public function callableMappingPageIds($mapValue)
    {
        return isset($mapValue['_id']) ? $mapValue['_id'] : 0;
    }

    /**
     * Get typo3 pages ES body.
     *
     * @param int $searchTerms
     * @param int $from
     * @param string $searchTerms
     * @param mixed $size
     *
     * @return array
     */
    public function getTypo3EsBody($size, $from, $searchTerms)
    {
        $pagesEsBody = [];
        $pagesEsBody['size'] = $size;
        $pagesEsBody['from'] = $from;
        $pagesEsBody['query']['bool']['should'] = $this->getTypo3Query($searchTerms);
        $pagesEsBody['query']['bool']['filter'] = ['term' => ['store_id' => $this->getStoreId()]];
        $pagesEsBody['sort'] = [['_score' => ['order' => 'desc']]];
        $pagesEsBody['min_score'] = 0.1;

        return $pagesEsBody;
    }

    /**
     * Get sorting
     *
     * @return array
     */
    public function getSorting()
    {
        return ['order' => self::SORTING_DATA];
    }

    /**
     * Get typo3 query.
     *
     * @param string $searchTerms
     *
     * @return array
     */
    public function getTypo3Query($searchTerms)
    {
        return [
            [
                'match_phrase' => [
                    'title.standard' => [
                        'query' => $searchTerms,
                        'boost' => 15,
                    ],
                ],
            ],
            [
                'regexp' => [
                    'title.untouched' => [
                        'value' => '.*'.$searchTerms.'.*',
                        "case_insensitive" => true,
                        'boost' => 15,
                    ],
                ],
            ],
            [
                'match_phrase' => [
                    'bodytext.standard' => [
                        'query' => $searchTerms,
                        'boost' => 20,
                    ],
                ],
            ],
            [
                'regexp' => [
                    'bodytext.untouched' => [
                        'value' => '.*'.$searchTerms.'.*',
                        "case_insensitive" => true,
                        'boost' => 20,
                    ],
                ],
            ],
        ];
    }

    /**
     * Get current store id
     *
     * @return int
     */
    public function getStoreId()
    {
        return $this->storeManager->getStore()->getId();
    }

    /**
     * Load the search request configuration (index, type, mapping, ...) using the search request container name.
     *
     *
     * @param int $storeId       Store id.
     * @param string  $containerName Search request container name.
     *
     * @throws \LogicException Thrown when the search container is not found into the configuration.
     *
     * @return ContainerConfigurationInterface
     */
    private function getRequestContainerConfiguration($storeId, $containerName)
    {
        if ($containerName === null) {
            throw new \LogicException('Request name is not set');
        }

        $config = $this->containerConfigFactory->create(
            ['containerName' => $containerName, 'storeId' => $storeId]
        );

        if ($config === null) {
            throw new \LogicException("No configuration exists for request {$containerName}");
        }

        return $config;
    }

    /**
     * Remove conflict typoe page url
     *
     * @param array $collection
     *
     * @return int
     */
    public function removeConflicPages($collection)
    {
        $skipRow = 1;
        if (isset($collection[$this->getTypo3UrlColumn()])) {
            $typo3Url = ltrim($collection[$this->getTypo3UrlColumn()], '/');
            if ($typo3Url) {
                $magentoUrl = $this->urlRewriteData->create()->getCollection();
                $magentoUrl->addFieldToFilter('entity_type', ['neq' => 'product']);
                $magentoUrl->addFieldToFilter('entity_type', ['neq' => 'cms-page']);
                $magentoUrl->addFieldToFilter('request_path', ['like' => '%' . $typo3Url . '%']);
                $pageResult = array_filter(array_map([$this, 'typo3MappingPageUrl'], $magentoUrl->getData(), [$typo3Url]));
                if (count($pageResult)) {
                    $skipRow = 0;
                }
            }
        }

        return $skipRow;
    }

    /**
     * Checked conflict typoe page url
     *
     * @param mixed $pageData
     * @param mixed $typo3Url
     *
     * @return int|null
     */
    public function typo3MappingPageUrl($pageData, $typo3Url)
    {
        if (isset($pageData['request_path'])) {
            $typo3Url = $typo3Url . '' . $this->getCategoryUrlSuffix();
            if ($typo3Url == $pageData['request_path']) {
                return 1;
            }
        }
    }

    /**
     * Get typo3 connection name form env file
     *
     * @param string $connectionName
     *
     * @return string
     */
    public function getTypo3EnvConnectionName($connectionName)
    {
        return str_replace(self::DEFAULT_CONNECTION_NAME, $connectionName, self::ENV_DEFAULT_DB);
    }

    /**
     * Get typo3 DB name form env file
     *
     * @param string $connectionName
     *
     * @return string
     */
    public function getTypo3EnvDBName()
    {
        return $this->deploymentConfig->get(
            $this->getTypo3EnvConnectionName(
                $this->getTypo3ConnectionName()
            )
        );
    }
}
