<?php
/*
 * web-vision GmbH
 * Do not edit or add to this file if you wish to upgrade Magento to newer
 * versions in the future. If you wish to customize Magento for your
 * needs please refer to https://www.web-vision.de for more information.
 * @category    WebVision
 * @copyright   Copyright (c) 2001-2022 web-vision GmbH (https://www.web-vision.de)
 * @author      Mahesh Makwana <mmakwana@web-vision.de>
 */
namespace WebVision\UnityElasticSearch\Model\Autocomplete\Typo3;

use Magento\Search\Model\Autocomplete\DataProviderInterface;
use Magento\Search\Model\QueryFactory;
use Smile\ElasticsuiteCatalog\Helper\Autocomplete as ConfigurationHelper;
use Smile\ElasticsuiteCore\Model\Autocomplete\Terms\DataProvider as TermDataProvider;
use WebVision\UnityElasticSearch\Helper\Data as Typo3Helper;
use WebVision\UnityElasticSearch\Model\ResourceModel\Typo3\Fulltext\CollectionFactory as Typo3CollectionFactory;

class DataProvider implements DataProviderInterface
{
    /**
     * Autocomplete type
     */
    const AUTOCOMPLETE_TYPE = 'typo3';

    /**
     * Autocomplete result item factory
     *
     * @var ItemFactory
     */
    protected $itemFactory;

    /**
     * Query factory
     *
     * @var QueryFactory
     */
    protected $queryFactory;

    /**
     * @var TermDataProvider
     */
    protected $termDataProvider;

    /**
     * @var Typo3CollectionFactory
     */
    protected $typo3CollectionFactory;

    /**
     * @var ConfigurationHelper
     */
    protected $configurationHelper;

    /**
     * @var Typo3Helper
     */
    protected $typo3Helper;

    /**
     * @var string Autocomplete result type
     */
    private $type;

    /**
     * Constructor.
     *
     * @param ItemFactory               $itemFactory               Suggest item factory.
     * @param QueryFactory              $queryFactory              Search query factory.
     * @param TermDataProvider          $termDataProvider          Search terms suggester.
     * @param Typo3CollectionFactory    $typo3CollectionFactory    Typo3 collection factory.
     * @param ConfigurationHelper       $configurationHelper       Autocomplete configuration helper.
     * @param string                    $type                      Autocomplete provider type.
     * @param Typo3Helper               $typo3Helper
     */
    public function __construct(
        ItemFactory $itemFactory,
        QueryFactory $queryFactory,
        TermDataProvider $termDataProvider,
        Typo3CollectionFactory $typo3CollectionFactory,
        ConfigurationHelper $configurationHelper,
        Typo3Helper $typo3Helper,
        $type = self::AUTOCOMPLETE_TYPE
    ) {
        $this->itemFactory = $itemFactory;
        $this->queryFactory = $queryFactory;
        $this->termDataProvider = $termDataProvider;
        $this->typo3CollectionFactory = $typo3CollectionFactory;
        $this->configurationHelper = $configurationHelper;
        $this->typo3Helper = $typo3Helper;
        $this->type = $type;
    }

    /**
     * @return string
     */
    public function getType()
    {
        return $this->type;
    }

    /**
     * {@inheritDoc}
     */
    public function getItems()
    {
        $result = [];

        $typo3Status = $this->typo3Helper->getTypo3SearchStatus();
        $autoSuggestionStatus = $this->configurationHelper->isEnabled($this->getType());

        if (($typo3Status) && ($autoSuggestionStatus)) {
            $typo3Collection = $this->getTypo3Collection();
            if ($typo3Collection) {
                foreach ($typo3Collection as $typo3) {
                    $result[] = $this->itemFactory->create([
                        'typo3' => $typo3,
                        'type' => $this->getType(),
                    ]);
                }
            }
        }

        return $result;
    }

    /**
     * List of search terms suggested by the search terms data provider.
     *
     * @return array
     */
    private function getSuggestedTerms()
    {
        $terms = array_map(
            function (\Magento\Search\Model\Autocomplete\Item $termItem) {
                return $termItem->getTitle();
            },
            $this->termDataProvider->getItems()
        );

        return $terms;
    }

    /**
     * Suggested typo3 page collection.
     * Returns null if no suggested search terms.
     *
     * @param bool $size
     *
     * @throws \Magento\Framework\Exception\LocalizedException
     *
     * @return \WebVision\UnityElasticSearch\Model\ResourceModel\Typo3\Fulltext\Collection|null
     */
    public function getTypo3Collection($size = true)
    {
        $typo3Collection = null;
        $terms = $this->queryFactory->get()->getQueryText();
        if ($terms) {
            $typo3Collection = $this->typo3CollectionFactory->create();
            if($size){
                $typo3Collection->setPageSize($this->getResultsPageSize());
            }
            $typo3Collection->setPageSearchTearm($this->queryFactory->get()->getQueryText());
            $typo3Collection->isSpellchecked(true);
        }

        return $typo3Collection;
    }

    /**
     * Retrieve number of typo3 pages to display in autocomplete results
     *
     * @return int
     */
    private function getResultsPageSize()
    {
        return $this->configurationHelper->getMaxSize($this->getType());
    }
}
