<?php
/*
 * web-vision GmbH
 * Do not edit or add to this file if you wish to upgrade Magento to newer
 * versions in the future. If you wish to customize Magento for your
 * needs please refer to https://www.web-vision.de for more information.
 * @category    WebVision
 * @copyright   Copyright (c) 2001-2022 web-vision GmbH (https://www.web-vision.de)
 * @author      Mahesh Makwana <mmakwana@web-vision.de>
 */
namespace WebVision\UnityElasticSearch\Model\ResourceModel\Typo3\Fulltext;

use Smile\ElasticsuiteCore\Search\Adapter\Elasticsuite\Response\QueryResponse;
use Smile\ElasticsuiteCore\Search\Request\QueryInterface;
use WebVision\UnityElasticSearch\Model\ResourceModel\Pages\Collection as PageCollection;

class Collection extends PageCollection
{
    /**
     * @var \Smile\ElasticsuiteCore\Search\Request\Builder
     */
    private $requestBuilder;

    /**
     * @var \Magento\Search\Model\SearchEngine
     */
    private $searchEngine;

    /**
     * @var string
     */
    protected $searchRequestName;

    /**
     * @var array
     */
    private $filters = [];

    /**
     * @var QueryInterface[]
     */
    private $queryFilters = [];

    /**
     * @var QueryResponse
     */
    private $queryResponse;

    /**
     * @var string|QueryInterface
     */
    private $query;

    /**
     * @var string
     */
    private $searchTerms;

    /**
     * @var ResourceConnection
     */
    private $resourceConnection;

    /**
     * @var Builder
     */
    private $mysqlQueryBuilder;

    /**
     * @var \Elasticsearch\Client
     */
    private $elasticsearchClient;

    /**
     * @var \WebVision\UnityElasticSearch\Helper\Data
     */
    private $webVisionEsHelper;

    /**
     * @var bool
     */
    private $isSpellchecked = false;

    /**
     * @var bool
     */
    private $typo3StoreId = null;

    /**
     * Collection constructor.
     *
     * @param \Magento\Framework\Data\Collection\EntityFactory $entityFactory
     * @param \Psr\Log\LoggerInterface $logger
     * @param \Magento\Framework\Data\Collection\Db\FetchStrategyInterface $fetchStrategy
     * @param \Magento\Framework\Event\ManagerInterface $eventManager
     * @param \Magento\Eav\Model\Config $eavConfig
     * @param \Magento\Framework\App\ResourceConnection $resource
     * @param \Magento\Eav\Model\EntityFactory $eavEntityFactory
     * @param \Magento\Eav\Model\ResourceModel\Helper $resourceHelper
     * @param \Magento\Framework\Validator\UniversalFactory $universalFactory
     * @param \Smile\ElasticsuiteCore\Search\Request\Builder $requestBuilder
     * @param \Magento\Search\Model\SearchEngine $searchEngine
     * @param \WebVision\UnityElasticSearch\Helper\Data $webVisionEsHelper
     * @param \Magento\Framework\DB\Adapter\AdapterInterface $connection
     * @param string $searchRequestName
     */
    public function __construct(
        \Magento\Framework\Data\Collection\EntityFactory $entityFactory,
        \Psr\Log\LoggerInterface $logger,
        \Magento\Framework\Data\Collection\Db\FetchStrategyInterface $fetchStrategy,
        \Magento\Framework\Event\ManagerInterface $eventManager,
        \Magento\Eav\Model\Config $eavConfig,
        \Magento\Framework\App\ResourceConnection $resource,
        \Magento\Eav\Model\EntityFactory $eavEntityFactory,
        \Magento\Eav\Model\ResourceModel\Helper $resourceHelper,
        \Magento\Framework\Validator\UniversalFactory $universalFactory,
        \Smile\ElasticsuiteCore\Search\Request\Builder $requestBuilder,
        \Magento\Search\Model\SearchEngine $searchEngine,
        \WebVision\UnityElasticSearch\Helper\Data $webVisionEsHelper,
        \Magento\Framework\DB\Adapter\AdapterInterface $connection = null,
        $searchRequestName = 'typo3_search_container'
    ) {
        parent::__construct(
            $entityFactory,
            $logger,
            $fetchStrategy,
            $eventManager,
            $connection
        );
        $this->requestBuilder = $requestBuilder;
        $this->searchEngine = $searchEngine;
        $this->searchRequestName = $searchRequestName;
        $this->webVisionEsHelper = $webVisionEsHelper;
    }

    /**
     * {@inheritDoc}
     */
    public function getSize()
    {
        if ($this->_totalRecords === null) {
            $this->_totalRecords = 0;
        }

        return $this->_totalRecords;
    }

    /**
     * {@inheritDoc}
     */
    public function setOrder($attribute, $dir = self::SORT_ORDER_DESC)
    {
        $this->_orders[$attribute] = $dir;

        return $this;
    }

    /**
     * {@inheritDoc}
     */
    public function addFieldToFilter($field, $condition = null)
    {
        $this->filters[$field] = $condition;

        return $this;
    }

    /**
     * {@inheritDoc}
     */
    public function addAttributeToSort($attribute, $dir = self::SORT_ORDER_ASC)
    {
        return $this->setOrder($attribute, $dir);
    }

    /**
     * Append a prebuilt (QueryInterface) query filter to the collection.
     *
     * @param QueryInterface $queryFilter Query filter.
     *
     * @return $this
     */
    public function addQueryFilter(QueryInterface $queryFilter)
    {
        $this->queryFilters[] = $queryFilter;

        return $this;
    }

    /**
     * Set search query filter in the collection.
     *
     * @param string|QueryInterface $query Search query text.
     *
     * @return \Smile\ElasticsuiteCatalog\Model\ResourceModel\Product\Fulltext\Collection
     */
    public function setSearchQuery($query)
    {
        $this->query = $query;

        return $this;
    }

    /**
     * Set search query filter for typo3 pages collection.
     *
     * @param string|QueryInterface $query Search query text.
     * @param mixed $searchData
     *
     * @return \Smile\ElasticsuiteCatalog\Model\ResourceModel\Product\Fulltext\Collection
     */
    public function setPageSearchTearm($searchData)
    {
        $this->searchTerms = $searchData;

        return $this;
    }

    /**
     * Add search query filter.
     *
     * @deprecated Replaced by setSearchQuery
     *
     * @param string $query Search query text.
     *
     * @return \Smile\ElasticsuiteCatalog\Model\ResourceModel\Product\Fulltext\Collection
     */
    public function addSearchFilter($query)
    {
        return $this->setSearchQuery($query);
    }

    /**
     * Return field faceted data from faceted search result.
     *
     * @param string $field Facet field.
     *
     * @return array
     */
    public function getFacetedData($field)
    {
        $this->_renderFilters();
        $result = [];
        $aggregations = $this->queryResponse->getAggregations();

        $bucket = $aggregations->getBucket($field);

        if ($bucket) {
            foreach ($bucket->getValues() as $value) {
                $metrics = $value->getMetrics();
                $result[$value->getValue()] = $metrics['count'];
            }
        }

        return $result;
    }

    /**
     * Indicates if the collection is spellchecked or not.
     *
     * @return bool
     */
    public function isSpellchecked()
    {
        return $this->isSpellchecked;
    }

    /**
     * @SuppressWarnings(PHPMD.CamelCaseMethodName)
     *
     * {@inheritdoc}
     */
    protected function _renderFiltersBefore()
    {
        // Pagination params.
        $size = $this->_pageSize ? $this->_pageSize : 20;
        $from = $size * (max(1, $this->_curPage) - 1);

        $pagesIds = $this->webVisionEsHelper->getTypo3ElasticsuiteData(
            $this->searchRequestName,
            $this->getStoreId(),
            $size,
            $from,
            $this->searchTerms
        );

        // Update the item count.
        if($this->_totalRecords = count($pagesIds)){

            $field = $this->webVisionEsHelper->getTypo3PrimaryKey();

            $this->getSelect()->where($field . ' IN (?)', ['in' => $pagesIds]);

            // Create the FIELD() expression for ORDER BY
            $fieldList = implode(',', array_map(function ($id) {
                return is_numeric($id) ? (int)$id : "'" . addslashes($id) . "'";
            }, $pagesIds));
            $fieldOrder = "FIELD($field, $fieldList)";
            $this->getSelect()->order($fieldOrder);
        }

        $this->_pageSize = false;
        $this->isSpellchecked = true;

        parent::_renderFiltersBefore();
    }

    /**
     * Get current store id
     *
     * @return int
     */
    public function getStoreId()
    {
        return $this->webVisionEsHelper->getStoreId();
    }
}
