<?php
/*
 * web-vision GmbH
 * Do not edit or add to this file if you wish to upgrade Magento to newer
 * versions in the future. If you wish to customize Magento for your
 * needs please refer to https://www.web-vision.de for more information.
 * @category    WebVision
 * @copyright   Copyright (c) 2001-2022 web-vision GmbH (https://www.web-vision.de)
 * @author      Mahesh Makwana <mmakwana@web-vision.de>
 */
namespace WebVision\UnityElasticSearch\Model\ResourceModel\Typo3\Indexer\Fulltext\Action;

use Magento\Framework\App\ResourceConnection;
use Magento\Framework\EntityManager\MetadataPool;
use Magento\Framework\Exception\InputException;
use Magento\Framework\Stdlib\DateTime\DateTime;
use Magento\Store\Model\StoreManagerInterface;
use Smile\ElasticsuiteCatalog\Model\ResourceModel\Eav\Indexer\Indexer;
use WebVision\UnityElasticSearch\Helper\Data as Typo3helper;
use WebVision\UnityElasticSearch\Model\ResourceModel\Pages\CollectionFactory as PageCollectionFactory;

class Full extends Indexer
{
    /**
     * @var \WebVision\UnityElasticSearch\Model\ResourceModel\Pages\CollectionFactory
     */
    private $pageCollectionFactory;

    /**
     * @var Typo3helper
     */
    private $typo3helper;

    /**
     * @var DateTime
     */
    protected $dateTime;

    /**
     * Constructor.
     *
     * @param ResourceConnection        $resource                  Resource connection.
     * @param StoreManagerInterface     $storeManager              Store manager.
     * @param MetadataPool              $metadataPool              Metadata pool.
     * @param PageCollectionFactory     $pageCollectionFactory     Page collection factory.
     */
    public function __construct(
        ResourceConnection $resource,
        StoreManagerInterface $storeManager,
        MetadataPool $metadataPool,
        PageCollectionFactory $pageCollectionFactory,
        Typo3helper $typo3helper,
        DateTime $dateTime
    ) {
        parent::__construct($resource, $storeManager, $metadataPool);
        $this->pageCollectionFactory = $pageCollectionFactory;
        $this->typo3helper = $typo3helper;
        $this->dateTime = $dateTime;
    }

    /**
     * Load a bulk of Page data.
     *
     * @param int     $storeId     Store id.
     * @param string  $categoryIds Product ids filter.
     * @param int $fromId      Load product with id greater than.
     * @param int $limit       Number of product to get loaded.
     *
     * @return array
     */
    public function getSearchablePages($storeId, $categoryIds = null, $fromId = 0, $limit = 100): array
    {
        $showPagesData = [];

        try {
            $primaryKey = $this->typo3helper->getTypo3PrimaryKey();
            $typo3DatabaseName = $this->typo3helper->getTypo3EnvDBName();
            $typo3StoreId = $this->typo3helper->getTypo3StoreFilterData($storeId);

            /**
             * @var \WebVision\UnityElasticSearch\Model\ResourceModel\Pages\CollectionFactory $pagesCollection
             */
            $pagesCollection = $this->pageCollectionFactory->create();

            // Add Filter
            $filterData = $this->typo3helper->getTypo3FilterData();
            if (count($filterData)) {
                foreach ($filterData as $filterKey => $filterVal) {
                    if (is_array($filterVal['attr_val'])) {
                        $pagesCollection->addFieldToFilter($filterVal['attr_code'], $filterVal['attr_val']);
                    } else {
                        $pagesCollection->addFieldToFilter($filterVal['attr_code'], [$filterVal['attr_condition'] => $filterVal['attr_val']]);
                    }
                }
            }

            $pagesCollection->addFieldToSelect(['title','slug','starttime','endtime']);
            $pagesCollection->getSelect()->joinLeft(
                ['pages_content' => $typo3DatabaseName.'.tt_content'],
                "`main_table`.`uid` =  `pages_content`.`pid` ",
                ["GROUP_CONCAT(`pages_content`.`bodytext`) AS bodytext"]
            );
            $pagesCollection->getSelect()->where("`main_table`.`deleted` = '0'");
            $pagesCollection->getSelect()->where("`main_table`.`hidden` = '0'");
            $pagesCollection->getSelect()->where("`main_table`.`sys_language_uid` = '".$typo3StoreId."'");
            $pagesCollection->getSelect()->where("`pages_content`.`sys_language_uid` = '".$typo3StoreId."'");
            $pagesCollection->getSelect()->where("`main_table`.`".$primaryKey."` > 0");
            $pagesCollection->getSelect()->group("main_table.uid");
            $pagesCollection->getSelect()->order('main_table.'.$primaryKey, 'ASC');

            $select = $pagesCollection->getSelect();
            $pagesCollection->getSelect()->where("`main_table`.`".$primaryKey."`" . ' > ?', $fromId)
                ->query();

            $pageData = $pagesCollection->toArray();

            if (isset($pageData['items'])) {
                foreach ($pageData['items'] as $pageKey => $pageVal) {
                    $skipRow = 1;
                    $timeStamp = $this->dateTime->gmtTimestamp($this->dateTime->date());

                    // remove page conflict
                    //$skipRow = $this->typo3helper->removeConflicPages($pageVal);

                    // Check start date
                    if ($pageVal['starttime']) {
                        if ($timeStamp < $pageVal['starttime']) {
                            $skipRow = 0;
                        }
                    }

                    // Check end date
                    if ($pageVal['endtime']) {
                        if ($timeStamp > $pageVal['endtime']) {
                            $skipRow = 0;
                        }
                    }

                    // Check available pages
                    if ($skipRow) {
                        $pageVal['store_id'] = $storeId;
                        unset($pageVal['starttime'], $pageVal['endtime']);
                        $showPagesData[] = $pageVal;
                    }
                }
            }
        } catch (\Exception $e) {
            throw new InputException(__($e->getMessage()));
        }

        return $showPagesData;
    }
}
