<?php

use Cemes\Pdo\Expression;

/**
 * web-vision GmbH
 *
 * NOTICE OF LICENSE
 *
 * <!--LICENSETEXT-->
 *
 * DISCLAIMER
 *
 * Do not edit or add to this file if you wish to upgrade Magento to newer
 * versions in the future. If you wish to customize Magento for your
 * needs please refer to http://www.web-vision.de for more information.
 *
 * @category    WebVision
 * @package     Fci_Model
 * @copyright   Copyright (c) 2001-2017 web-vision GmbH (http://www.web-vision.de)
 * @license     <!--LICENSEURL-->
 * @author      Mahesh Makwana <mmakwana@web-vision.de>
 */
class Fci_Model_Resource_Product extends Fci_Model_Resource_AbstractResource
{
    /**
     * @var string
     */
    public static $_confProductSku;

    /**
     * @var int
     */
    public static $_confProductId;

    /**
     * @var string
     */
    public static $_productUrlSuffix = ".html";


    /**
     * Fci_Model_Resource_Category constructor.
     */
    public function __construct()
    {
        parent::__construct();
        $this->_entityTable = 'catalog_product_entity';
    }

    /**
     * @inheritDoc
     */
    public function insert(Fci_Model_AbstractEntity $product)
    {
        // Start time tracking
        Cemes_Registry::get('insert_timer')->start();

        try{
            /** @var Fci_Model_Product $product */

            if (!$product->getData('sku')
                || !$product->getData('name')
                || !$product->getData('name')->getValue()
                || !$product->getData('price')
            ) {
                $missing = ['sku', 'name', 'price'];
                $mappings = $this->_config->getMappings();
                if ($mappings) {
                    foreach ($missing as $id => $field) {
                        if (in_array($field, $mappings, true)) {
                            $missing[$id] = str_replace('_', ' ', array_search($field, $mappings, true));
                        }
                    }
                }
                $context = [
                    'sku' => $product->getData('sku'),
                    'missing' => $missing,
                ];
                $this->_logger->error(
                    'Produkt mit der SKU "{sku}" kann nicht importiert werden, da eines der folgenden Pflichtfelder leer ist. {missing}',
                    $context
                );

                // Stop time tracking
                Cemes_Registry::get('insert_timer')->stop();
                return $this;
            }

            $this->_beforeSave($product);

            $insert = $this->_mysql->insert($this->getEntityTable())
                ->set('attribute_set_id', $product->getData('attribute_set_id'))
                ->set('type_id', $product->getData('type_id'))
                ->set('sku', $product->getData('sku'))
                ->set('has_options', $product->getData('has_options'))
                ->set('created_at', new Expression('NOW()'))
                ->set('updated_at', new Expression('NOW()'));

            if ($this->_fieldExistsInTable($this->getEntityTable(), 'entity_type_id')) {
                $insert->set('entity_type_id', $product->getData('entity_type_id'));
            }

            $insert->query();

            $product->setId($this->_mysql->lastInsertId());

            // ========== import Product URL ==========
            $this->setProductUrlkey($product);

            $product->getAttributeResource()->insert($product);

            // ========== relation between configurable and simple product ==========
            // configurable products logic
            if ($product->getTypeId() === 'simple') { // simple
                $configurableProductId = null;
                if ($product->hasData('configurable_product_id')) {
                    $configurableProductId = $product->getData('configurable_product_id');
                } elseif (strpos($product->getSku(), static::$_confProductSku ?? '') !== false) {
                    $configurableProductId = static::$_confProductId;
                }

                if (null !== $configurableProductId) {
                    $this->_mysql->insert('catalog_product_relation')
                        ->set('parent_id', $configurableProductId)
                        ->set('child_id', $product->getId())
                        ->execute();
                    $this->_mysql->insert('catalog_product_super_link')
                        ->set('parent_id', $configurableProductId)
                        ->set('product_id', $product->getId())
                        ->execute();
                }
            } elseif ($product->getTypeId() === 'configurable') { // configurable
                // save sku and id globally to check if next simple product is a subproduct
                static::$_confProductSku = $product->getSku();
                static::$_confProductId = $product->getId();

                if (is_array($product->getData('product_options'))) {
                    // save the attributes that are configurable
                    $position = 0;
                    foreach ($product->getData('product_options') as $optionLabel => $optionId) {
                        $this->_mysql->insert('catalog_product_super_attribute')
                            ->set('product_id', static::$_confProductId)
                            ->set('attribute_id', $optionId)
                            ->set('position', $position++)
                            ->execute();
                        $sid = $this->_mysql->select('catalog_product_super_attribute', ['product_super_attribute_id'])
                            ->where('product_id', static::$_confProductId)
                            ->where('attribute_id', $optionId)
                            ->fetchEntry('product_super_attribute_id');
                        $this->_mysql->insert('catalog_product_super_attribute_label')
                            ->set('product_super_attribute_id', $sid)
                            ->set('value', $optionLabel)
                            ->execute();
                    }
                }
            }

            // ========== product links ==========
            $linkResource = new Fci_Model_Resource_Product_Link();
            $linkResource->processInsert($product);

            // ========== stock data ==========
            $repository = new Fci_Model_Resource_Product_Stock();
            $repository->insert($product);

            // ========== website product relation ==========
            $insert = $this->_mysql->insert('catalog_product_website')
                ->columns(['product_id', 'website_id']);
            foreach ($product->getData('websites') as $websiteId) {
                $insert->values([$product->getId(), $websiteId]);
            }
            $insert->execute();

            // ========== category product relation ==========
            $insert = $this->_mysql->insert('catalog_category_product')
                ->columns(['product_id', 'category_id', 'position']);
            foreach ($product->getData('category_ids') as $category) {
                $insert->values([$product->getId(), $category['id'], $category['position']]);
            }
            $insert->execute();

            // ========== import gallery ==========
            $repository = new Fci_Model_Resource_Product_Gallery();
            $repository->insert($product);

            // ========== import tier_price ==========
            if ($product->getData('tier_price') && count($product->getData('tier_price')) !== 0) {
                $insert = $this->_mysql->insert('catalog_product_entity_tier_price')
                    ->columns(['entity_id', 'all_groups', 'customer_group_id', 'qty', 'value', 'website_id']);
                foreach ($product->getData('tier_price') as $tierPrice) {
                    $insert->values(
                        [
                            $product->getId(),
                            0,
                            $tierPrice['customer_group'],
                            $tierPrice['qty'],
                            $tierPrice['price'],
                            $tierPrice['website'],
                        ]
                    );
                }
                $insert->execute();
            }

            // ========== import group_price ==========
            if ($product->getData('group_price') && count($product->getData('group_price')) !== 0) {
                $insert = $this->_mysql->insert('catalog_product_entity_group_price')
                    ->columns(['entity_id', 'all_groups', 'customer_group_id', 'value', 'website_id']);
                foreach ($product->getData('group_price') as $groupPrice) {
                    $insert->values(
                        [
                            $product->getId(),
                            0,
                            $groupPrice['customer_group'],
                            $groupPrice['price'],
                            $groupPrice['website'],
                        ]
                    );
                }
                $insert->execute();
            }

            // ========== import simple product options ==========
            $repository = new Fci_Model_Resource_Product_Option();
            $repository->insert($product);

            $this->_afterSave($product);

            static::$_processedEntities++;
            static::$_insertedEntities++;

            $context = [
                'entitycount' => static::getProcessedEntities(),
                'sku'         => $product->getSku(),
            ];
            $this->_logger->success(
                'Produkt #{entitycount} mit der SKU "{sku}" erfolgreich eingetragen.',
                $context
            );

            // manage Advanced URL data
            $this->manageAdvancedUrlData($product);

        }  catch (\Cemes\Pdo\DatabaseException $e) {
            $this->_logger->error($e->getMessage());
        }  catch(Exception $e) {
            $this->_logger->error($e->getMessage());
        }

        // Stop time tracking
        Cemes_Registry::get('insert_timer')->stop();

        return $this;
    }

    /**
     * Generated Product URL Key
     *
     * @param Fci_Model_AbstractEntity $product
     *
     * @return void
     */
    public function setProductUrlkey(&$product)
    {
        return false; // Disbale URL key generation for now

        if ($product->getData('url_key')) {
            $productUrlSuffix = static::$_productUrlSuffix;
            $path = $product->getData('url_key')->getValue()."".$productUrlSuffix;
            $context = [
                'sku'  => $product->getSku(),
            ];
            foreach ($product->getData('stores') as $storeKey => $storeId) {

                // Check URL is new or existing
                $existinUrls = $this->_mysql->select('url_rewrite', ['url_rewrite_id', 'entity_id', 'request_path', 'store_id'])
                            ->where('entity_type', 'product')
                            ->where('store_id', $storeId)
                            ->where('request_path', $path , 'like')
                            ->fetchAll();

                // Check for existing product data
                if ($product->getData('entity_id')) {
                    $existinProductsUrls = $this->_mysql->select('url_rewrite', ['url_rewrite_id', 'entity_id', 'request_path', 'store_id'])
                        ->where('entity_type', 'product')
                        ->where('entity_id', $product->getData('entity_id'))
                        ->where('store_id', $storeId)
                        ->fetchAll();

                    // Removed duplicate values from array
                    $existinUrls = array_merge($existinUrls, $existinProductsUrls);
                    $existinUrlsUpdate = array_map("unserialize", array_unique(array_map("serialize", $existinUrls)));
                    if(count($existinUrlsUpdate)){
                        $existinUrls = $existinUrlsUpdate;
                    }
                }

                // generate product urls
                if(count($existinUrls) == 0){
                    // New Product URL
                    $insert = $this->_mysql->insert('url_rewrite')
                        ->columns(
                            [
                                'entity_type',
                                'entity_id',
                                'request_path',
                                'target_path',
                                'redirect_type',
                                'store_id',
                                'is_autogenerated'
                            ]
                        );
                    $insert->values(
                        [
                            'product',
                            $product->getData('entity_id'),
                            $path,
                            'catalog/product/view/id/'.$product->getData('entity_id'),
                            0,
                            $storeId,
                            1,
                        ]
                    );

                    $insert->execute();

                    $this->_logger->success(
                        '#SKU "{sku}" Die neue Produkt-URL wurde erfolgreich generiert: '.$path,
                        $context
                    );

                } elseif ((count($existinUrls))&&($product->getData('entity_id'))) {
                    // Update into Existing URL if product url is exist
                    $urlUpdateStatus = false;
                    foreach ($existinUrls as $existinUrl) {
                        if ($product->getData('entity_id') == $existinUrl['entity_id']) {
                            try{
                                $this->_mysql->update('url_rewrite')
                                        ->set('request_path', $path)
                                        ->where('url_rewrite_id', $existinUrl['url_rewrite_id'])
                                        ->execute();
                                $this->_logger->success(
                                    '#SKU "{sku}" Die Produkt-URL wurde erfolgreich aktualisiert: '.$path,
                                    $context
                                );
                                $urlUpdateStatus = true;
                            } catch(Exception $e) {
                                $this->_logger->error(
                                    '#SKU "{sku}" Das Problem mit der Produkt-URL:'.$e->getMessage(),
                                    $context
                                );
                            }
                            break;
                        }

                    }

                    // Update the url with store code to make it different
                    if (!$urlUpdateStatus) {
                        $this->setUniqueProductUrlKey($product, $path, $storeId);
                    }
                } else {
                    $this->_logger->error(
                        '#SKU "{sku}" Es gibt ein Problem mit der Produkt-URL: '.$path,
                        $context
                    );
                }
            }
        }
    }

    /**
     * Generated Product URL with unique Key
     *
     * @param Fci_Model_AbstractEntity $product
     * @param string $path
     * @param string $storeId
     *
     * @return void
     */
    public function setUniqueProductUrlKey(&$product, $path, $storeId)
    {
        $urlUpdateStatus = false;
        $context = [
            'sku'  => $product->getSku(),
        ];
        // Check URL is new or existing
        $existinUrls = $this->_mysql->select('url_rewrite', ['url_rewrite_id', 'entity_id', 'request_path', 'store_id'])
                    ->where('entity_type', 'product')
                    ->where('store_id', $storeId)
                    ->where('request_path', $path , 'like')
                    ->fetchAll();

        // If same Id already exit then update it
        if ((count($existinUrls))&&($product->getData('entity_id'))) {

            $productUrlSuffix = static::$_productUrlSuffix;
            $newPath = $product->getData('url_key')->getValue()."-".$storeId.$productUrlSuffix;

            // Update if duplicate ids are found into same product
            foreach ($existinUrls as $existinUrl) {
                if ($product->getData('entity_id') == $existinUrl['entity_id']) {
                    try{
                        $this->_mysql->update('url_rewrite')
                                ->set('request_path', $newPath)
                                ->where('url_rewrite_id', $existinUrl['url_rewrite_id'])
                                ->execute();
                        $urlUpdateStatus = true;
                        $this->_logger->success(
                            '#SKU "{sku}" Aktualisierung der Produkt-URL mit Store-ID: '.$newPath,
                            $context
                        );

                        // Update new url key to product
                        $this->updateNewProductUrlKey($product, $newPath);

                        break;
                    } catch(Exception $e) {
                        $this->_logger->error(
                            '#SKU "{sku}" Es gibt ein Problem mit der Produkt-URL:'.$newPath,
                            $context
                        );
                    }
                }
            }

            // Update if duplicate ids are found in another product
            if (!$urlUpdateStatus) {
                $existinProductUrls = $this->_mysql->select('url_rewrite', ['url_rewrite_id', 'entity_id', 'request_path', 'store_id'])
                        ->where('entity_type', 'product')
                        ->where('entity_id', $product->getData('entity_id'))
                        ->where('store_id', $storeId)
                        ->fetchAll();

                if (count($existinProductUrls) == 1) {
                    try{
                        $this->_mysql->update('url_rewrite')
                            ->set('request_path', $newPath)
                            ->where('url_rewrite_id', $existinProductUrls[0]['url_rewrite_id'])
                            ->execute();
                        $this->_logger->success(
                            '#SKU "{sku}" Aktualisierung der Produkt-URL mit Store-ID: '.$newPath,
                            $context
                        );

                        // Update new url key to product
                        $this->updateNewProductUrlKey($product, $newPath);

                    } catch(Exception $e) {
                        $this->_logger->error(
                            '#SKU "{sku}" Es gibt ein Problem mit der Produkt-URL:'.$newPath,
                            $context
                        );
                    }
                } else {
                    $this->_logger->error(
                        '#SKU "{sku}" Die Generierung der Produkt-URL ist fehlgeschlagen. : '.$newPath,
                        $context
                    );
                }
            }
        } else {
            $this->_logger->error(
                '#SKU "{sku}" Die Generierung der Produkt-URL ist fehlgeschlagen: '.$path,
                $context
            );
        }
    }

    /**
     * Save new product url key into product
     *
     * @param Fci_Model_AbstractEntity $product
     * @param string $urlKey
     *
     * @return void
     */
    public function updateNewProductUrlKey(&$product, $urlKey)
    {
        $updatedUrlKey = rtrim($urlKey, static::$_productUrlSuffix);
        $urlKeyAttribute = $this->_cache->getProductAttribute('url_key');
        $urlKeyAttribute->setValue($updatedUrlKey);
        $product->setData('url_key', $urlKeyAttribute);
    }

    /**
     * @inheritDoc
     */
    public function update(Fci_Model_AbstractEntity $product)
    {
        // Start time tracking
        Cemes_Registry::get('update_timer')->start();

        try{
            /** @var Fci_Model_Product $product */

            if (!$product->getData('sku')) {
                $this->_logger->error('Produkt kann nicht aktualisiert werden, da keine SKU existiert');

                // Stop time tracking
                Cemes_Registry::get('update_timer')->stop();
                return $this;
            }

            $this->_beforeSave($product);

            // ========== remove special price ==========
            if ($this->_config->getUnsetSpecialPrice() && !$product->hasData('special_price'))
            {
                $this->_mysql->delete('catalog_product_entity_decimal')
                    ->where('attribute_id', $this->_cache->getData('productAttributes/special_price/attribute_id'))
                    ->where('entity_id', $product->getId())
                    ->execute();

                $this->_mysql->delete('catalog_product_entity_datetime')
                    ->where('attribute_id', array(
                        $this->_cache->getData('productAttributes/special_from_date/attribute_id'),
                        $this->_cache->getData('productAttributes/special_to_date/attribute_id')), 'in')
                    ->where('entity_id', $product->getId())
                    ->execute();

                Cemes_Registry::get('logger')->notice('Special price without value in the CSV file is unset.');
            }

            // ========== catalog_product_entity ==========
            // updated_at date
            $this->_mysql->update($this->getEntityTable())
                ->set('updated_at', new Expression('NOW()'))
                ->where('entity_id', $product->getId())
                ->execute();

            // ========== relation between configurable and simple product ==========
            // configurable products logic
            if ($product->getTypeId() === 'simple') { // simple
                $configurableProductId = null;
                if ($product->hasData('configurable_product_id')) {
                    $configurableProductId = $product->getData('configurable_product_id');
                } elseif (strpos($product->getSku(), static::$_confProductSku ?? '') !== false) {
                    $configurableProductId = static::$_confProductId;
                }

                if (null !== $configurableProductId) {
                    $this->_mysql->select('catalog_product_relation')
                        ->where('parent_id', $configurableProductId)
                        ->where('child_id', $product->getId())
                        ->query();
                    if ($this->_mysql->count() === 0) {
                        $this->_mysql->insert('catalog_product_relation')
                            ->set('parent_id', $configurableProductId)
                            ->set('child_id', $product->getId())
                            ->execute();
                    }
                    $this->_mysql->select('catalog_product_super_link')
                        ->where('parent_id', $configurableProductId)
                        ->where('product_id', $product->getId())
                        ->query();
                    if ($this->_mysql->count() === 0) {
                        $this->_mysql->insert('catalog_product_super_link')
                            ->set('parent_id', $configurableProductId)
                            ->set('product_id', $product->getId())
                            ->execute();
                    }
                }
            } elseif ($product->getTypeId() === 'configurable') { // configurable
                // save sku and id globally to check if next simple product is a subproduct
                static::$_confProductSku = $product->getSku();
                static::$_confProductId = $product->getId();

                if (is_array($product->getData('product_options'))) {
                    // save the attributes that are configurable
                    foreach ($product->getData('product_options') as $optionLabel => $optionId) {
                        $this->_mysql->select('catalog_product_super_attribute')
                            ->where('product_id', static::$_confProductId)
                            ->where('attribute_id', $optionId)
                            ->query();
                        if ($this->_mysql->count() === 0) {
                            $this->_mysql->insert('catalog_product_super_attribute')
                                ->set('product_id', static::$_confProductId)
                                ->set('attribute_id', $optionId)
                                ->execute();
                            $sid = $this->_mysql->select('catalog_product_super_attribute', ['product_super_attribute_id'])
                                ->where('product_id', static::$_confProductId)
                                ->where('attribute_id', $optionId)
                                ->fetchEntry('product_super_attribute_id');
                            $this->_mysql->insert('catalog_product_super_attribute_label')
                                ->set('product_super_attribute_id', $sid)
                                ->set('value', $optionLabel)
                                ->execute();
                        }
                    }
                }
            }

            // ========== import Product URL ==========
            $this->setProductUrlkey($product);

            // ========== create attribute data ==========
            $product->getAttributeResource()->update($product);

            // ========== product links ==========
            $linkResource = new Fci_Model_Resource_Product_Link();
            $linkResource->processUpdate($product);

            // ========== stock data ==========
            $repository = new Fci_Model_Resource_Product_Stock();
            $repository->update($product);

            // ========== website product relation ==========
            $delete = $this->_mysql->delete('catalog_product_website')
                ->where('product_id', $product->getId());
            if ($this->_config->getClearExistingWebsites() === '0') {
                // only delete entries we will add again in the next step
                $delete->where('website_id', $product->getData('websites'), 'in');
            }
            $delete->execute();

            $insert = $this->_mysql->insert('catalog_product_website')
                ->columns(['product_id', 'website_id']);
            foreach ($product->getData('websites') as $websiteId) {
                $insert->values([$product->getId(), $websiteId]);
            }

            $insert->execute();

            // ========== category product relation ==========
            if ($product->hasData('category_ids')) {
                $categoryDataArray = $product->getData('category_ids');
                $catIds = [];
                $catIdsWithPosition = [];
                $catIdsWithoutPosition = [];
                foreach ($categoryDataArray as $categoryData) {
                    $catIds[] = $categoryData['id'];

                    if ($categoryData['position'] !== $this->_config->getDefaultProductPosition()) {
                        $catIdsWithPosition[] = $categoryData;
                    } else {
                        $catIdsWithoutPosition[] = $categoryData;
                    }
                }

                // category product
                // delete all relations that are not going to be updated or inserted
                $delete = $this->_mysql->delete('catalog_category_product')
                    ->where('product_id', $product->getId());
                if (!empty($catIds)) {
                    $delete->where('category_id', $catIds, 'nin');
                }
                $delete->execute();

                // update or insert relations that have a specific position
                if (!empty($catIdsWithPosition)) {
                    $replace = $this->_mysql->replace('catalog_category_product')
                        ->columns(['product_id', 'category_id', 'position']);
                    foreach ($catIdsWithPosition as $category) {
                        $replace->values([$product->getId(), $category['id'], $category['position']]);
                    }
                    $replace->execute();
                }

                // insert relations that are not already present and have no specific position
                if (!empty($catIdsWithoutPosition)) {
                    $entries = $this->_mysql->select('catalog_category_product', ['category_id'])
                        ->where('product_id', $product->getId())
                        ->where('category_id', $catIds, 'in')
                        ->fetchAll('category_id', PDO::FETCH_ASSOC);

                    $replace = $this->_mysql->replace('catalog_category_product')
                        ->columns(['product_id', 'category_id', 'position']);
                    $newEntriesPresent = false;
                    foreach ($catIdsWithoutPosition as $category) {
                        if (!$entries || !in_array($category['id'], $entries, false)) {
                            $replace->values([$product->getId(), $category['id'], $category['position']]);
                            $newEntriesPresent = true;
                        }
                    }

                    if ($newEntriesPresent) {
                        $replace->execute();
                    }
                }
            } elseif (in_array('categories', Fci_Model::$csvFields, true)
                || in_array('category_ids', Fci_Model::$csvFields, true)) {
                // category product
                $this->_mysql->delete('catalog_category_product')
                    ->where('product_id', $product->getId())
                    ->execute();
            }

            // ========== import gallery ==========
            $repository = new Fci_Model_Resource_Product_Gallery();
            $repository->update($product);

            // ========== import tier_price ==========
            if ($product->getData('tier_price') && count($product->getData('tier_price')) !== 0) {
                $this->_mysql->delete('catalog_product_entity_tier_price')
                    ->where('entity_id', $product->getId())
                    ->execute();
                $insert = $this->_mysql->insert('catalog_product_entity_tier_price')
                    ->columns(['entity_id', 'all_groups', 'customer_group_id', 'qty', 'value', 'website_id']);
                foreach ($product->getData('tier_price') as $tierPrice) {
                    $insert->values(
                        [
                            $product->getId(),
                            0,
                            $tierPrice['customer_group'],
                            $tierPrice['qty'],
                            $tierPrice['price'],
                            $tierPrice['website'],
                        ]
                    );
                }
                $insert->execute();
            }

            // ========== import group_price ==========
            if ($product->getData('group_price') && count($product->getData('group_price')) !== 0) {
                $this->_mysql->delete('catalog_product_entity_group_price')
                    ->where('entity_id', $product->getId())
                    ->execute();
                $insert = $this->_mysql->insert('catalog_product_entity_group_price')
                    ->columns(['entity_id', 'all_groups', 'customer_group_id', 'value', 'website_id']);
                foreach ($product->getData('group_price') as $groupPrice) {
                    $insert->values(
                        [
                            $product->getId(),
                            0,
                            $groupPrice['customer_group'],
                            $groupPrice['price'],
                            $groupPrice['website'],
                        ]
                    );
                }
                $insert->execute();
            }

            // ========== import simple product options ==========
            if ($product->getData('options') && count($product->getData('options')) !== 0) {
                $repository = new Fci_Model_Resource_Product_Option();
                $repository->update($product);
            }

            $this->_afterSave($product);

            static::$_processedEntities++;
            static::$_updatedEntities++;

            $context = [
                'entitycount' => static::getProcessedEntities(),
                'sku'         => $product->getSku(),
            ];
            $this->_logger->success(
                'Produkt #{entitycount} mit der SKU "{sku}" erfolgreich aktualisiert.',
                $context
            );

            // manage Advanced URL data
            $this->manageAdvancedUrlData($product);

        }  catch (\Cemes\Pdo\DatabaseException $e) {
            $this->_logger->error($e->getMessage());
        }  catch(Exception $e) {
            $this->_logger->error($e->getMessage());
        }

        // Stop time tracking
        Cemes_Registry::get('update_timer')->stop();

        return $this;
    }

    /**
     * Manage Advanced URL data
     *
     * @param Object $product
     * @return void
     */
    public function manageAdvancedUrlData($product)
    {
        $stores = $product->getStores();
        if (($productUrlKey = $product->getUrlKey()?->getValue()) && (isset($stores[0]))) {
            $productId = $product->getEntityId();
            $productSku = $product->getSku();
            $storeId = $stores[0];

            // Prepare the SQL with ON DUPLICATE KEY UPDATE
            $sql = "INSERT INTO extendware_advance_url_main
                        (product_id, product_sku, store_id, product_urlkey, manage_url_status)
                    VALUES (?, ?, ?, ?, 1)
                    ON DUPLICATE KEY UPDATE
                        product_urlkey = VALUES(product_urlkey),
                        manage_url_status = VALUES(manage_url_status)";

            // Prepare statement
            $stmt = $this->_mysql->prepare($sql);

            // Execute with parameters
            $stmt->execute([
                $productId,
                $productSku,
                $storeId,
                $productUrlKey
            ]);
        }
    }
}
