require.config({"config": {
        "jsbuild":{"mage/validation.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\ndefine([\n    'jquery',\n    'moment',\n    'mageUtils',\n    'jquery-ui-modules/widget',\n    'jquery/validate',\n    'mage/translate'\n], function ($, moment, utils) {\n    'use strict';\n\n    var creditCartTypes, rules, showLabel, originValidateDelegate;\n\n    $.extend(true, $, {\n        // @TODO: Move methods 'isEmpty', 'isEmptyNoTrim', 'parseNumber', 'stripHtml' in file with utility functions\n        mage: {\n            /**\n             * Check if string is empty with trim\n             * @param {String} value\n             */\n            isEmpty: function (value) {\n                return value === '' || value === undefined ||\n                    value == null || value.length === 0 || /^\\s+$/.test(value);\n            },\n\n            /**\n             * Check if string is empty no trim\n             * @param {String} value\n             */\n            isEmptyNoTrim: function (value) {\n                return value === '' || value == null || value.length === 0;\n            },\n\n            /**\n             * Checks if {value} is between numbers {from} and {to}\n             * @param {String} value\n             * @param {String} from\n             * @param {String} to\n             * @returns {Boolean}\n             */\n            isBetween: function (value, from, to) {\n                return ($.mage.isEmpty(from) || value >= $.mage.parseNumber(from)) &&\n                    ($.mage.isEmpty(to) || value <= $.mage.parseNumber(to));\n            },\n\n            /**\n             * Parse price string\n             * @param {String} value\n             */\n            parseNumber: function (value) {\n                var isDot, isComa;\n\n                if (typeof value !== 'string') {\n                    return parseFloat(value);\n                }\n                isDot = value.indexOf('.');\n                isComa = value.indexOf(',');\n\n                if (isDot !== -1 && isComa !== -1) {\n                    if (isComa > isDot) {\n                        value = value.replace('.', '').replace(',', '.');\n                    } else {\n                        value = value.replace(',', '');\n                    }\n                } else if (isComa !== -1) {\n                    value = value.replace(',', '.');\n                }\n\n                return parseFloat(value);\n            },\n\n            /**\n             * Removes HTML tags and space characters, numbers and punctuation.\n             *\n             * @param {String} value - Value being stripped.\n             * @return {String}\n             */\n            stripHtml: function (value) {\n                return value.replace(/<.[^<>]*?>/g, ' ').replace(/&nbsp;|&#160;/gi, ' ')\n                    .replace(/[0-9.(),;:!?%#$'\"_+=\\/-]*/g, '');\n            }\n        }\n    });\n\n    /**\n     * @param {String} name\n     * @param {*} method\n     * @param {*} message\n     * @param {*} dontSkip\n     */\n    $.validator.addMethod = function (name, method, message, dontSkip) {\n        $.validator.methods[name] = method;\n        $.validator.messages[name] = message !== undefined ? message : $.validator.messages[name];\n\n        if (method.length < 3 || dontSkip) {\n            $.validator.addClassRules(name, $.validator.normalizeRule(name));\n        }\n    };\n\n    /**\n     * Javascript object with credit card types\n     * 0 - regexp for card number\n     * 1 - regexp for cvn\n     * 2 - check or not credit card number trough Luhn algorithm by\n     */\n    creditCartTypes = {\n        'SO': [\n            new RegExp('^(6334[5-9]([0-9]{11}|[0-9]{13,14}))|(6767([0-9]{12}|[0-9]{14,15}))$'),\n            new RegExp('^([0-9]{3}|[0-9]{4})?$'),\n            true\n        ],\n        'SM': [\n            new RegExp('(^(5[0678])[0-9]{11,18}$)|(^(6[^05])[0-9]{11,18}$)|' +\n                '(^(601)[^1][0-9]{9,16}$)|(^(6011)[0-9]{9,11}$)|(^(6011)[0-9]{13,16}$)|' +\n                '(^(65)[0-9]{11,13}$)|(^(65)[0-9]{15,18}$)|(^(49030)[2-9]([0-9]{10}$|[0-9]{12,13}$))|' +\n                '(^(49033)[5-9]([0-9]{10}$|[0-9]{12,13}$))|(^(49110)[1-2]([0-9]{10}$|[0-9]{12,13}$))|' +\n                '(^(49117)[4-9]([0-9]{10}$|[0-9]{12,13}$))|(^(49118)[0-2]([0-9]{10}$|[0-9]{12,13}$))|' +\n                '(^(4936)([0-9]{12}$|[0-9]{14,15}$))'), new RegExp('^([0-9]{3}|[0-9]{4})?$'),\n            true\n        ],\n        'VI': [new RegExp('^4[0-9]{12}([0-9]{3})?$'), new RegExp('^[0-9]{3}$'), true],\n        'MC': [\n            new RegExp('^(?:5[1-5][0-9]{2}|222[1-9]|22[3-9][0-9]|2[3-6][0-9]{2}|27[01][0-9]|2720)[0-9]{12}$'),\n            new RegExp('^[0-9]{3}$'),\n            true\n        ],\n        'AE': [new RegExp('^3[47][0-9]{13}$'), new RegExp('^[0-9]{4}$'), true],\n        'DI': [new RegExp('^(6011(0|[2-4]|74|7[7-9]|8[6-9]|9)|6(4[4-9]|5))\\\\d*$'), new RegExp('^[0-9]{3}$'), true],\n        'JCB': [new RegExp('^35(2[8-9]|[3-8])\\\\d*$'), new RegExp('^[0-9]{3}$'), true],\n        'DN': [new RegExp('^(3(0[0-5]|095|6|[8-9]))\\\\d*$'), new RegExp('^[0-9]{3}$'), true],\n        'UN': [\n            new RegExp('^(622(1(2[6-9]|[3-9])|[3-8]|9([[0-1]|2[0-5]))|62[4-6]|628([2-8]))\\\\d*?$'),\n            new RegExp('^[0-9]{3}$'),\n            true\n        ],\n        'MI': [new RegExp('^(5(0|[6-9])|63|67(?!59|6770|6774))\\\\d*$'), new RegExp('^[0-9]{3}$'), true],\n        'MD': [new RegExp('^6759(?!24|38|40|6[3-9]|70|76)|676770|676774\\\\d*$'), new RegExp('^[0-9]{3}$'), true]\n    };\n\n    /**\n     * validate credit card number using mod10\n     * @param {String} s\n     * @return {Boolean}\n     */\n    function validateCreditCard(s) {\n        // remove non-numerics\n        var v = '0123456789',\n            w = '',\n            i, j, k, m, c, a, x;\n\n        for (i = 0; i < s.length; i++) {\n            x = s.charAt(i);\n\n            if (v.indexOf(x, 0) !== -1) {\n                w += x;\n            }\n        }\n        // validate number\n        j = w.length / 2;\n        k = Math.floor(j);\n        m = Math.ceil(j) - k;\n        c = 0;\n\n        for (i = 0; i < k; i++) {\n            a = w.charAt(i * 2 + m) * 2;\n            c += a > 9 ? Math.floor(a / 10 + a % 10) : a;\n        }\n\n        for (i = 0; i < k + m; i++) {\n            c += w.charAt(i * 2 + 1 - m) * 1;\n        }\n\n        return c % 10 === 0;\n    }\n\n    /**\n     * validate all table required inputs at once, using single hidden input\n     * @param {String} value\n     * @param {HTMLElement} element\n     *\n     * @return {Boolean}\n     */\n    function tableSingleValidation(value, element) {\n        var empty = $(element).closest('table')\n            .find('input.required-option:visible')\n            .filter(function (i, el) {\n                if ($(el).is('disabled')) {\n                    return $.mage.isEmpty(el.value);\n                }\n            })\n            .length;\n\n        return empty === 0;\n    }\n\n    /**\n     *\n     * @param {float} qty\n     * @param {float} qtyIncrements\n     * @returns {float}\n     */\n    function resolveModulo(qty, qtyIncrements) {\n        var divideEpsilon = 10000,\n            epsilon,\n            remainder;\n\n        while (qtyIncrements < 1) {\n            qty *= 10;\n            qtyIncrements *= 10;\n        }\n\n        epsilon = qtyIncrements / divideEpsilon;\n        remainder = qty % qtyIncrements;\n\n        if (Math.abs(remainder - qtyIncrements) < epsilon ||\n            Math.abs(remainder) < epsilon) {\n            remainder = 0;\n        }\n\n        return remainder;\n    }\n\n    /**\n     * Collection of validation rules including rules from additional-methods.js\n     * @type {Object}\n     */\n    rules = {\n        'max-words': [\n            function (value, element, params) {\n                return this.optional(element) || $.mage.stripHtml(value).match(/\\b\\w+\\b/g).length <= params;\n            },\n            $.mage.__('Please enter {0} words or less.')\n        ],\n        'min-words': [\n            function (value, element, params) {\n                return this.optional(element) || $.mage.stripHtml(value).match(/\\b\\w+\\b/g).length >= params;\n            },\n            $.mage.__('Please enter at least {0} words.')\n        ],\n        'range-words': [\n            function (value, element, params) {\n                return this.optional(element) ||\n                    $.mage.stripHtml(value).match(/\\b\\w+\\b/g).length >= params[0] &&\n                    value.match(/bw+b/g).length < params[1];\n            },\n            $.mage.__('Please enter between {0} and {1} words.')\n        ],\n        'letters-with-basic-punc': [\n            function (value, element) {\n                return this.optional(element) || /^[a-z\\-.,()'\\\"\\s]+$/i.test(value);\n            },\n            $.mage.__('Letters or punctuation only please')\n        ],\n        'alphanumeric': [\n            function (value, element) {\n                return this.optional(element) || /^\\w+$/i.test(value);\n            },\n            $.mage.__('Letters, numbers, spaces or underscores only please')\n        ],\n        'letters-only': [\n            function (value, element) {\n                return this.optional(element) || /^[a-z]+$/i.test(value);\n            },\n            $.mage.__('Letters only please')\n        ],\n        'no-whitespace': [\n            function (value, element) {\n                return this.optional(element) || /^\\S+$/i.test(value);\n            },\n            $.mage.__('No white space please')\n        ],\n        'no-marginal-whitespace': [\n            function (value, element) {\n                return this.optional(element) || !/^\\s+|\\s+$/i.test(value);\n            },\n            $.mage.__('No marginal white space please')\n        ],\n        'zip-range': [\n            function (value, element) {\n                return this.optional(element) || /^90[2-5]-\\d{2}-\\d{4}$/.test(value);\n            },\n            $.mage.__('Your ZIP-code must be in the range 902xx-xxxx to 905-xx-xxxx')\n        ],\n        'integer': [\n            function (value, element) {\n                return this.optional(element) || /^-?\\d+$/.test(value);\n            },\n            $.mage.__('A positive or negative non-decimal number please')\n        ],\n        'vinUS': [\n            function (v) {\n                var i, n, d, f, cd, cdv, LL, VL, FL, rs;\n\n                /* eslint-disable max-depth */\n                if (v.length !== 17) {\n                    return false;\n                }\n\n                LL = ['A', 'B', 'C', 'D', 'E', 'F', 'G', 'H', 'J', 'K', 'L',\n                    'M', 'N', 'P', 'R', 'S', 'T', 'U', 'V', 'W', 'X', 'Y', 'Z'];\n                VL = [1, 2, 3, 4, 5, 6, 7, 8, 1, 2, 3, 4, 5, 7, 9, 2, 3, 4, 5, 6, 7, 8, 9];\n                FL = [8, 7, 6, 5, 4, 3, 2, 10, 0, 9, 8, 7, 6, 5, 4, 3, 2];\n                rs = 0;\n\n                for (i = 0; i < 17; i++) {\n                    f = FL[i];\n                    d = v.slice(i, i + 1);\n\n                    if (i === 8) {\n                        cdv = d;\n                    }\n\n                    if (!isNaN(d)) {\n                        d *= f;\n                    } else {\n                        for (n = 0; n < LL.length; n++) {\n                            if (d.toUpperCase() === LL[n]) {\n                                d = VL[n];\n                                d *= f;\n\n                                if (isNaN(cdv) && n === 8) {\n                                    cdv = LL[n];\n                                }\n                                break;\n                            }\n                        }\n                    }\n                    rs += d;\n                }\n\n                /* eslint-enable max-depth */\n                cd = rs % 11;\n\n                if (cd === 10) {\n                    cd = 'X';\n                }\n\n                if (cd === cdv) {\n                    return true;\n                }\n\n                return false;\n            },\n            $.mage.__('The specified vehicle identification number (VIN) is invalid.')\n        ],\n        'dateITA': [\n            function (value, element) {\n                var check = false,\n                    re = /^\\d{1,2}\\/\\d{1,2}\\/\\d{4}$/,\n                    adata, gg, mm, aaaa, xdata;\n\n                if (re.test(value)) {\n                    adata = value.split('/');\n                    gg = parseInt(adata[0], 10);\n                    mm = parseInt(adata[1], 10);\n                    aaaa = parseInt(adata[2], 10);\n                    xdata = new Date(aaaa, mm - 1, gg);\n\n                    if (xdata.getFullYear() === aaaa &&\n                        xdata.getMonth() === mm - 1 &&\n                        xdata.getDate() === gg\n                    ) {\n                        check = true;\n                    } else {\n                        check = false;\n                    }\n                } else {\n                    check = false;\n                }\n\n                return this.optional(element) || check;\n            },\n            $.mage.__('Please enter a correct date')\n        ],\n        'dateNL': [\n            function (value, element) {\n                return this.optional(element) || /^\\d\\d?[\\.\\/-]\\d\\d?[\\.\\/-]\\d\\d\\d?\\d?$/.test(value);\n            },\n            'Vul hier een geldige datum in.'\n        ],\n        'time': [\n            function (value, element) {\n                return this.optional(element) || /^([01]\\d|2[0-3])(:[0-5]\\d){0,2}$/.test(value);\n            },\n            $.mage.__('Please enter a valid time, between 00:00 and 23:59')\n        ],\n        'time12h': [\n            function (value, element) {\n                return this.optional(element) || /^((0?[1-9]|1[012])(:[0-5]\\d){0,2}(\\s[AP]M))$/i.test(value);\n            },\n            $.mage.__('Please enter a valid time, between 00:00 am and 12:00 pm')\n        ],\n        'phoneUS': [\n            function (phoneNumber, element) {\n                phoneNumber = phoneNumber.replace(/\\s+/g, '');\n\n                return this.optional(element) || phoneNumber.length > 9 &&\n                    phoneNumber.match(/^(1-?)?(\\([2-9]\\d{2}\\)|[2-9]\\d{2})-?[2-9]\\d{2}-?\\d{4}$/);\n            },\n            $.mage.__('Please specify a valid phone number')\n        ],\n        'phoneUK': [\n            function (phoneNumber, element) {\n                return this.optional(element) || phoneNumber.length > 9 &&\n                    phoneNumber.match(/^(\\(?(0|\\+44)[1-9]{1}\\d{1,4}?\\)?\\s?\\d{3,4}\\s?\\d{3,4})$/);\n            },\n            $.mage.__('Please specify a valid phone number')\n        ],\n        'mobileUK': [\n            function (phoneNumber, element) {\n                return this.optional(element) || phoneNumber.length > 9 &&\n                    phoneNumber.match(/^((0|\\+44)7\\d{3}\\s?\\d{6})$/);\n            },\n            $.mage.__('Please specify a valid mobile number')\n        ],\n        'stripped-min-length': [\n            function (value, element, param) {\n                return value.length >= param;\n            },\n            $.mage.__('Please enter at least {0} characters')\n        ],\n\n        /* detect chars that would require more than 3 bytes */\n        'validate-no-utf8mb4-characters': [\n            function (value) {\n                var validator = this,\n                    message = $.mage.__('Please remove invalid characters: {0}.'),\n                    matches = value.match(/(?:[\\uD800-\\uDBFF][\\uDC00-\\uDFFF])/g),\n                    result = matches === null;\n\n                if (!result) {\n                    validator.charErrorMessage = message.replace('{0}', matches.join());\n                }\n\n                return result;\n            }, function () {\n                return this.charErrorMessage;\n            }\n        ],\n\n        /* eslint-disable max-len */\n        'email2': [\n            function (value, element) {\n                return this.optional(element) ||\n                    /^((([a-z]|\\d|[!#\\$%&'\\*\\+\\-\\/=\\?\\^_`{\\|}~]|[\\u00A0-\\uD7FF\\uF900-\\uFDCF\\uFDF0-\\uFFEF])+(\\.([a-z]|\\d|[!#\\$%&'\\*\\+\\-\\/=\\?\\^_`{\\|}~]|[\\u00A0-\\uD7FF\\uF900-\\uFDCF\\uFDF0-\\uFFEF])+)*)|((\\x22)((((\\x20|\\x09)*(\\x0d\\x0a))?(\\x20|\\x09)+)?(([\\x01-\\x08\\x0b\\x0c\\x0e-\\x1f\\x7f]|\\x21|[\\x23-\\x5b]|[\\x5d-\\x7e]|[\\u00A0-\\uD7FF\\uF900-\\uFDCF\\uFDF0-\\uFFEF])|(\\\\([\\x01-\\x09\\x0b\\x0c\\x0d-\\x7f]|[\\u00A0-\\uD7FF\\uF900-\\uFDCF\\uFDF0-\\uFFEF]))))*(((\\x20|\\x09)*(\\x0d\\x0a))?(\\x20|\\x09)+)?(\\x22)))@((([a-z]|\\d|[\\u00A0-\\uD7FF\\uF900-\\uFDCF\\uFDF0-\\uFFEF])|(([a-z]|\\d|[\\u00A0-\\uD7FF\\uF900-\\uFDCF\\uFDF0-\\uFFEF])([a-z]|\\d|-|\\.|_|~|[\\u00A0-\\uD7FF\\uF900-\\uFDCF\\uFDF0-\\uFFEF])*([a-z]|\\d|[\\u00A0-\\uD7FF\\uF900-\\uFDCF\\uFDF0-\\uFFEF])))\\.)*(([a-z]|[\\u00A0-\\uD7FF\\uF900-\\uFDCF\\uFDF0-\\uFFEF])|(([a-z]|[\\u00A0-\\uD7FF\\uF900-\\uFDCF\\uFDF0-\\uFFEF])([a-z]|\\d|-|\\.|_|~|[\\u00A0-\\uD7FF\\uF900-\\uFDCF\\uFDF0-\\uFFEF])*([a-z]|[\\u00A0-\\uD7FF\\uF900-\\uFDCF\\uFDF0-\\uFFEF])))\\.?$/i.test(value);\n            },\n            $.validator.messages.email\n        ],\n        'url2': [\n            function (value, element) {\n                return this.optional(element) || /^(https?|ftp):\\/\\/(((([a-z]|\\d|-|\\.|_|~|[\\u00A0-\\uD7FF\\uF900-\\uFDCF\\uFDF0-\\uFFEF])|(%[\\da-f]{2})|[!\\$&'\\(\\)\\*\\+,;=]|:)*@)?(((\\d|[1-9]\\d|1\\d\\d|2[0-4]\\d|25[0-5])\\.(\\d|[1-9]\\d|1\\d\\d|2[0-4]\\d|25[0-5])\\.(\\d|[1-9]\\d|1\\d\\d|2[0-4]\\d|25[0-5])\\.(\\d|[1-9]\\d|1\\d\\d|2[0-4]\\d|25[0-5]))|((([a-z]|\\d|[\\u00A0-\\uD7FF\\uF900-\\uFDCF\\uFDF0-\\uFFEF])|(([a-z]|\\d|[\\u00A0-\\uD7FF\\uF900-\\uFDCF\\uFDF0-\\uFFEF])([a-z]|\\d|-|\\.|_|~|[\\u00A0-\\uD7FF\\uF900-\\uFDCF\\uFDF0-\\uFFEF])*([a-z]|\\d|[\\u00A0-\\uD7FF\\uF900-\\uFDCF\\uFDF0-\\uFFEF])))\\.)*(([a-z]|[\\u00A0-\\uD7FF\\uF900-\\uFDCF\\uFDF0-\\uFFEF])|(([a-z]|[\\u00A0-\\uD7FF\\uF900-\\uFDCF\\uFDF0-\\uFFEF])([a-z]|\\d|-|\\.|_|~|[\\u00A0-\\uD7FF\\uF900-\\uFDCF\\uFDF0-\\uFFEF])*([a-z]|[\\u00A0-\\uD7FF\\uF900-\\uFDCF\\uFDF0-\\uFFEF])))\\.?)(:\\d*)?)(\\/((([a-z]|\\d|-|\\.|_|~|[\\u00A0-\\uD7FF\\uF900-\\uFDCF\\uFDF0-\\uFFEF])|(%[\\da-f]{2})|[!\\$&'\\(\\)\\*\\+,;=]|:|@)+(\\/(([a-z]|\\d|-|\\.|_|~|[\\u00A0-\\uD7FF\\uF900-\\uFDCF\\uFDF0-\\uFFEF])|(%[\\da-f]{2})|[!\\$&'\\(\\)\\*\\+,;=]|:|@)*)*)?)?(\\?((([a-z]|\\d|-|\\.|_|~|[\\u00A0-\\uD7FF\\uF900-\\uFDCF\\uFDF0-\\uFFEF])|(%[\\da-f]{2})|[!\\$&'\\(\\)\\*\\+,;=]|:|@)|[\\uE000-\\uF8FF]|\\/|\\?)*)?(\\#((([a-z]|\\d|-|\\.|_|~|[\\u00A0-\\uD7FF\\uF900-\\uFDCF\\uFDF0-\\uFFEF])|(%[\\da-f]{2})|[!\\$&'\\(\\)\\*\\+,;=]|:|@)|\\/|\\?)*)?$/i.test(value);\n            },\n            $.validator.messages.url\n        ],\n\n        /* eslint-enable max-len */\n        'credit-card-types': [\n            function (value, element, param) {\n                var validTypes;\n\n                if (/[^0-9-]+/.test(value)) {\n                    return false;\n                }\n                value = value.replace(/\\D/g, '');\n\n                validTypes = 0x0000;\n\n                if (param.mastercard) {\n                    validTypes |= 0x0001;\n                }\n\n                if (param.visa) {\n                    validTypes |= 0x0002;\n                }\n\n                if (param.amex) {\n                    validTypes |= 0x0004;\n                }\n\n                if (param.dinersclub) {\n                    validTypes |= 0x0008;\n                }\n\n                if (param.enroute) {\n                    validTypes |= 0x0010;\n                }\n\n                if (param.discover) {\n                    validTypes |= 0x0020;\n                }\n\n                if (param.jcb) {\n                    validTypes |= 0x0040;\n                }\n\n                if (param.unknown) {\n                    validTypes |= 0x0080;\n                }\n\n                if (param.all) {\n                    validTypes = 0x0001 | 0x0002 | 0x0004 | 0x0008 | 0x0010 | 0x0020 | 0x0040 | 0x0080;\n                }\n\n                if (validTypes & 0x0001 && /^(51|52|53|54|55)/.test(value)) { //mastercard\n                    return value.length === 16;\n                }\n\n                if (validTypes & 0x0002 && /^(4)/.test(value)) { //visa\n                    return value.length === 16;\n                }\n\n                if (validTypes & 0x0004 && /^(34|37)/.test(value)) { //amex\n                    return value.length === 15;\n                }\n\n                if (validTypes & 0x0008 && /^(300|301|302|303|304|305|36|38)/.test(value)) { //dinersclub\n                    return value.length === 14;\n                }\n\n                if (validTypes & 0x0010 && /^(2014|2149)/.test(value)) { //enroute\n                    return value.length === 15;\n                }\n\n                if (validTypes & 0x0020 && /^(6011)/.test(value)) { //discover\n                    return value.length === 16;\n                }\n\n                if (validTypes & 0x0040 && /^(3)/.test(value)) { //jcb\n                    return value.length === 16;\n                }\n\n                if (validTypes & 0x0040 && /^(2131|1800)/.test(value)) { //jcb\n                    return value.length === 15;\n                }\n\n                if (validTypes & 0x0080) { //unknown\n                    return true;\n                }\n\n                return false;\n            },\n            $.mage.__('Please enter a valid credit card number.')\n        ],\n\n        /* eslint-disable max-len */\n        'ipv4': [\n            function (value, element) {\n                return this.optional(element) ||\n                    /^(25[0-5]|2[0-4][0-9]|[01]?[0-9][0-9]?)\\.(25[0-5]|2[0-4][0-9]|[01]?[0-9][0-9]?)\\.(25[0-5]|2[0-4][0-9]|[01]?[0-9][0-9]?)\\.(25[0-5]|2[0-4][0-9]|[01]?[0-9][0-9]?)$/i.test(value);\n            },\n            $.mage.__('Please enter a valid IP v4 address.')\n        ],\n        'ipv6': [\n            function (value, element) {\n                return this.optional(element) || /^((([0-9A-Fa-f]{1,4}:){7}[0-9A-Fa-f]{1,4})|(([0-9A-Fa-f]{1,4}:){6}:[0-9A-Fa-f]{1,4})|(([0-9A-Fa-f]{1,4}:){5}:([0-9A-Fa-f]{1,4}:)?[0-9A-Fa-f]{1,4})|(([0-9A-Fa-f]{1,4}:){4}:([0-9A-Fa-f]{1,4}:){0,2}[0-9A-Fa-f]{1,4})|(([0-9A-Fa-f]{1,4}:){3}:([0-9A-Fa-f]{1,4}:){0,3}[0-9A-Fa-f]{1,4})|(([0-9A-Fa-f]{1,4}:){2}:([0-9A-Fa-f]{1,4}:){0,4}[0-9A-Fa-f]{1,4})|(([0-9A-Fa-f]{1,4}:){6}((\\b((25[0-5])|(1\\d{2})|(2[0-4]\\d)|(\\d{1,2}))\\b)\\.){3}(\\b((25[0-5])|(1\\d{2})|(2[0-4]\\d)|(\\d{1,2}))\\b))|(([0-9A-Fa-f]{1,4}:){0,5}:((\\b((25[0-5])|(1\\d{2})|(2[0-4]\\d)|(\\d{1,2}))\\b)\\.){3}(\\b((25[0-5])|(1\\d{2})|(2[0-4]\\d)|(\\d{1,2}))\\b))|(::([0-9A-Fa-f]{1,4}:){0,5}((\\b((25[0-5])|(1\\d{2})|(2[0-4]\\d)|(\\d{1,2}))\\b)\\.){3}(\\b((25[0-5])|(1\\d{2})|(2[0-4]\\d)|(\\d{1,2}))\\b))|([0-9A-Fa-f]{1,4}::([0-9A-Fa-f]{1,4}:){0,5}[0-9A-Fa-f]{1,4})|(::([0-9A-Fa-f]{1,4}:){0,6}[0-9A-Fa-f]{1,4})|(([0-9A-Fa-f]{1,4}:){1,7}:))$/i.test(value);\n            },\n            $.mage.__('Please enter a valid IP v6 address.')\n        ],\n\n        /* eslint-enable max-len */\n        'pattern': [\n            function (value, element, param) {\n                return this.optional(element) || new RegExp(param).test(value);\n            },\n            $.mage.__('Invalid format.')\n        ],\n        'allow-container-className': [\n            function (element) {\n                if (element.type === 'radio' || element.type === 'checkbox') {\n                    return $(element).hasClass('change-container-classname');\n                }\n            },\n            ''\n        ],\n        'validate-no-html-tags': [\n            function (value) {\n                return !/<(\\/)?\\w+/.test(value);\n            },\n            $.mage.__('HTML tags are not allowed.')\n        ],\n        'validate-select': [\n            function (value) {\n                return value !== 'none' && value != null && value.length !== 0;\n            },\n            $.mage.__('Please select an option.')\n        ],\n        'validate-no-empty': [\n            function (value) {\n                return !$.mage.isEmpty(value);\n            },\n            $.mage.__('Empty Value.')\n        ],\n        'validate-alphanum-with-spaces': [\n            function (v) {\n                return $.mage.isEmptyNoTrim(v) || /^[a-zA-Z0-9 ]+$/.test(v);\n            },\n            $.mage.__('Please use only letters (a-z or A-Z), numbers (0-9) or spaces only in this field.')\n        ],\n        'validate-data': [\n            function (v) {\n                return $.mage.isEmptyNoTrim(v) || /^[A-Za-z]+[A-Za-z0-9_]+$/.test(v);\n            },\n            $.mage.__('Please use only letters (a-z or A-Z), numbers (0-9) or underscore (_) in this field, and the first character should be a letter.') //eslint-disable-line max-len\n        ],\n        'validate-street': [\n            function (v) {\n                return $.mage.isEmptyNoTrim(v) || /^[ \\w]{3,}([A-Za-z]\\.)?([ \\w]*\\#\\d+)?(\\r\\n| )[ \\w]{3,}/.test(v);\n            },\n            $.mage.__('Please use only letters (a-z or A-Z), numbers (0-9), spaces and \"#\" in this field.')\n        ],\n        'validate-phoneStrict': [\n            function (v) {\n                return $.mage.isEmptyNoTrim(v) || /^(\\()?\\d{3}(\\))?(-|\\s)?\\d{3}(-|\\s)\\d{4}$/.test(v);\n            },\n            $.mage.__('Please enter a valid phone number. For example (123) 456-7890 or 123-456-7890.')\n        ],\n        'validate-phoneLax': [\n            function (v) {\n                return $.mage.isEmptyNoTrim(v) ||\n                    /^((\\d[\\-. ]?)?((\\(\\d{3}\\))|\\d{3}))?[\\-. ]?\\d{3}[\\-. ]?\\d{4}$/.test(v);\n            },\n            $.mage.__('Please enter a valid phone number. For example (123) 456-7890 or 123-456-7890.')\n        ],\n        'validate-fax': [\n            function (v) {\n                return $.mage.isEmptyNoTrim(v) || /^(\\()?\\d{3}(\\))?(-|\\s)?\\d{3}(-|\\s)\\d{4}$/.test(v);\n            },\n            $.mage.__('Please enter a valid fax number (Ex: 123-456-7890).')\n        ],\n        'validate-email': [\n            function (v) {\n                return $.mage.isEmptyNoTrim(v) || /^([a-z0-9,!\\#\\$%&'\\*\\+\\/=\\?\\^_`\\{\\|\\}~-]|[\\u00A0-\\uD7FF\\uF900-\\uFDCF\\uFDF0-\\uFFEF])+(\\.([a-z0-9,!\\#\\$%&'\\*\\+\\/=\\?\\^_`\\{\\|\\}~-]|[\\u00A0-\\uD7FF\\uF900-\\uFDCF\\uFDF0-\\uFFEF])+)*@([a-z0-9-]|[\\u00A0-\\uD7FF\\uF900-\\uFDCF\\uFDF0-\\uFFEF])+(\\.([a-z0-9-]|[\\u00A0-\\uD7FF\\uF900-\\uFDCF\\uFDF0-\\uFFEF])+)*\\.(([a-z]|[\\u00A0-\\uD7FF\\uF900-\\uFDCF\\uFDF0-\\uFFEF]){2,})$/i.test(v); //eslint-disable-line max-len\n            },\n            $.mage.__('Please enter a valid email address (Ex: johndoe@domain.com).')\n        ],\n        //replace jquery.validation.js email validation rule\n        'email' : [\n            function (v) {\n                return $.mage.isEmptyNoTrim(v) || /^([a-z0-9,!\\#\\$%&'\\*\\+\\/=\\?\\^_`\\{\\|\\}~-]|[\\u00A0-\\uD7FF\\uF900-\\uFDCF\\uFDF0-\\uFFEF])+(\\.([a-z0-9,!\\#\\$%&'\\*\\+\\/=\\?\\^_`\\{\\|\\}~-]|[\\u00A0-\\uD7FF\\uF900-\\uFDCF\\uFDF0-\\uFFEF])+)*@([a-z0-9-]|[\\u00A0-\\uD7FF\\uF900-\\uFDCF\\uFDF0-\\uFFEF])+(\\.([a-z0-9-]|[\\u00A0-\\uD7FF\\uF900-\\uFDCF\\uFDF0-\\uFFEF])+)*\\.(([a-z]|[\\u00A0-\\uD7FF\\uF900-\\uFDCF\\uFDF0-\\uFFEF]){2,})$/i.test(v); //eslint-disable-line max-len\n            },\n            $.mage.__('Please enter a valid email address.')\n        ],\n        'validate-emailSender': [\n            function (v) {\n                return $.mage.isEmptyNoTrim(v) || /^(?!.*:)[\\S ]+$/.test(v);\n            },\n            $.mage.__('Please use only visible characters and spaces. The colon character is not allowed.')\n        ],\n        'validate-password': [\n            function (v) {\n                var pass;\n\n                if (v == null) {\n                    return false;\n                }\n                //strip leading and trailing spaces\n                pass = v.trim();\n\n                if (!pass.length) {\n                    return true;\n                }\n\n                return !(pass.length > 0 && pass.length < 6);\n            },\n            $.mage.__('Please enter 6 or more characters. Leading and trailing spaces will be ignored.')\n        ],\n        'validate-admin-password': [\n            function (v) {\n                var pass;\n\n                if (v == null) {\n                    return false;\n                }\n                pass = v.trim();\n                // strip leading and trailing spaces\n                if (pass.length === 0) {\n                    return true;\n                }\n\n                if (!/[a-z]/i.test(v) || !/[0-9]/.test(v)) {\n                    return false;\n                }\n\n                if (pass.length < 7) {\n                    return false;\n                }\n\n                return true;\n            },\n            $.mage.__('Please enter 7 or more characters, using both numeric and alphabetic.')\n        ],\n        'validate-customer-password': [\n            function (v, elm) {\n                var validator = this,\n                    counter = 0,\n                    passwordMinLength = $(elm).data('password-min-length'),\n                    passwordMinCharacterSets = $(elm).data('password-min-character-sets'),\n                    pass = v.trim(),\n                    result = pass.length >= passwordMinLength;\n\n                if (result === false) {\n                    validator.passwordErrorMessage = $.mage.__('Minimum length of this field must be equal or greater than %1 symbols. Leading and trailing spaces will be ignored.').replace('%1', passwordMinLength); //eslint-disable-line max-len\n\n                    return result;\n                }\n\n                if (pass.match(/\\d+/)) {\n                    counter++;\n                }\n\n                if (pass.match(/[a-z]+/)) {\n                    counter++;\n                }\n\n                if (pass.match(/[A-Z]+/)) {\n                    counter++;\n                }\n\n                if (pass.match(/[^a-zA-Z0-9]+/)) {\n                    counter++;\n                }\n\n                if (counter < passwordMinCharacterSets) {\n                    result = false;\n                    validator.passwordErrorMessage = $.mage.__('Minimum of different classes of characters in password is %1. Classes of characters: Lower Case, Upper Case, Digits, Special Characters.').replace('%1', passwordMinCharacterSets); //eslint-disable-line max-len\n                }\n\n                return result;\n            }, function () {\n                return this.passwordErrorMessage;\n            }\n        ],\n        'validate-url': [\n            function (v) {\n                if ($.mage.isEmptyNoTrim(v)) {\n                    return true;\n                }\n                v = (v || '').replace(/^\\s+/, '').replace(/\\s+$/, '');\n\n                return (/^(http|https|ftp):\\/\\/(([A-Z0-9]([A-Z0-9_-]*[A-Z0-9]|))(\\.[A-Z0-9]([A-Z0-9_-]*[A-Z0-9]|))*)(:(\\d+))?(\\/[A-Z0-9~](([A-Z0-9_~-]|\\.)*[A-Z0-9~]|))*\\/?(.*)?$/i).test(v); //eslint-disable-line max-len\n\n            },\n            $.mage.__('Please enter a valid URL. Protocol is required (http://, https:// or ftp://).')\n        ],\n        'validate-clean-url': [\n            function (v) {\n                return $.mage.isEmptyNoTrim(v) || /^(http|https|ftp):\\/\\/(([A-Z0-9][A-Z0-9_-]*)(\\.[A-Z0-9][A-Z0-9_-]*)+.(com|org|net|dk|at|us|tv|info|uk|co.uk|biz|se)$)(:(\\d+))?\\/?/i.test(v) || /^(www)((\\.[A-Z0-9][A-Z0-9_-]*)+.(com|org|net|dk|at|us|tv|info|uk|co.uk|biz|se)$)(:(\\d+))?\\/?/i.test(v); //eslint-disable-line max-len\n\n            },\n            $.mage.__('Please enter a valid URL. For example http://www.example.com or www.example.com.')\n        ],\n        'validate-xml-identifier': [\n            function (v) {\n                return $.mage.isEmptyNoTrim(v) || /^[A-Z][A-Z0-9_\\/-]*$/i.test(v);\n\n            },\n            $.mage.__('Please enter a valid XML-identifier (Ex: something_1, block5, id-4).')\n        ],\n        'validate-ssn': [\n            function (v) {\n                return $.mage.isEmptyNoTrim(v) || /^\\d{3}-?\\d{2}-?\\d{4}$/.test(v);\n\n            },\n            $.mage.__('Please enter a valid social security number (Ex: 123-45-6789).')\n        ],\n        'validate-zip-us': [\n            function (v) {\n                return $.mage.isEmptyNoTrim(v) || /(^\\d{5}$)|(^\\d{5}-\\d{4}$)/.test(v);\n\n            },\n            $.mage.__('Please enter a valid zip code (Ex: 90602 or 90602-1234).')\n        ],\n        'validate-date-au': [\n            function (v) {\n                var regex, d;\n\n                if ($.mage.isEmptyNoTrim(v)) {\n                    return true;\n                }\n                regex = /^(\\d{2})\\/(\\d{2})\\/(\\d{4})$/;\n\n                if ($.mage.isEmpty(v) || !regex.test(v)) {\n                    return false;\n                }\n                d = new Date(v.replace(regex, '$2/$1/$3'));\n\n                return parseInt(RegExp.$2, 10) === 1 + d.getMonth() &&\n                    parseInt(RegExp.$1, 10) === d.getDate() &&\n                    parseInt(RegExp.$3, 10) === d.getFullYear();\n\n            },\n            $.mage.__('Please use this date format: dd/mm/yyyy. For example 17/03/2006 for the 17th of March, 2006.')\n        ],\n        'validate-currency-dollar': [\n            function (v) {\n                return $.mage.isEmptyNoTrim(v) || /^\\$?\\-?([1-9]{1}[0-9]{0,2}(\\,[0-9]{3})*(\\.[0-9]{0,2})?|[1-9]{1}\\d*(\\.[0-9]{0,2})?|0(\\.[0-9]{0,2})?|(\\.[0-9]{1,2})?)$/.test(v); //eslint-disable-line max-len\n\n            },\n            $.mage.__('Please enter a valid $ amount. For example $100.00.')\n        ],\n        'validate-not-negative-number': [\n            function (v) {\n                if ($.mage.isEmptyNoTrim(v)) {\n                    return true;\n                }\n                v = $.mage.parseNumber(v);\n\n                return !isNaN(v) && v >= 0;\n\n            },\n            $.mage.__('Please enter a number 0 or greater in this field.')\n        ],\n        // validate-not-negative-number should be replaced in all places with this one and then removed\n        'validate-zero-or-greater': [\n            function (v) {\n                if ($.mage.isEmptyNoTrim(v)) {\n                    return true;\n                }\n                v = $.mage.parseNumber(v);\n\n                return !isNaN(v) && v >= 0;\n\n            },\n            $.mage.__('Please enter a number 0 or greater in this field.')\n        ],\n        'validate-greater-than-zero': [\n            function (v) {\n                if ($.mage.isEmptyNoTrim(v)) {\n                    return true;\n                }\n                v = $.mage.parseNumber(v);\n\n                return !isNaN(v) && v > 0;\n            },\n            $.mage.__('Please enter a number greater than 0 in this field.')\n        ],\n        'validate-css-length': [\n            function (v) {\n                if (v !== '') {\n                    return (/^[0-9]*\\.*[0-9]+(px|pc|pt|ex|em|mm|cm|in|%)?$/).test(v);\n                }\n\n                return true;\n            },\n            $.mage.__('Please input a valid CSS-length (Ex: 100px, 77pt, 20em, .5ex or 50%).')\n        ],\n        // Additional methods\n        'validate-number': [\n            function (v) {\n                return $.mage.isEmptyNoTrim(v) || !isNaN($.mage.parseNumber(v)) && /^\\s*-?\\d*(\\.\\d*)?\\s*$/.test(v);\n            },\n            $.mage.__('Please enter a valid number in this field.')\n        ],\n        'required-number': [\n            function (v) {\n                return !!v.length;\n            },\n            $.mage.__('Please enter a valid number in this field.')\n        ],\n        'validate-number-range': [\n            function (v, elm, param) {\n                var numValue, dataAttrRange, classNameRange, result, range, m, classes, ii;\n\n                if ($.mage.isEmptyNoTrim(v)) {\n                    return true;\n                }\n\n                numValue = $.mage.parseNumber(v);\n\n                if (isNaN(numValue)) {\n                    return false;\n                }\n\n                dataAttrRange = /^(-?[\\d.,]+)?-(-?[\\d.,]+)?$/;\n                classNameRange = /^number-range-(-?[\\d.,]+)?-(-?[\\d.,]+)?$/;\n                result = true;\n                range = param;\n\n                if (typeof range === 'string') {\n                    m = dataAttrRange.exec(range);\n\n                    if (m) {\n                        result = result && $.mage.isBetween(numValue, m[1], m[2]);\n                    } else {\n                        result = false;\n                    }\n                } else if (elm && elm.className) {\n                    classes = elm.className.split(' ');\n                    ii = classes.length;\n\n                    while (ii--) {\n                        range = classes[ii];\n                        m = classNameRange.exec(range);\n\n                        if (m) { //eslint-disable-line max-depth\n                            result = result && $.mage.isBetween(numValue, m[1], m[2]);\n                            break;\n                        }\n                    }\n                }\n\n                return result;\n            },\n            $.mage.__('The value is not within the specified range.'),\n            true\n        ],\n        'validate-digits': [\n            function (v) {\n                return $.mage.isEmptyNoTrim(v) || !/[^\\d]/.test(v);\n            },\n            $.mage.__('Please enter a valid number in this field.')\n        ],\n        'validate-forbidden-extensions': [\n            function (v, elem) {\n                var forbiddenExtensions = $(elem).attr('data-validation-params'),\n                    forbiddenExtensionsArray = forbiddenExtensions.split(','),\n                    extensionsArray = v.split(','),\n                    result = true;\n\n                this.validateExtensionsMessage = $.mage.__('Forbidden extensions has been used. Avoid usage of ') +\n                    forbiddenExtensions;\n\n                $.each(extensionsArray, function (key, extension) {\n                    if (forbiddenExtensionsArray.indexOf(extension) !== -1) {\n                        result = false;\n                    }\n                });\n\n                return result;\n            }, function () {\n                return this.validateExtensionsMessage;\n            }\n        ],\n        'validate-digits-range': [\n            function (v, elm, param) {\n                var numValue, dataAttrRange, classNameRange, result, range, m, classes, ii;\n\n                if ($.mage.isEmptyNoTrim(v)) {\n                    return true;\n                }\n\n                numValue = $.mage.parseNumber(v);\n\n                if (isNaN(numValue)) {\n                    return false;\n                }\n\n                dataAttrRange = /^(-?\\d+)?-(-?\\d+)?$/;\n                classNameRange = /^digits-range-(-?\\d+)?-(-?\\d+)?$/;\n                result = true;\n                range = param;\n\n                if (typeof range === 'string') {\n                    m = dataAttrRange.exec(range);\n\n                    if (m) {\n                        result = result && $.mage.isBetween(numValue, m[1], m[2]);\n                    } else {\n                        result = false;\n                    }\n                } else if (elm && elm.className) {\n                    classes = elm.className.split(' ');\n                    ii = classes.length;\n\n                    while (ii--) {\n                        range = classes[ii];\n                        m = classNameRange.exec(range);\n\n                        if (m) { //eslint-disable-line max-depth\n                            result = result && $.mage.isBetween(numValue, m[1], m[2]);\n                            break;\n                        }\n                    }\n                }\n\n                return result;\n            },\n            $.mage.__('The value is not within the specified range.'),\n            true\n        ],\n        'validate-range': [\n            function (v, elm) {\n                var minValue, maxValue, ranges, reRange, result, values,\n                    i, name, validRange, minValidRange, maxValidRange;\n\n                if ($.mage.isEmptyNoTrim(v)) {\n                    return true;\n                } else if ($.validator.methods['validate-digits'] && $.validator.methods['validate-digits'](v)) {\n                    minValue = maxValue = $.mage.parseNumber(v);\n                } else {\n                    ranges = /^(-?\\d+)?-(-?\\d+)?$/.exec(v);\n\n                    if (ranges) {\n                        minValue = $.mage.parseNumber(ranges[1]);\n                        maxValue = $.mage.parseNumber(ranges[2]);\n\n                        if (minValue > maxValue) { //eslint-disable-line max-depth\n                            return false;\n                        }\n                    } else {\n                        return false;\n                    }\n                }\n                reRange = /^range-(-?\\d+)?-(-?\\d+)?$/;\n                result = true;\n                values = $(elm).prop('class').split(' ');\n\n                for (i = values.length - 1; i >= 0; i--) {\n                    name = values[i];\n                    validRange = reRange.exec(name);\n\n                    if (validRange) {\n                        minValidRange = $.mage.parseNumber(validRange[1]);\n                        maxValidRange = $.mage.parseNumber(validRange[2]);\n                        result = result &&\n                            (isNaN(minValidRange) || minValue >= minValidRange) &&\n                            (isNaN(maxValidRange) || maxValue <= maxValidRange);\n                    }\n                }\n\n                return result;\n            },\n            $.mage.__('The value is not within the specified range.')\n        ],\n        'validate-alpha': [\n            function (v) {\n                return $.mage.isEmptyNoTrim(v) || /^[a-zA-Z]+$/.test(v);\n            },\n            $.mage.__('Please use letters only (a-z or A-Z) in this field.')\n        ],\n        'validate-code': [\n            function (v) {\n                return $.mage.isEmptyNoTrim(v) || /^[a-zA-Z]+[a-zA-Z0-9_]+$/.test(v);\n            },\n            $.mage.__('Please use only letters (a-z or A-Z), numbers (0-9) or underscore (_) in this field, and the first character should be a letter.') //eslint-disable-line max-len\n        ],\n        'validate-alphanum': [\n            function (v) {\n                return $.mage.isEmptyNoTrim(v) || /^[a-zA-Z0-9]+$/.test(v);\n            },\n            $.mage.__('Please use only letters (a-z or A-Z) or numbers (0-9) in this field. No spaces or other characters are allowed.') //eslint-disable-line max-len\n        ],\n        'validate-not-number-first': [\n            function (value) {\n                return $.mage.isEmptyNoTrim(value) || /^[^0-9-\\.].*$/.test(value.trim());\n            },\n            $.mage.__('First character must be letter.')\n        ],\n        'validate-date': [\n            function (value, params, additionalParams) {\n                var test = moment(value, utils.convertToMomentFormat(additionalParams.dateFormat));\n\n                return $.mage.isEmptyNoTrim(value) || test.isValid();\n            },\n            $.mage.__('Please enter a valid date.')\n\n        ],\n        'validate-date-range': [\n            function (v, elm) {\n                var m = /\\bdate-range-(\\w+)-(\\w+)\\b/.exec(elm.className),\n                    currentYear, normalizedTime, dependentElements;\n\n                if (!m || m[2] === 'to' || $.mage.isEmptyNoTrim(v)) {\n                    return true;\n                }\n\n                currentYear = new Date().getFullYear() + '';\n\n                /**\n                 * @param {String} vd\n                 * @return {Number}\n                 */\n                normalizedTime = function (vd) {\n                    vd = vd.split(/[.\\/]/);\n\n                    if (vd[2] && vd[2].length < 4) {\n                        vd[2] = currentYear.substr(0, vd[2].length) + vd[2];\n                    }\n\n                    return new Date(vd.join('/')).getTime();\n                };\n\n                dependentElements = $(elm.form).find('.validate-date-range.date-range-' + m[1] + '-to');\n\n                return !dependentElements.length || $.mage.isEmptyNoTrim(dependentElements[0].value) ||\n                    normalizedTime(v) <= normalizedTime(dependentElements[0].value);\n            },\n            $.mage.__('Make sure the To Date is later than or the same as the From Date.')\n        ],\n        'validate-cpassword': [\n            function () {\n                var conf = $('#confirmation').length > 0 ? $('#confirmation') : $($('.validate-cpassword')[0]),\n                    pass = false,\n                    passwordElements, i, passwordElement;\n\n                if ($('#password')) {\n                    pass = $('#password');\n                }\n                passwordElements = $('.validate-password');\n\n                for (i = 0; i < passwordElements.length; i++) {\n                    passwordElement = $(passwordElements[i]);\n\n                    if (passwordElement.closest('form').attr('id') === conf.closest('form').attr('id')) {\n                        pass = passwordElement;\n                    }\n                }\n\n                if ($('.validate-admin-password').length) {\n                    pass = $($('.validate-admin-password')[0]);\n                }\n\n                return pass.val() === conf.val();\n            },\n            $.mage.__('Please make sure your passwords match.')\n        ],\n        'validate-identifier': [\n            function (v) {\n                return $.mage.isEmptyNoTrim(v) || /^[a-z0-9][a-z0-9_\\/-]+(\\.[a-z0-9_-]+)?$/.test(v);\n            },\n            $.mage.__('Please enter a valid URL Key (Ex: \"example-page\", \"example-page.html\" or \"anotherlevel/example-page\").') //eslint-disable-line max-len\n        ],\n        'validate-zip-international': [\n\n            /*function(v) {\n             // @TODO: Cleanup\n             return Validation.get('IsEmpty').test(v) ||\n             /(^[A-z0-9]{2,10}([\\s]{0,1}|[\\-]{0,1})[A-z0-9]{2,10}$)/.test(v);\n             }*/\n            function () {\n                return true;\n            },\n            $.mage.__('Please enter a valid zip code.')\n        ],\n        'validate-one-required': [\n            function (v, elm) {\n                var p = $(elm).parent(),\n                    options = p.find('input');\n\n                return options.map(function (el) {\n                    return $(el).val();\n                }).length > 0;\n            },\n            $.mage.__('Please select one of the options above.')\n        ],\n        'validate-state': [\n            function (v) {\n                return v !== 0;\n            },\n            $.mage.__('Please select State/Province.')\n        ],\n        'required-file': [\n            function (v, elm) {\n                var result = !$.mage.isEmptyNoTrim(v),\n                    ovId;\n\n                if (!result) {\n                    ovId = $('#' + $(elm).attr('id') + '_value');\n\n                    if (ovId.length > 0) {\n                        result = !$.mage.isEmptyNoTrim(ovId.val());\n                    }\n                }\n\n                return result;\n            },\n            $.mage.__('Please select a file.')\n        ],\n        'validate-ajax-error': [\n            function (v, element) {\n                element = $(element);\n                element.on('change.ajaxError', function () {\n                    element.removeClass('validate-ajax-error');\n                    element.off('change.ajaxError');\n                });\n\n                return !element.hasClass('validate-ajax-error');\n            },\n            ''\n        ],\n        'validate-optional-datetime': [\n            function (v, elm, param) {\n                var dateTimeParts = $('.datetime-picker[id^=\"options_' + param + '\"]'),\n                    hasWithValue = false,\n                    hasWithNoValue = false,\n                    pattern = /day_part$/i,\n                    i;\n\n                for (i = 0; i < dateTimeParts.length; i++) {\n                    if (!pattern.test($(dateTimeParts[i]).attr('id'))) {\n                        if ($(dateTimeParts[i]).val() === 's') { //eslint-disable-line max-depth\n                            hasWithValue = true;\n                        } else {\n                            hasWithNoValue = true;\n                        }\n                    }\n                }\n\n                return hasWithValue ^ hasWithNoValue;\n            },\n            $.mage.__('The field isn\\'t complete.')\n        ],\n        'validate-required-datetime': [\n            function (v, elm, param) {\n                var dateTimeParts = $('.datetime-picker[id^=\"options_' + param + '\"]'),\n                    i;\n\n                for (i = 0; i < dateTimeParts.length; i++) {\n                    if (dateTimeParts[i].value === '') {\n                        return false;\n                    }\n                }\n\n                return true;\n            },\n            $.mage.__('This is a required field.')\n        ],\n        'validate-one-required-by-name': [\n            function (v, elm, selector) {\n                var name = elm.name.replace(/([\\\\\"])/g, '\\\\$1'),\n                    container = this.currentForm;\n\n                selector = selector === true ? 'input[name=\"' + name + '\"]:checked' : selector;\n\n                return !!container.querySelectorAll(selector).length;\n            },\n            $.mage.__('Please select one of the options.')\n        ],\n        'less-than-equals-to': [\n            function (value, element, params) {\n                if ($.isNumeric($(params).val()) && $.isNumeric(value)) {\n                    this.lteToVal = $(params).val();\n\n                    return parseFloat(value) <= parseFloat($(params).val());\n                }\n\n                return true;\n            },\n            function () {\n                var message = $.mage.__('Please enter a value less than or equal to %s.');\n\n                return message.replace('%s', this.lteToVal);\n            }\n        ],\n        'greater-than-equals-to': [\n            function (value, element, params) {\n                if ($.isNumeric($(params).val()) && $.isNumeric(value)) {\n                    this.gteToVal = $(params).val();\n\n                    return parseFloat(value) >= parseFloat($(params).val());\n                }\n\n                return true;\n            },\n            function () {\n                var message = $.mage.__('Please enter a value greater than or equal to %s.');\n\n                return message.replace('%s', this.gteToVal);\n            }\n        ],\n        'validate-emails': [\n            function (value) {\n                var validRegexp, emails, i;\n\n                if ($.mage.isEmpty(value)) {\n                    return true;\n                }\n                validRegexp = /^([a-z0-9,!\\#\\$%&'\\*\\+\\/=\\?\\^_`\\{\\|\\}~-]|[\\u00A0-\\uD7FF\\uF900-\\uFDCF\\uFDF0-\\uFFEF])+(\\.([a-z0-9,!\\#\\$%&'\\*\\+\\/=\\?\\^_`\\{\\|\\}~-]|[\\u00A0-\\uD7FF\\uF900-\\uFDCF\\uFDF0-\\uFFEF])+)*@([a-z0-9-]|[\\u00A0-\\uD7FF\\uF900-\\uFDCF\\uFDF0-\\uFFEF])+(\\.([a-z0-9-]|[\\u00A0-\\uD7FF\\uF900-\\uFDCF\\uFDF0-\\uFFEF])+)*\\.(([a-z]|[\\u00A0-\\uD7FF\\uF900-\\uFDCF\\uFDF0-\\uFFEF]){2,})$/i; //eslint-disable-line max-len\n                emails = value.split(/[\\s\\n\\,]+/g);\n\n                for (i = 0; i < emails.length; i++) {\n                    if (!validRegexp.test(emails[i].trim())) {\n                        return false;\n                    }\n                }\n\n                return true;\n            },\n            $.mage.__('Please enter valid email addresses, separated by commas. For example, johndoe@domain.com, johnsmith@domain.com.') //eslint-disable-line max-len\n        ],\n\n        'validate-cc-type-select': [\n\n            /**\n             * Validate credit card type matches credit card number\n             * @param {*} value - select credit card type\n             * @param {*} element - element contains the select box for credit card types\n             * @param {*} params - selector for credit card number\n             * @return {Boolean}\n             */\n            function (value, element, params) {\n                if (value && params && creditCartTypes[value]) {\n                    return creditCartTypes[value][0].test($(params).val().replace(/\\s+/g, ''));\n                }\n\n                return false;\n            },\n            $.mage.__('Card type does not match credit card number.')\n        ],\n        'validate-cc-number': [\n\n            /**\n             * Validate credit card number based on mod 10.\n             *\n             * @param {*} value - credit card number\n             * @return {Boolean}\n             */\n            function (value) {\n                if (value) {\n                    return validateCreditCard(value);\n                }\n\n                return false;\n            },\n            $.mage.__('Please enter a valid credit card number.')\n        ],\n        'validate-cc-type': [\n\n            /**\n             * Validate credit card number is for the correct credit card type.\n             *\n             * @param {String} value - credit card number\n             * @param {*} element - element contains credit card number\n             * @param {*} params - selector for credit card type\n             * @return {Boolean}\n             */\n            function (value, element, params) {\n                var ccType;\n\n                if (value && params) {\n                    ccType = $(params).val();\n                    value = value.replace(/\\s/g, '').replace(/\\-/g, '');\n\n                    if (creditCartTypes[ccType] && creditCartTypes[ccType][0]) {\n                        return creditCartTypes[ccType][0].test(value);\n                    } else if (creditCartTypes[ccType] && !creditCartTypes[ccType][0]) {\n                        return true;\n                    }\n                }\n\n                return false;\n            },\n            $.mage.__('Credit card number does not match credit card type.')\n        ],\n        'validate-cc-exp': [\n\n            /**\n             * Validate credit card expiration date, make sure it's within the year and not before current month.\n             *\n             * @param {*} value - month\n             * @param {*} element - element contains month\n             * @param {*} params - year selector\n             * @return {Boolean}\n             */\n            function (value, element, params) {\n                var isValid = false,\n                    month, year, currentTime, currentMonth, currentYear;\n\n                if (value && params) {\n                    month = value;\n                    year = $(params).val();\n                    currentTime = new Date();\n                    currentMonth = currentTime.getMonth() + 1;\n                    currentYear = currentTime.getFullYear();\n\n                    isValid = !year || year > currentYear || year == currentYear && month >= currentMonth; //eslint-disable-line\n                }\n\n                return isValid;\n            },\n            $.mage.__('Incorrect credit card expiration date.')\n        ],\n        'validate-cc-cvn': [\n\n            /**\n             * Validate credit card cvn based on credit card type.\n             *\n             * @param {*} value - credit card cvn\n             * @param {*} element - element contains credit card cvn\n             * @param {*} params - credit card type selector\n             * @return {*}\n             */\n            function (value, element, params) {\n                var ccType;\n\n                if (value && params) {\n                    ccType = $(params).val();\n\n                    if (creditCartTypes[ccType] && creditCartTypes[ccType][0]) {\n                        return creditCartTypes[ccType][1].test(value);\n                    }\n                }\n\n                return false;\n            },\n            $.mage.__('Please enter a valid credit card verification number.')\n        ],\n        'validate-cc-ukss': [\n\n            /**\n             * Validate Switch/Solo/Maestro issue number and start date is filled.\n             *\n             * @param {*} value - input field value\n             * @return {*}\n             */\n            function (value) {\n                return value;\n            },\n            $.mage.__('Please enter issue number or start date for switch/solo card type.')\n        ],\n        'validate-length': [\n            function (v, elm) {\n                var reMax = new RegExp(/^maximum-length-[0-9]+$/),\n                    reMin = new RegExp(/^minimum-length-[0-9]+$/),\n                    validator = this,\n                    result = true,\n                    length = 0;\n\n                $.each(elm.className.split(' '), function (index, name) {\n                    if (name.match(reMax) && result) {\n                        length = name.split('-')[2];\n                        result = v.length <= length;\n                        validator.validateMessage =\n                            $.mage.__('Please enter less or equal than %1 symbols.').replace('%1', length);\n                    }\n\n                    if (name.match(reMin) && result && !$.mage.isEmpty(v)) {\n                        length = name.split('-')[2];\n                        result = v.length >= length;\n                        validator.validateMessage =\n                            $.mage.__('Please enter more or equal than %1 symbols.').replace('%1', length);\n                    }\n                });\n\n                return result;\n            }, function () {\n                return this.validateMessage;\n            }\n        ],\n        'required-entry': [\n            function (value) {\n                return !$.mage.isEmpty(value);\n            }, $.mage.__('This is a required field.')\n        ],\n        'not-negative-amount': [\n            function (v) {\n                if (v.length) {\n                    return (/^\\s*\\d+([,.]\\d+)*\\s*%?\\s*$/).test(v);\n                }\n\n                return true;\n            },\n            $.mage.__('Please enter positive number in this field.')\n        ],\n        'validate-per-page-value-list': [\n            function (v) {\n                var isValid = true,\n                    values = v.split(','),\n                    i;\n\n                if ($.mage.isEmpty(v)) {\n                    return isValid;\n                }\n\n                for (i = 0; i < values.length; i++) {\n                    if (!/^[0-9]+$/.test(values[i])) {\n                        isValid = false;\n                    }\n                }\n\n                return isValid;\n            },\n            $.mage.__('Please enter a valid value, ex: 10,20,30')\n        ],\n        'validate-per-page-value': [\n            function (v, elm) {\n                var values;\n\n                if ($.mage.isEmpty(v)) {\n                    return false;\n                }\n                values = $('#' + elm.id + '_values').val().split(',');\n\n                return values.indexOf(v) !== -1;\n            },\n            $.mage.__('Please enter a valid value from list')\n        ],\n        'validate-new-password': [\n            function (v) {\n                if ($.validator.methods['validate-password'] && !$.validator.methods['validate-password'](v)) {\n                    return false;\n                }\n\n                if ($.mage.isEmpty(v) && v !== '') {\n                    return false;\n                }\n\n                return true;\n            },\n            $.mage.__('Please enter 6 or more characters. Leading and trailing spaces will be ignored.')\n        ],\n        'required-if-not-specified': [\n            function (value, element, params) {\n                var valid = false,\n                    alternate = $(params),\n                    alternateValue;\n\n                if (alternate.length > 0) {\n                    valid = this.check(alternate);\n                    // if valid, it may be blank, so check for that\n                    if (valid) {\n                        alternateValue = alternate.val();\n\n                        if (typeof alternateValue == 'undefined' || alternateValue.length === 0) { //eslint-disable-line\n                            valid = false;\n                        }\n                    }\n                }\n\n                if (!valid) {\n                    valid = !this.optional(element);\n                }\n\n                return valid;\n            },\n            $.mage.__('This is a required field.')\n        ],\n        'required-if-all-sku-empty-and-file-not-loaded': [\n            function (value, element, params) {\n                var valid = false,\n                    alternate = $(params.specifiedId),\n                    alternateValue;\n\n                if (alternate.length > 0) {\n                    valid = this.check(alternate);\n                    // if valid, it may be blank, so check for that\n                    if (valid) {\n                        alternateValue = alternate.val();\n\n                        if (typeof alternateValue == 'undefined' || alternateValue.length === 0) { //eslint-disable-line\n                            valid = false;\n                        }\n                    }\n                }\n\n                if (!valid) {\n                    valid = !this.optional(element);\n                }\n\n                $('input[' + params.dataSku + '=true]').each(function () {\n                    if ($(this).val() !== '') {\n                        valid = true;\n                    }\n                });\n\n                return valid;\n            },\n            $.mage.__('Please enter valid SKU key.')\n        ],\n        'required-if-specified': [\n            function (value, element, params) {\n                var valid = true,\n                    dependent = $(params),\n                    dependentValue;\n\n                if (dependent.length > 0) {\n                    valid = this.check(dependent);\n                    // if valid, it may be blank, so check for that\n                    if (valid) {\n                        dependentValue = dependent.val();\n                        valid = typeof dependentValue != 'undefined' && dependentValue.length > 0;\n                    }\n                }\n\n                if (valid) {\n                    valid = !this.optional(element);\n                } else {\n                    valid = true; // dependent was not valid, so don't even check\n                }\n\n                return valid;\n            },\n            $.mage.__('This is a required field.')\n        ],\n        'required-number-if-specified': [\n            function (value, element, params) {\n                var valid = true,\n                    dependent = $(params),\n                    depeValue;\n\n                if (dependent.length) {\n                    valid = this.check(dependent);\n\n                    if (valid) {\n                        depeValue = dependent[0].value;\n                        valid = !!(depeValue && depeValue.length);\n                    }\n                }\n\n                return valid ? !!value.length : true;\n            },\n            $.mage.__('Please enter a valid number.')\n        ],\n        'datetime-validation': [\n            function (value, element) {\n                var isValid = true;\n\n                if ($(element).val().length === 0) {\n                    isValid = false;\n                    $(element).addClass('mage-error');\n                }\n\n                return isValid;\n            },\n            $.mage.__('This is required field')\n        ],\n        'required-text-swatch-entry': [\n            tableSingleValidation,\n            $.mage.__('Admin is a required field in each row.')\n        ],\n        'required-visual-swatch-entry': [\n            tableSingleValidation,\n            $.mage.__('Admin is a required field in each row.')\n        ],\n        'required-dropdown-attribute-entry': [\n            tableSingleValidation,\n            $.mage.__('Admin is a required field in each row.')\n        ],\n        'validate-item-quantity': [\n            function (value, element, params) {\n                var validator = this,\n                    result = false,\n                    // obtain values for validation\n                    qty = $.mage.parseNumber(value),\n                    isMinAllowedValid = typeof params.minAllowed === 'undefined' ||\n                        qty >= $.mage.parseNumber(params.minAllowed),\n                    isMaxAllowedValid = typeof params.maxAllowed === 'undefined' ||\n                        qty <= $.mage.parseNumber(params.maxAllowed),\n                    isQtyIncrementsValid = typeof params.qtyIncrements === 'undefined' ||\n                        resolveModulo(qty, $.mage.parseNumber(params.qtyIncrements)) === 0.0;\n\n                result = qty > 0;\n\n                if (result === false) {\n                    validator.itemQtyErrorMessage = $.mage.__('Please enter a quantity greater than 0.');//eslint-disable-line max-len\n\n                    return result;\n                }\n\n                result = isMinAllowedValid;\n\n                if (result === false) {\n                    validator.itemQtyErrorMessage = $.mage.__('The fewest you may purchase is %1.').replace('%1', params.minAllowed);//eslint-disable-line max-len\n\n                    return result;\n                }\n\n                result = isMaxAllowedValid;\n\n                if (result === false) {\n                    validator.itemQtyErrorMessage = $.mage.__('The maximum you may purchase is %1.').replace('%1', params.maxAllowed);//eslint-disable-line max-len\n\n                    return result;\n                }\n\n                result = isQtyIncrementsValid;\n\n                if (result === false) {\n                    validator.itemQtyErrorMessage = $.mage.__('You can buy this product only in quantities of %1 at a time.').replace('%1', params.qtyIncrements);//eslint-disable-line max-len\n\n                    return result;\n                }\n\n                return result;\n            }, function () {\n                return this.itemQtyErrorMessage;\n            }\n        ],\n        'password-not-equal-to-user-name': [\n            function (value, element, params) {\n                if (typeof params === 'string') {\n                    return value.toLowerCase() !== params.toLowerCase();\n                }\n\n                return true;\n            },\n            $.mage.__('The password can\\'t be the same as the email address. Create a new password and try again.')\n        ]\n    };\n\n    $.each(rules, function (i, rule) {\n        rule.unshift(i);\n        $.validator.addMethod.apply($.validator, rule);\n    });\n    $.validator.addClassRules({\n        'required-option': {\n            required: true\n        },\n        'required-options-count': {\n            required: true\n        },\n        'validate-both-passwords': {\n            'validate-cpassword': true\n        }\n    });\n    $.validator.messages = $.extend($.validator.messages, {\n        required: $.mage.__('This is a required field.'),\n        remote: $.mage.__('Please fix this field.'),\n        email: $.mage.__('Please enter a valid email address.'),\n        url: $.mage.__('Please enter a valid URL.'),\n        date: $.mage.__('Please enter a valid date.'),\n        dateISO: $.mage.__('Please enter a valid date (ISO).'),\n        number: $.mage.__('Please enter a valid number.'),\n        digits: $.mage.__('Please enter only digits.'),\n        creditcard: $.mage.__('Please enter a valid credit card number.'),\n        equalTo: $.mage.__('Please enter the same value again.'),\n        maxlength: $.validator.format($.mage.__('Please enter no more than {0} characters.')),\n        minlength: $.validator.format($.mage.__('Please enter at least {0} characters.')),\n        rangelength: $.validator.format($.mage.__('Please enter a value between {0} and {1} characters long.')),\n        range: $.validator.format($.mage.__('Please enter a value between {0} and {1}.')),\n        max: $.validator.format($.mage.__('Please enter a value less than or equal to {0}.')),\n        min: $.validator.format($.mage.__('Please enter a value greater than or equal to {0}.'))\n    });\n\n    if ($.metadata) {\n        // Setting the type as html5 to enable data-validate attribute\n        $.metadata.setType('html5');\n    }\n\n    showLabel = $.validator.prototype.showLabel;\n    $.extend(true, $.validator.prototype, {\n        /**\n         * @param {*} element\n         * @param {*} message\n         */\n        showLabel: function (element, message) {\n            var label, elem;\n\n            showLabel.call(this, element, message);\n\n            // ARIA (adding aria-invalid & aria-describedby)\n            label = this.errorsFor(element);\n            elem = $(element);\n\n            if (!label.attr('id')) {\n                label.attr('id', this.idOrName(element) + '-error');\n            }\n            elem.attr('aria-invalid', 'true')\n                .attr('aria-describedby', label.attr('id'));\n        }\n    });\n\n    /**\n     * Validate form field without instantiating validate plug-in.\n     *\n     * @param {Element|String} element - DOM element or selector\n     * @return {Boolean} validation result\n     */\n    $.validator.validateElement = function (element) {\n        var form, validator, valid, classes;\n\n        element = $(element);\n        form = element.get(0).form;\n        validator = form ? $(form).data('validator') : null;\n\n        if (validator) {\n            return validator.element(element.get(0));\n        }\n        valid = true;\n        classes = element.prop('class').split(' ');\n        $.each(classes, $.proxy(function (i, className) {\n            if (this.methods[className] && !this.methods[className](element.val(), element.get(0))) {\n                valid = false;\n\n                return valid;\n            }\n        }, this));\n\n        return valid;\n    };\n\n    originValidateDelegate = $.fn.validateDelegate;\n\n    /**\n     * @return {*}\n     */\n    $.fn.validateDelegate = function () {\n        if (!this[0].form) {\n            return this;\n        }\n\n        return originValidateDelegate.apply(this, arguments);\n    };\n\n    /**\n     * Validate single element.\n     *\n     * @param {Element} element\n     * @param {Object} config\n     * @returns {*}\n     */\n    $.validator.validateSingleElement = function (element, config) {\n        var errors = {},\n            valid = true,\n            validateConfig = {\n                errorElement: 'label',\n                ignore: '.ignore-validate',\n                hideError: false\n            },\n            form, validator, classes, elementValue;\n\n        $.extend(validateConfig, config);\n        element = $(element).not(validateConfig.ignore);\n\n        if (!element.length) {\n            return true;\n        }\n\n        form = element.get(0).form;\n        validator = form ? $(form).data('validator') : null;\n\n        if (validator) {\n            return validator.element(element.get(0));\n        }\n\n        classes = element.prop('class').split(' ');\n        validator = element.parent().data('validator') ||\n            $.mage.validation(validateConfig, element.parent()).validate;\n\n        element.removeClass(validator.settings.errorClass);\n        validator.toHide = validator.toShow;\n        validator.hideErrors();\n        validator.toShow = validator.toHide = $([]);\n\n        $.each(classes, $.proxy(function (i, className) {\n            elementValue = element.val();\n\n            if (element.is(':checkbox') || element.is(':radio')) {\n                elementValue = element.is(':checked') || null;\n            }\n\n            if (this.methods[className] && !this.methods[className](elementValue, element.get(0))) {\n                valid = false;\n                errors[element.get(0).name] = this.messages[className];\n                validator.invalid[element.get(0).name] = true;\n\n                if (!validateConfig.hideError) {\n                    validator.showErrors(errors);\n                }\n\n                return valid;\n            }\n        }, this));\n\n        return valid;\n    };\n\n    $.widget('mage.validation', {\n        options: {\n            meta: 'validate',\n            onfocusout: false,\n            onkeyup: false,\n            onclick: false,\n            ignoreTitle: true,\n            errorClass: 'mage-error',\n            errorElement: 'div',\n\n            /**\n             * @param {*} error\n             * @param {*} element\n             */\n            errorPlacement: function (error, element) {\n                var errorPlacement = element,\n                    fieldWrapper;\n\n                // logic for date-picker error placement\n                if (element.hasClass('_has-datepicker')) {\n                    errorPlacement = element.siblings('button');\n                }\n                // logic for field wrapper\n                fieldWrapper = element.closest('.addon');\n\n                if (fieldWrapper.length) {\n                    errorPlacement = fieldWrapper.after(error);\n                }\n                //logic for checkboxes/radio\n                if (element.is(':checkbox') || element.is(':radio')) {\n                    errorPlacement = element.parents('.control').children().last();\n\n                    //fallback if group does not have .control parent\n                    if (!errorPlacement.length) {\n                        errorPlacement = element.siblings('label').last();\n                    }\n                }\n                //logic for control with tooltip\n                if (element.siblings('.tooltip').length) {\n                    errorPlacement = element.siblings('.tooltip');\n                }\n                //logic for select with tooltip in after element\n                if (element.next().find('.tooltip').length) {\n                    errorPlacement = element.next();\n                }\n                errorPlacement.after(error);\n            }\n        },\n\n        /**\n         * Check if form pass validation rules without submit.\n         *\n         * @return boolean\n         */\n        isValid: function () {\n            return this.element.valid();\n        },\n\n        /**\n         * Remove validation error messages\n         */\n        clearError: function () {\n            if (arguments.length) {\n                $.each(arguments, $.proxy(function (index, item) {\n                    this.validate.prepareElement(item);\n                    this.validate.hideErrors();\n                }, this));\n            } else {\n                this.validate.resetForm();\n            }\n        },\n\n        /**\n         * Validation creation.\n         *\n         * @protected\n         */\n        _create: function () {\n            this.validate = this.element.validate(this.options);\n\n            // ARIA (adding aria-required attribute)\n            this.element\n                .find('.field.required')\n                .find('.control')\n                .find('input, select, textarea')\n                .attr('aria-required', 'true');\n            this.element\n                .find('.field.additional')\n                .find('.control')\n                .find('input, select, textarea')\n                .removeAttr('aria-required');\n            this._listenFormValidate();\n        },\n\n        /**\n         * Validation listening.\n         *\n         * @protected\n         */\n        _listenFormValidate: function () {\n            $('form').on('invalid-form.validate', this.listenFormValidateHandler);\n        },\n\n        /**\n         * Handle form validation. Focus on first invalid form field.\n         *\n         * @param {jQuery.Event} event\n         * @param {Object} validation\n         */\n        listenFormValidateHandler: function (event, validation) {\n            var firstActive = $(validation.errorList[0].element || []),\n                lastActive = $(validation.findLastActive() ||\n                    validation.errorList.length && validation.errorList[0].element || []),\n                windowHeight = $(window).height(),\n                parent, successList;\n\n            if (lastActive.is(':hidden')) {\n                parent = lastActive.parent();\n                $('html, body').animate({\n                    scrollTop: parent.offset().top - windowHeight / 2\n                });\n            }\n\n            // ARIA (removing aria attributes if success)\n            successList = validation.successList;\n\n            if (successList.length) {\n                $.each(successList, function () {\n                    $(this)\n                        .removeAttr('aria-describedby')\n                        .removeAttr('aria-invalid');\n                });\n            }\n\n            if (firstActive.length) {\n                $('html, body').stop().animate({\n                    scrollTop: firstActive.parent().offset().top - windowHeight / 2\n                });\n                firstActive.trigger('focus');\n            }\n        }\n    });\n\n    return $.mage.validation;\n});\n","mage/trim-input.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\ndefine([\n    'jquery',\n    'jquery-ui-modules/widget'\n], function ($) {\n    'use strict';\n\n    $.widget('mage.trimInput', {\n        options: {\n            cache: {}\n        },\n\n        /**\n         * Widget initialization\n         * @private\n         */\n        _create: function () {\n            this.options.cache.input = $(this.element);\n            this._bind();\n        },\n\n        /**\n         * Event binding, will monitor change, keyup and paste events.\n         * @private\n         */\n        _bind: function () {\n            if (this.options.cache.input.length) {\n                this._on(this.options.cache.input, {\n                    'change': this._trimInput,\n                    'keyup': this._trimInput,\n                    'paste': this._trimInput\n                });\n            }\n        },\n\n        /**\n         * Trim value\n         * @private\n         */\n        _trimInput: function () {\n            // Safari caret position workaround: storing carter position\n            var caretStart, caretEnd, input;\n\n            caretStart = this.options.cache.input.get(0).selectionStart;\n            caretEnd = this.options.cache.input.get(0).selectionEnd;\n\n            input = this._getInputValue().trim();\n\n            this.options.cache.input.val(input);\n\n            // Safari caret position workaround: setting caret position to previously stored values\n            if (caretStart !== null && caretEnd !== null) {\n                this.options.cache.input.get(0).setSelectionRange(caretStart, caretEnd);\n            }\n        },\n\n        /**\n         * Get input value\n         * @returns {*}\n         * @private\n         */\n        _getInputValue: function () {\n            return this.options.cache.input.val();\n        }\n    });\n\n    return $.mage.trimInput;\n});\n","mage/touch-slider.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\ndefine([\n    'jquery',\n    'underscore',\n    'jquery-ui-modules/slider'\n], function ($, _) {\n    'use strict';\n\n    /**\n     * Adds support for touch events for regular jQuery UI slider.\n     */\n    $.widget('mage.touchSlider', $.ui.slider, {\n\n        /**\n         * Creates instance of widget.\n         *\n         * @override\n         */\n        _create: function () {\n            _.bindAll(\n                this,\n                '_mouseDown',\n                '_mouseMove',\n                '_onTouchEnd'\n            );\n\n            return this._superApply(arguments);\n        },\n\n        /**\n         * Initializes mouse events on element.\n         * @override\n         */\n        _mouseInit: function () {\n            var result = this._superApply(arguments);\n\n            this.element\n                .off('mousedown.' + this.widgetName)\n                .on('touchstart.' + this.widgetName, this._mouseDown);\n\n            return result;\n        },\n\n        /**\n         * Elements' 'mousedown' event handler polyfill.\n         * @override\n         */\n        _mouseDown: function (event) {\n            var prevDelegate = this._mouseMoveDelegate,\n                result;\n\n            event = this._touchToMouse(event);\n            result = this._super(event);\n\n            if (prevDelegate === this._mouseMoveDelegate) {\n                return result;\n            }\n\n            $(document)\n                .off('mousemove.' + this.widgetName)\n                .off('mouseup.' + this.widgetName);\n\n            $(document)\n                .on('touchmove.' + this.widgetName, this._mouseMove)\n                .on('touchend.' + this.widgetName, this._onTouchEnd)\n                .on('tochleave.' + this.widgetName, this._onTouchEnd);\n\n            return result;\n        },\n\n        /**\n         * Documents' 'mousemove' event handler polyfill.\n         *\n         * @override\n         * @param {Event} event - Touch event object.\n         */\n        _mouseMove: function (event) {\n            event = this._touchToMouse(event);\n\n            return this._super(event);\n        },\n\n        /**\n         * Documents' 'touchend' event handler.\n         */\n        _onTouchEnd: function (event) {\n            $(document).trigger('mouseup');\n\n            return this._mouseUp(event);\n        },\n\n        /**\n         * Removes previously assigned touch handlers.\n         *\n         * @override\n         */\n        _mouseUp: function () {\n            this._removeTouchHandlers();\n\n            return this._superApply(arguments);\n        },\n\n        /**\n         * Removes previously assigned touch handlers.\n         *\n         * @override\n         */\n        _mouseDestroy: function () {\n            this._removeTouchHandlers();\n\n            return this._superApply(arguments);\n        },\n\n        /**\n         * Removes touch events from document object.\n         */\n        _removeTouchHandlers: function () {\n            $(document)\n                .off('touchmove.' + this.widgetName)\n                .off('touchend.' + this.widgetName)\n                .off('touchleave.' + this.widgetName);\n        },\n\n        /**\n         * Adds properties to the touch event to mimic mouse event.\n         *\n         * @param {Event} event - Touch event object.\n         * @returns {Event}\n         */\n        _touchToMouse: function (event) {\n            var orig = event.originalEvent,\n                touch = orig.touches[0];\n\n            return _.extend(event, {\n                which:      1,\n                pageX:      touch.pageX,\n                pageY:      touch.pageY,\n                clientX:    touch.clientX,\n                clientY:    touch.clientY,\n                screenX:    touch.screenX,\n                screenY:    touch.screenY\n            });\n        }\n    });\n\n    return $.mage.touchSlider;\n});\n","mage/ie-class-fixer.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\ndefine([], function () {\n    'use strict';\n\n    if (navigator.userAgent.match(/Trident.*rv[ :]*11\\./)) {\n        document.documentElement.classList.add('ie11');\n    }\n});\n","mage/calendar.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\n/*eslint max-depth: 0*/\ndefine([\n    'jquery',\n    'jquery-ui-modules/widget',\n    'jquery-ui-modules/datepicker',\n    'jquery-ui-modules/timepicker'\n], function ($) {\n    'use strict';\n\n    var calendarBasePrototype,\n        datepickerPrototype = $.datepicker.constructor.prototype;\n\n    $.datepicker.markerClassName = '_has-datepicker';\n\n    /**\n     * Extend JQuery date picker prototype with store local time methods\n     */\n    $.extend(datepickerPrototype, {\n        /**\n         * Get date/time according to store settings.\n         * We use serverTimezoneOffset (in seconds) instead of serverTimezoneSeconds\n         * in order to have ability to know actual store time even if page hadn't been reloaded\n         * @returns {Date}\n         */\n        _getTimezoneDate: function (options) {\n            // local time in ms\n            var ms = Date.now();\n\n            options = options || $.calendarConfig || {};\n\n            // Adjust milliseconds according to store timezone offset,\n            // mind the GMT zero offset\n            if (typeof options.serverTimezoneOffset !== 'undefined') {\n                // Make UTC time and add store timezone offset in seconds\n                ms += new Date().getTimezoneOffset() * 60 * 1000 + options.serverTimezoneOffset * 1000;\n            } else if (typeof options.serverTimezoneSeconds !== 'undefined') {\n                //Set milliseconds according to client local timezone offset\n                ms = (options.serverTimezoneSeconds + new Date().getTimezoneOffset() * 60) * 1000;\n            }\n\n            return new Date(ms);\n        },\n\n        /**\n         * Set date/time according to store settings.\n         * @param {String|Object} target - the target input field or division or span\n         */\n        _setTimezoneDateDatepicker: function (target) {\n            this._setDateDatepicker(target, this._getTimezoneDate());\n        }\n    });\n\n    /**\n     * Widget calendar\n     */\n    $.widget('mage.calendar', {\n        options: {\n            autoComplete: true\n        },\n\n        /**\n         * Merge global options with options passed to widget invoke\n         * @protected\n         */\n        _create: function () {\n            this._enableAMPM();\n            this.options = $.extend(\n                {},\n                $.calendarConfig ? $.calendarConfig : {},\n                this.options.showsTime ? {\n                    showTime: true,\n                    showHour: true,\n                    showMinute: true\n                } : {},\n                this.options\n            );\n            this._initPicker(this.element);\n            this._overwriteGenerateHtml();\n        },\n\n        /**\n         * Get picker name\n         * @protected\n         */\n        _picker: function () {\n            return this.options.showsTime ? 'datetimepicker' : 'datepicker';\n        },\n\n        /**\n         * Fix for Timepicker - Set ampm option for Timepicker if timeformat contains string 'tt'\n         * @protected\n         */\n        _enableAMPM: function () {\n            if (this.options.timeFormat && this.options.timeFormat.indexOf('tt') >= 0) {\n                this.options.ampm = true;\n            }\n        },\n\n        /**\n         * Wrapper for overwrite jQuery UI datepicker function.\n         */\n        _overwriteGenerateHtml: function () {\n            /**\n             * Overwrite jQuery UI datepicker function.\n             * Reason: magento date could be set before calendar show\n             * but local date will be styled as current in original _generateHTML\n             *\n             * @param {Object} inst - instance datepicker.\n             * @return {String} html template\n             */\n            $.datepicker.constructor.prototype._generateHTML = function (inst) {\n                var today = this._getTimezoneDate(),\n                    isRTL = this._get(inst, 'isRTL'),\n                    showButtonPanel = this._get(inst, 'showButtonPanel'),\n                    hideIfNoPrevNext = this._get(inst, 'hideIfNoPrevNext'),\n                    navigationAsDateFormat = this._get(inst, 'navigationAsDateFormat'),\n                    numMonths = this._getNumberOfMonths(inst),\n                    showCurrentAtPos = this._get(inst, 'showCurrentAtPos'),\n                    stepMonths = this._get(inst, 'stepMonths'),\n                    isMultiMonth = parseInt(numMonths[0], 10) !== 1 || parseInt(numMonths[1], 10) !== 1,\n                    currentDate = this._daylightSavingAdjust(!inst.currentDay ? new Date(9999, 9, 9) :\n                        new Date(inst.currentYear, inst.currentMonth, inst.currentDay)),\n                    minDate = this._getMinMaxDate(inst, 'min'),\n                    maxDate = this._getMinMaxDate(inst, 'max'),\n                    drawMonth = inst.drawMonth - showCurrentAtPos,\n                    drawYear = inst.drawYear,\n                    maxDraw,\n                    prevText = this._get(inst, 'prevText'),\n                    prev,\n                    nextText = this._get(inst, 'nextText'),\n                    next,\n                    currentText = this._get(inst, 'currentText'),\n                    gotoDate,\n                    controls,\n                    buttonPanel,\n                    firstDay,\n                    showWeek = this._get(inst, 'showWeek'),\n                    dayNames = this._get(inst, 'dayNames'),\n                    dayNamesMin = this._get(inst, 'dayNamesMin'),\n                    monthNames = this._get(inst, 'monthNames'),\n                    monthNamesShort =  this._get(inst, 'monthNamesShort'),\n                    beforeShowDay = this._get(inst, 'beforeShowDay'),\n                    showOtherMonths = this._get(inst, 'showOtherMonths'),\n                    selectOtherMonths = this._get(inst, 'selectOtherMonths'),\n                    defaultDate = this._getDefaultDate(inst),\n                    html = '',\n                    row = 0,\n                    col = 0,\n                    selectedDate,\n                    cornerClass = ' ui-corner-all',\n                    group = '',\n                    calender = '',\n                    dow = 0,\n                    thead,\n                    day,\n                    daysInMonth,\n                    leadDays,\n                    curRows,\n                    numRows,\n                    printDate,\n                    dRow = 0,\n                    tbody,\n                    daySettings,\n                    otherMonth,\n                    unselectable;\n\n                if (drawMonth < 0) {\n                    drawMonth += 12;\n                    drawYear--;\n                }\n\n                if (maxDate) {\n                    maxDraw = this._daylightSavingAdjust(new Date(maxDate.getFullYear(),\n                        maxDate.getMonth() - numMonths[0] * numMonths[1] + 1, maxDate.getDate()));\n                    maxDraw = minDate && maxDraw < minDate ? minDate : maxDraw;\n\n                    while (this._daylightSavingAdjust(new Date(drawYear, drawMonth, 1)) > maxDraw) {\n                        drawMonth--;\n\n                        if (drawMonth < 0) {\n                            drawMonth = 11;\n                            drawYear--;\n\n                        }\n                    }\n                }\n                inst.drawMonth = drawMonth;\n                inst.drawYear = drawYear;\n                prevText = !navigationAsDateFormat ? prevText : this.formatDate(prevText,\n                    this._daylightSavingAdjust(new Date(drawYear, drawMonth - stepMonths, 1)),\n                    this._getFormatConfig(inst));\n                prev = this._canAdjustMonth(inst, -1, drawYear, drawMonth) ?\n                    '<a class=\"ui-datepicker-prev ui-corner-all\" data-handler=\"prev\" data-event=\"click\"' +\n                    ' title=\"' + prevText + '\">' +\n                    '<span class=\"ui-icon ui-icon-circle-triangle-' + (isRTL ? 'e' : 'w') + '\">' +\n                    '' + prevText + '</span></a>'\n                    : hideIfNoPrevNext ? ''\n                        :   '<a class=\"ui-datepicker-prev ui-corner-all ui-state-disabled\" title=\"' +\n                            '' + prevText + '\"><span class=\"ui-icon ui-icon-circle-triangle-' +\n                            '' + (isRTL ? 'e' : 'w') + '\">' + prevText + '</span></a>';\n                nextText = !navigationAsDateFormat ?\n                    nextText\n                    :   this.formatDate(nextText,\n                        this._daylightSavingAdjust(new Date(drawYear, drawMonth + stepMonths, 1)),\n                        this._getFormatConfig(inst));\n                next = this._canAdjustMonth(inst, +1, drawYear, drawMonth) ?\n                    '<a class=\"ui-datepicker-next ui-corner-all\" data-handler=\"next\" data-event=\"click\"' +\n                    'title=\"' + nextText + '\"><span class=\"ui-icon ui-icon-circle-triangle-' +\n                    '' + (isRTL ? 'w' : 'e') + '\">' + nextText + '</span></a>'\n                    : hideIfNoPrevNext ? ''\n                        :   '<a class=\"ui-datepicker-next ui-corner-all ui-state-disabled\" title=\"' + nextText + '\">' +\n                            '<span class=\"ui-icon ui-icon-circle-triangle-' + (isRTL ? 'w' : 'e') + '\">' + nextText +\n                            '</span></a>';\n                gotoDate = this._get(inst, 'gotoCurrent') && inst.currentDay ? currentDate : today;\n                currentText = !navigationAsDateFormat ? currentText :\n                    this.formatDate(currentText, gotoDate, this._getFormatConfig(inst));\n                controls = !inst.inline ?\n                    '<button type=\"button\" class=\"ui-datepicker-close ui-state-default ui-priority-primary ' +\n                    'ui-corner-all\" data-handler=\"hide\" data-event=\"click\">' +\n                    this._get(inst, 'closeText') + '</button>'\n                    : '';\n                buttonPanel = showButtonPanel ?\n                    '<div class=\"ui-datepicker-buttonpane ui-widget-content\">' + (isRTL ? controls : '') +\n                    (this._isInRange(inst, gotoDate) ? '<button type=\"button\" class=\"ui-datepicker-current ' +\n                    'ui-state-default ui-priority-secondary ui-corner-all\" data-handler=\"today\" data-event=\"click\"' +\n                    '>' + currentText + '</button>' : '') + (isRTL ? '' : controls) + '</div>' : '';\n                firstDay = parseInt(this._get(inst, 'firstDay'), 10);\n                firstDay = isNaN(firstDay) ? 0 : firstDay;\n\n                for (row = 0; row < numMonths[0]; row++) {\n                    this.maxRows = 4;\n\n                    for (col = 0; col < numMonths[1]; col++) {\n                        selectedDate = this._daylightSavingAdjust(new Date(drawYear, drawMonth, inst.selectedDay));\n\n                        calender = '';\n\n                        if (isMultiMonth) {\n                            calender += '<div class=\"ui-datepicker-group';\n\n                            if (numMonths[1] > 1) {\n                                switch (col) {\n                                    case 0: calender += ' ui-datepicker-group-first';\n                                        cornerClass = ' ui-corner-' + (isRTL ? 'right' : 'left');\n                                        break;\n\n                                    case numMonths[1] - 1: calender += ' ui-datepicker-group-last';\n                                        cornerClass = ' ui-corner-' + (isRTL ? 'left' : 'right');\n                                        break;\n\n                                    default: calender += ' ui-datepicker-group-middle'; cornerClass = '';\n                                }\n                            }\n                            calender += '\">';\n                        }\n                        calender += '<div class=\"ui-datepicker-header ' +\n                            'ui-widget-header ui-helper-clearfix' + cornerClass + '\">' +\n                            (/all|left/.test(cornerClass) && parseInt(row, 10) === 0 ? isRTL ? next : prev : '') +\n                            (/all|right/.test(cornerClass) && parseInt(row, 10) === 0 ? isRTL ? prev : next : '') +\n                            this._generateMonthYearHeader(inst, drawMonth, drawYear, minDate, maxDate,\n                            row > 0 || col > 0, monthNames, monthNamesShort) + // draw month headers\n                            '</div><table class=\"ui-datepicker-calendar\"><thead>' +\n                            '<tr>';\n                        thead = showWeek ?\n                            '<th class=\"ui-datepicker-week-col\">' + this._get(inst, 'weekHeader') + '</th>' : '';\n\n                        for (dow = 0; dow < 7; dow++) { // days of the week\n                            day = (dow + firstDay) % 7;\n                            thead += '<th' + ((dow + firstDay + 6) % 7 >= 5 ?\n                                ' class=\"ui-datepicker-week-end\"' : '') + '>' +\n                                '<span title=\"' + dayNames[day] + '\">' + dayNamesMin[day] + '</span></th>';\n                        }\n                        calender += thead + '</tr></thead><tbody>';\n                        daysInMonth = this._getDaysInMonth(drawYear, drawMonth);\n\n                        if (drawYear === inst.selectedYear && drawMonth === inst.selectedMonth) {\n                            inst.selectedDay = Math.min(inst.selectedDay, daysInMonth);\n                        }\n                        leadDays = (this._getFirstDayOfMonth(drawYear, drawMonth) - firstDay + 7) % 7;\n                        curRows = Math.ceil((leadDays + daysInMonth) / 7); // calculate the number of rows to generate\n                        numRows = isMultiMonth ? this.maxRows > curRows ? this.maxRows : curRows : curRows;\n                        this.maxRows = numRows;\n                        printDate = this._daylightSavingAdjust(new Date(drawYear, drawMonth, 1 - leadDays));\n\n                        for (dRow = 0; dRow < numRows; dRow++) { // create date picker rows\n                            calender += '<tr>';\n                            tbody = !showWeek ? '' : '<td class=\"ui-datepicker-week-col\">' +\n                            this._get(inst, 'calculateWeek')(printDate) + '</td>';\n\n                            for (dow = 0; dow < 7; dow++) { // create date picker days\n                                daySettings = beforeShowDay ?\n                                    beforeShowDay.apply(inst.input ? inst.input[0] : null, [printDate]) : [true, ''];\n                                otherMonth = printDate.getMonth() !== drawMonth;\n                                unselectable = otherMonth && !selectOtherMonths || !daySettings[0] ||\n                                minDate && printDate < minDate || maxDate && printDate > maxDate;\n                                tbody += '<td class=\"' +\n                                ((dow + firstDay + 6) % 7 >= 5 ? ' ui-datepicker-week-end' : '') + // highlight weekends\n                                (otherMonth ? ' ui-datepicker-other-month' : '') + // highlight days from other months\n                                (printDate.getTime() === selectedDate.getTime() &&\n                                drawMonth === inst.selectedMonth && inst._keyEvent || // user pressed key\n                                defaultDate.getTime() === printDate.getTime() &&\n                                defaultDate.getTime() === selectedDate.getTime() ?\n                                    // or defaultDate is current printedDate and defaultDate is selectedDate\n                                ' ' + this._dayOverClass : '') + // highlight selected day\n                                (unselectable ? ' ' + this._unselectableClass + ' ui-state-disabled' : '') +\n                                (otherMonth && !showOtherMonths ? '' : ' ' + daySettings[1] + // highlight custom dates\n                                (printDate.getTime() === currentDate.getTime() ? ' ' + this._currentClass : '') +\n                                (printDate.getDate() === today.getDate() && printDate.getMonth() === today.getMonth() &&\n                                printDate.getYear() === today.getYear() ? ' ui-datepicker-today' : '')) + '\"' +\n                                ((!otherMonth || showOtherMonths) && daySettings[2] ?\n                                ' title=\"' + daySettings[2] + '\"' : '') + // cell title\n                                (unselectable ? '' : ' data-handler=\"selectDay\" data-event=\"click\" data-month=\"' +\n                                '' + printDate.getMonth() + '\" data-year=\"' + printDate.getFullYear() + '\"') + '>' +\n                                (otherMonth && !showOtherMonths ? '&#xa0;' : // display for other months\n                                    unselectable ? '<span class=\"ui-state-default\">' + printDate.getDate() + '</span>'\n                                        : '<a class=\"ui-state-default' +\n                                    (printDate.getTime() === today.getTime() ? ' ' : '') +\n                                    (printDate.getTime() === currentDate.getTime() ? ' ui-state-active' : '') +\n                                    (otherMonth ? ' ui-priority-secondary' : '') +\n                                    '\" data-date=\"' + printDate.getDate() + '\" href=\"#\">' +\n                                        printDate.getDate() + '</a>') + '</td>';\n                                printDate.setDate(printDate.getDate() + 1);\n                                printDate = this._daylightSavingAdjust(printDate);\n                            }\n                            calender += tbody + '</tr>';\n                        }\n                        drawMonth++;\n\n                        if (drawMonth > 11) {\n                            drawMonth = 0;\n                            drawYear++;\n                        }\n                        calender += '</tbody></table>' + (isMultiMonth ? '</div>' +\n                        (numMonths[0] > 0 && col === numMonths[1] - 1 ? '<div class=\"ui-datepicker-row-break\"></div>'\n                            : '') : '');\n                        group += calender;\n                    }\n                    html += group;\n                }\n                html += buttonPanel + ($.ui.ie6 && !inst.inline ?\n                    '<iframe src=\"javascript:false;\" class=\"ui-datepicker-cover\" frameborder=\"0\"></iframe>' : '');\n                inst._keyEvent = false;\n\n                return html;\n            };\n        },\n\n        /**\n         * Set current date if the date is not set\n         * @protected\n         * @param {Object} element\n         */\n        _setCurrentDate: function (element) {\n            if (!element.val()) {\n                element[this._picker()]('setTimezoneDate').val('');\n            }\n        },\n\n        /**\n         * Init Datetimepicker\n         * @protected\n         * @param {Object} element\n         */\n        _initPicker: function (element) {\n            var picker = element[this._picker()](this.options),\n                pickerButtonText = picker.next('.ui-datepicker-trigger')\n                    .find('img')\n                    .attr('title');\n\n            picker.next('.ui-datepicker-trigger')\n                .addClass('v-middle')\n                .text('') // Remove jQuery UI datepicker generated image\n                .append('<span>' + pickerButtonText + '</span>');\n\n            $(element).attr('autocomplete', this.options.autoComplete ? 'on' : 'off');\n\n            this._setCurrentDate(element);\n        },\n\n        /**\n         * destroy instance of datetimepicker\n         */\n        _destroy: function () {\n            this.element[this._picker()]('destroy');\n            this._super();\n        },\n\n        /**\n         * Method is kept for backward compatibility and unit-tests acceptance\n         * see \\mage\\calendar\\calendar-test.js\n         * @return {Object} date\n         */\n        getTimezoneDate: function () {\n            return datepickerPrototype._getTimezoneDate.call(this, this.options);\n        }\n    });\n\n    calendarBasePrototype = $.mage.calendar.prototype;\n\n    /**\n     * Extension for Calendar - date and time format convert functionality\n     * @var {Object}\n     */\n    $.widget('mage.calendar', $.extend({}, calendarBasePrototype,\n            /** @lends {$.mage.calendar.prototype} */ {\n                /**\n                 * key - backend format, value - jquery format\n                 * @type {Object}\n                 * @private\n                 */\n                dateTimeFormat: {\n                    date: {\n                        'EEEE': 'DD',\n                        'EEE': 'D',\n                        'EE': 'D',\n                        'E': 'D',\n                        'D': 'o',\n                        'MMMM': 'MM',\n                        'MMM': 'M',\n                        'MM': 'mm',\n                        'M': 'mm',\n                        'yyyy': 'yy',\n                        'y': 'yy',\n                        'Y': 'yy',\n                        'yy': 'yy' // Always long year format on frontend\n                    },\n                    time: {\n                        'a': 'TT'\n                    }\n                },\n\n                /**\n                 * Add Date and Time converting to _create method\n                 * @protected\n                 */\n                _create: function () {\n                    if (this.options.dateFormat) {\n                        this.options.dateFormat = this._convertFormat(this.options.dateFormat, 'date');\n                    }\n\n                    if (this.options.timeFormat) {\n                        this.options.timeFormat = this._convertFormat(this.options.timeFormat, 'time');\n                    }\n                    calendarBasePrototype._create.apply(this, arguments);\n                },\n\n                /**\n                 * Converting date or time format\n                 * @protected\n                 * @param {String} format\n                 * @param {String} type\n                 * @return {String}\n                 */\n                _convertFormat: function (format, type) {\n                    var symbols = format.match(/([a-z]+)/ig),\n                        separators = format.match(/([^a-z]+)/ig),\n                        self = this,\n                        convertedFormat = '';\n\n                    if (symbols) {\n                        $.each(symbols, function (key, val) {\n                            convertedFormat +=\n                                (self.dateTimeFormat[type][val] || val) +\n                                (separators[key] || '');\n                        });\n                    }\n\n                    return convertedFormat;\n                }\n            })\n    );\n\n    /**\n     * Widget dateRange\n     * @extends $.mage.calendar\n     */\n    $.widget('mage.dateRange', $.mage.calendar, {\n\n        /**\n         * creates two instances of datetimepicker for date range selection\n         * @protected\n         */\n        _initPicker: function () {\n            var from,\n                to;\n\n            if (this.options.from && this.options.to) {\n                from = this.element.find('#' + this.options.from.id);\n                to = this.element.find('#' + this.options.to.id);\n                this.options.onSelect = $.proxy(function (selectedDate) {\n                    to[this._picker()]('option', 'minDate', selectedDate);\n                }, this);\n                $.mage.calendar.prototype._initPicker.call(this, from);\n                from.on('change', $.proxy(function () {\n                    to[this._picker()]('option', 'minDate', from[this._picker()]('getDate'));\n                }, this));\n                this.options.onSelect = $.proxy(function (selectedDate) {\n                    from[this._picker()]('option', 'maxDate', selectedDate);\n                }, this);\n                $.mage.calendar.prototype._initPicker.call(this, to);\n                to.on('change', $.proxy(function () {\n                    from[this._picker()]('option', 'maxDate', to[this._picker()]('getDate'));\n                }, this));\n            }\n        },\n\n        /**\n         * destroy two instances of datetimepicker\n         */\n        _destroy: function () {\n            if (this.options.from) {\n                this.element.find('#' + this.options.from.id)[this._picker()]('destroy');\n            }\n\n            if (this.options.to) {\n                this.element.find('#' + this.options.to.id)[this._picker()]('destroy');\n            }\n            this._super();\n        }\n    });\n\n    // Overrides the \"today\" button functionality to select today's date when clicked.\n    $.datepicker._gotoTodayOriginal = $.datepicker._gotoToday;\n\n    /**\n     * overwrite jQuery UI _showDatepicker function for proper HTML generation conditions.\n     *\n     */\n    $.datepicker._showDatepickerOriginal = $.datepicker._showDatepicker;\n\n    /**\n     * Triggers original method showDataPicker for rendering calendar\n     * @param {HTMLObject} input\n     * @private\n     */\n    $.datepicker._showDatepicker = function (input) {\n        if (!input.disabled) {\n            $.datepicker._showDatepickerOriginal.call(this, input);\n        }\n    };\n\n    /**\n     * _gotoToday\n     * @param {Object} el\n     */\n    $.datepicker._gotoToday = function (el) {\n        //Set date/time according to timezone offset\n        $(el).datepicker('setTimezoneDate')\n            // To ensure that user can re-select date field without clicking outside it first.\n            .trigger('blur').trigger('change');\n    };\n\n    return {\n        dateRange:  $.mage.dateRange,\n        calendar:   $.mage.calendar\n    };\n});\n","mage/url.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\n/* eslint-disable strict */\ndefine([], function () {\n    var baseUrl = '';\n\n    return {\n        /**\n         * @param {String} url\n         */\n        setBaseUrl: function (url) {\n            baseUrl = url;\n        },\n\n        /**\n         * @param {String} path\n         * @return {*}\n         */\n        build: function (path) {\n            if (path.indexOf(baseUrl) !== -1) {\n                return path;\n            }\n\n            return baseUrl + path;\n        }\n    };\n});\n","mage/template.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\ndefine([\n    'underscore'\n], function (_) {\n    'use strict';\n\n    /**\n     * Checks if provided string is a valid DOM selector.\n     *\n     * @param {String} selector - Selector to be checked.\n     * @returns {Boolean}\n     */\n    function isSelector(selector) {\n        try {\n            document.querySelector(selector);\n\n            return true;\n        } catch (e) {\n            return false;\n        }\n    }\n\n    /**\n     * Unescapes characters used in underscore templates.\n     *\n     * @param {String} str - String to be processed.\n     * @returns {String}\n     */\n    function unescape(str) {\n        return str.replace(/&lt;%|%3C%/g, '<%').replace(/%&gt;|%%3E/g, '%>');\n    }\n\n    /**\n     * If 'tmpl' is a valid selector, returns target node's innerHTML if found.\n     * Else, returns empty string and emits console warning.\n     * If 'tmpl' is not a selector, returns 'tmpl' as is.\n     *\n     * @param {String} tmpl\n     * @returns {String}\n     */\n    function getTmplString(tmpl) {\n        if (isSelector(tmpl)) {\n            tmpl = document.querySelector(tmpl);\n\n            if (tmpl) {\n                tmpl = tmpl.innerHTML.trim();\n            } else {\n                console.warn('No template was found by selector: ' + tmpl);\n\n                tmpl = '';\n            }\n        }\n\n        return unescape(tmpl);\n    }\n\n    /**\n     * Compiles or renders template provided either\n     * by selector or by the template string.\n     *\n     * @param {String} tmpl - Template string or selector.\n     * @param {(Object|Array|Function)} [data] - Data object with which to render template.\n     * @returns {String|Function}\n     */\n    return function (tmpl, data) {\n        var render;\n\n        tmpl   = getTmplString(tmpl);\n        render = _.template(tmpl);\n\n        return !_.isUndefined(data) ?\n            render(data) :\n            render;\n    };\n});\n","mage/storage.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\ndefine(['jquery', 'mage/url'], function ($, urlBuilder) {\n    'use strict';\n\n    return {\n        /**\n         * Perform asynchronous GET request to server.\n         * @param {String} url\n         * @param {Boolean} global\n         * @param {String} contentType\n         * @param {Object} headers\n         * @returns {Deferred}\n         */\n        get: function (url, global, contentType, headers) {\n            headers = headers || {};\n            global = global === undefined ? true : global;\n            contentType = contentType || 'application/json';\n\n            return $.ajax({\n                url: urlBuilder.build(url),\n                type: 'GET',\n                global: global,\n                contentType: contentType,\n                headers: headers\n            });\n        },\n\n        /**\n         * Perform asynchronous POST request to server.\n         * @param {String} url\n         * @param {String} data\n         * @param {Boolean} global\n         * @param {String} contentType\n         * @param {Object} headers\n         * @param {Boolean} async\n         * @returns {Deferred}\n         */\n        post: function (url, data, global, contentType, headers, async) {\n            headers = headers || {};\n            global = global === undefined ? true : global;\n            contentType = contentType || 'application/json';\n            async = async === undefined ? true : async;\n\n            return $.ajax({\n                url: urlBuilder.build(url),\n                type: 'POST',\n                data: data,\n                global: global,\n                contentType: contentType,\n                headers: headers,\n                async: async\n            });\n        },\n\n        /**\n         * Perform asynchronous PUT request to server.\n         * @param {String} url\n         * @param {String} data\n         * @param {Boolean} global\n         * @param {String} contentType\n         * @param {Object} headers\n         * @returns {Deferred}\n         */\n        put: function (url, data, global, contentType, headers) {\n            var ajaxSettings = {};\n\n            headers = headers || {};\n            global = global === undefined ? true : global;\n            contentType = contentType || 'application/json';\n            ajaxSettings.url = urlBuilder.build(url);\n            ajaxSettings.type = 'PUT';\n            ajaxSettings.data = data;\n            ajaxSettings.global = global;\n            ajaxSettings.contentType = contentType;\n            ajaxSettings.headers = headers;\n\n            return $.ajax(ajaxSettings);\n        },\n\n        /**\n         * Perform asynchronous DELETE request to server.\n         * @param {String} url\n         * @param {Boolean} global\n         * @param {String} contentType\n         * @param {Object} headers\n         * @returns {Deferred}\n         */\n        delete: function (url, global, contentType, headers) {\n            headers = headers || {};\n            global = global === undefined ? true : global;\n            contentType = contentType || 'application/json';\n\n            return $.ajax({\n                url: urlBuilder.build(url),\n                type: 'DELETE',\n                global: global,\n                contentType: contentType,\n                headers: headers\n            });\n        }\n    };\n});\n","mage/accordion.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\ndefine([\n    'jquery',\n    'mage/tabs'\n], function ($, tabs) {\n    'use strict';\n\n    $.widget('mage.accordion', tabs, {\n        options: {\n            active: [0],\n            multipleCollapsible: false,\n            openOnFocus: false\n        },\n\n        /**\n         * @private\n         */\n        _callCollapsible: function () {\n            var self = this,\n                disabled = false,\n                active = false;\n\n            if (typeof this.options.active === 'string') {\n                this.options.active = this.options.active.split(' ').map(function (item) {\n                    return parseInt(item, 10);\n                });\n            }\n\n            $.each(this.collapsibles, function (i) {\n                disabled = active = false;\n\n                if ($.inArray(i, self.options.disabled) !== -1) {\n                    disabled = true;\n                }\n\n                if ($.inArray(i, self.options.active) !== -1) {\n                    active = true;\n                }\n                self._instantiateCollapsible(this, i, active, disabled);\n            });\n        },\n\n        /**\n         * Overwrites default functionality to provide the option to activate/deactivate multiple sections simultaneous\n         * @param {*} action\n         * @param {*} index\n         * @private\n         */\n        _toggleActivate: function (action, index) {\n            var self = this;\n\n            if (Array.isArray(index && this.options.multipleCollapsible)) {\n                $.each(index, function () {\n                    self.collapsibles.eq(this).collapsible(action);\n                });\n            } else if (index === undefined && this.options.multipleCollapsible) {\n                this.collapsibles.collapsible(action);\n            } else {\n                this._super(action, index);\n            }\n        },\n\n        /**\n         * If the Accordion allows multiple section to be active at the same time, if deep linking is used\n         * sections that don't contain the id from anchor shouldn't be closed, otherwise the accordion uses the\n         * tabs behavior\n         * @private\n         */\n        _handleDeepLinking: function () {\n            if (!this.options.multipleCollapsible) {\n                this._super();\n            }\n        },\n\n        /**\n         * Prevent default behavior that closes the other sections when one gets activated if the Accordion allows\n         * multiple sections simultaneous\n         * @private\n         */\n        _closeOthers: function () {\n            var self = this;\n\n            if (!this.options.multipleCollapsible) {\n                $.each(this.collapsibles, function () {\n                    $(this).on('beforeOpen', function () {\n                        self.collapsibles.not(this).collapsible('deactivate');\n                    });\n                });\n            }\n            $.each(this.collapsibles, function () {\n                $(this).on('beforeOpen', function () {\n                    var section = $(this);\n\n                    section.addClass('allow').prevAll().addClass('allow');\n                    section.nextAll().removeClass('allow');\n                });\n            });\n        }\n    });\n\n    return $.mage.accordion;\n});\n","mage/apply/main.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\ndefine([\n    'underscore',\n    'jquery',\n    './scripts'\n], function (_, $, processScripts) {\n    'use strict';\n\n    var dataAttr = 'data-mage-init',\n        nodeSelector = '[' + dataAttr + ']';\n\n    /**\n     * Initializes components assigned to a specified element via data-* attribute.\n     *\n     * @param {HTMLElement} el - Element to initialize components with.\n     * @param {Object|String} config - Initial components' config.\n     * @param {String} component - Components' path.\n     */\n    function init(el, config, component) {\n        require([component], function (fn) {\n            var $el;\n\n            if (typeof fn === 'object') {\n                fn = fn[component].bind(fn);\n            }\n\n            if (_.isFunction(fn)) {\n                fn = fn.bind(null, config, el);\n            } else {\n                $el = $(el);\n\n                if ($el[component]) {\n                    // eslint-disable-next-line jquery-no-bind-unbind\n                    fn = $el[component].bind($el, config);\n                }\n            }\n            // Init module in separate task to prevent blocking main thread.\n            setTimeout(fn);\n        }, function (error) {\n            if ('console' in window && typeof window.console.error === 'function') {\n                console.error(error);\n            }\n\n            return true;\n        });\n    }\n\n    /**\n     * Parses elements 'data-mage-init' attribute as a valid JSON data.\n     * Note: data-mage-init attribute will be removed.\n     *\n     * @param {HTMLElement} el - Element whose attribute should be parsed.\n     * @returns {Object}\n     */\n    function getData(el) {\n        var data = el.getAttribute(dataAttr);\n\n        el.removeAttribute(dataAttr);\n\n        return {\n            el: el,\n            data: JSON.parse(data)\n        };\n    }\n\n    return {\n        /**\n         * Initializes components assigned to HTML elements via [data-mage-init].\n         *\n         * @example Sample 'data-mage-init' declaration.\n         *      data-mage-init='{\"path/to/component\": {\"foo\": \"bar\"}}'\n         */\n        apply: function (context) {\n            var virtuals = processScripts(!context ? document : context),\n                nodes = document.querySelectorAll(nodeSelector);\n\n            _.toArray(nodes)\n                .map(getData)\n                .concat(virtuals)\n                .forEach(function (itemContainer) {\n                    var element = itemContainer.el;\n\n                    _.each(itemContainer.data, function (obj, key) {\n                            if (obj.mixins) {\n                                require(obj.mixins, function () { //eslint-disable-line max-nested-callbacks\n                                    var i, len;\n\n                                    for (i = 0, len = arguments.length; i < len; i++) {\n                                        $.extend(\n                                            true,\n                                            itemContainer.data[key],\n                                            arguments[i](itemContainer.data[key], element)\n                                        );\n                                    }\n\n                                    delete obj.mixins;\n                                    init.call(null, element, obj, key);\n                                });\n                            } else {\n                                init.call(null, element, obj, key);\n                            }\n\n                        }\n                    );\n\n                });\n        },\n        applyFor: init\n    };\n});\n","mage/apply/scripts.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\ndefine([\n    'underscore',\n    'jquery'\n], function (_, $) {\n    'use strict';\n\n    var scriptSelector = 'script[type=\"text/x-magento-init\"]',\n        dataAttr = 'data-mage-init',\n        virtuals = [];\n\n    /**\n     * Adds components to the virtual list.\n     *\n     * @param {Object} components\n     */\n    function addVirtual(components) {\n        virtuals.push({\n            el: false,\n            data: components\n        });\n    }\n\n    /**\n     * Merges provided data with a current data\n     * of a elements' \"data-mage-init\" attribute.\n     *\n     * @param {Object} components - Object with components and theirs configuration.\n     * @param {HTMLElement} elem - Element whose data should be modified.\n     */\n    function setData(components, elem) {\n        var data = elem.getAttribute(dataAttr);\n\n        data = data ? JSON.parse(data) : {};\n        _.each(components, function (obj, key) {\n            if (_.has(obj, 'mixins')) {\n                data[key] = data[key] || {};\n                data[key].mixins = data[key].mixins || [];\n                data[key].mixins = data[key].mixins.concat(obj.mixins);\n                delete obj.mixins;\n            }\n        });\n\n        data = $.extend(true, data, components);\n        data = JSON.stringify(data);\n        elem.setAttribute(dataAttr, data);\n    }\n\n    /**\n     * Search for the elements by privded selector and extends theirs data.\n     *\n     * @param {Object} components - Object with components and theirs configuration.\n     * @param {String} selector - Selector for the elements.\n     */\n    function processElems(components, selector) {\n        var elems,\n            iterator;\n\n        if (selector === '*') {\n            addVirtual(components);\n\n            return;\n        }\n\n        elems = document.querySelectorAll(selector);\n        iterator = setData.bind(null, components);\n\n        _.toArray(elems).forEach(iterator);\n    }\n\n    /**\n     * Parses content of a provided script node.\n     * Note: node will be removed from DOM.\n     *\n     * @param {HTMLScriptElement} node - Node to be processed.\n     * @returns {Object}\n     */\n    function getNodeData(node) {\n        var data = node.textContent;\n\n        node.parentNode.removeChild(node);\n\n        return JSON.parse(data);\n    }\n\n    /**\n     * Parses 'script' tags with a custom type attribute and moves it's data\n     * to a 'data-mage-init' attribute of an element found by provided selector.\n     * Note: All found script nodes will be removed from DOM.\n     *\n     * @returns {Array} An array of components not assigned to the specific element.\n     *\n     * @example Sample declaration.\n     *      <script type=\"text/x-magento-init\">\n     *          {\n     *              \"body\": {\n     *                  \"path/to/component\": {\"foo\": \"bar\"}\n     *              }\n     *          }\n     *      </script>\n     *\n     * @example Providing data without selector.\n     *      {\n     *          \"*\": {\n     *              \"path/to/component\": {\"bar\": \"baz\"}\n     *          }\n     *      }\n     */\n    return function () {\n        var nodes = document.querySelectorAll(scriptSelector);\n\n        _.toArray(nodes)\n            .map(getNodeData)\n            .forEach(function (item) {\n                _.each(item, processElems);\n            });\n\n        return virtuals.splice(0, virtuals.length);\n    };\n});\n","mage/requirejs/text.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n/* inspired by http://github.com/requirejs/text */\n/*global XDomainRequest */\n\ndefine(['module'], function (module) {\n    'use strict';\n\n    var xmlRegExp = /^\\s*<\\?xml(\\s)+version=[\\'\\\"](\\d)*.(\\d)*[\\'\\\"](\\s)*\\?>/im,\n        bodyRegExp = /<body[^>]*>\\s*([\\s\\S]+)\\s*<\\/body>/im,\n        stripReg = /!strip$/i,\n        defaultConfig = module.config && module.config() || {};\n\n    /**\n     * Strips <?xml ...?> declarations so that external SVG and XML documents can be\n     * added to a document without worry.\n     * Also, if the string is an HTML document, only the part inside the body tag is returned.\n     *\n     * @param {String} external\n     * @returns {String}\n     */\n    function stripContent(external) {\n        var matches;\n\n        if (!external) {\n            return '';\n        }\n\n        matches = external.match(bodyRegExp);\n        external = matches ?\n            matches[1] :\n            external.replace(xmlRegExp, '');\n\n        return external;\n    }\n\n    /**\n     * Checks that url match current location\n     *\n     * @param {String} url\n     * @returns {Boolean}\n     */\n    function sameDomain(url) {\n        var uProtocol, uHostName, uPort,\n            xdRegExp = /^([\\w:]+)?\\/\\/([^\\/\\\\]+)/i,\n            location = window.location,\n            match = xdRegExp.exec(url);\n\n        if (!match) {\n            return true;\n        }\n        uProtocol = match[1];\n        uHostName = match[2];\n\n        uHostName = uHostName.split(':');\n        uPort = uHostName[1] || '';\n        uHostName = uHostName[0];\n\n        return (!uProtocol || uProtocol === location.protocol) &&\n            (!uHostName || uHostName.toLowerCase() === location.hostname.toLowerCase()) &&\n            (!uPort && !uHostName || uPort === location.port);\n    }\n\n    /**\n     * @returns {XMLHttpRequest|XDomainRequest|null}\n     */\n    function createRequest(url) {\n        var xhr = new XMLHttpRequest();\n\n        if (!sameDomain(url) && typeof XDomainRequest !== 'undefined') {\n            xhr = new XDomainRequest();\n        }\n\n        return xhr;\n    }\n\n    /**\n     * XHR requester. Returns value to callback.\n     *\n     * @param {String} url\n     * @param {Function} callback\n     * @param {Function} fail\n     * @param {Object} headers\n     */\n    function getContent(url, callback, fail, headers) {\n        var xhr = createRequest(url),\n            header;\n\n        xhr.open('GET', url);\n\n        /*eslint-disable max-depth */\n        if ('setRequestHeader' in xhr && headers) {\n            for (header in headers) {\n                if (headers.hasOwnProperty(header)) {\n                    xhr.setRequestHeader(header.toLowerCase(), headers[header]);\n                }\n            }\n        }\n\n        /**\n         * @inheritdoc\n         */\n        xhr.onreadystatechange = function () {\n            var status, err;\n\n            //Do not explicitly handle errors, those should be\n            //visible via console output in the browser.\n            if (xhr.readyState === 4) {\n                status = xhr.status || 0;\n\n                if (status > 399 && status < 600) {\n                    //An http 4xx or 5xx error. Signal an error.\n                    err = new Error(url + ' HTTP status: ' + status);\n                    err.xhr = xhr;\n\n                    if (fail) {\n                        fail(err);\n                    }\n                } else {\n                    callback(xhr.responseText);\n\n                    if (defaultConfig.onXhrComplete) {\n                        defaultConfig.onXhrComplete(xhr, url);\n                    }\n                }\n            }\n        };\n\n        /*eslint-enable max-depth */\n\n        if (defaultConfig.onXhr) {\n            defaultConfig.onXhr(xhr, url);\n        }\n\n        xhr.send();\n    }\n\n    /**\n     * Main method used by RequireJs.\n     *\n     * @param {String} name - has format: some.module.filext!strip\n     * @param {Function} req\n     * @param {Function|undefined} onLoad\n     */\n    function loadContent(name, req, onLoad) {\n\n        var toStrip = stripReg.test(name),\n            url = req.toUrl(name.replace(stripReg, '')),\n            headers = defaultConfig.headers;\n\n        getContent(url, function (content) {\n                content = toStrip ? stripContent(content) : content;\n                onLoad(content);\n            }, onLoad.error, headers);\n    }\n\n    return {\n        load: loadContent,\n        get: getContent\n    };\n});\n","mage/requirejs/resolver.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\ndefine([\n    'underscore',\n    'domReady!'\n], function (_) {\n    'use strict';\n\n    var context = require.s.contexts._,\n        execCb = context.execCb,\n        registry = context.registry,\n        callbacks = [],\n        retries = 10,\n        updateDelay = 1,\n        ready,\n        update;\n\n    /**\n     * Checks if provided callback already exists in the callbacks list.\n     *\n     * @param {Object} callback - Callback object to be checked.\n     * @returns {Boolean}\n     */\n    function isSubscribed(callback) {\n        return !!_.findWhere(callbacks, callback);\n    }\n\n    /**\n     * Checks if provided module is rejected during load.\n     *\n     * @param {Object} module - Module to be checked.\n     * @return {Boolean}\n     */\n    function isRejected(module) {\n        return registry[module.id] && (registry[module.id].inited || registry[module.id].error);\n    }\n\n    /**\n     * Checks if provided module had path fallback triggered.\n     *\n     * @param {Object} module - Module to be checked.\n     * @return {Boolean}\n     */\n    function isPathFallback(module) {\n        return registry[module.id] && registry[module.id].events.error;\n    }\n\n    /**\n     * Checks if provided module has unresolved dependencies.\n     *\n     * @param {Object} module - Module to be checked.\n     * @returns {Boolean}\n     */\n    function isPending(module) {\n        if (!module.depCount) {\n            return false;\n        }\n\n        return module.depCount >\n            _.filter(module.depMaps, isRejected).length + _.filter(module.depMaps, isPathFallback).length;\n    }\n\n    /**\n     * Checks if requirejs's registry object contains pending modules.\n     *\n     * @returns {Boolean}\n     */\n    function hasPending() {\n        return _.some(registry, isPending);\n    }\n\n    /**\n     * Checks if 'resolver' module is in ready\n     * state and that there are no pending modules.\n     *\n     * @returns {Boolean}\n     */\n    function isReady() {\n        return ready && !hasPending();\n    }\n\n    /**\n     * Invokes provided callback handler.\n     *\n     * @param {Object} callback\n     */\n    function invoke(callback) {\n        callback.handler.call(callback.ctx);\n    }\n\n    /**\n     * Sets 'resolver' module to a ready state\n     * and invokes pending callbacks.\n     */\n    function resolve() {\n        ready = true;\n\n        callbacks.splice(0).forEach(invoke);\n    }\n\n    /**\n     * Drops 'ready' flag and runs the update process.\n     */\n    function tick() {\n        ready = false;\n\n        update(retries);\n    }\n\n    /**\n     * Adds callback which will be invoked\n     * when all of the pending modules are initiated.\n     *\n     * @param {Function} handler - 'Ready' event handler function.\n     * @param {Object} [ctx] - Optional context with which handler\n     *      will be invoked.\n     */\n    function subscribe(handler, ctx) {\n        var callback = {\n            handler: handler,\n            ctx: ctx\n        };\n\n        if (!isSubscribed(callback)) {\n            callbacks.push(callback);\n\n            if (isReady()) {\n                _.defer(tick);\n            }\n        }\n    }\n\n    /**\n     * Checks for all modules to be initiated\n     * and invokes pending callbacks if it's so.\n     *\n     * @param {Number} [retry] - Number of retries\n     *      that will be used to repeat the 'update' function\n     *      invokation in case if there are no pending requests.\n     */\n    update = _.debounce(function (retry) {\n        if (!hasPending()) {\n            retry ? update(--retry) : resolve();\n        }\n    }, updateDelay);\n\n    /**\n     * Overrides requirejs's original 'execCb' method\n     * in order to track pending modules.\n     *\n     * @returns {*} Result of original method call.\n     */\n    context.execCb = function () {\n        var exported = execCb.apply(context, arguments);\n\n        tick();\n\n        return exported;\n    };\n\n    return subscribe;\n});\n","mage/requirejs/baseUrlResolver.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\n/**\n * Sample configuration:\n *\n require.config({\n        \"config\": {\n            \"baseUrlInterceptor\": {\n                \"Magento_Ui/js/lib/knockout/bindings/collapsible.js\": \"../../../../frontend/Magento/luma/en_US/\"\n            }\n        }\n    });\n */\n\n/* global jsSuffixRegExp */\n/* eslint-disable max-depth */\ndefine('baseUrlInterceptor', [\n    'module'\n], function (module) {\n    'use strict';\n\n    /**\n     * RequireJS Context object\n     */\n    var ctx = require.s.contexts._,\n\n        /**\n         * Original function\n         *\n         * @type {Function}\n         */\n        origNameToUrl = ctx.nameToUrl,\n\n        /**\n         * Original function\n         *\n         * @type {Function}\n         */\n        newContextConstr = require.s.newContext;\n\n    /**\n     * Remove dots from URL\n     *\n     * @param {Array} ary\n     */\n    function trimDots(ary) {\n        var i, part, length = ary.length;\n\n        for (i = 0; i < length; i++) {\n            part = ary[i];\n\n            if (part === '.') {\n                ary.splice(i, 1);\n                i -= 1;\n            } else if (part === '..') {\n                if (i === 1 && (ary[2] === '..' || ary[0] === '..')) {\n                    //End of the line. Keep at least one non-dot\n                    //path segment at the front so it can be mapped\n                    //correctly to disk. Otherwise, there is likely\n                    //no path mapping for a path starting with '..'.\n                    //This can still fail, but catches the most reasonable\n                    //uses of ..\n                    break;\n                } else if (i > 0) {\n                    ary.splice(i - 1, 2);\n                    i -= 2;\n                }\n            }\n        }\n    }\n\n    /**\n     * Normalize URL string (remove '/../')\n     *\n     * @param {String} name\n     * @param {String} baseName\n     * @param {Object} applyMap\n     * @param {Object} localContext\n     * @returns {*}\n     */\n    function normalize(name, baseName, applyMap, localContext) {\n        var lastIndex,\n            baseParts = baseName && baseName.split('/'),\n            normalizedBaseParts = baseParts;\n\n        //Adjust any relative paths.\n        if (name && name.charAt(0) === '.') {\n            //If have a base name, try to normalize against it,\n            //otherwise, assume it is a top-level require that will\n            //be relative to baseUrl in the end.\n            if (baseName) {\n                //Convert baseName to array, and lop off the last part,\n                //so that . matches that 'directory' and not name of the baseName's\n                //module. For instance, baseName of 'one/two/three', maps to\n                //'one/two/three.js', but we want the directory, 'one/two' for\n                //this normalization.\n                normalizedBaseParts = baseParts.slice(0, baseParts.length - 1);\n                name = name.split('/');\n                lastIndex = name.length - 1;\n\n                // If wanting node ID compatibility, strip .js from end\n                // of IDs. Have to do this here, and not in nameToUrl\n                // because node allows either .js or non .js to map\n                // to same file.\n                if (localContext.nodeIdCompat && jsSuffixRegExp.test(name[lastIndex])) {\n                    name[lastIndex] = name[lastIndex].replace(jsSuffixRegExp, '');\n                }\n\n                name = normalizedBaseParts.concat(name);\n                trimDots(name);\n                name = name.join('/');\n            } else if (name.indexOf('./') === 0) {\n                // No baseName, so this is ID is resolved relative\n                // to baseUrl, pull off the leading dot.\n                name = name.substring(2);\n            }\n        }\n\n        return name;\n    }\n\n    /**\n     * Get full url.\n     *\n     * @param {Object} context\n     * @param {String} url\n     * @return {String}\n     */\n    function getUrl(context, url) {\n        var baseUrl = context.config.baseUrl,\n            newConfig = context.config,\n            modulePath = url.replace(baseUrl, ''),\n            newBaseUrl,\n            rewrite = module.config()[modulePath];\n\n        if (!rewrite) {\n            return url;\n        }\n\n        newBaseUrl = normalize(rewrite, baseUrl, undefined, newConfig);\n\n        return newBaseUrl + modulePath;\n    }\n\n    /**\n     * Replace original function.\n     *\n     * @returns {*}\n     */\n    ctx.nameToUrl = function () {\n        return getUrl(ctx, origNameToUrl.apply(ctx, arguments));\n    };\n\n    /**\n     * Replace original function.\n     *\n     * @return {*}\n     */\n    require.s.newContext = function () {\n        var newCtx = newContextConstr.apply(require.s, arguments),\n            newOrigNameToUrl = newCtx.nameToUrl;\n\n        /**\n         * New implementation of native function.\n         *\n         * @returns {String}\n         */\n        newCtx.nameToUrl = function () {\n            return getUrl(newCtx, newOrigNameToUrl.apply(newCtx, arguments));\n        };\n\n        return newCtx;\n    };\n});\n\nrequire(['baseUrlInterceptor'], function () {\n    'use strict';\n\n});\n","mage/msie/file-reader.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\ndefine([\n    'jquery'\n], function ($) {\n    'use strict';\n\n    /**\n     * Init \"readAsBinaryString\" function for FileReader class.\n     * It need for IE11\n     * @param {Blob} fileData\n     */\n    var readAsBinaryStringIEFunc = function (fileData) {\n        var binary = '',\n            self = this,\n            reader = new FileReader();\n\n        /**\n         * Read file as binary string\n         */\n        reader.onload = function () {\n            var bytes, length, index;\n\n            /* eslint-disable no-undef */\n            bytes = new Uint8Array(reader.result);\n            /* eslint-enable */\n            length = bytes.length;\n\n            for (index = 0; index < length; index++) {\n                binary += String.fromCharCode(bytes[index]);\n            }\n            //self.result  - readonly so assign binary\n            self.content = binary;\n            $(self).trigger('onload');\n        };\n        reader.readAsArrayBuffer(fileData);\n    };\n\n    if (typeof FileReader.prototype.readAsBinaryString === 'undefined') {\n        FileReader.prototype.readAsBinaryString = readAsBinaryStringIEFunc;\n    }\n});\n","mage/utils/wrapper.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\n/**\n * Utility methods used to wrap and extend functions.\n *\n * @example Usage of a 'wrap' method with arguments delegation.\n *      var multiply = function (a, b) {\n *          return a * b;\n *      };\n *\n *      multiply = module.wrap(multiply, function (orig) {\n *          return 'Result is: ' + orig();\n *      });\n *\n *      multiply(2, 2);\n *      => 'Result is: 4'\n *\n * @example Usage of 'wrapSuper' method.\n *      var multiply = function (a, b) {\n *         return a * b;\n *      };\n *\n *      var obj = {\n *          multiply: module.wrapSuper(multiply, function () {\n *              return 'Result is: ' + this._super();\n *          });\n *      };\n *\n *      obj.multiply(2, 2);\n *      => 'Result is: 4'\n */\ndefine([\n    'underscore'\n], function (_) {\n    'use strict';\n\n    /**\n     * Checks if string has a '_super' substring.\n     */\n    var superReg = /\\b_super\\b/;\n\n    return {\n\n        /**\n         * Wraps target function with a specified wrapper, which will receive\n         * reference to the original function as a first argument.\n         *\n         * @param {Function} target - Function to be wrapped.\n         * @param {Function} wrapper - Wrapper function.\n         * @returns {Function} Wrapper function.\n         */\n        wrap: function (target, wrapper) {\n            if (!_.isFunction(target) || !_.isFunction(wrapper)) {\n                return wrapper;\n            }\n\n            return function () {\n                var args    = _.toArray(arguments),\n                    ctx     = this,\n                    _super;\n\n                /**\n                 * Function that will be passed to the wrapper.\n                 * If no arguments will be passed to it, then the original\n                 * function will be called with an arguments of a wrapper function.\n                 */\n                _super = function () {\n                    var superArgs = arguments.length ? arguments : args.slice(1);\n\n                    return target.apply(ctx, superArgs);\n                };\n\n                args.unshift(_super);\n\n                return wrapper.apply(ctx, args);\n            };\n        },\n\n        /**\n         * Wraps the incoming function to implement support of the '_super' method.\n         *\n         * @param {Function} target - Function to be wrapped.\n         * @param {Function} wrapper - Wrapper function.\n         * @returns {Function} Wrapped function.\n         */\n        wrapSuper: function (target, wrapper) {\n            if (!this.hasSuper(wrapper) || !_.isFunction(target)) {\n                return wrapper;\n            }\n\n            return function () {\n                var _super  = this._super,\n                    args    = arguments,\n                    result;\n\n                /**\n                 * Temporary define '_super' method which\n                 * contains call to the original function.\n                 */\n                this._super = function () {\n                    var superArgs = arguments.length ? arguments : args;\n\n                    return target.apply(this, superArgs);\n                };\n\n                result = wrapper.apply(this, args);\n\n                this._super = _super;\n\n                return result;\n            };\n        },\n\n        /**\n         * Checks wether the incoming method contains calls of the '_super' method.\n         *\n         * @param {Function} fn - Function to be checked.\n         * @returns {Boolean}\n         */\n        hasSuper: function (fn) {\n            return _.isFunction(fn) && superReg.test(fn);\n        },\n\n        /**\n         * Extends target object with provided extenders.\n         * If property in target and extender objects is a function,\n         * then it will be wrapped using 'wrap' method.\n         *\n         * @param {Object} target - Object to be extended.\n         * @param {...Object} extenders - Multiple extenders objects.\n         * @returns {Object} Modified target object.\n         */\n        extend: function (target) {\n            var extenders = _.toArray(arguments).slice(1),\n                iterator = this._extend.bind(this, target);\n\n            extenders.forEach(iterator);\n\n            return target;\n        },\n\n        /**\n         * Same as the 'extend' method, but operates only on one extender object.\n         *\n         * @private\n         * @param {Object} target\n         * @param {Object} extender\n         */\n        _extend: function (target, extender) {\n            _.each(extender, function (value, key) {\n                target[key] = this.wrap(target[key], extender[key]);\n            }, this);\n        }\n    };\n});\n","mage/utils/misc.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\ndefine([\n    'underscore',\n    'jquery',\n    'mage/utils/objects'\n], function (_, $, utils) {\n    'use strict';\n\n    var defaultAttributes,\n        ajaxSettings,\n        map;\n\n    defaultAttributes = {\n        method: 'post',\n        enctype: 'multipart/form-data'\n    };\n\n    ajaxSettings = {\n        default: {\n            method: 'POST',\n            cache: false,\n            processData: false,\n            contentType: false\n        },\n        simple: {\n            method: 'POST',\n            dataType: 'json'\n        }\n    };\n\n    map = {\n        'D': 'DDD',\n        'dd': 'DD',\n        'd': 'D',\n        'EEEE': 'dddd',\n        'EEE': 'ddd',\n        'e': 'd',\n        'yyyy': 'YYYY',\n        'yy': 'YY',\n        'y': 'YYYY',\n        'a': 'A'\n    };\n\n    return {\n\n        /**\n         * Generates a unique identifier.\n         *\n         * @param {Number} [size=7] - Length of a resulting identifier.\n         * @returns {String}\n         */\n        uniqueid: function (size) {\n            var code = Math.random() * 25 + 65 | 0,\n                idstr = String.fromCharCode(code);\n\n            size = size || 7;\n\n            while (idstr.length < size) {\n                code = Math.floor(Math.random() * 42 + 48);\n\n                if (code < 58 || code > 64) {\n                    idstr += String.fromCharCode(code);\n                }\n            }\n\n            return idstr;\n        },\n\n        /**\n         * Limits function call.\n         *\n         * @param {Object} owner\n         * @param {String} target\n         * @param {Number} limit\n         */\n        limit: function (owner, target, limit) {\n            var fn = owner[target];\n\n            owner[target] = _.debounce(fn.bind(owner), limit);\n        },\n\n        /**\n         * Converts mage date format to a moment.js format.\n         *\n         * @param {String} mageFormat\n         * @returns {String}\n         */\n        normalizeDate: function (mageFormat) {\n            var result = mageFormat;\n\n            _.each(map, function (moment, mage) {\n                result = result.replace(\n                    new RegExp(mage + '(?=([^\\u0027]*\\u0027[^\\u0027]*\\u0027)*[^\\u0027]*$)'),\n                    moment\n                );\n            });\n            result = result.replace(/'(.*?)'/g, '[$1]');\n            return result;\n        },\n\n        /**\n         * Puts provided value in range of min and max parameters.\n         *\n         * @param {Number} value - Value to be located.\n         * @param {Number} min - Min value.\n         * @param {Number} max - Max value.\n         * @returns {Number}\n         */\n        inRange: function (value, min, max) {\n            return Math.min(Math.max(min, value), max);\n        },\n\n        /**\n         * Serializes and sends data via POST request.\n         *\n         * @param {Object} options - Options object that consists of\n         *      a 'url' and 'data' properties.\n         * @param {Object} attrs - Attributes that will be added to virtual form.\n         */\n        submit: function (options, attrs) {\n            var form        = document.createElement('form'),\n                data        = utils.serialize(options.data),\n                attributes  = _.extend({}, defaultAttributes, attrs || {});\n\n            if (!attributes.action) {\n                attributes.action = options.url;\n            }\n\n            data['form_key'] = window.FORM_KEY;\n\n            _.each(attributes, function (value, name) {\n                form.setAttribute(name, value);\n            });\n\n            data = _.map(\n                data,\n                function (value, name) {\n                    return '<input type=\"hidden\" ' +\n                        'name=\"' + _.escape(name) + '\" ' +\n                        'value=\"' + _.escape(value) + '\"' +\n                        ' />';\n                }\n            ).join('');\n\n            form.insertAdjacentHTML('afterbegin', data);\n            document.body.appendChild(form);\n\n            form.submit();\n        },\n\n        /**\n         * Serializes and sends data via AJAX POST request.\n         *\n         * @param {Object} options - Options object that consists of\n         *      a 'url' and 'data' properties.\n         * @param {Object} config\n         */\n        ajaxSubmit: function (options, config) {\n            var t = new Date().getTime(),\n                settings;\n\n            options.data['form_key'] = window.FORM_KEY;\n            options.data = this.prepareFormData(options.data, config.ajaxSaveType);\n            settings = _.extend({}, ajaxSettings[config.ajaxSaveType], options || {});\n\n            if (!config.ignoreProcessEvents) {\n                $('body').trigger('processStart');\n            }\n\n            return $.ajax(settings)\n                .done(function (data) {\n                    if (config.response) {\n                        data.t = t;\n                        config.response.data(data);\n                        config.response.status(undefined);\n                        config.response.status(!data.error);\n                    }\n                })\n                .fail(function () {\n                    if (config.response) {\n                        config.response.status(undefined);\n                        config.response.status(false);\n                        config.response.data({\n                            error: true,\n                            messages: 'Something went wrong.',\n                            t: t\n                        });\n                    }\n                })\n                .always(function () {\n                    if (!config.ignoreProcessEvents) {\n                        $('body').trigger('processStop');\n                    }\n                });\n        },\n\n        /**\n         * Creates FormData object and append this data.\n         *\n         * @param {Object} data\n         * @param {String} type\n         * @returns {FormData}\n         */\n        prepareFormData: function (data, type) {\n            var formData;\n\n            if (type === 'default') {\n                formData = new FormData();\n                _.each(utils.serialize(data), function (val, name) {\n                    formData.append(name, val);\n                });\n            } else if (type === 'simple') {\n                formData = utils.serialize(data);\n            }\n\n            return formData;\n        },\n\n        /**\n         * Filters data object. Finds properties with suffix\n         * and sets their values to properties with the same name without suffix.\n         *\n         * @param {Object} data - The data object that should be filtered\n         * @param {String} suffix - The string by which data object should be filtered\n         * @param {String} separator - The string that is separator between property and suffix\n         *\n         * @returns {Object} Filtered data object\n         */\n        filterFormData: function (data, suffix, separator) {\n            data = data || {};\n            suffix = suffix || 'prepared-for-send';\n            separator = separator || '-';\n\n            _.each(data, function (value, key) {\n                if (_.isObject(value) && !Array.isArray(value)) {\n                    this.filterFormData(value, suffix, separator);\n                } else if (_.isString(key) && ~key.indexOf(suffix)) {\n                    data[key.split(separator)[0]] = value;\n                    delete data[key];\n                }\n            }, this);\n\n            return data;\n        },\n\n        /**\n         * Replaces special characters with their corresponding HTML entities.\n         *\n         * @param {String} string - Text to escape.\n         * @returns {String} Escaped text.\n         */\n        escape: function (string) {\n            return string ? $('<p></p>').text(string).html().replace(/\"/g, '&quot;') : string;\n        },\n\n        /**\n         * Replaces symbol codes with their unescaped counterparts.\n         *\n         * @param {String} data\n         *\n         * @returns {String}\n         */\n        unescape: function (data) {\n            var unescaped = _.unescape(data),\n                mapCharacters = {\n                    '&#039;': '\\''\n                };\n\n            _.each(mapCharacters, function (value, key) {\n                unescaped = unescaped.replace(key, value);\n            });\n\n            return unescaped;\n        },\n\n        /**\n         * Converts PHP IntlFormatter format to moment format.\n         *\n         * @param {String} format - PHP format\n         * @returns {String} - moment compatible formatting\n         */\n        convertToMomentFormat: function (format) {\n            var newFormat;\n\n            newFormat = format.replace(/yyyy|yy|y/, 'YYYY'); // replace the year\n            newFormat = newFormat.replace(/dd|d/g, 'DD'); // replace the date\n\n            return newFormat;\n        },\n\n        /**\n         * Get Url Parameters.\n         *\n         * @param {String} url - Url string\n         * @returns {Object}\n         */\n        getUrlParameters: function (url) {\n            var params = {},\n                queries = url.split('?'),\n                temp,\n                i,\n                l;\n\n            if (!queries[1]) {\n                return params;\n            }\n\n            queries = queries[1].split('&');\n\n            for (i = 0, l = queries.length; i < l; i++) {\n                temp = queries[i].split('=');\n\n                if (temp[1]) {\n                    params[temp[0]] = decodeURIComponent(temp[1].replace(/\\+/g, '%20'));\n                } else {\n                    params[temp[0]] = '';\n                }\n            }\n\n            return params;\n        }\n    };\n});\n","mage/utils/arrays.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\ndefine([\n    'underscore',\n    './strings'\n], function (_, utils) {\n    'use strict';\n\n    /**\n     * Defines index of an item in a specified container.\n     *\n     * @param {*} item - Item whose index should be defined.\n     * @param {Array} container - Container upon which to perform search.\n     * @returns {Number}\n     */\n    function getIndex(item, container) {\n        var index = container.indexOf(item);\n\n        if (~index) {\n            return index;\n        }\n\n        return _.findIndex(container, function (value) {\n            return value && value.name === item;\n        });\n    }\n\n    return {\n        /**\n         * Facade method to remove/add value from/to array\n         * without creating a new instance.\n         *\n         * @param {Array} arr - Array to be modified.\n         * @param {*} value - Value to add/remove.\n         * @param {Boolean} add - Flag that specfies operation.\n         * @returns {Utils} Chainable.\n         */\n        toggle: function (arr, value, add) {\n            return add ?\n                this.add(arr, value) :\n                this.remove(arr, value);\n        },\n\n        /**\n         * Removes the incoming value from array in case\n         * without creating a new instance of it.\n         *\n         * @param {Array} arr - Array to be modified.\n         * @param {*} value - Value to be removed.\n         * @returns {Utils} Chainable.\n         */\n        remove: function (arr, value) {\n            var index = arr.indexOf(value);\n\n            if (~index) {\n                arr.splice(index, 1);\n            }\n\n            return this;\n        },\n\n        /**\n         * Adds the incoming value to array if\n         * it's not alredy present in there.\n         *\n         * @param {Array} arr - Array to be modifed.\n         * @param {...*} arguments - Values to be added.\n         * @returns {Utils} Chainable.\n         */\n        add: function (arr) {\n            var values = _.toArray(arguments).slice(1);\n\n            values.forEach(function (value) {\n                if (!~arr.indexOf(value)) {\n                    arr.push(value);\n                }\n            });\n\n            return this;\n        },\n\n        /**\n         * Inserts specified item into container at a specified position.\n         *\n         * @param {*} item - Item to be inserted into container.\n         * @param {Array} container - Container of items.\n         * @param {*} [position=-1] - Position at which item should be inserted.\n         *      Position can represent:\n         *          - specific index in container\n         *          - item which might already be present in container\n         *          - structure with one of these properties: after, before\n         * @returns {Boolean|*}\n         *      - true if element has changed its' position\n         *      - false if nothing has changed\n         *      - inserted value if it wasn't present in container\n         */\n        insert: function (item, container, position) {\n            var currentIndex = getIndex(item, container),\n                newIndex,\n                target;\n\n            if (typeof position === 'undefined') {\n                position = -1;\n            } else if (typeof position === 'string') {\n                position = isNaN(+position) ? position : +position;\n            }\n\n            newIndex = position;\n\n            if (~currentIndex) {\n                target = container.splice(currentIndex, 1)[0];\n\n                if (typeof item === 'string') {\n                    item = target;\n                }\n            }\n\n            if (typeof position !== 'number') {\n                target = position.after || position.before || position;\n\n                newIndex = getIndex(target, container);\n\n                if (~newIndex && (position.after || newIndex >= currentIndex)) {\n                    newIndex++;\n                }\n            }\n\n            if (newIndex < 0) {\n                newIndex += container.length + 1;\n            }\n\n            container[newIndex] ?\n                container.splice(newIndex, 0, item) :\n                container[newIndex] = item;\n\n            return !~currentIndex ? item : currentIndex !== newIndex;\n        },\n\n        /**\n         * @param {Array} elems\n         * @param {Number} offset\n         * @return {Number|*}\n         */\n        formatOffset: function (elems, offset) {\n            if (utils.isEmpty(offset)) {\n                offset = -1;\n            }\n\n            offset = +offset;\n\n            if (offset < 0) {\n                offset += elems.length + 1;\n            }\n\n            return offset;\n        }\n    };\n});\n","mage/utils/main.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\ndefine(function (require) {\n    'use strict';\n\n    var utils = {},\n        _ = require('underscore'),\n        root = typeof self == 'object' && self.self === self && self ||\n            typeof global == 'object' && global.global === global && global ||\n            Function('return this')() || {};\n\n    root._ = _;\n\n    return _.extend(\n        utils,\n        require('./arrays'),\n        require('./compare'),\n        require('./misc'),\n        require('./objects'),\n        require('./strings'),\n        require('./template')\n    );\n});\n","mage/utils/objects.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\ndefine([\n    'ko',\n    'jquery',\n    'underscore',\n    'mage/utils/strings'\n], function (ko, $, _, stringUtils) {\n    'use strict';\n\n    var primitives = [\n        'undefined',\n        'boolean',\n        'number',\n        'string'\n    ];\n\n    /**\n     * Sets nested property of a specified object.\n     * @private\n     *\n     * @param {Object} parent - Object to look inside for the properties.\n     * @param {Array} path - Splitted path the property.\n     * @param {*} value - Value of the last property in 'path' array.\n     * returns {*} New value for the property.\n     */\n    function setNested(parent, path, value) {\n        var last = path.pop(),\n            len = path.length,\n            pi = 0,\n            part = path[pi];\n\n        for (; pi < len; part = path[++pi]) {\n            if (!_.isObject(parent[part])) {\n                parent[part] = {};\n            }\n\n            parent = parent[part];\n        }\n\n        if (typeof parent[last] === 'function') {\n            parent[last](value);\n        } else {\n            parent[last] = value;\n        }\n\n        return value;\n    }\n\n    /**\n     * Retrieves value of a nested property.\n     * @private\n     *\n     * @param {Object} parent - Object to look inside for the properties.\n     * @param {Array} path - Splitted path the property.\n     * @returns {*} Value of the property.\n     */\n    function getNested(parent, path) {\n        var exists = true,\n            len = path.length,\n            pi = 0;\n\n        for (; pi < len && exists; pi++) {\n            parent = parent[path[pi]];\n\n            if (typeof parent === 'undefined') {\n                exists = false;\n            }\n        }\n\n        if (exists) {\n            if (ko.isObservable(parent)) {\n                parent = parent();\n            }\n\n            return parent;\n        }\n    }\n\n    /**\n     * Removes property from a specified object.\n     * @private\n     *\n     * @param {Object} parent - Object from which to remove property.\n     * @param {Array} path - Splitted path to the property.\n     */\n    function removeNested(parent, path) {\n        var field = path.pop();\n\n        parent = getNested(parent, path);\n\n        if (_.isObject(parent)) {\n            delete parent[field];\n        }\n    }\n\n    return {\n\n        /**\n         * Retrieves or defines objects' property by a composite path.\n         *\n         * @param {Object} data - Container for the properties specified in path.\n         * @param {String} path - Objects' properties divided by dots.\n         * @param {*} [value] - New value for the last property.\n         * @returns {*} Returns value of the last property in chain.\n         *\n         * @example\n         *      utils.nested({}, 'one.two', 3);\n         *      => { one: {two: 3} }\n         */\n        nested: function (data, path, value) {\n            var action = arguments.length > 2 ? setNested : getNested;\n\n            path = path ? path.split('.') : [];\n\n            return action(data, path, value);\n        },\n\n        /**\n         * Removes nested property from an object.\n         *\n         * @param {Object} data - Data source.\n         * @param {String} path - Path to the property e.g. 'one.two.three'\n         */\n        nestedRemove: function (data, path) {\n            path = path.split('.');\n\n            removeNested(data, path);\n        },\n\n        /**\n         * Flattens objects' nested properties.\n         *\n         * @param {Object} data - Object to flatten.\n         * @param {String} [separator='.'] - Objects' keys separator.\n         * @returns {Object} Flattened object.\n         *\n         * @example Example with a default separator.\n         *      utils.flatten({one: { two: { three: 'value'} }});\n         *      => { 'one.two.three': 'value' };\n         *\n         * @example Example with a custom separator.\n         *      utils.flatten({one: { two: { three: 'value'} }}, '=>');\n         *      => {'one=>two=>three': 'value'};\n         */\n        flatten: function (data, separator, parent, result) {\n            separator = separator || '.';\n            result = result || {};\n\n            if (!data) {\n                return result;\n            }\n\n            // UnderscoreJS each breaks when an object has a length property so we use Object.keys\n            _.each(Object.keys(data), function (name) {\n                var node = data[name];\n\n                if ({}.toString.call(node) === '[object Function]') {\n                    return;\n                }\n\n                if (parent) {\n                    name = parent + separator + name;\n                }\n\n                typeof node === 'object' ?\n                    this.flatten(node, separator, name, result) :\n                    result[name] = node;\n\n            }, this);\n\n            return result;\n        },\n\n        /**\n         * Opposite operation of the 'flatten' method.\n         *\n         * @param {Object} data - Previously flattened object.\n         * @param {String} [separator='.'] - Keys separator.\n         * @returns {Object} Object with nested properties.\n         *\n         * @example Example using custom separator.\n         *      utils.unflatten({'one=>two': 'value'}, '=>');\n         *      => {\n         *          one: { two: 'value' }\n         *      };\n         */\n        unflatten: function (data, separator) {\n            var result = {};\n\n            separator = separator || '.';\n\n            _.each(data, function (value, nodes) {\n                nodes = nodes.split(separator);\n\n                setNested(result, nodes, value);\n            });\n\n            return result;\n        },\n\n        /**\n         * Same operation as 'flatten' method,\n         * but returns objects' keys wrapped in '[]'.\n         *\n         * @param {Object} data - Object that should be serialized.\n         * @returns {Object} Serialized data.\n         *\n         * @example\n         *      utils.serialize({one: { two: { three: 'value'} }});\n         *      => { 'one[two][three]': 'value' }\n         */\n        serialize: function (data) {\n            var result = {};\n\n            data = this.flatten(data);\n\n            _.each(data, function (value, keys) {\n                keys = stringUtils.serializeName(keys);\n                value = _.isUndefined(value) ? '' : value;\n\n                result[keys] = value;\n            }, this);\n\n            return result;\n        },\n\n        /**\n         * Performs deep extend of specified objects.\n         *\n         * @returns {Object|Array} Extended object.\n         */\n        extend: function () {\n            var args = _.toArray(arguments);\n\n            args.unshift(true);\n\n            return $.extend.apply($, args);\n        },\n\n        /**\n         * Performs a deep clone of a specified object.\n         *\n         * @param {(Object|Array)} data - Data that should be copied.\n         * @returns {Object|Array} Cloned object.\n         */\n        copy: function (data) {\n            var result = data,\n                isArray = Array.isArray(data),\n                placeholder;\n\n            if (this.isObject(data) || isArray) {\n                placeholder = isArray ? [] : {};\n                result = this.extend(placeholder, data);\n            }\n\n            return result;\n        },\n\n        /**\n         * Performs a deep clone of a specified object.\n         * Doesn't save links to original object.\n         *\n         * @param {*} original - Object to clone\n         * @returns {*}\n         */\n        hardCopy: function (original) {\n            if (original === null || typeof original !== 'object') {\n                return original;\n            }\n\n            return JSON.parse(JSON.stringify(original));\n        },\n\n        /**\n         * Removes specified nested properties from the target object.\n         *\n         * @param {Object} target - Object whose properties should be removed.\n         * @param {(...String|Array|Object)} list - List that specifies properties to be removed.\n         * @returns {Object} Modified object.\n         *\n         * @example Basic usage\n         *      var obj = {a: {b: 2}, c: 'a'};\n         *\n         *      omit(obj, 'a.b');\n         *      => {'a.b': 2};\n         *      obj => {a: {}, c: 'a'};\n         *\n         * @example Various syntaxes that would return same result\n         *      omit(obj, ['a.b', 'c']);\n         *      omit(obj, 'a.b', 'c');\n         *      omit(obj, {'a.b': true, 'c': true});\n         */\n        omit: function (target, list) {\n            var removed = {},\n                ignored = list;\n\n            if (this.isObject(list)) {\n                ignored = [];\n\n                _.each(list, function (value, key) {\n                    if (value) {\n                        ignored.push(key);\n                    }\n                });\n            } else if (_.isString(list)) {\n                ignored = _.toArray(arguments).slice(1);\n            }\n\n            _.each(ignored, function (path) {\n                var value = this.nested(target, path);\n\n                if (!_.isUndefined(value)) {\n                    removed[path] = value;\n\n                    this.nestedRemove(target, path);\n                }\n            }, this);\n\n            return removed;\n        },\n\n        /**\n         * Checks if provided value is a plain object.\n         *\n         * @param {*} value - Value to be checked.\n         * @returns {Boolean}\n         */\n        isObject: function (value) {\n            var objProto = Object.prototype;\n\n            return typeof value == 'object' ?\n            objProto.toString.call(value) === '[object Object]' :\n                false;\n        },\n\n        /**\n         *\n         * @param {*} value\n         * @returns {Boolean}\n         */\n        isPrimitive: function (value) {\n            return value === null || ~primitives.indexOf(typeof value);\n        },\n\n        /**\n         * Iterates over obj props/array elems recursively, applying action to each one\n         *\n         * @param {Object|Array} data - Data to be iterated.\n         * @param {Function} action - Callback to be called with each item as an argument.\n         * @param {Number} [maxDepth=7] - Max recursion depth.\n         */\n        forEachRecursive: function (data, action, maxDepth) {\n            maxDepth = typeof maxDepth === 'number' && !isNaN(maxDepth) ? maxDepth - 1 : 7;\n\n            if (!_.isFunction(action) || _.isFunction(data) || maxDepth < 0) {\n                return;\n            }\n\n            if (!_.isObject(data)) {\n                action(data);\n\n                return;\n            }\n\n            _.each(data, function (value) {\n                this.forEachRecursive(value, action, maxDepth);\n            }, this);\n\n            action(data);\n        },\n\n        /**\n         * Maps obj props/array elems recursively\n         *\n         * @param {Object|Array} data - Data to be iterated.\n         * @param {Function} action - Callback to transform each item.\n         * @param {Number} [maxDepth=7] - Max recursion depth.\n         *\n         * @returns {Object|Array}\n         */\n        mapRecursive: function (data, action, maxDepth) {\n            var newData;\n\n            maxDepth = typeof maxDepth === 'number' && !isNaN(maxDepth) ? maxDepth - 1 : 7;\n\n            if (!_.isFunction(action) || _.isFunction(data) || maxDepth < 0) {\n                return data;\n            }\n\n            if (!_.isObject(data)) {\n                return action(data);\n            }\n\n            if (_.isArray(data)) {\n                newData = _.map(data, function (item) {\n                    return this.mapRecursive(item, action, maxDepth);\n                }, this);\n\n                return action(newData);\n            }\n\n            newData = _.mapObject(data, function (val, key) {\n                if (data.hasOwnProperty(key)) {\n                    return this.mapRecursive(val, action, maxDepth);\n                }\n\n                return val;\n            }, this);\n\n            return action(newData);\n        },\n\n        /**\n         * Removes empty(in common sence) obj props/array elems\n         *\n         * @param {*} data - Data to be cleaned.\n         * @returns {*}\n         */\n        removeEmptyValues: function (data) {\n            if (!_.isObject(data)) {\n                return data;\n            }\n\n            if (_.isArray(data)) {\n                return data.filter(function (item) {\n                    return !this.isEmptyObj(item);\n                }, this);\n            }\n\n            return _.omit(data, this.isEmptyObj.bind(this));\n        },\n\n        /**\n         * Checks that argument of any type is empty in common sence:\n         * empty string, string with spaces only, object without own props, empty array, null or undefined\n         *\n         * @param {*} val - Value to be checked.\n         * @returns {Boolean}\n         */\n        isEmptyObj: function (val) {\n\n            return _.isObject(val) && _.isEmpty(val) ||\n            this.isEmpty(val) ||\n            val && val.trim && this.isEmpty(val.trim());\n        }\n    };\n});\n\n","mage/utils/compare.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\ndefine([\n    'underscore',\n    'mage/utils/objects'\n], function (_, utils) {\n    'use strict';\n\n    var result = [];\n\n    /**\n     * Checks if all of the provided arrays contains equal values.\n     *\n     * @param {(Boolean|Array)} [keepOrder=false]\n     * @param {Array} target\n     * @returns {Boolean}\n     */\n    function equalArrays(keepOrder, target) {\n        var args = _.toArray(arguments),\n            arrays;\n\n        if (!Array.isArray(keepOrder)) {\n            arrays      = args.slice(2);\n        } else {\n            target      = keepOrder;\n            keepOrder   = false;\n            arrays      = args.slice(1);\n        }\n\n        if (!arrays.length) {\n            return true;\n        }\n\n        return arrays.every(function (array) {\n            if (array === target) {\n                return true;\n            } else if (array.length !== target.length) {\n                return false;\n            } else if (!keepOrder) {\n                return !_.difference(target, array).length;\n            }\n\n            return array.every(function (value, index) {\n                return target[index] === value;\n            });\n        });\n    }\n\n    /**\n     * Checks if two values are different.\n     *\n     * @param {*} a - First value.\n     * @param {*} b - Second value.\n     * @returns {Boolean}\n     */\n    function isDifferent(a, b) {\n        var oldIsPrimitive = utils.isPrimitive(a);\n\n        if (Array.isArray(a) && Array.isArray(b)) {\n            return !equalArrays(true, a, b);\n        }\n\n        return oldIsPrimitive ? a !== b : true;\n    }\n\n    /**\n     * @param {String} prefix\n     * @param {String} part\n     */\n    function getPath(prefix, part) {\n        return prefix ? prefix + '.' + part : part;\n    }\n\n    /**\n     * Checks if object has own specified property.\n     *\n     * @param {*} obj - Value to be checked.\n     * @param {String} key - Key of the property.\n     * @returns {Boolean}\n     */\n    function hasOwn(obj, key) {\n        return Object.prototype.hasOwnProperty.call(obj, key);\n    }\n\n    /**\n     * @param {Array} changes\n     */\n    function getContainers(changes) {\n        var containers  = {},\n            indexed     = _.indexBy(changes, 'path');\n\n        _.each(indexed, function (change, name) {\n            var path;\n\n            name.split('.').forEach(function (part) {\n                path = getPath(path, part);\n\n                if (path in indexed) {\n                    return;\n                }\n\n                (containers[path] = containers[path] || []).push(change);\n            });\n        });\n\n        return containers;\n    }\n\n    /**\n     * @param {String} path\n     * @param {String} name\n     * @param {String} type\n     * @param {String} newValue\n     * @param {String} oldValue\n     */\n    function addChange(path, name, type, newValue, oldValue) {\n        var data;\n\n        data = {\n            path: path,\n            name: name,\n            type: type\n        };\n\n        if (type !== 'remove') {\n            data.value = newValue;\n            data.oldValue = oldValue;\n        } else {\n            data.oldValue = newValue;\n        }\n\n        result.push(data);\n    }\n\n    /**\n     * @param {String} ns\n     * @param {String} name\n     * @param {String} type\n     * @param {String} iterator\n     * @param {String} placeholder\n     */\n    function setAll(ns, name, type, iterator, placeholder) {\n        var key;\n\n        if (arguments.length > 4) {\n            type === 'add' ?\n                addChange(ns, name, 'update', iterator, placeholder) :\n                addChange(ns, name, 'update', placeholder, iterator);\n        } else {\n            addChange(ns, name, type, iterator);\n        }\n\n        if (!utils.isObject(iterator)) {\n            return;\n        }\n\n        for (key in iterator) {\n            if (hasOwn(iterator, key)) {\n                setAll(getPath(ns, key), key, type, iterator[key]);\n            }\n        }\n    }\n\n    /*eslint-disable max-depth*/\n    /**\n     * @param {Object} old\n     * @param {Object} current\n     * @param {String} ns\n     * @param {String} name\n     */\n    function compare(old, current, ns, name) {\n        var key,\n            oldIsObj = utils.isObject(old),\n            newIsObj = utils.isObject(current);\n\n        if (oldIsObj && newIsObj) {\n            for (key in old) {\n                if (hasOwn(old, key) && !hasOwn(current, key)) {\n                    setAll(getPath(ns, key), key, 'remove', old[key]);\n                }\n            }\n\n            for (key in current) {\n                if (hasOwn(current, key)) {\n                    hasOwn(old, key) ?\n                        compare(old[key], current[key], getPath(ns, key), key) :\n                        setAll(getPath(ns, key), key, 'add', current[key]);\n                }\n            }\n        } else if (oldIsObj) {\n            setAll(ns, name, 'remove', old, current);\n        } else if (newIsObj) {\n            setAll(ns, name, 'add', current, old);\n        } else if (isDifferent(old, current)) {\n            addChange(ns, name, 'update', current, old);\n        }\n    }\n\n    /*eslint-enable max-depth*/\n\n    return {\n\n        /**\n         *\n         * @returns {Object}\n         */\n        compare: function () {\n            var changes;\n\n            compare.apply(null, arguments);\n\n            changes = result.splice(0);\n\n            return {\n                containers: getContainers(changes),\n                changes: changes,\n                equal: !changes.length\n            };\n        },\n\n        equalArrays: equalArrays\n    };\n});\n","mage/utils/strings.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\ndefine([\n    'underscore'\n], function (_) {\n    'use strict';\n\n    var jsonRe = /^(?:\\{[\\w\\W]*\\}|\\[[\\w\\W]*\\])$/;\n\n    return {\n\n        /**\n         * Attempts to convert string to one of the primitive values,\n         * or to parse it as a valid json object.\n         *\n         * @param {String} str - String to be processed.\n         * @returns {*}\n         */\n        castString: function (str) {\n            try {\n                str = str === 'true' ? true :\n                    str === 'false' ? false :\n                        str === 'null' ? null :\n                            +str + '' === str ? +str :\n                                jsonRe.test(str) ? JSON.parse(str) :\n                                    str;\n            } catch (e) {\n            }\n\n            return str;\n        },\n\n        /**\n         * Splits string by separator if it's possible,\n         * otherwise returns the incoming value.\n         *\n         * @param {(String|Array|*)} str - String to split.\n         * @param {String} [separator=' '] - Seperator based on which to split the string.\n         * @returns {Array|*} Splitted string or the incoming value.\n         */\n        stringToArray: function (str, separator) {\n            separator = separator || ' ';\n\n            return typeof str === 'string' ?\n                str.split(separator) :\n                str;\n        },\n\n        /**\n         * Converts the incoming string which consists\n         * of a specified delimiters into a format commonly used in form elements.\n         *\n         * @param {String} name - The incoming string.\n         * @param {String} [separator='.']\n         * @returns {String} Serialized string.\n         *\n         * @example\n         *      utils.serializeName('one.two.three');\n         *      => 'one[two][three]';\n         */\n        serializeName: function (name, separator) {\n            var result;\n\n            separator = separator || '.';\n            name = name.split(separator);\n\n            result = name.shift();\n\n            name.forEach(function (part) {\n                result += '[' + part + ']';\n            });\n\n            return result;\n        },\n\n        /**\n         * Checks wether the incoming value is not empty,\n         * e.g. not 'null' or 'undefined'\n         *\n         * @param {*} value - Value to check.\n         * @returns {Boolean}\n         */\n        isEmpty: function (value) {\n            return value === '' || _.isUndefined(value) || _.isNull(value);\n        },\n\n        /**\n         * Adds 'prefix' to the 'part' value if it was provided.\n         *\n         * @param {String} prefix\n         * @param {String} part\n         * @returns {String}\n         */\n        fullPath: function (prefix, part) {\n            return prefix ? prefix + '.' + part : part;\n        },\n\n        /**\n         * Splits incoming string and returns its' part specified by offset.\n         *\n         * @param {String} parts\n         * @param {Number} [offset]\n         * @param {String} [delimiter=.]\n         * @returns {String}\n         */\n        getPart: function (parts, offset, delimiter) {\n            delimiter = delimiter || '.';\n            parts = parts.split(delimiter);\n            offset = this.formatOffset(parts, offset);\n\n            parts.splice(offset, 1);\n\n            return parts.join(delimiter) || '';\n        },\n\n        /**\n         * Converts nameThroughCamelCase to name-through-minus\n         *\n         * @param {String} string\n         * @returns {String}\n         */\n        camelCaseToMinus: function camelCaseToMinus(string) {\n            return ('' + string)\n                .split('')\n                .map(function (symbol, index) {\n                    return index ?\n                        symbol.toUpperCase() === symbol ?\n                        '-' + symbol.toLowerCase() :\n                            symbol :\n                        symbol.toLowerCase();\n                })\n                .join('');\n        },\n\n        /**\n         * Converts name-through-minus to nameThroughCamelCase\n         *\n         * @param {String} string\n         * @returns {String}\n         */\n        minusToCamelCase: function minusToCamelCase(string) {\n            return ('' + string)\n                .split('-')\n                .map(function (part, index) {\n                    return index ? part.charAt(0).toUpperCase() + part.slice(1) : part;\n                })\n                .join('');\n        }\n    };\n});\n","mage/utils/template.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\n/* eslint-disable no-shadow */\n\ndefine([\n    'jquery',\n    'underscore',\n    'mage/utils/objects',\n    'mage/utils/strings'\n], function ($, _, utils, stringUtils) {\n    'use strict';\n\n    var tmplSettings = _.templateSettings,\n        interpolate = /\\$\\{([\\s\\S]+?)\\}/g,\n        opener = '${',\n        template,\n        hasStringTmpls;\n\n    /**\n     * Identifies whether ES6 templates are supported.\n     */\n    hasStringTmpls = (function () {\n        var testString = 'var foo = \"bar\"; return `${ foo }` === foo';\n\n        try {\n            return Function(testString)();\n        } catch (e) {\n            return false;\n        }\n    })();\n\n    /**\n     * Objects can specify how to use templating for their properties - getting that configuration.\n     *\n     * To disable rendering for all properties of your object add __disableTmpl: true.\n     * To disable for specific property add __disableTmpl: {propertyName: true}.\n     * To limit recursion for a specific property add __disableTmpl: {propertyName: numberOfCycles}.\n     *\n     * @param {String} tmpl\n     * @param {Object | undefined} target\n     * @returns {Boolean|Object}\n     */\n    function isTmplIgnored(tmpl, target) {\n        var parsedTmpl;\n\n        try {\n            parsedTmpl = JSON.parse(tmpl);\n\n            if (typeof parsedTmpl === 'object') {\n                return tmpl.includes('__disableTmpl');\n            }\n        } catch (e) {\n        }\n\n        if (typeof target !== 'undefined') {\n            if (typeof target === 'object' && target.hasOwnProperty('__disableTmpl')) {\n                return target.__disableTmpl;\n            }\n        }\n\n        return false;\n\n    }\n\n    if (hasStringTmpls) {\n\n        /*eslint-disable no-unused-vars, no-eval*/\n        /**\n         * Evaluates template string using ES6 templates.\n         *\n         * @param {String} tmpl - Template string.\n         * @param {Object} $ - Data object used in a template.\n         * @returns {String} Compiled template.\n         */\n        template = function (tmpl, $) {\n            return eval('`' + tmpl + '`');\n        };\n\n        /*eslint-enable no-unused-vars, no-eval*/\n    } else {\n\n        /**\n         * Fallback function used when ES6 templates are not supported.\n         * Uses underscore templates renderer.\n         *\n         * @param {String} tmpl - Template string.\n         * @param {Object} data - Data object used in a template.\n         * @returns {String} Compiled template.\n         */\n        template = function (tmpl, data) {\n            var cached = tmplSettings.interpolate;\n\n            tmplSettings.interpolate = interpolate;\n\n            tmpl = _.template(tmpl, {\n                variable: '$'\n            })(data);\n\n            tmplSettings.interpolate = cached;\n\n            return tmpl;\n        };\n    }\n\n    /**\n     * Checks if provided value contains template syntax.\n     *\n     * @param {*} value - Value to be checked.\n     * @returns {Boolean}\n     */\n    function isTemplate(value) {\n        return typeof value === 'string' &&\n            value.indexOf(opener) !== -1 &&\n            // the below pattern almost always indicates an accident which should not cause template evaluation\n            // refuse to evaluate\n            value.indexOf('${{') === -1;\n    }\n\n    /**\n     * Iteratively processes provided string\n     * until no templates syntax will be found.\n     *\n     * @param {String} tmpl - Template string.\n     * @param {Object} data - Data object used in a template.\n     * @param {Boolean} [castString=false] - Flag that indicates whether template\n     *      should be casted after evaluation to a value of another type or\n     *      that it should be leaved as a string.\n     * @param {Number|undefined} maxCycles - Maximum number of rendering cycles, can be 0.\n     * @returns {*} Compiled template.\n     */\n    function render(tmpl, data, castString, maxCycles) {\n        var last = tmpl,\n            cycles = 0;\n\n        while (~tmpl.indexOf(opener) && (typeof maxCycles === 'undefined' || cycles < maxCycles)) {\n            if (!isTmplIgnored(tmpl)) {\n                tmpl = template(tmpl, data);\n            }\n\n            if (tmpl === last) {\n                break;\n            }\n\n            last = tmpl;\n            cycles++;\n        }\n\n        return castString ?\n            stringUtils.castString(tmpl) :\n            tmpl;\n    }\n\n    return {\n\n        /**\n         * Applies provided data to the template.\n         *\n         * @param {Object|String} tmpl\n         * @param {Object} [data] - Data object to match with template.\n         * @param {Boolean} [castString=false] - Flag that indicates whether template\n         *      should be casted after evaluation to a value of another type or\n         *      that it should be leaved as a string.\n         * @returns {*}\n         *\n         * @example Template defined as a string.\n         *      var source = { foo: 'Random Stuff', bar: 'Some' };\n         *\n         *      utils.template('${ $.bar } ${ $.foo }', source);\n         *      => 'Some Random Stuff';\n         *\n         * @example Template defined as an object.\n         *      var tmpl = {\n         *              key: {'${ $.$data.bar }': '${ $.$data.foo }'},\n         *              foo: 'bar',\n         *              x1: 2, x2: 5,\n         *              delta: '${ $.x2 - $.x1 }',\n         *              baz: 'Upper ${ $.foo.toUpperCase() }'\n         *      };\n         *\n         *      utils.template(tmpl, source);\n         *      => {\n         *          key: {'Some': 'Random Stuff'},\n         *          foo: 'bar',\n         *          x1: 2, x2: 5,\n         *          delta: 3,\n         *          baz: 'Upper BAR'\n         *      };\n         */\n        template: function (tmpl, data, castString, dontClone) {\n            if (typeof tmpl === 'string') {\n                return render(tmpl, data, castString);\n            }\n\n            if (!dontClone) {\n                tmpl = utils.copy(tmpl);\n            }\n\n            tmpl.$data = data || {};\n\n            /**\n             * Template iterator function.\n             */\n            _.each(tmpl, function iterate(value, key, list) {\n                var disabled,\n                    maxCycles;\n\n                if (key === '$data') {\n                    return;\n                }\n\n                if (isTemplate(key)) {\n                    delete list[key];\n\n                    key = render(key, tmpl);\n                    list[key] = value;\n                }\n\n                if (isTemplate(value)) {\n                    //Getting template disabling settings, can be true for all disabled and separate settings\n                    //for each property.\n                    disabled = isTmplIgnored(value, list);\n\n                    if (typeof disabled === 'object' && disabled.hasOwnProperty(key) && disabled[key] !== false) {\n                        //Checking if specific settings for a property provided.\n                        maxCycles = disabled[key];\n                    }\n\n                    if (disabled === true || maxCycles === true) {\n                        //Rendering for all properties is disabled.\n                        maxCycles = 0;\n                    }\n\n                    list[key] = render(value, tmpl, castString, maxCycles);\n                } else if ($.isPlainObject(value) || Array.isArray(value)) {\n                    _.each(value, iterate);\n                }\n            });\n\n            delete tmpl.$data;\n\n            return tmpl;\n        }\n    };\n});\n","mage/backend/notification.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\ndefine([\n    'jquery',\n    'mage/template',\n    'jquery/ui'\n], function ($, mageTemplate) {\n    'use strict';\n\n    $.widget('mage.notification', {\n        options: {\n            templates: {\n                global: '<div data-role=\"messages\" id=\"messages\">' +\n                    '<div class=\"message <% if (data.error) { %>error<% } %>\"><div><%- data.message %></div></div>' +\n                '</div>',\n                error: '<div data-role=\"messages\" id=\"messages\">' +\n                    '<div class=\"messages\"><div class=\"message message-error error\">' +\n                        '<div data-ui-id=\"messages-message-error\"><%- data.message %></div></div>' +\n                    '</div></div>'\n            }\n        },\n        placeholder: '[data-role=messages]',\n\n        /**\n         * Notification creation\n         * @protected\n         */\n        _create: function () {\n            $(document).on('ajaxComplete ajaxError', $.proxy(this._add, this));\n        },\n\n        /**\n         * Add new message\n         * @protected\n         * @param {Object} event - object\n         * @param {Object} jqXHR - The jQuery XMLHttpRequest object returned by $.ajax()\n         */\n        _add: function (event, jqXHR) {\n            var response;\n\n            try {\n                response = JSON.parse(jqXHR.responseText);\n\n                if (response && response.error && response['html_message']) {\n                    $(this.placeholder).html(response['html_message']);\n                }\n            } catch (e) {}\n        },\n\n        /**\n         * Adds new message.\n         *\n         * @param {Object} data - Data with a message to be displayed.\n         */\n        add: function (data) {\n            var template = data.error ? this.options.templates.error : this.options.templates.global,\n                message = mageTemplate(template, {\n                    data: data\n                }),\n                messageContainer;\n\n            if (typeof data.insertMethod === 'function') {\n                data.insertMethod(message);\n            } else {\n                messageContainer = data.messageContainer || this.placeholder;\n                $(messageContainer).prepend(message);\n            }\n\n            return this;\n        },\n\n        /**\n         * Removes error messages.\n         */\n        clear: function () {\n            $(this.placeholder).html('');\n        }\n    });\n\n    return $.mage.notification;\n});\n","mage/backend/action-link.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\ndefine([\n    'jquery',\n    'jquery/ui'\n], function ($) {\n    'use strict';\n\n    $.widget('mage.actionLink', {\n        /**\n         * Button creation\n         * @protected\n         */\n        _create: function () {\n            this._bind();\n        },\n\n        /**\n         * Bind handler on button click\n         * @protected\n         */\n        _bind: function () {\n            var keyCode = $.ui.keyCode;\n\n            this._on({\n                /**\n                 * @param {jQuery.Event} e\n                 */\n                mousedown: function (e) {\n                    this._stopPropogation(e);\n                },\n\n                /**\n                 * @param {jQuery.Event} e\n                 */\n                mouseup: function (e) {\n                    this._stopPropogation(e);\n                },\n\n                /**\n                 * @param {jQuery.Event} e\n                 */\n                click: function (e) {\n                    this._stopPropogation(e);\n                    this._triggerEvent();\n                },\n\n                /**\n                 * @param {jQuery.Event} e\n                 */\n                keydown: function (e) {\n                    switch (e.keyCode) {\n                        case keyCode.ENTER:\n                        case keyCode.NUMPAD_ENTER:\n                            this._stopPropogation(e);\n                            this._triggerEvent();\n                            break;\n                    }\n                },\n\n                /**\n                 * @param {jQuery.Event} e\n                 */\n                keyup: function (e) {\n                    switch (e.keyCode) {\n                        case keyCode.ENTER:\n                        case keyCode.NUMPAD_ENTER:\n                            this._stopPropogation(e);\n                            break;\n                    }\n                }\n            });\n        },\n\n        /**\n         * @param {Object} e - event object\n         * @private\n         */\n        _stopPropogation: function (e) {\n            e.stopImmediatePropagation();\n            e.preventDefault();\n        },\n\n        /**\n         * @private\n         */\n        _triggerEvent: function () {\n            $(this.options.related || this.element)\n                .trigger(this.options.event, this.options.eventData ? [this.options.eventData] : [{}]);\n        }\n    });\n\n    return $.mage.actionLink;\n});\n","mage/backend/bootstrap.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\n/* global FORM_KEY */\ndefine([\n    'jquery',\n    'mage/apply/main',\n    'mage/backend/notification',\n    'Magento_Ui/js/lib/knockout/bootstrap',\n    'mage/mage',\n    'mage/translate'\n], function ($, mage, notification) {\n    'use strict';\n\n    var bootstrap;\n\n    $.ajaxSetup({\n        /*\n         * @type {string}\n         */\n        type: 'POST',\n\n        /**\n         * Ajax before send callback.\n         *\n         * @param {Object} jqXHR - The jQuery XMLHttpRequest object returned by $.ajax()\n         * @param {Object} settings\n         */\n        beforeSend: function (jqXHR, settings) {\n            var formKey = typeof FORM_KEY !== 'undefined' ? FORM_KEY : null;\n\n            if (!settings.url.match(new RegExp('[?&]isAjax=true',''))) {\n                settings.url = settings.url.match(\n                    new RegExp('\\\\?', 'g')) ?\n                    settings.url + '&isAjax=true' :\n                    settings.url + '?isAjax=true';\n            }\n\n            if (!settings.data) {\n                settings.data = {\n                    'form_key': formKey\n                };\n            } else if (typeof settings.data === 'string' &&\n                settings.data.indexOf('form_key=') === -1) {\n                settings.data += '&' + $.param({\n                    'form_key': formKey\n                });\n            } else if ($.isPlainObject(settings.data) && !settings.data['form_key']) {\n                settings.data['form_key'] = formKey;\n            }\n        },\n\n        /**\n         * Ajax complete callback.\n         *\n         * @param {Object} jqXHR - The jQuery XMLHttpRequest object returned by $.ajax()\n         */\n        complete: function (jqXHR) {\n            var jsonObject;\n\n            if (jqXHR.readyState === 4) {\n                try {\n                    jsonObject = JSON.parse(jqXHR.responseText);\n\n                    if (jsonObject.ajaxExpired && jsonObject.ajaxRedirect) { //eslint-disable-line max-depth\n                        window.location.replace(jsonObject.ajaxRedirect);\n                    }\n                } catch (e) {}\n            }\n        },\n\n        /**\n         * Error callback.\n         *\n         * @param {Object} jqXHR - The jQuery XMLHttpRequest object returned by $.ajax()\n         * @param {String} textStatus\n         */\n        error: function (jqXHR, textStatus) {\n            $('body').notification('clear');\n            if (jqXHR.readyState !== 0 && textStatus !== 'abort') {\n                $('body').notification('add', {\n                    error: true,\n                    message: $.mage.__(\n                        'A technical problem with the server created an error. ' +\n                        'Try again to continue what you were doing. If the problem persists, try again later.'\n                    ),\n\n                    /**\n                     * @param {String} message\n                     */\n                    insertMethod: function (message) {\n                        var $wrapper = $('<div></div>').html(message);\n\n                        $('.page-main-actions').after($wrapper);\n                    }\n                });\n            }\n        }\n    });\n\n    /**\n     * Bootstrap application.\n     */\n    bootstrap = function () {\n        /**\n         * Init all components defined via data-mage-init attribute\n         * and subscribe init action on contentUpdated event\n         */\n        mage.apply();\n\n        /*\n         * Initialization of notification widget\n         */\n        notification({}, $('body'));\n    };\n\n    $(bootstrap);\n});\n","mage/backend/tabs.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\n/* global FORM_KEY */\ndefine([\n    'jquery',\n    'jquery/ui',\n    'jquery/ui-modules/widgets/tabs'\n], function ($) {\n    'use strict';\n\n    var rhash, isLocal;\n\n    // mage.tabs base functionality\n    $.widget('mage.tabs', $.ui.tabs, {\n        options: {\n            spinner: false,\n            groups: null,\n            tabPanelClass: '',\n            excludedPanel: ''\n        },\n\n        /**\n         * Tabs creation\n         * @protected\n         */\n        _create: function () {\n            var activeIndex = this._getTabIndex(this.options.active);\n\n            this.options.active = activeIndex >= 0 ? activeIndex : 0;\n            this._super();\n        },\n\n        /**\n         * @override\n         * @private\n         * @return {Array} Array of DOM-elements\n         */\n        _getList: function () {\n            if (this.options.groups) {\n                return this.element.find(this.options.groups);\n            }\n\n            return this._super();\n        },\n\n        /**\n         * Get active anchor\n         * @return {Element}\n         */\n        activeAnchor: function () {\n            return this.anchors.eq(this.option('active'));\n        },\n\n        /**\n         * Get tab index by tab id\n         * @protected\n         * @param {String} id - id of tab\n         * @return {Number}\n         */\n        _getTabIndex: function (id) {\n            var anchors = this.anchors ?\n                this.anchors :\n                this._getList().find('> li > a[href]');\n\n            return anchors.index($('#' + id));\n        },\n\n        /**\n         * Switch between tabs\n         * @protected\n         * @param {Object} event - event object\n         * @param {undefined|Object} eventData\n         */\n        _toggle: function (event, eventData) {\n            var anchor = $(eventData.newTab).find('a');\n\n            if ($(eventData.newTab).find('a').data().tabType === 'link') {\n                location.href = anchor.prop('href');\n            } else {\n                this._superApply(arguments);\n            }\n        }\n    });\n\n    rhash = /#.*$/;\n\n    /**\n     * @param {*} anchor\n     * @return {Boolean}\n     */\n    isLocal = function (anchor) {\n        return anchor.hash.length > 1 &&\n            anchor.href.replace(rhash, '') ===\n                location.href.replace(rhash, '')\n                    // support: Safari 5.1\n                    // Safari 5.1 doesn't encode spaces in window.location\n                    // but it does encode spaces from anchors (#8777)\n                    .replace(/\\s/g, '%20');\n    };\n\n    // Extension for mage.tabs - Move panels in destination element\n    $.widget('mage.tabs', $.mage.tabs, {\n        /**\n         * Move panels in destination element on creation\n         * @protected\n         * @override\n         */\n        _create: function () {\n            this._super();\n            this._movePanelsInDestination(this.panels);\n        },\n\n        /**\n         * Get panel for tab. If panel no exist in tabs container, then find panel in destination element\n         * @protected\n         * @override\n         * @param {Element} tab - tab \"li\" DOM-element\n         * @return {Element}\n         */\n        _getPanelForTab: function (tab) {\n            var panel = this._superApply(arguments),\n                id;\n\n            if (!panel.length) {\n                id = $(tab).attr('aria-controls');\n                panel = $(this.options.destination).find(this._sanitizeSelector('#' + id));\n            }\n\n            return panel;\n        },\n\n        /**\n         * @private\n         */\n        _processTabs: function () {\n            var that = this;\n\n            this.tablist = this._getList()\n                .addClass('ui-tabs-nav ui-helper-reset ui-helper-clearfix ui-widget-header ui-corner-all')\n                .attr('role', 'tablist');\n\n            this.tabs = this.tablist.find('> li:has(a[href])')\n                .addClass('ui-state-default ui-corner-top')\n                .attr({\n                    role: 'tab',\n                    tabIndex: -1\n                });\n\n            this.anchors = this.tabs.map(function () {\n                return $('a', this)[ 0 ];\n            })\n                .addClass('ui-tabs-anchor')\n                .attr({\n                    role: 'presentation',\n                    tabIndex: -1\n                });\n\n            this.panels = $();\n\n            this.anchors.each(function (i, anchor) {\n                var selector, panel, panelId,\n                    anchorId = $(anchor).uniqueId().attr('id'),\n                    tab = $(anchor).closest('li'),\n                    originalAriaControls = tab.attr('aria-controls');\n\n                // inline tab\n                if (isLocal(anchor)) {\n                    selector = anchor.hash;\n                    panel = that.document.find(that._sanitizeSelector(selector));\n                    // remote tab\n                } else {\n                    panelId = tab.attr('aria-controls') || $({}).uniqueId()[ 0 ].id;\n                    selector = '#' + panelId;\n                    panel = that.element.find(selector);\n\n                    if (!panel.length) {\n                        panel = that._createPanel(panelId);\n                        panel.insertAfter(that.panels[ i - 1 ] || that.tablist);\n                    }\n                    panel.attr('aria-live', 'polite');\n                }\n\n                if (panel.length) {\n                    that.panels = that.panels.add(panel);\n                }\n\n                if (originalAriaControls) {\n                    tab.data('ui-tabs-aria-controls', originalAriaControls);\n                }\n                tab.attr({\n                    'aria-controls': selector.substring(1),\n                    'aria-labelledby': anchorId\n                });\n                panel.attr('aria-labelledby', anchorId);\n\n                if (that.options.excludedPanel.indexOf(anchorId + '_content') < 0) {\n                    panel.addClass(that.options.tabPanelClass);\n                }\n            });\n\n            this.panels\n                .addClass('ui-tabs-panel ui-widget-content ui-corner-bottom')\n                .attr('role', 'tabpanel');\n        },\n\n        /**\n         * Move panels in destination element\n         * @protected\n         * @override\n         */\n        _movePanelsInDestination: function (panels) {\n            if (this.options.destination && !panels.parents(this.options.destination).length) {\n                this.element.trigger('beforePanelsMove', panels);\n\n                panels.find('script:not([type]), script[type=\"text/javascript\"]').remove();\n\n                panels.appendTo(this.options.destination)\n                    .each($.proxy(function (i, panel) {\n                        $(panel).trigger('move.tabs', this.anchors.eq(i));\n                    }, this));\n            }\n        },\n\n        /**\n         * Move panels in destination element on tabs switching\n         * @protected\n         * @override\n         * @param {Object} event - event object\n         * @param {Object} eventData\n         */\n        _toggle: function (event, eventData) {\n            this._movePanelsInDestination(eventData.newPanel);\n            this._superApply(arguments);\n        }\n    });\n\n    // Extension for mage.tabs - Ajax functionality for tabs\n    $.widget('mage.tabs', $.mage.tabs, {\n        options: {\n            /**\n             * Add form key to ajax call\n             * @param {Object} event - event object\n             * @param {Object} ui\n             */\n            beforeLoad: function (event, ui) {\n                ui.ajaxSettings.type = 'POST';\n                ui.ajaxSettings.hasContent = true;\n                ui.jqXHR.setRequestHeader('Content-Type', ui.ajaxSettings.contentType);\n                ui.ajaxSettings.data = jQuery.param(\n                    {\n                        isAjax: true,\n                        'form_key': typeof FORM_KEY !== 'undefined' ? FORM_KEY : null\n                    },\n                    ui.ajaxSettings.traditional\n                );\n            },\n\n            /**\n             * Replacing href attribute with loaded panel id\n             * @param {Object} event - event object\n             * @param {Object} ui\n             */\n            load: function (event, ui) {\n                var panel = $(ui.panel);\n\n                $(ui.tab).prop('href', '#' + panel.prop('id'));\n                panel.trigger('contentUpdated');\n            }\n        }\n    });\n\n    // Extension for mage.tabs - Attach event handlers to tabs\n    $.widget('mage.tabs', $.mage.tabs, {\n        options: {\n            tabIdArgument: 'tab',\n            tabsBlockPrefix: null\n        },\n\n        /**\n         * Attach event handlers to tabs, on creation\n         * @protected\n         * @override\n         */\n        _refresh: function () {\n            this._super();\n            $.each(this.tabs, $.proxy(function (i, tab) {\n                $(this._getPanelForTab(tab))\n                    .off('changed' + this.eventNamespace)\n                    .off('highlight.validate' + this.eventNamespace)\n                    .off('focusin' + this.eventNamespace)\n\n                    .on('changed' + this.eventNamespace, {\n                        index: i\n                    }, $.proxy(this._onContentChange, this))\n                    .on('highlight.validate' + this.eventNamespace, {\n                        index: i\n                    }, $.proxy(this._onInvalid, this))\n                    .on('focusin' + this.eventNamespace, {\n                        index: i\n                    }, $.proxy(this._onFocus, this));\n            }, this));\n\n            ($(this.options.destination).is('form') ?\n                $(this.options.destination) :\n                $(this.options.destination).closest('form'))\n                    .off('beforeSubmit' + this.eventNamespace)\n                    .on('beforeSubmit' + this.eventNamespace, $.proxy(this._onBeforeSubmit, this));\n        },\n\n        /**\n         * Mark tab as changed if some field inside tab panel is changed\n         * @protected\n         * @param {Object} e - event object\n         */\n        _onContentChange: function (e) {\n            var cssChanged = '_changed';\n\n            this.anchors.eq(e.data.index).addClass(cssChanged);\n            this._updateNavTitleMessages(e, cssChanged);\n        },\n\n        /**\n         * Clone messages (tooltips) from anchor to parent element\n         * @protected\n         * @param {Object} e - event object\n         * @param {String} messageType - changed or error\n         */\n        _updateNavTitleMessages: function (e, messageType) {\n            var curAnchor = this.anchors.eq(e.data.index),\n                curItem = curAnchor.parents('[data-role=\"container\"]').find('[data-role=\"title\"]'),\n                curItemMessages = curItem.find('[data-role=\"title-messages\"]'),\n                activeClass = '_active';\n\n            if (curItemMessages.is(':empty')) {\n                curAnchor\n                    .find('[data-role=\"item-messages\"]')\n                    .clone()\n                    .appendTo(curItemMessages);\n            }\n\n            curItemMessages.find('.' + messageType).addClass(activeClass);\n        },\n\n        /**\n         * Mark tab as error if some field inside tab panel is not passed validation\n         * @param {Object} e - event object\n         * @protected\n         */\n        _onInvalid: function (e) {\n            var cssError = '_error',\n                fakeEvent = e;\n\n            fakeEvent.currentTarget = $(this.anchors).eq(e.data.index);\n            this._eventHandler(fakeEvent);\n            this.anchors.eq(e.data.index).addClass(cssError).find('.' + cssError).show();\n            this._updateNavTitleMessages(e, cssError);\n        },\n\n        /**\n         * Show tab panel if focus event triggered of some field inside tab panel\n         * @param {Object} e - event object\n         * @protected\n         */\n        _onFocus: function (e) {\n            this.option('_active', e.data.index);\n        },\n\n        /**\n         * Add active tab id in data object when \"beforeSubmit\" event is triggered\n         * @param {Object} e - event object\n         * @param {Object} data - event data object\n         * @protected\n         */\n        _onBeforeSubmit: function (e, data) { //eslint-disable-line no-unused-vars\n            var activeAnchor = this.activeAnchor(),\n                activeTabId = activeAnchor.prop('id'),\n                options;\n\n            if (this.options.tabsBlockPrefix) {\n                if (activeAnchor.is('[id*=\"' + this.options.tabsBlockPrefix + '\"]')) {\n                    activeTabId = activeAnchor.prop('id').substr(this.options.tabsBlockPrefix.length);\n                }\n            }\n            $(this.anchors).removeClass('error');\n            options = {\n                action: {\n                    args: {}\n                }\n            };\n            options.action.args[this.options.tabIdArgument] = activeTabId;\n        }\n    });\n\n    // Extension for mage.tabs - Shadow tabs functionality\n    $.widget('mage.tabs', $.mage.tabs, {\n        /**\n         * Add shadow tabs functionality on creation\n         * @protected\n         * @override\n         */\n        _refresh: function () {\n            var anchors, shadowTabs, tabs;\n\n            this._super();\n            anchors = this.anchors;\n            shadowTabs = this.options.shadowTabs;\n            tabs = this.tabs;\n\n            if (shadowTabs) {\n                anchors.each($.proxy(function (i, anchor) {\n                    var anchorId = $(anchor).prop('id');\n\n                    if (shadowTabs[anchorId]) {\n                        $(anchor).parents('li').on('click', $.proxy(function () {\n                            $.each(shadowTabs[anchorId], $.proxy(function (key, id) {\n                                this.load($(tabs).index($('#' + id).parents('li')), {});\n                            }, this));\n                        }, this));\n                    }\n                }, this));\n            }\n        }\n    });\n\n    return $.mage.tabs;\n});\n","mage/backend/suggest.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\ndefine([\n    'jquery',\n    'mage/template',\n    'mage/mage',\n    'jquery/ui',\n    'mage/backend/menu',\n    'mage/translate'\n], function ($, mageTemplate) {\n    'use strict';\n\n    /**\n     * Implement base functionality\n     */\n    $.widget('mage.suggest', {\n        widgetEventPrefix: 'suggest',\n        options: {\n            template: '<% if (data.items.length) { %>' +\n                '<% if (!data.term && !data.allShown() && data.recentShown()) { %>' +\n                '<h5 class=\"title\"><%- data.recentTitle %></h5>' +\n                '<% } %>' +\n                '<ul data-mage-init=\\'{\"menu\":[]}\\'>' +\n                '<% _.each(data.items, function(value){ %>' +\n                '<% if (!data.itemSelected(value)) { %><li <%= data.optionData(value) %>>' +\n                '<a href=\"#\"><%- value.label %></a></li><% } %>' +\n                '<% }); %>' +\n                '<% if (!data.term && !data.allShown() && data.recentShown()) { %>' +\n                '<li data-mage-init=\\'{\"actionLink\":{\"event\":\"showAll\"}}\\' class=\"show-all\">' +\n                '<a href=\"#\"><%- data.showAllTitle %></a></li>' +\n                '<% } %>' +\n                '</ul><% } else { %><span class=\"mage-suggest-no-records\"><%- data.noRecordsText %></span><% } %>',\n            minLength: 1,\n\n            /**\n             * @type {(String|Array)}\n             */\n            source: null,\n            delay: 500,\n            loadingClass: 'mage-suggest-state-loading',\n            events: {},\n            appendMethod: 'after',\n            controls: {\n                selector: ':ui-menu, :mage-menu',\n                eventsMap: {\n                    focus: ['menufocus'],\n                    blur: ['menublur'],\n                    select: ['menuselect']\n                }\n            },\n            termAjaxArgument: 'label_part',\n            filterProperty: 'label',\n            className: null,\n            inputWrapper: '<div class=\"mage-suggest\"><div class=\"mage-suggest-inner\"></div></div>',\n            dropdownWrapper: '<div class=\"mage-suggest-dropdown\"></div>',\n            preventClickPropagation: true,\n            currentlySelected: null,\n            submitInputOnEnter: true\n        },\n\n        /**\n         * Component's constructor\n         * @private\n         */\n        _create: function () {\n            this._term = null;\n            this._nonSelectedItem = {\n                id: '',\n                label: ''\n            };\n            this.templates = {};\n            this._renderedContext = null;\n            this._selectedItem = this._nonSelectedItem;\n            this._control = this.options.controls || {};\n            this._setTemplate();\n            this._prepareValueField();\n            this._render();\n            this._bind();\n        },\n\n        /**\n         * Render base elements for suggest component\n         * @private\n         */\n        _render: function () {\n            var wrapper;\n\n            this.dropdown = $(this.options.dropdownWrapper).hide();\n            wrapper = this.options.className ?\n                $(this.options.inputWrapper).addClass(this.options.className) :\n                $(this.options.inputWrapper);\n            this.element\n                .wrap(wrapper)[this.options.appendMethod](this.dropdown)\n                .attr('autocomplete', 'off');\n        },\n\n        /**\n         * Define a field for storing item id (find in DOM or create a new one)\n         * @private\n         */\n        _prepareValueField: function () {\n            if (this.options.valueField) {\n                this.valueField = $(this.options.valueField);\n            } else {\n                this.valueField = this._createValueField()\n                    .insertBefore(this.element)\n                    .attr('name', this.element.attr('name'));\n                this.element.removeAttr('name');\n            }\n        },\n\n        /**\n         * Create value field which keeps a id for selected option\n         * can be overridden in descendants\n         * @return {jQuery}\n         * @private\n         */\n        _createValueField: function () {\n            return $('<input/>', {\n                type: 'hidden'\n            });\n        },\n\n        /**\n         * Component's destructor\n         * @private\n         */\n        _destroy: function () {\n            this.element\n                .unwrap()\n                .removeAttr('autocomplete');\n\n            if (!this.options.valueField) {\n                this.element.attr('name', this.valueField.attr('name'));\n                this.valueField.remove();\n            }\n\n            this.dropdown.remove();\n            this._off(this.element, 'keydown keyup blur');\n        },\n\n        /**\n         * Return actual value of an \"input\"-element\n         * @return {String}\n         * @private\n         */\n        _value: function () {\n            return this.element[this.element.is(':input') ? 'val' : 'text']().trim();\n        },\n\n        /**\n         * Pass original event to a control component for handling it as it's own event\n         * @param {Object} event - event object\n         * @private\n         */\n        _proxyEvents: function (event) {\n            var fakeEvent = $.extend({}, $.Event(event.type), {\n                    ctrlKey: event.ctrlKey,\n                    keyCode: event.keyCode,\n                    which: event.keyCode\n                }),\n                target = this._control.selector ? this.dropdown.find(this._control.selector) : this.dropdown;\n\n            target.trigger(fakeEvent);\n        },\n\n        /**\n         * Bind handlers on specific events\n         * @private\n         */\n        _bind: function () {\n            this._on($.extend({\n                /**\n                 * @param {jQuery.Event} event\n                 */\n                keydown: function (event) {\n                    var keyCode = $.ui.keyCode,\n                        suggestList,\n                        hasSuggestedItems,\n                        hasSelectedItems,\n                        selectedItem;\n\n                    switch (event.keyCode) {\n                        case keyCode.PAGE_UP:\n                        case keyCode.UP:\n                            if (!event.shiftKey) {\n                                event.preventDefault();\n                                this._proxyEvents(event);\n                            }\n\n                            suggestList = event.currentTarget.parentNode.getElementsByTagName('ul')[0];\n                            hasSuggestedItems = event.currentTarget\n                                    .parentNode.getElementsByTagName('ul')[0].children.length >= 0;\n\n                            if (hasSuggestedItems) {\n                                selectedItem =  $(suggestList.getElementsByClassName('_active')[0])\n                                    .removeClass('_active').prev().addClass('_active');\n                                event.currentTarget.value = selectedItem.find('a').text();\n                            }\n\n                            break;\n\n                        case keyCode.PAGE_DOWN:\n                        case keyCode.DOWN:\n                            if (!event.shiftKey) {\n                                event.preventDefault();\n                                this._proxyEvents(event);\n                            }\n\n                            suggestList = event.currentTarget.parentNode.getElementsByTagName('ul')[0];\n                            hasSuggestedItems = event.currentTarget\n                                .parentNode.getElementsByTagName('ul')[0].children.length >= 0;\n\n                            if (hasSuggestedItems) {\n                                hasSelectedItems = suggestList.getElementsByClassName('_active').length === 0;\n\n                                if (hasSelectedItems) { //eslint-disable-line max-depth\n                                    selectedItem = $(suggestList.children[0]).addClass('_active');\n                                    event.currentTarget.value = selectedItem.find('a').text();\n                                } else {\n                                    selectedItem = $(suggestList.getElementsByClassName('_active')[0])\n                                        .removeClass('_active').next().addClass('_active');\n                                    event.currentTarget.value = selectedItem.find('a').text();\n                                }\n                            }\n                            break;\n\n                        case keyCode.TAB:\n                            if (this.isDropdownShown()) {\n                                this._onSelectItem(event, null);\n                                event.preventDefault();\n                            }\n                            break;\n\n                        case keyCode.ENTER:\n                        case keyCode.NUMPAD_ENTER:\n                            this._toggleEnter(event);\n\n                            if (this.isDropdownShown() && this._focused) {\n                                this._proxyEvents(event);\n                                event.preventDefault();\n                            }\n                            break;\n\n                        case keyCode.ESCAPE:\n                            if (this.isDropdownShown()) {\n                                event.stopPropagation();\n                            }\n                            this.close(event);\n                            this._blurItem();\n                            break;\n                    }\n                },\n\n                /**\n                 * @param {jQuery.Event} event\n                 */\n                keyup: function (event) {\n                    var keyCode = $.ui.keyCode;\n\n                    switch (event.keyCode) {\n                        case keyCode.HOME:\n                        case keyCode.END:\n                        case keyCode.PAGE_UP:\n                        case keyCode.PAGE_DOWN:\n                        case keyCode.ESCAPE:\n                        case keyCode.UP:\n                        case keyCode.DOWN:\n                        case keyCode.LEFT:\n                        case keyCode.RIGHT:\n                        case keyCode.TAB:\n                            break;\n\n                        case keyCode.ENTER:\n                        case keyCode.NUMPAD_ENTER:\n                            if (this.isDropdownShown()) {\n                                event.preventDefault();\n                            }\n                            break;\n                        default:\n                            this.search(event);\n                    }\n                },\n\n                /**\n                 * @param {jQuery.Event} event\n                 */\n                blur: function (event) {\n                    if (!this.preventBlur) {\n                        this._abortSearch();\n                        this.close(event);\n                        this._change(event);\n                    } else {\n                        this.element.trigger('focus');\n                    }\n                },\n                cut: this.search,\n                paste: this.search,\n                input: this.search,\n                selectItem: this._onSelectItem,\n                click: this.search\n            }, this.options.events));\n\n            this._bindSubmit();\n            this._bindDropdown();\n        },\n\n        /**\n         * @param {Object} event\n         * @private\n         */\n        _toggleEnter: function (event) {\n            var suggestList,\n                activeItems,\n                selectedItem;\n\n            if (!this.options.submitInputOnEnter) {\n                event.preventDefault();\n            }\n\n            suggestList = $(event.currentTarget.parentNode).find('ul').first();\n            activeItems = suggestList.find('._active');\n\n            if (activeItems.length >= 0) {\n                selectedItem = activeItems.first();\n\n                if (selectedItem.find('a') && selectedItem.find('a').attr('href') !== undefined) {\n                    window.location = selectedItem.find('a').attr('href');\n                    event.preventDefault();\n                }\n            }\n        },\n\n        /**\n         * Bind handlers for submit on enter\n         * @private\n         */\n        _bindSubmit: function () {\n            this.element.parents('form').on('submit', function (event) {\n                if (!this.submitInputOnEnter) {\n                    event.preventDefault();\n                }\n            });\n        },\n\n        /**\n         * @param {Object} e - event object\n         * @private\n         */\n        _change: function (e) {\n            if (this._term !== this._value()) {\n                this._trigger('change', e);\n            }\n        },\n\n        /**\n         * Bind handlers for dropdown element on specific events\n         * @private\n         */\n        _bindDropdown: function () {\n            var events = {\n                /**\n                 * @param {jQuery.Event} e\n                 */\n                click: function (e) {\n                    // prevent default browser's behavior of changing location by anchor href\n                    e.preventDefault();\n                },\n\n                /**\n                 * @param {jQuery.Event} e\n                 */\n                mousedown: function (e) {\n                    e.preventDefault();\n                }\n            };\n\n            $.each(this._control.eventsMap, $.proxy(function (suggestEvent, controlEvents) {\n                $.each(controlEvents, $.proxy(function (i, handlerName) {\n                    switch (suggestEvent) {\n                        case 'select':\n                            events[handlerName] = this._onSelectItem;\n                            break;\n\n                        case 'focus':\n                            events[handlerName] = this._focusItem;\n                            break;\n\n                        case 'blur':\n                            events[handlerName] = this._blurItem;\n                            break;\n                    }\n                }, this));\n            }, this));\n\n            if (this.options.preventClickPropagation) {\n                this._on(this.dropdown, events);\n            }\n            // Fix for IE 8\n            this._on(this.dropdown, {\n                /**\n                 * Mousedown.\n                 */\n                mousedown: function () {\n                    this.preventBlur = true;\n                },\n\n                /**\n                 * Mouseup.\n                 */\n                mouseup: function () {\n                    this.preventBlur = false;\n                }\n            });\n        },\n\n        /**\n         * @override\n         */\n        _trigger: function (type, event) {\n            var result = this._superApply(arguments);\n\n            if (result === false && event) {\n                event.stopImmediatePropagation();\n                event.preventDefault();\n            }\n\n            return result;\n        },\n\n        /**\n         * Handle focus event of options item\n         * @param {Object} e - event object\n         * @param {Object} ui - object that can contain information about focused item\n         * @private\n         */\n        _focusItem: function (e, ui) {\n            if (ui && ui.item) {\n                this._focused = $(ui.item).prop('tagName') ?\n                    this._readItemData(ui.item) :\n                    ui.item;\n\n                this.element.val(this._focused.label);\n                this._trigger('focus', e, {\n                    item: this._focused\n                });\n            }\n        },\n\n        /**\n         * Handle blur event of options item\n         * @private\n         */\n        _blurItem: function () {\n            this._focused = null;\n            this.element.val(this._term);\n        },\n\n        /**\n         * @param {Object} e - event object\n         * @param {Object} item\n         * @private\n         */\n        _onSelectItem: function (e, item) {\n            if (item && typeof item === 'object' && $(e.target).is(this.element)) {\n                this._focusItem(e, {\n                    item: item\n                });\n            }\n\n            if (this._trigger('beforeselect', e || null, {\n                    item: this._focused\n                }) === false) {\n                return;\n            }\n            this._selectItem(e);\n            this._blurItem();\n            this._trigger('select', e || null, {\n                item: this._selectedItem\n            });\n        },\n\n        /**\n         * Save selected item and hide dropdown\n         * @private\n         * @param {Object} e - event object\n         */\n        _selectItem: function (e) {\n            if (this._focused) {\n                this._selectedItem = this._focused;\n\n                if (this._selectedItem !== this._nonSelectedItem) {\n                    this._term = this._selectedItem.label;\n                    this.valueField.val(this._selectedItem.id);\n                    this.close(e);\n                }\n            }\n        },\n\n        /**\n         * Read option data from item element\n         * @param {Element} element\n         * @return {Object}\n         * @private\n         */\n        _readItemData: function (element) {\n            return element.data('suggestOption') || this._nonSelectedItem;\n        },\n\n        /**\n         * Check if dropdown is shown\n         * @return {Boolean}\n         */\n        isDropdownShown: function () {\n            return this.dropdown.is(':visible');\n        },\n\n        /**\n         * Open dropdown\n         * @private\n         * @param {Object} e - event object\n         */\n        open: function (e) {\n            if (!this.isDropdownShown()) {\n                this.element.addClass('_suggest-dropdown-open');\n                this.dropdown.show();\n                this._trigger('open', e);\n            }\n        },\n\n        /**\n         * Close and clear dropdown content\n         * @private\n         * @param {Object} e - event object\n         */\n        close: function (e) {\n            this._renderedContext = null;\n\n            if (this.dropdown.length) {\n                this.element.removeClass('_suggest-dropdown-open');\n                this.dropdown.hide().empty();\n            }\n\n            this._trigger('close', e);\n        },\n\n        /**\n         * Acquire content template\n         * @private\n         */\n        _setTemplate: function () {\n            this.templateName = 'suggest' + Math.random().toString(36).substr(2);\n\n            this.templates[this.templateName] = mageTemplate(this.options.template);\n        },\n\n        /**\n         * Execute search process\n         * @public\n         * @param {Object} e - event object\n         */\n        search: function (e) {\n            var term = this._value();\n\n            if ((this._term !== term || term.length === 0) && !this.preventBlur) {\n                this._term = term;\n\n                if (typeof term === 'string' && term.length >= this.options.minLength) {\n                    if (this._trigger('search', e) === false) { //eslint-disable-line max-depth\n                        return;\n                    }\n                    this._search(e, term, {});\n                } else {\n                    this._selectedItem = this._nonSelectedItem;\n                    this._resetSuggestValue();\n                }\n            }\n        },\n\n        /**\n         * Clear suggest hidden input\n         * @private\n         */\n        _resetSuggestValue: function () {\n            this.valueField.val(this._nonSelectedItem.id);\n        },\n\n        /**\n         * Actual search method, can be overridden in descendants\n         * @param {Object} e - event object\n         * @param {String} term - search phrase\n         * @param {Object} context - search context\n         * @private\n         */\n        _search: function (e, term, context) {\n            var response = $.proxy(function (items) {\n                return this._processResponse(e, items, context || {});\n            }, this);\n\n            this.element.addClass(this.options.loadingClass);\n\n            if (this.options.delay) {\n                if (typeof this.options.data !== 'undefined') {\n                    response(this.filter(this.options.data, term));\n                }\n                clearTimeout(this._searchTimeout);\n                this._searchTimeout = this._delay(function () {\n                    this._source(term, response);\n                }, this.options.delay);\n            } else {\n                this._source(term, response);\n            }\n        },\n\n        /**\n         * Extend basic context with additional data (search results, search term)\n         * @param {Object} context\n         * @return {Object}\n         * @private\n         */\n        _prepareDropdownContext: function (context) {\n            return $.extend(context, {\n                items: this._items,\n                term: this._term,\n\n                /**\n                 * @param {Object} item\n                 * @return {String}\n                 */\n                optionData: function (item) {\n                    return 'data-suggest-option=\"' +\n                        $('<div>').text(JSON.stringify(item)).html().replace(/\"/g, '&quot;') + '\"';\n                },\n                itemSelected: $.proxy(this._isItemSelected, this),\n                noRecordsText: $.mage.__('No records found.')\n            });\n        },\n\n        /**\n         * @param {Object} item\n         * @return {Boolean}\n         * @private\n         */\n        _isItemSelected: function (item) {\n            return item.id == (this._selectedItem && this._selectedItem.id ? //eslint-disable-line eqeqeq\n                this._selectedItem.id :\n                this.options.currentlySelected);\n        },\n\n        /**\n         * Render content of suggest's dropdown\n         * @param {Object} e - event object\n         * @param {Array} items - list of label+id objects\n         * @param {Object} context - template's context\n         * @private\n         */\n        _renderDropdown: function (e, items, context) {\n            var tmpl = this.templates[this.templateName];\n\n            this._items = items;\n\n            tmpl = tmpl({\n                data: this._prepareDropdownContext(context)\n            });\n\n            $(tmpl).appendTo(this.dropdown.empty());\n\n            this.dropdown.trigger('contentUpdated')\n                .find(this._control.selector).on('focus', function (event) {\n                    event.preventDefault();\n                });\n\n            this._renderedContext = context;\n            this.element.removeClass(this.options.loadingClass);\n            this.open(e);\n        },\n\n        /**\n         * @param {Object} e\n         * @param {Object} items\n         * @param {Object} context\n         * @private\n         */\n        _processResponse: function (e, items, context) {\n            var renderer = $.proxy(function (i) {\n                return this._renderDropdown(e, i, context || {});\n            }, this);\n\n            if (this._trigger('response', e, [items, renderer]) === false) {\n                return;\n            }\n            this._renderDropdown(e, items, context);\n        },\n\n        /**\n         * Implement search process via spesific source\n         * @param {String} term - search phrase\n         * @param {Function} response - search results handler, process search result\n         * @private\n         */\n        _source: function (term, response) {\n            var o = this.options,\n                ajaxData;\n\n            if (Array.isArray(o.source)) {\n                response(this.filter(o.source, term));\n            } else if (typeof o.source === 'string') {\n                ajaxData = {};\n                ajaxData[this.options.termAjaxArgument] = term;\n\n                this._xhr = $.ajax($.extend(true, {\n                    url: o.source,\n                    type: 'POST',\n                    dataType: 'json',\n                    data: ajaxData,\n                    success: $.proxy(function (items) {\n                        this.options.data = items;\n                        response.apply(response, arguments);\n                    }, this)\n                }, o.ajaxOptions || {}));\n            } else if (typeof o.source === 'function') {\n                o.source.apply(o.source, arguments);\n            }\n        },\n\n        /**\n         * Abort search process\n         * @private\n         */\n        _abortSearch: function () {\n            this.element.removeClass(this.options.loadingClass);\n            clearTimeout(this._searchTimeout);\n        },\n\n        /**\n         * Perform filtering in advance loaded items and returns search result\n         * @param {Array} items - all available items\n         * @param {String} term - search phrase\n         * @return {Object}\n         */\n        filter: function (items, term) {\n            var matcher = new RegExp(term.replace(/[\\-\\/\\\\\\^$*+?.()|\\[\\]{}]/g, '\\\\$&'), 'i'),\n                itemsArray = Array.isArray(items) ? items : $.map(items, function (element) {\n                    return element;\n                }),\n                property = this.options.filterProperty;\n\n            return $.grep(\n                itemsArray,\n                function (value) {\n                    return matcher.test(value[property] || value.id || value);\n                }\n            );\n        }\n    });\n\n    /**\n     * Implement show all functionality and storing and display recent searches\n     */\n    $.widget('mage.suggest', $.mage.suggest, {\n        options: {\n            showRecent: false,\n            showAll: false,\n            storageKey: 'suggest',\n            storageLimit: 10\n        },\n\n        /**\n         * @override\n         */\n        _create: function () {\n            var recentItems;\n\n            if (this.options.showRecent && window.localStorage) {\n                recentItems = JSON.parse(localStorage.getItem(this.options.storageKey));\n\n                /**\n                 * @type {Array} - list of recently searched items\n                 * @private\n                 */\n                this._recentItems = Array.isArray(recentItems) ? recentItems : [];\n            }\n            this._super();\n        },\n\n        /**\n         * @override\n         */\n        _bind: function () {\n            this._super();\n            this._on(this.dropdown, {\n                /**\n                 * @param {jQuery.Event} e\n                 */\n                showAll: function (e) {\n                    e.stopImmediatePropagation();\n                    e.preventDefault();\n                    this.element.trigger('showAll');\n                }\n            });\n\n            if (this.options.showRecent || this.options.showAll) {\n                this._on({\n                    /**\n                     * @param {jQuery.Event} e\n                     */\n                    focus: function (e) {\n                        if (!this.isDropdownShown()) {\n                            this.search(e);\n                        }\n                    },\n                    showAll: this._showAll\n                });\n            }\n        },\n\n        /**\n         * @private\n         * @param {Object} e - event object\n         */\n        _showAll: function (e) {\n            this._abortSearch();\n            this._search(e, '', {\n                _allShown: true\n            });\n        },\n\n        /**\n         * @override\n         */\n        search: function (e) {\n            if (!this._value()) {\n\n                if (this.options.showRecent) {\n\n                    if (this._recentItems.length) { //eslint-disable-line max-depth\n                        this._processResponse(e, this._recentItems, {});\n                    } else {\n                        this._showAll(e);\n                    }\n                } else if (this.options.showAll) {\n                    this._showAll(e);\n                }\n            }\n            this._superApply(arguments);\n        },\n\n        /**\n         * @override\n         */\n        _selectItem: function () {\n            this._superApply(arguments);\n\n            if (this._selectedItem && this._selectedItem.id && this.options.showRecent) {\n                this._addRecent(this._selectedItem);\n            }\n        },\n\n        /**\n         * @override\n         */\n        _prepareDropdownContext: function () {\n            var context = this._superApply(arguments);\n\n            return $.extend(context, {\n                recentShown: $.proxy(function () {\n                    return this.options.showRecent;\n                }, this),\n                recentTitle: $.mage.__('Recent items'),\n                showAllTitle: $.mage.__('Show all...'),\n\n                /**\n                 * @return {Boolean}\n                 */\n                allShown: function () {\n                    return !!context._allShown;\n                }\n            });\n        },\n\n        /**\n         * Add selected item of search result into storage of recents\n         * @param {Object} item - label+id object\n         * @private\n         */\n        _addRecent: function (item) {\n            this._recentItems = $.grep(this._recentItems, function (obj) {\n                return obj.id !== item.id;\n            });\n            this._recentItems.unshift(item);\n            this._recentItems = this._recentItems.slice(0, this.options.storageLimit);\n            localStorage.setItem(this.options.storageKey, JSON.stringify(this._recentItems));\n        }\n    });\n\n    /**\n     * Implement multi suggest functionality\n     */\n    $.widget('mage.suggest', $.mage.suggest, {\n        options: {\n            multiSuggestWrapper: '<ul class=\"mage-suggest-choices\">' +\n                '<li class=\"mage-suggest-search-field\" data-role=\"parent-choice-element\"><' +\n                'label class=\"mage-suggest-search-label\"></label></li></ul>',\n            choiceTemplate: '<li class=\"mage-suggest-choice button\"><div><%- text %></div>' +\n            '<span class=\"mage-suggest-choice-close\" tabindex=\"-1\" ' +\n            'data-mage-init=\\'{\"actionLink\":{\"event\":\"removeOption\"}}\\'></span></li>',\n            selectedClass: 'mage-suggest-selected'\n        },\n\n        /**\n         * @override\n         */\n        _create: function () {\n            this.choiceTmpl = mageTemplate(this.options.choiceTemplate);\n\n            this._super();\n\n            if (this.options.multiselect) {\n                this.valueField.hide();\n            }\n        },\n\n        /**\n         * @override\n         */\n        _render: function () {\n            this._super();\n\n            if (this.options.multiselect) {\n                this._renderMultiselect();\n            }\n        },\n\n        /**\n         * Render selected options\n         * @private\n         */\n        _renderMultiselect: function () {\n            var that = this;\n\n            this.element.wrap(this.options.multiSuggestWrapper);\n            this.elementWrapper = this.element.closest('[data-role=\"parent-choice-element\"]');\n            $(function () {\n                that._getOptions()\n                    .each(function (i, option) {\n                        option = $(option);\n                        that._createOption({\n                            id: option.val(),\n                            label: option.text()\n                        });\n                    });\n            });\n        },\n\n        /**\n         * @return {Array} array of DOM-elements\n         * @private\n         */\n        _getOptions: function () {\n            return this.valueField.find('option');\n        },\n\n        /**\n         * @override\n         */\n        _bind: function () {\n            this._super();\n\n            if (this.options.multiselect) {\n                this._on({\n                    /**\n                     * @param {jQuery.Event} event\n                     */\n                    keydown: function (event) {\n                        if (event.keyCode === $.ui.keyCode.BACKSPACE) {\n                            if (!this._value()) {\n                                this._removeLastAdded(event);\n                            }\n                        }\n                    },\n                    removeOption: this.removeOption\n                });\n            }\n        },\n\n        /**\n         * @param {Array} items\n         * @return {Array}\n         * @private\n         */\n        _filterSelected: function (items) {\n            var options = this._getOptions();\n\n            return $.grep(items, function (value) {\n                var itemSelected = false;\n\n                $.each(options, function () {\n                    if (value.id == $(this).val()) { //eslint-disable-line eqeqeq\n                        itemSelected = true;\n                    }\n                });\n\n                return !itemSelected;\n            });\n        },\n\n        /**\n         * @override\n         */\n        _processResponse: function (e, items, context) {\n            if (this.options.multiselect) {\n                items = this._filterSelected(items, context);\n            }\n            this._superApply([e, items, context]);\n        },\n\n        /**\n         * @override\n         */\n        _prepareValueField: function () {\n            this._super();\n\n            if (this.options.multiselect && !this.options.valueField && this.options.selectedItems) {\n                $.each(this.options.selectedItems, $.proxy(function (i, item) {\n                    this._addOption(item);\n                }, this));\n            }\n        },\n\n        /**\n         * If \"multiselect\" option is set, then do not need to clear value for hidden select, to avoid losing of\n         *      previously selected items\n         * @override\n         */\n        _resetSuggestValue: function () {\n            if (!this.options.multiselect) {\n                this._super();\n            }\n        },\n\n        /**\n         * @override\n         */\n        _createValueField: function () {\n            if (this.options.multiselect) {\n                return $('<select></select>', {\n                    type: 'hidden',\n                    multiple: 'multiple'\n                });\n            }\n\n            return this._super();\n        },\n\n        /**\n         * @override\n         */\n        _selectItem: function (e) {\n            if (this.options.multiselect) {\n                if (this._focused) {\n                    this._selectedItem = this._focused;\n\n                    /* eslint-disable max-depth */\n                    if (this._selectedItem !== this._nonSelectedItem) {\n                        this._term = '';\n                        this.element.val(this._term);\n\n                        if (this._isItemSelected(this._selectedItem)) {\n                            $(e.target).removeClass(this.options.selectedClass);\n                            this.removeOption(e, this._selectedItem);\n                            this._selectedItem = this._nonSelectedItem;\n                        } else {\n                            $(e.target).addClass(this.options.selectedClass);\n                            this._addOption(e, this._selectedItem);\n                        }\n                    }\n\n                    /* eslint-enable max-depth */\n                }\n                this.close(e);\n            } else {\n                this._superApply(arguments);\n            }\n        },\n\n        /**\n         * @override\n         */\n        _isItemSelected: function (item) {\n            if (this.options.multiselect) {\n                return this.valueField.find('option[value=' + item.id + ']').length > 0;\n            }\n\n            return this._superApply(arguments);\n        },\n\n        /**\n         *\n         * @param {Object} item\n         * @return {Element}\n         * @private\n         */\n        _createOption: function (item) {\n            var option = this._getOption(item);\n\n            if (!option.length) {\n                option = $('<option>', {\n                    value: item.id,\n                    selected: true\n                }).text(item.label);\n            }\n\n            return option.data('renderedOption', this._renderOption(item));\n        },\n\n        /**\n         * Add selected item in to select options\n         * @param {Object} e - event object\n         * @param {*} item\n         * @private\n         */\n        _addOption: function (e, item) {\n            this.valueField.append(this._createOption(item).data('selectTarget', $(e.target)));\n        },\n\n        /**\n         * @param {Object|Element} item\n         * @return {Element}\n         * @private\n         */\n        _getOption: function (item) {\n            return $(item).prop('tagName') ?\n                $(item) :\n                this.valueField.find('option[value=' + item.id + ']');\n        },\n\n        /**\n         * Remove last added option\n         * @private\n         * @param {Object} e - event object\n         */\n        _removeLastAdded: function (e) {\n            var lastAdded = this._getOptions().last();\n\n            if (lastAdded.length) {\n                this.removeOption(e, lastAdded);\n            }\n        },\n\n        /**\n         * Remove item from select options\n         * @param {Object} e - event object\n         * @param {Object} item\n         * @private\n         */\n        removeOption: function (e, item) {\n            var option = this._getOption(item),\n                selectTarget = option.data('selectTarget');\n\n            if (selectTarget && selectTarget.length) {\n                selectTarget.removeClass(this.options.selectedClass);\n            }\n\n            option.data('renderedOption').remove();\n            option.remove();\n        },\n\n        /**\n         * Render visual element of selected item\n         * @param {Object} item - selected item\n         * @private\n         */\n        _renderOption: function (item) {\n            var tmpl = this.choiceTmpl({\n                text: item.label\n            });\n\n            return $(tmpl)\n                .insertBefore(this.elementWrapper)\n                .trigger('contentUpdated')\n                .on('removeOption', $.proxy(function (e) {\n                    this.removeOption(e, item);\n                }, this));\n        }\n    });\n\n    return $.mage.suggest;\n});\n","mage/backend/tree-suggest.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\ndefine([\n    'jquery',\n    'jquery/ui',\n    'jquery/jstree/jquery.jstree',\n    'mage/backend/suggest'\n], function ($) {\n    'use strict';\n\n    /* jscs:disable requireCamelCaseOrUpperCaseIdentifiers */\n    var hover_node, dehover_node, select_node, init;\n\n    $.extend(true, $, {\n        // @TODO: Move method 'treeToList' in file with utility functions\n        mage: {\n            /**\n             * @param {Array} list\n             * @param {Object} nodes\n             * @param {*} level\n             * @param {*} path\n             * @return {*}\n             */\n            treeToList: function (list, nodes, level, path) {\n                $.each(nodes, function () {\n                    if (typeof this === 'object') {\n                        list.push({\n                            label: this.label,\n                            id: this.id,\n                            level: level,\n                            item: this,\n                            path: path + this.label\n                        });\n\n                        if (this.children) {\n                            $.mage.treeToList(list, this.children, level + 1, path + this.label + ' / ');\n                        }\n                    }\n                });\n\n                return list;\n            }\n        }\n    });\n\n    hover_node = $.jstree.core.prototype.hover_node;\n    dehover_node = $.jstree.core.prototype.dehover_node;\n    select_node = $.jstree.core.prototype.select_node;\n    init = $.jstree.core.prototype.init;\n\n    $.extend(true, $.jstree.core.prototype, {\n        /**\n         * @override\n         */\n        init: function () {\n            this.get_container()\n                .show()\n                .on('keydown', $.proxy(function (e) {\n                    var o;\n\n                    if (e.keyCode === $.ui.keyCode.ENTER) {\n                        o = this.data.ui.hovered || this.data.ui.last_selected || -1;\n                        this.select_node(o, true);\n                    }\n                }, this));\n            init.call(this);\n        },\n\n        /**\n         * @override\n         */\n        hover_node: function (obj) {\n            hover_node.apply(this, arguments);\n            obj = this._get_node(obj);\n\n            if (!obj.length) {\n                return false;\n            }\n            this.get_container().trigger('hover_node', [{\n                item: obj.find('a:first')\n            }]);\n        },\n\n        /**\n         * @override\n         */\n        dehover_node: function () {\n            dehover_node.call(this);\n            this.get_container().trigger('dehover_node');\n        },\n\n        /**\n         * @override\n         */\n        select_node: function (o) {\n            var node;\n\n            select_node.apply(this, arguments);\n            node = this._get_node(o);\n\n            (node ? $(node) : this.data.ui.last_selected)\n                .trigger('select_tree_node');\n        }\n    });\n\n    $.widget('mage.treeSuggest', $.mage.suggest, {\n        widgetEventPrefix: 'suggest',\n        options: {\n            template:\n                '<% if (data.items.length) { %>' +\n                    '<% if (data.allShown()) { %>' +\n                        '<% if (typeof data.nested === \"undefined\") { %>' +\n                            '<div style=\"display:none;\" data-mage-init=\"{&quot;jstree&quot;:{&quot;plugins&quot;:[&quot;themes&quot;,&quot;html_data&quot;,&quot;ui&quot;,&quot;hotkeys&quot;],&quot;themes&quot;:{&quot;theme&quot;:&quot;default&quot;,&quot;dots&quot;:false,&quot;icons&quot;:false}}}\">' + //eslint-disable-line max-len\n                        '<% } %>' +\n                        '<ul>' +\n                            '<% _.each(data.items, function(value) { %>' +\n                                '<li class=\"<% if (data.itemSelected(value)) { %>mage-suggest-selected<% } %>' +\n                '                   <% if (value.is_active == 0) { %> mage-suggest-not-active<% } %>\">' +\n                                    '<a href=\"#\" <%= data.optionData(value) %>><%- value.label %></a>' +\n                                    '<% if (value.children && value.children.length) { %>' +\n                                        '<%= data.renderTreeLevel(value.children) %>' +\n                                    '<% } %>' +\n                                '</li>' +\n                            '<% }); %>' +\n                        '</ul>' +\n                        '<% if (typeof data.nested === \"undefined\") { %>' +\n                            '</div>' +\n                        '<% } %>' +\n                    '<% } else { %>' +\n                        '<ul data-mage-init=\"{&quot;menu&quot;:[]}\">' +\n                            '<% _.each(data.items, function(value) { %>' +\n                                '<% if (!data.itemSelected(value)) {%>' +\n                                    '<li <%= data.optionData(value) %>>' +\n                                        '<a href=\"#\">' +\n                                            '<span class=\"category-label\"><%- value.label %></span>' +\n                                            '<span class=\"category-path\"><%- value.path %></span>' +\n                                        '</a>' +\n                                    '</li>' +\n                                '<% } %>' +\n                            '<% }); %>' +\n                        '</ul>' +\n                    '<% } %>' +\n                '<% } else { %>' +\n                    '<span class=\"mage-suggest-no-records\"><%- data.noRecordsText %></span>' +\n                '<% } %>',\n            controls: {\n                selector: ':ui-menu, :mage-menu, .jstree',\n                eventsMap: {\n                    focus: ['menufocus', 'hover_node'],\n                    blur: ['menublur', 'dehover_node'],\n                    select: ['menuselect', 'select_tree_node']\n                }\n            }\n        },\n\n        /**\n         * @override\n         */\n        _bind: function () {\n            this._super();\n            this._on({\n                /**\n                 * @param {jQuery.Event} event\n                 */\n                keydown: function (event) {\n                    var keyCode = $.ui.keyCode;\n\n                    switch (event.keyCode) {\n                        case keyCode.LEFT:\n                        case keyCode.RIGHT:\n\n                            if (this.isDropdownShown()) {\n                                event.preventDefault();\n                                this._proxyEvents(event);\n                            }\n                            break;\n                    }\n                }\n            });\n        },\n\n        /**\n         * @override\n         */\n        close: function (e) {\n            var eType = e ? e.type : null;\n\n            if (eType === 'select_tree_node') {\n                this.element.focus();\n            } else {\n                this._superApply(arguments);\n            }\n        },\n\n        /**\n         * @override\n         */\n        _filterSelected: function (items, context) {\n            if (context._allShown) {\n                return items;\n            }\n\n            return this._superApply(arguments);\n        },\n\n        /**\n         * @override\n         */\n        _prepareDropdownContext: function () {\n            var context = this._superApply(arguments),\n                optionData = context.optionData,\n                templates = this.templates,\n                tmplName = this.templateName;\n\n            /**\n             * @param {Object} item\n             * @return {*|String}\n             */\n            context.optionData = function (item) {\n                item = $.extend({}, item);\n                delete item.children;\n\n                return optionData(item);\n            };\n\n            return $.extend(context, {\n                /**\n                 * @param {*} children\n                 * @return {*|jQuery}\n                 */\n                renderTreeLevel: function (children) {\n                    var _context = $.extend({}, this, {\n                        items: children,\n                        nested: true\n                    }),\n                    tmpl = templates[tmplName];\n\n                    tmpl = tmpl({\n                        data: _context\n                    });\n\n                    return $('<div>').append($(tmpl)).html();\n                }\n            });\n        },\n\n        /**\n         * @override\n         */\n        _processResponse: function (e, items, context) {\n            var control;\n\n            if (context && !context._allShown) {\n                items = this.filter($.mage.treeToList([], items, 0, ''), this._term);\n            }\n            control = this.dropdown.find(this._control.selector);\n\n            if (control.length && control.hasClass('jstree')) {\n                control.jstree('destroy');\n            }\n            this._superApply([e, items, context]);\n        }\n    });\n\n    return $.mage.treeSuggest;\n});\n","mage/backend/floating-header.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\ndefine([\n    'jquery',\n    'jquery/ui'\n], function ($) {\n    'use strict';\n\n    $.widget('mage.floatingHeader', {\n        options: {\n            placeholderAttrs: {\n                'class': 'page-actions-placeholder'\n            },\n            fixedClass: '_fixed',\n            hiddenClass: '_hidden',\n            title: '.page-title-wrapper .page-title',\n            pageMainActions: '.page-main-actions',\n            contains: '[data-role=modal]'\n        },\n\n        /**\n         * Widget initialization\n         * @private\n         */\n        _create: function () {\n            var title = $(this.options.title).text(),\n                wrapped = this.element.find('.page-actions-buttons').children();\n\n            if (this.element.parents(this.options.contains).length) {\n                return this;\n            }\n\n            this._setVars();\n            this._bind();\n            this.element.find('script').remove();\n\n            if (wrapped.length) {\n                wrapped\n                    .unwrap()   // .page-actions-buttons\n                    .unwrap();  // .page-actions-inner\n            }\n            this.element.wrapInner($('<div></div>', {\n                'class': 'page-actions-buttons'\n            }));\n            this.element.wrapInner($('<div></div>', {\n                'class': 'page-actions-inner', 'data-title': title\n            }));\n            this.element.removeClass('floating-header');\n        },\n\n        /**\n         * Set privat variables on load, for performance purposes\n         * @private\n         */\n        _setVars: function () {\n            this._placeholder = this.element.before($('<div/>', this.options.placeholderAttrs)).prev();\n            this._offsetTop = this._placeholder.offset().top;\n            this._height = this.element\n                .parents(this.options.pageMainActions)\n                .outerHeight();\n        },\n\n        /**\n         * Event binding, will monitor scroll and resize events (resize events left for backward compat)\n         * @private\n         */\n        _bind: function () {\n            this._on(window, {\n                scroll: this._handlePageScroll,\n                resize: this._handlePageScroll\n            });\n        },\n\n        /**\n         * Event handler for setting fixed positioning\n         * @private\n         */\n        _handlePageScroll: function () {\n            var isActive = $(window).scrollTop() > this._offsetTop;\n\n            if (isActive) {\n                this.element\n                    .addClass(this.options.fixedClass)\n                    .parents(this.options.pageMainActions)\n                    .addClass(this.options.hiddenClass);\n            } else {\n                this.element\n                    .removeClass(this.options.fixedClass)\n                    .parents(this.options.pageMainActions)\n                    .removeClass(this.options.hiddenClass);\n            }\n\n            this._placeholder.height(isActive ? this._height : '');\n        },\n\n        /**\n         * Widget destroy functionality\n         * @private\n         */\n        _destroy: function () {\n            if (this._placeholder) {\n                this._placeholder.remove();\n            }\n            this._off($(window));\n        }\n    });\n\n    return $.mage.floatingHeader;\n});\n","mage/backend/editablemultiselect.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\n/**\n * @deprecated since version 2.2.0\n */\n/* global EditableMultiselect */\n/* eslint-disable strict */\ndefine([\n    'jquery',\n    'Magento_Ui/js/modal/alert',\n    'Magento_Ui/js/modal/confirm',\n    'jquery/editableMultiselect/js/jquery.editable',\n    'jquery/editableMultiselect/js/jquery.multiselect'\n], function ($, alert, confirm) {\n    /**\n     * Editable multiselect wrapper for multiselects\n     * This class is defined in global scope ('var' is not needed).\n     *\n     *  @param {Object} settings - settings object.\n     *  @param {String} settings.add_button_caption - caption of the 'Add New Value' button\n     *  @param {String} settings.new_url - URL to which new request has to be submitted\n     *  @param {String} settings.save_url - URL to which save request has to be submitted\n     *  @param {String} settings.delete_url - URL to which delete request has to be submitted\n     *  @param {String} settings.delete_confirm_message - confirmation message that is shown to user during\n     *      delete operation\n     *  @param {String} settings.target_select_id - HTML ID of target select element\n     *  @param {Hash} settings.submit_data - extra parameters to send with new/edit/delete requests\n     *  @param {String} settings.entity_value_name - name of the request parameter that represents select option text\n     *  @param {String} settings.entity_id_name - name of the request parameter that represents select option value\n     *  @param {Boolean} settings.is_entry_editable - flag that shows if user can add/edit/remove data\n     *\n     * @constructor\n     */\n    window.EditableMultiselect = function (settings) {\n\n        this.settings = settings || {};\n        this.addButtonCaption = this.settings['add_button_caption'] || 'Add new value';\n        this.newUrl = this.settings['new_url'];\n        this.saveUrl = this.settings['save_url'];\n        this.deleteUrl = this.settings['delete_url'];\n        this.deleteConfirmMessage = this.settings['delete_confirm_message'];\n        this.targetSelectId = this.settings['target_select_id'];\n        this.submitData = this.settings['submit_data'] || {};\n        this.entityIdName = this.settings['entity_id_name'] || 'entity_id';\n        this.entityValueName = this.settings['entity_value_name'] || 'entity_value';\n        this.isEntityEditable = this.settings['is_entity_editable'] || false;\n\n        /**\n         * Initialize editable multiselect (make it visible in UI)\n         */\n        EditableMultiselect.prototype.init = function () {\n            var self = this,\n                mselectOptions = {\n                    addText: this.addButtonCaption,\n\n                    /**\n                     * @param {*} value\n                     * @param {*} options\n                     */\n                    mselectInputSubmitCallback: function (value, options) {\n                        self.createEntity(value, options);\n                    }\n                },\n                mselectList;\n\n            if (!this.isEntityEditable) {\n                // Override default layout of editable multiselect\n                mselectOptions.layout = '<section class=\"block %mselectListClass%\">' +\n                    '<div class=\"block-content\"><div class=\"%mselectItemsWrapperClass%\">' +\n                    '%items%' +\n                    '</div></div>' +\n                    '<div class=\"%mselectInputContainerClass%\">' +\n                    '<input type=\"text\" class=\"%mselectInputClass%\" title=\"%inputTitle%\"/>' +\n                    '<span class=\"%mselectButtonCancelClass%\" title=\"%cancelText%\"></span>' +\n                    '<span class=\"%mselectButtonSaveClass%\" title=\"Add\"></span>' +\n                    '</div>' +\n                    '</section>';\n            }\n\n            $('#' + this.targetSelectId).multiselect(mselectOptions);\n\n            // Make multiselect editable if needed\n            if (this.isEntityEditable) {\n                this.makeMultiselectEditable();\n\n                // Root element of HTML markup that represents select element in UI\n                mselectList = $('#' + this.targetSelectId).next();\n                this.attachEventsToControls(mselectList);\n            }\n        };\n\n        /**\n         * Attach required event handlers to control elements of editable multiselect\n         *\n         * @param {Object} mselectList\n         */\n        EditableMultiselect.prototype.attachEventsToControls = function (mselectList) {\n            mselectList.on('click.mselect-delete', '.mselect-delete', {\n                container: this\n            }, function (event) {\n                // Pass the clicked button to container\n                event.data.container.deleteEntity({\n                    'delete_button': this\n                });\n            });\n\n            mselectList.on('click.mselect-checked', '.mselect-list-item input', {\n                container: this\n            }, function (event) {\n                var el = $(this),\n                    checkedClassName = 'mselect-checked';\n\n                el[el.is(':checked') ? 'addClass' : 'removeClass'](checkedClassName);\n                event.data.container.makeMultiselectEditable();\n            });\n\n            mselectList.on('click.mselect-edit', '.mselect-edit', {\n                container: this\n            }, function (event) {\n                event.data.container.makeMultiselectEditable();\n                $(this).parent().find('label span').trigger('dblclick');\n            });\n        };\n\n        /**\n         * Make multiselect editable\n         */\n        EditableMultiselect.prototype.makeMultiselectEditable = function () {\n            var entityIdName = this.entityIdName,\n                entityValueName = this.entityValueName,\n                selectList = $('#' + this.targetSelectId).next();\n\n            selectList.find('.mselect-list-item:not(.mselect-list-item-not-editable) label span').editable(this.saveUrl,\n            {\n                type: 'text',\n                submit: '<button class=\"mselect-save\" title=\"Save\" type=\"submit\"></button>',\n                cancel: '<span class=\"mselect-cancel\" title=\"Cancel\"></span>',\n                event: 'dblclick',\n                placeholder: '',\n\n                /**\n                 * Is checked.\n                 */\n                isChecked: function () {\n                    var that = $(this),\n                        checked;\n\n                    if (!that.closest('.mselect-list-item').hasClass('mselect-disabled')) {\n                        checked = that.parent().find('[type=checkbox]').prop('disabled');\n                        that.parent().find('[type=checkbox]').prop({\n                            disabled: !checked\n                        });\n                    }\n                },\n\n                /**\n                 * @param {*} value\n                 * @param {Object} sett\n                 * @return {*}\n                 */\n                data: function (value, sett) {\n                    var retval;\n\n                    sett.isChecked.apply(this, [sett]);\n\n                    if (typeof value === 'string') {\n                        retval = value.unescapeHTML();\n\n                        return retval;\n                    }\n\n                    return value;\n                },\n                submitdata: this.submitData,\n                onblur: 'cancel',\n                name: entityValueName,\n                ajaxoptions: {\n                    dataType: 'json'\n                },\n\n                /**\n                 * @param {Object} sett\n                 * @param {*} original\n                 */\n                onsubmit: function (sett, original) {\n                    var select = $(original).closest('.mselect-list').prev(),\n                        current = $(original).closest('.mselect-list-item').index(),\n                        entityId = select.find('option').eq(current).val(),\n                        entityInfo = {};\n\n                    entityInfo[entityIdName] = entityId;\n                    sett.submitdata = $.extend(sett.submitdata || {}, entityInfo);\n                },\n\n                /**\n                 * @param {Object} result\n                 * @param {Object} sett\n                 */\n                callback: function (result, sett) {\n                    var select, current;\n\n                    sett.isChecked.apply(this, [sett]);\n                    select = $(this).closest('.mselect-list').prev();\n                    current = $(this).closest('.mselect-list-item').index();\n\n                    if (result.success) {\n                        if (typeof result[entityValueName] === 'string') {\n                            select.find('option').eq(current).val(result[entityIdName]).text(result[entityValueName]);\n                            $(this).html(result[entityValueName].escapeHTML());\n                        }\n                    } else {\n                        alert({\n                            content: result['error_message']\n                        });\n                    }\n                }\n            });\n        };\n\n        /**\n         * Callback function that is called when admin adds new value to select\n         *\n         * @param {*} value\n         * @param {Object} options - list of settings of multiselect\n         */\n        EditableMultiselect.prototype.createEntity = function (value, options) {\n            var select, entityIdName, entityValueName, entityInfo, postData, ajaxOptions;\n\n            if (!value) {\n                return;\n            }\n\n            select = $('#' + this.targetSelectId),\n            entityIdName = this.entityIdName,\n            entityValueName = this.entityValueName,\n            entityInfo = {};\n            entityInfo[entityIdName] = null;\n            entityInfo[entityValueName] = value;\n\n            postData = $.extend(entityInfo, this.submitData);\n\n            ajaxOptions = {\n                type: 'POST',\n                data: postData,\n                dataType: 'json',\n                url: this.newUrl,\n\n                /**\n                 * @param {Object} result\n                 */\n                success: function (result) {\n                    var resultEntityValueName, mselectItemHtml, sectionBlock, itemsWrapper, inputSelector;\n\n                    if (result.success) {\n                        resultEntityValueName = '';\n\n                        if (typeof result[entityValueName] === 'string') {\n                            resultEntityValueName = result[entityValueName].escapeHTML();\n                        } else {\n                            resultEntityValueName = result[entityValueName];\n                        }\n                        // Add item to initial select element\n                        select.append('<option value=\"' + result[entityIdName] + '\" selected=\"selected\">' +\n                        resultEntityValueName + '</option>');\n                        // Add editable multiselect item\n                        mselectItemHtml = $(options.item.replace(/%value%|%label%/gi, resultEntityValueName)\n                            .replace(/%mselectDisabledClass%|%iseditable%|%isremovable%/gi, '')\n                            .replace(/%mselectListItemClass%/gi, options.mselectListItemClass))\n                            .find('[type=checkbox]')\n                            .addClass(options.mselectCheckedClass)\n                            .prop('checked', true)\n                            .end();\n                        sectionBlock = select.nextAll('section.block:first');\n                        itemsWrapper = sectionBlock.find('.' + options.mselectItemsWrapperClass + '');\n\n                        if (itemsWrapper.children('.' + options.mselectListItemClass + '').length) {\n                            itemsWrapper.children('.' + options.mselectListItemClass + ':last').after(mselectItemHtml);\n                        } else {\n                            itemsWrapper.prepend(mselectItemHtml);\n                        }\n                        // Trigger blur event on input field, that is used to add new value, to hide it\n                        inputSelector = '.' + options.mselectInputContainerClass + ' [type=text].' +\n                            options.mselectInputClass + '';\n                        sectionBlock.find(inputSelector).trigger('blur');\n                    } else {\n                        alert({\n                            content: result['error_message']\n                        });\n                    }\n                }\n            };\n            $.ajax(ajaxOptions);\n        };\n\n        /**\n         * Callback function that is called when user tries to delete value from select\n         *\n         * @param {Object} options\n         */\n        EditableMultiselect.prototype.deleteEntity = function (options) {\n            var self = this;\n\n            if (options['delete_button']) {\n                confirm({\n                    content: this.deleteConfirmMessage,\n                    actions: {\n                        /**\n                         * Confirm.\n                         */\n                        confirm: function () {\n                            // Button that has been clicked\n                            var deleteButton = $(options['delete_button']),\n                                index = deleteButton.parent().index(),\n                                select = deleteButton.closest('.mselect-list').prev(),\n                                entityId = select.find('option').eq(index).val(),\n                                entityInfo = {},\n                                postData, ajaxOptions;\n\n                            entityInfo[self.entityIdName] = entityId;\n                            postData = $.extend(entityInfo, self.submitData);\n\n                            ajaxOptions = {\n                                type: 'POST',\n                                data: postData,\n                                dataType: 'json',\n                                url: self.deleteUrl,\n\n                                /**\n                                 * @param {Object} result\n                                 */\n                                success: function (result) {\n                                    if (result.success) {\n                                        deleteButton.parent().remove();\n                                        select.find('option').eq(index).remove();\n                                    } else {\n                                        alert({\n                                            content: result['error_message']\n                                        });\n                                    }\n                                }\n                            };\n                            $.ajax(ajaxOptions);\n                        }\n                    }\n                });\n            }\n        };\n    };\n});\n","mage/backend/validation.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\n/* global BASE_URL, alertAlreadyDisplayed */\ndefine([\n    'jquery',\n    'underscore',\n    'Magento_Ui/js/modal/alert',\n    'jquery/ui',\n    'jquery/validate',\n    'mage/translate',\n    'mage/validation'\n], function ($, _, alert) {\n    'use strict';\n\n    $.extend(true, $.validator.prototype, {\n        /**\n         * Focus invalid fields\n         */\n        focusInvalid: function () {\n            if (this.settings.focusInvalid) {\n                try {\n                    $(this.errorList.length && this.errorList[0].element || [])\n                        .trigger('focus')\n                        .trigger('focusin');\n                } catch (e) {\n                    // ignore IE throwing errors when focusing hidden elements\n                }\n            }\n        },\n\n        /**\n         * Elements.\n         */\n        elements: function () {\n            var validator = this,\n                rulesCache = {};\n\n            // select all valid inputs inside the form (no submit or reset buttons)\n            return $(this.currentForm)\n                .find('input, select, textarea')\n                .not(this.settings.forceIgnore)\n                .not(':submit, :reset, :image, [disabled]')\n                .not(this.settings.ignore)\n                .filter(function () {\n                    if (!this.name && validator.settings.debug && window.console) {\n                        console.error('%o has no name assigned', this);\n                    }\n\n                    // select only the first element for each name, and only those with rules specified\n                    if (this.name in rulesCache || !validator.objectLength($(this).rules())) {\n                        return false;\n                    }\n\n                    rulesCache[this.name] = true;\n\n                    return true;\n                });\n        }\n    });\n\n    $.extend($.fn, {\n        /**\n         * ValidationDelegate overridden for those cases where the form is located in another form,\n         *     to avoid not correct working of validate plug-in\n         * @override\n         * @param {String} delegate - selector, if event target matched against this selector,\n         *     then event will be delegated\n         * @param {String} type - event type\n         * @param {Function} handler - event handler\n         * @return {Element}\n         */\n        validateDelegate: function (delegate, type, handler) {\n            return this.on(type, $.proxy(function (event) {\n                var target = $(event.target),\n                    form = target[0].form;\n\n                if (form && $(form).is(this) && $.data(form, 'validator') && target.is(delegate)) {\n                    return handler.apply(target, arguments);\n                }\n            }, this));\n        }\n    });\n\n    $.widget('mage.validation', $.mage.validation, {\n        options: {\n            messagesId: 'messages',\n            forceIgnore: '',\n            ignore: ':disabled, .ignore-validate, .no-display.template, ' +\n                ':disabled input, .ignore-validate input, .no-display.template input, ' +\n                ':disabled select, .ignore-validate select, .no-display.template select, ' +\n                ':disabled textarea, .ignore-validate textarea, .no-display.template textarea',\n            errorElement: 'label',\n            errorUrl: typeof BASE_URL !== 'undefined' ? BASE_URL : null,\n\n            /**\n             * @param {HTMLElement} element\n             */\n            highlight: function (element) {\n                if ($.validator.defaults.highlight && typeof $.validator.defaults.highlight === 'function') {\n                    $.validator.defaults.highlight.apply(this, arguments);\n                }\n                $(element).trigger('highlight.validate');\n            },\n\n            /**\n             * @param {HTMLElement} element\n             */\n            unhighlight: function (element) {\n                if ($.validator.defaults.unhighlight && typeof $.validator.defaults.unhighlight === 'function') {\n                    $.validator.defaults.unhighlight.apply(this, arguments);\n                }\n                $(element).trigger('unhighlight.validate');\n            }\n        },\n\n        /**\n         * Validation creation\n         * @protected\n         */\n        _create: function () {\n            if (!this.options.submitHandler && typeof this.options.submitHandler !== 'function') {\n                if (!this.options.frontendOnly && this.options.validationUrl) {\n                    this.options.submitHandler = $.proxy(this._ajaxValidate, this);\n                } else {\n                    this.options.submitHandler = $.proxy(this._submit, this);\n                }\n            }\n            this.element.on('resetElement', function (e) {\n                $(e.target).rules('remove');\n            });\n            this._super('_create');\n        },\n\n        /**\n         * ajax validation\n         * @protected\n         */\n        _ajaxValidate: function () {\n            $.ajax({\n                url: this.options.validationUrl,\n                type: 'POST',\n                dataType: 'json',\n                data: this.element.serialize(),\n                context: $('body'),\n                success: $.proxy(this._onSuccess, this),\n                error: $.proxy(this._onError, this),\n                showLoader: true,\n                dontHide: false\n            });\n        },\n\n        /**\n         * Process ajax success.\n         *\n         * @protected\n         * @param {Object} response\n         */\n        _onSuccess: function (response) {\n            if (!response.error) {\n                this._submit();\n            } else {\n                this._showErrors(response);\n                $(this.element[0]).trigger('afterValidate.error');\n                $('body').trigger('processStop');\n            }\n        },\n\n        /**\n         * Submitting a form.\n         * @private\n         */\n        _submit: function () {\n            $(this.element[0]).trigger('afterValidate.beforeSubmit');\n            this.element[0].submit();\n        },\n\n        /**\n         * Displays errors after backend validation.\n         *\n         * @param {Object} data - Data that came from backend.\n         */\n        _showErrors: function (data) {\n            $('body').notification('clear')\n                .notification('add', {\n                    error: data.error,\n                    message: data.message,\n\n                    /**\n                     * @param {*} message\n                     */\n                    insertMethod: function (message) {\n                        $('.messages:first').html(message);\n                    }\n                });\n        },\n\n        /**\n         * Tries to retrieve element either by id or by inputs' name property.\n         * @param {String} code - String to search by.\n         * @returns {jQuery} jQuery element.\n         */\n        _getByCode: function (code) {\n            var parent = this.element[0],\n                element;\n\n            element = parent.querySelector('#' + code) || parent.querySelector('input[name=' + code + ']');\n\n            return $(element);\n        },\n\n        /**\n         * Process ajax error\n         * @protected\n         */\n        _onError: function () {\n            $(this.element[0]).trigger('afterValidate.error');\n            $('body').trigger('processStop');\n\n            if (this.options.errorUrl) {\n                location.href = this.options.errorUrl;\n            }\n        }\n    });\n\n    _.each({\n        'validate-greater-zero-based-on-option': [\n            function (v, el) {\n                var optionType = $(el)\n                    .closest('.form-list')\n                    .prev('.fieldset-alt')\n                    .find('select.select-product-option-type'),\n                    optionTypeVal = optionType.val();\n\n                v = Number(v) || 0;\n\n                if (optionType && (optionTypeVal == 'checkbox' || optionTypeVal == 'multi') && v <= 0) { //eslint-disable-line\n                    return false;\n                }\n\n                return true;\n            },\n            $.mage.__('Please enter a number greater 0 in this field.')\n        ],\n        'validate-rating': [\n            function () {\n                var ratings = $('#detailed_rating').find('.field-rating'),\n                    noError = true;\n\n                ratings.each(function (index, rating) {\n                    noError = noError && $(rating).find('input:checked').length > 0;\n                });\n\n                return noError;\n            },\n            $.mage.__('Please select one of each ratings above.')\n        ],\n        'validate-downloadable-file': [\n            function (v, element) {\n                var elmParent = $(element).parent(),\n                    linkType = elmParent.find('input[value=\"file\"]'),\n                    newFileContainer;\n\n                if (linkType.is(':checked') && (v === '' || v === '[]')) {\n                    newFileContainer = elmParent.find('.new-file');\n\n                    if (!alertAlreadyDisplayed && (newFileContainer.empty() || newFileContainer.is(':visible'))) {\n                        window.alertAlreadyDisplayed = true;\n                        alert({\n                            content: $.mage.__('There are files that were selected but not uploaded yet. ' +\n                            'Please upload or remove them first')\n                        });\n                    }\n\n                    return false;\n                }\n\n                return true;\n            },\n            'Please upload a file.'\n        ],\n        'validate-downloadable-url': [\n            function (v, element) {\n                var linkType = $(element).parent().find('input[value=\"url\"]');\n\n                if (linkType.is(':checked') && v === '') {\n                    return false;\n                }\n\n                return true;\n            },\n            $.mage.__('Please specify Url.')\n        ]\n    }, function (rule, i) {\n        rule.unshift(i);\n        $.validator.addMethod.apply($.validator, rule);\n    });\n\n    return $.mage.validation;\n});\n","mage/backend/button.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\ndefine([\n    'jquery',\n    'jquery/ui'\n], function ($) {\n    'use strict';\n\n    $.widget('ui.button', $.ui.button, {\n        options: {\n            eventData: {},\n            waitTillResolved: true\n        },\n\n        /**\n         * Button creation.\n         * @protected\n         */\n        _create: function () {\n            if (this.options.event) {\n                this.options.target = this.options.target || this.element;\n                this._bind();\n            }\n\n            this._super();\n        },\n\n        /**\n         * Bind handler on button click.\n         * @protected\n         */\n        _bind: function () {\n            this.element\n                .off('click.button')\n                .on('click.button', $.proxy(this._click, this));\n        },\n\n        /**\n         * Button click handler.\n         * @protected\n         */\n        _click: function () {\n            var options = this.options;\n\n            $(options.target).trigger(options.event, [options.eventData]);\n        }\n    });\n\n    return $.ui.button;\n});\n","mage/backend/form.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\ndefine([\n    'jquery',\n    'jquery/ui'\n], function ($) {\n    'use strict';\n\n    $.widget('mage.form', {\n        options: {\n            handlersData: {\n                save: {},\n                saveAndContinueEdit: {\n                    action: {\n                        args: {\n                            back: 'edit'\n                        }\n                    }\n                },\n                preview: {\n                    target: '_blank'\n                }\n            }\n        },\n\n        /**\n         * Form creation\n         * @protected\n         */\n        _create: function () {\n            this._bind();\n        },\n\n        /**\n         * Set form attributes to initial state\n         * @protected\n         */\n        _rollback: function () {\n            if (this.oldAttributes) {\n                this.element.prop(this.oldAttributes);\n            }\n        },\n\n        /**\n         * Check if field value is changed\n         * @protected\n         * @param {Object} e - event object\n         */\n        _changesObserver: function (e) {\n            var target = $(e.target),\n                changed;\n\n            if (e.type === 'focus' || e.type === 'focusin') {\n                this.currentField = {\n                    statuses: {\n                        checked: target.is(':checked'),\n                        selected: target.is(':selected')\n                    },\n                    val: target.val()\n                };\n\n            } else {\n                if (this.currentField) { //eslint-disable-line no-lonely-if\n                    changed = target.val() !== this.currentField.val ||\n                        target.is(':checked') !== this.currentField.statuses.checked ||\n                        target.is(':selected') !== this.currentField.statuses.selected;\n\n                    if (changed) { //eslint-disable-line max-depth\n                        target.trigger('changed');\n                    }\n                }\n            }\n        },\n\n        /**\n         * Get array with handler names\n         * @protected\n         * @return {Array} Array of handler names\n         */\n        _getHandlers: function () {\n            var handlers = [];\n\n            $.each(this.options.handlersData, function (key) {\n                handlers.push(key);\n            });\n\n            return handlers;\n        },\n\n        /**\n         * Store initial value of form attribute\n         * @param {String} attrName - name of attribute\n         * @protected\n         */\n        _storeAttribute: function (attrName) {\n            var prop;\n\n            this.oldAttributes = this.oldAttributes || {};\n\n            if (!this.oldAttributes[attrName]) {\n                prop = this.element.attr(attrName);\n                this.oldAttributes[attrName] = prop ? prop : '';\n            }\n        },\n\n        /**\n         * Bind handlers\n         * @protected\n         */\n        _bind: function () {\n            this.element\n                .on(this._getHandlers().join(' '), $.proxy(this._submit, this))\n                .on('focus blur focusin focusout', $.proxy(this._changesObserver, this));\n        },\n\n        /**\n         * Get action url for form\n         * @param {Object|String} data - object with parameters for action url or url string\n         * @return {String} action url\n         */\n        _getActionUrl: function (data) {\n            if (typeof data === 'object') {\n                return this._buildURL(this.oldAttributes.action, data.args);\n            }\n\n            return typeof data === 'string' ? data : this.oldAttributes.action;\n        },\n\n        /**\n         * Add additional parameters into URL\n         * @param {String} url - original url\n         * @param {Object} params - object with parameters for action url\n         * @return {String} action url\n         * @private\n         */\n        _buildURL: function (url, params) {\n            var concat = /\\?/.test(url) ? ['&', '='] : ['/', '/'];\n\n            url = url.replace(/[\\/&]+$/, '');\n            $.each(params, function (key, value) {\n                url += concat[0] + key + concat[1] + window.encodeURIComponent(value);\n            });\n\n            return url + (concat[0] === '/' ? '/' : '');\n        },\n\n        /**\n         * Prepare data for form attributes\n         * @protected\n         * @param {Object} data\n         * @return {Object}\n         */\n        _processData: function (data) {\n            $.each(data, $.proxy(function (attrName, attrValue) {\n                this._storeAttribute(attrName);\n\n                if (attrName === 'action') {\n                    data[attrName] = this._getActionUrl(attrValue);\n                }\n            }, this));\n\n            return data;\n        },\n\n        /**\n         * Get additional data before form submit\n         * @protected\n         * @param {String} handlerName\n         * @param {Object} data\n         */\n        _beforeSubmit: function (handlerName, data) {\n            var submitData = {},\n                event = new $.Event('beforeSubmit');\n\n            this.element.trigger(event, [submitData, handlerName]);\n            data = $.extend(\n                true, {},\n                this.options.handlersData[handlerName] || {},\n                submitData,\n                data\n            );\n            this.element.prop(this._processData(data));\n\n            return !event.isDefaultPrevented();\n        },\n\n        /**\n         * Submit the form\n         * @param {Object} e - event object\n         * @param {Object} data - event data object\n         */\n        _submit: function (e, data) {\n            this._rollback();\n\n            if (this._beforeSubmit(e.type, data) !== false) {\n                this.element.trigger('submit', e);\n            }\n        }\n    });\n\n    return $.mage.form;\n});\n","mage/backend/menu.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\ndefine([\n    'jquery',\n    'jquery/ui'\n], function ($) {\n    'use strict';\n\n    $.widget('mage.menu', {\n        widgetEventPrefix: 'menu',\n        version: '1.10.1',\n        defaultElement: '<ul>',\n        delay: 300,\n        options: {\n            icons: {\n                submenu: 'ui-icon-carat-1-e'\n            },\n            menus: 'ul',\n            position: {\n                my: 'left top',\n                at: 'right top'\n            },\n            role: 'menu',\n\n            // callbacks\n            blur: null,\n            focus: null,\n            select: null\n        },\n\n        /**\n         * @private\n         */\n        _create: function () {\n            this.activeMenu = this.element;\n            // flag used to prevent firing of the click handler\n            // as the event bubbles up through nested menus\n            this.mouseHandled = false;\n            this.element\n                .uniqueId()\n                .addClass('ui-menu ui-widget ui-widget-content ui-corner-all')\n                .toggleClass('ui-menu-icons', !!this.element.find('.ui-icon').length)\n                .attr({\n                    role: this.options.role,\n                    tabIndex: 0\n                })\n                // need to catch all clicks on disabled menu\n                // not possible through _on\n                .on('click' + this.eventNamespace, $.proxy(function (event) {\n                    if (this.options.disabled) {\n                        event.preventDefault();\n                    }\n                }, this));\n\n            if (this.options.disabled) {\n                this.element\n                    .addClass('ui-state-disabled')\n                    .attr('aria-disabled', 'true');\n            }\n\n            this._on({\n                /**\n                 * Prevent focus from sticking to links inside menu after clicking\n                 * them (focus should always stay on UL during navigation).\n                 */\n                'mousedown .ui-menu-item > a': function (event) {\n                    event.preventDefault();\n                },\n\n                /**\n                 * Prevent focus from sticking to links inside menu after clicking\n                 * them (focus should always stay on UL during navigation).\n                 */\n                'click .ui-state-disabled > a': function (event) {\n                    event.preventDefault();\n                },\n\n                /**\n                 * @param {jQuery.Event} event\n                 */\n                'click .ui-menu-item:has(a)': function (event) {\n                    var target = $(event.target).closest('.ui-menu-item');\n\n                    if (!this.mouseHandled && target.not('.ui-state-disabled').length) {\n                        this.mouseHandled = true;\n\n                        this.select(event);\n                        // Open submenu on click\n                        if (target.has('.ui-menu').length) {\n                            this.expand(event);\n                        } else if (!this.element.is(':focus')) {\n                            // Redirect focus to the menu\n                            this.element.trigger('focus', [true]);\n\n                            // If the active item is on the top level, let it stay active.\n                            // Otherwise, blur the active item since it is no longer visible.\n                            if (this.active && this.active.parents('.ui-menu').length === 1) { //eslint-disable-line\n                                clearTimeout(this.timer);\n                            }\n                        }\n                    }\n                },\n\n                /**\n                 * @param {jQuery.Event} event\n                 */\n                'mouseenter .ui-menu-item': function (event) {\n                    var target = $(event.currentTarget);\n\n                    // Remove ui-state-active class from siblings of the newly focused menu item\n                    // to avoid a jump caused by adjacent elements both having a class with a border\n                    target.siblings().children('.ui-state-active').removeClass('ui-state-active');\n                    this.focus(event, target);\n                },\n                mouseleave: 'collapseAll',\n                'mouseleave .ui-menu': 'collapseAll',\n\n                /**\n                 * @param {jQuery.Event} event\n                 * @param {*} keepActiveItem\n                 */\n                focus: function (event, keepActiveItem) {\n                    // If there's already an active item, keep it active\n                    // If not, activate the first item\n                    var item = this.active || this.element.children('.ui-menu-item').eq(0);\n\n                    if (!keepActiveItem) {\n                        this.focus(event, item);\n                    }\n                },\n\n                /**\n                 * @param {jQuery.Event} event\n                 */\n                blur: function (event) {\n                    this._delay(function () {\n                        if (!$.contains(this.element[0], this.document[0].activeElement)) {\n                            this.collapseAll(event);\n                        }\n                    });\n                },\n                keydown: '_keydown'\n            });\n\n            this.refresh();\n\n            // Clicks outside of a menu collapse any open menus\n            this._on(this.document, {\n                /**\n                 * @param {jQuery.Event} event\n                 */\n                click: function (event) {\n                    if (!$(event.target).closest('.ui-menu').length) {\n                        this.collapseAll(event);\n                    }\n\n                    // Reset the mouseHandled flag\n                    this.mouseHandled = false;\n                }\n            });\n        },\n\n        /**\n         * @private\n         */\n        _destroy: function () {\n            // Destroy (sub)menus\n            this.element\n                .removeAttr('aria-activedescendant')\n                .find('.ui-menu').addBack()\n                .removeClass('ui-menu ui-widget ui-widget-content ui-corner-all ui-menu-icons')\n                .removeAttr('role')\n                .removeAttr('tabIndex')\n                .removeAttr('aria-labelledby')\n                .removeAttr('aria-expanded')\n                .removeAttr('aria-hidden')\n                .removeAttr('aria-disabled')\n                .removeUniqueId()\n                .show();\n\n            // Destroy menu items\n            this.element.find('.ui-menu-item')\n                .removeClass('ui-menu-item')\n                .removeAttr('role')\n                .removeAttr('aria-disabled')\n                .children('a')\n                .removeUniqueId()\n                .removeClass('ui-corner-all ui-state-hover')\n                .removeAttr('tabIndex')\n                .removeAttr('role')\n                .removeAttr('aria-haspopup')\n                .children().each(function () {\n                    var elem = $(this);\n\n                    if (elem.data('ui-menu-submenu-carat')) {\n                        elem.remove();\n                    }\n                });\n\n            // Destroy menu dividers\n            this.element.find('.ui-menu-divider').removeClass('ui-menu-divider ui-widget-content');\n        },\n\n        /**\n         * @param {jQuery.Event} event\n         * @private\n         */\n        _keydown: function (event) {\n            var match, prev, character, skip, regex,\n                preventDefault = true;\n\n            /**\n             * @param {String} value\n             */\n            function escape(value) {\n                return value.replace(/[\\-\\[\\]{}()*+?.,\\\\\\^$|#\\s]/g, '\\\\$&');\n            }\n\n            switch (event.keyCode) {\n                case $.ui.keyCode.PAGE_UP:\n                    this.previousPage(event);\n                    break;\n\n                case $.ui.keyCode.PAGE_DOWN:\n                    this.nextPage(event);\n                    break;\n\n                case $.ui.keyCode.HOME:\n                    this._move('first', 'first', event);\n                    break;\n\n                case $.ui.keyCode.END:\n                    this._move('last', 'last', event);\n                    break;\n\n                case $.ui.keyCode.UP:\n                    this.previous(event);\n                    break;\n\n                case $.ui.keyCode.DOWN:\n                    this.next(event);\n                    break;\n\n                case $.ui.keyCode.LEFT:\n                    this.collapse(event);\n                    break;\n\n                case $.ui.keyCode.RIGHT:\n                    if (this.active && !this.active.is('.ui-state-disabled')) {\n                        this.expand(event);\n                    }\n                    break;\n\n                case $.ui.keyCode.ENTER:\n                case $.ui.keyCode.SPACE:\n                    this._activate(event);\n                    break;\n\n                case $.ui.keyCode.ESCAPE:\n                    this.collapse(event);\n                    break;\n\n                default:\n                    preventDefault = false;\n                    prev = this.previousFilter || '';\n                    character = String.fromCharCode(event.keyCode);\n                    skip = false;\n\n                    clearTimeout(this.filterTimer);\n\n                    if (character === prev) {\n                        skip = true;\n                    } else {\n                        character = prev + character;\n                    }\n\n                    regex = new RegExp('^' + escape(character), 'i');\n                    match = this.activeMenu.children('.ui-menu-item').filter(function () {\n                        return regex.test($(this).children('a').text());\n                    });\n                    match = skip && match.index(this.active.next()) !== -1 ?\n                        this.active.nextAll('.ui-menu-item') :\n                        match;\n\n                    // If no matches on the current filter, reset to the last character pressed\n                    // to move down the menu to the first item that starts with that character\n                    if (!match.length) {\n                        character = String.fromCharCode(event.keyCode);\n                        regex = new RegExp('^' + escape(character), 'i');\n                        match = this.activeMenu.children('.ui-menu-item').filter(function () {\n                            return regex.test($(this).children('a').text());\n                        });\n                    }\n\n                    if (match.length) {\n                        this.focus(event, match);\n\n                        if (match.length > 1) { //eslint-disable-line max-depth\n                            this.previousFilter = character;\n                            this.filterTimer = this._delay(function () {\n                                delete this.previousFilter;\n                            }, 1000);\n                        } else {\n                            delete this.previousFilter;\n                        }\n                    } else {\n                        delete this.previousFilter;\n                    }\n            }\n\n            if (preventDefault) {\n                event.preventDefault();\n            }\n        },\n\n        /**\n         * @param {jQuery.Event} event\n         * @private\n         */\n        _activate: function (event) {\n            if (!this.active.is('.ui-state-disabled')) {\n                if (this.active.children('a[aria-haspopup=\"true\"]').length) {\n                    this.expand(event);\n                } else {\n                    this.select(event);\n                }\n            }\n        },\n\n        /**\n         * Refresh.\n         */\n        refresh: function () {\n            var menus,\n                icon = this.options.icons.submenu,\n                submenus = this.element.find(this.options.menus);\n\n            // Initialize nested menus\n            submenus.filter(':not(.ui-menu)')\n                .addClass('ui-menu ui-widget ui-widget-content ui-corner-all')\n                .hide()\n                .attr({\n                    role: this.options.role,\n                    'aria-hidden': 'true',\n                    'aria-expanded': 'false'\n                })\n                .each(function () {\n                    var menu = $(this),\n                        item = menu.prev('a'),\n                        submenuCarat = $('<span>')\n                            .addClass('ui-menu-icon ui-icon ' + icon)\n                            .data('ui-menu-submenu-carat', true);\n\n                    item\n                        .attr('aria-haspopup', 'true')\n                        .prepend(submenuCarat);\n                    menu.attr('aria-labelledby', item.attr('id'));\n                });\n\n            menus = submenus.add(this.element);\n\n            // Don't refresh list items that are already adapted\n            menus.children(':not(.ui-menu-item):has(a)')\n                .addClass('ui-menu-item')\n                .attr('role', 'presentation')\n                .children('a')\n                .uniqueId()\n                .addClass('ui-corner-all')\n                .attr({\n                    tabIndex: -1,\n                    role: this._itemRole()\n                });\n\n            // Initialize unlinked menu-items containing spaces and/or dashes only as dividers\n            menus.children(':not(.ui-menu-item)').each(function () {\n                var item = $(this);\n\n                // hyphen, em dash, en dash\n                if (!/[^\\-\\u2014\\u2013\\s]/.test(item.text())) {\n                    item.addClass('ui-widget-content ui-menu-divider');\n                }\n            });\n\n            // Add aria-disabled attribute to any disabled menu item\n            menus.children('.ui-state-disabled').attr('aria-disabled', 'true');\n\n            // If the active item has been removed, blur the menu\n            if (this.active && !$.contains(this.element[0], this.active[0])) {\n                this.blur();\n            }\n        },\n\n        /**\n         * @return {*}\n         * @private\n         */\n        _itemRole: function () {\n            return {\n                menu: 'menuitem',\n                listbox: 'option'\n            }[this.options.role];\n        },\n\n        /**\n         * @param {String} key\n         * @param {*} value\n         * @private\n         */\n        _setOption: function (key, value) {\n            if (key === 'icons') {\n                this.element.find('.ui-menu-icon')\n                    .removeClass(this.options.icons.submenu)\n                    .addClass(value.submenu);\n            }\n            this._super(key, value);\n        },\n\n        /**\n         * @param {jQuery.Event} event\n         * @param {Object} item\n         */\n        focus: function (event, item) {\n            var nested, focused;\n\n            this.blur(event, event && event.type === 'focus');\n\n            this._scrollIntoView(item);\n\n            this.active = item.first();\n            focused = this.active.children('a').addClass('ui-state-focus');\n            // Only update aria-activedescendant if there's a role\n            // otherwise we assume focus is managed elsewhere\n            if (this.options.role) {\n                this.element.attr('aria-activedescendant', focused.attr('id'));\n            }\n\n            // Highlight active parent menu item, if any\n            this.active\n                .parent()\n                .closest('.ui-menu-item')\n                .children('a:first')\n                .addClass('ui-state-active');\n\n            if (event && event.type === 'keydown') {\n                this._close();\n            } else {\n                this.timer = this._delay(function () {\n                    this._close();\n                }, this.delay);\n            }\n\n            nested = item.children('.ui-menu');\n\n            if (nested.length && /^mouse/.test(event.type)) {\n                this._startOpening(nested);\n            }\n            this.activeMenu = item.parent();\n\n            this._trigger('focus', event, {\n                item: item\n            });\n        },\n\n        /**\n         * @param {Object} item\n         * @private\n         */\n        _scrollIntoView: function (item) {\n            var borderTop, paddingTop, offset, scroll, elementHeight, itemHeight;\n\n            if (this._hasScroll()) {\n                borderTop = parseFloat($.css(this.activeMenu[0], 'borderTopWidth')) || 0;\n                paddingTop = parseFloat($.css(this.activeMenu[0], 'paddingTop')) || 0;\n                offset = item.offset().top - this.activeMenu.offset().top - borderTop - paddingTop;\n                scroll = this.activeMenu.scrollTop();\n                elementHeight = this.activeMenu.height();\n                itemHeight = item.height();\n\n                if (offset < 0) {\n                    this.activeMenu.scrollTop(scroll + offset);\n                } else if (offset + itemHeight > elementHeight) {\n                    this.activeMenu.scrollTop(scroll + offset - elementHeight + itemHeight);\n                }\n            }\n        },\n\n        /**\n         * @param {jQuery.Event} event\n         * @param {*} fromFocus\n         */\n        blur: function (event, fromFocus) {\n            if (!fromFocus) {\n                clearTimeout(this.timer);\n            }\n\n            if (!this.active) {\n                return;\n            }\n\n            this.active.children('a').removeClass('ui-state-focus');\n            this.active = null;\n\n            this._trigger('blur', event, {\n                item: this.active\n            });\n        },\n\n        /**\n         * @param {*} submenu\n         * @private\n         */\n        _startOpening: function (submenu) {\n            clearTimeout(this.timer);\n\n            // Don't open if already open fixes a Firefox bug that caused a .5 pixel\n            // shift in the submenu position when mousing over the carat icon\n            if (submenu.attr('aria-hidden') !== 'true') {\n                return;\n            }\n\n            this.timer = this._delay(function () {\n                this._close();\n                this._open(submenu);\n            }, this.delay);\n        },\n\n        /**\n         * @param {*} submenu\n         * @private\n         */\n        _open: function (submenu) {\n            var position = $.extend({\n                of: this.active\n            }, this.options.position);\n\n            clearTimeout(this.timer);\n            this.element.find('.ui-menu').not(submenu.parents('.ui-menu'))\n                .hide()\n                .attr('aria-hidden', 'true');\n\n            submenu\n                .show()\n                .removeAttr('aria-hidden')\n                .attr('aria-expanded', 'true')\n                .position(position);\n        },\n\n        /**\n         * @param {jQuery.Event} event\n         * @param {*} all\n         */\n        collapseAll: function (event, all) {\n            clearTimeout(this.timer);\n            this.timer = this._delay(function () {\n                // If we were passed an event, look for the submenu that contains the event\n                var currentMenu = all ? this.element :\n                    $(event && event.target).closest(this.element.find('.ui-menu'));\n\n                // If we found no valid submenu ancestor, use the main menu to close all sub menus anyway\n                if (!currentMenu.length) {\n                    currentMenu = this.element;\n                }\n\n                this._close(currentMenu);\n\n                this.blur(event);\n                this.activeMenu = currentMenu;\n            }, this.delay);\n        },\n\n        // With no arguments, closes the currently active menu - if nothing is active\n        // it closes all menus.  If passed an argument, it will search for menus BELOW\n        /**\n         * With no arguments, closes the currently active menu - if nothing is active\n         * it closes all menus.  If passed an argument, it will search for menus BELOW.\n         *\n         * @param {*} startMenu\n         * @private\n         */\n        _close: function (startMenu) {\n            if (!startMenu) {\n                startMenu = this.active ? this.active.parent() : this.element;\n            }\n\n            startMenu\n                .find('.ui-menu')\n                .hide()\n                .attr('aria-hidden', 'true')\n                .attr('aria-expanded', 'false')\n                .end()\n                .find('a.ui-state-active')\n                .removeClass('ui-state-active');\n        },\n\n        /**\n         * @param {jQuery.Event} event\n         */\n        collapse: function (event) {\n            var newItem = this.active &&\n                this.active.parent().closest('.ui-menu-item', this.element);\n\n            if (newItem && newItem.length) {\n                this._close();\n                this.focus(event, newItem);\n            }\n        },\n\n        /**\n         * @param {jQuery.Event} event\n         */\n        expand: function (event) {\n            var newItem = this.active &&\n                this.active\n                    .children('.ui-menu ')\n                    .children('.ui-menu-item')\n                    .first();\n\n            if (newItem && newItem.length) {\n                this._open(newItem.parent());\n\n                // Delay so Firefox will not hide activedescendant change in expanding submenu from AT\n                this._delay(function () {\n                    this.focus(event, newItem);\n                });\n            }\n        },\n\n        /**\n         * @param {jQuery.Event} event\n         */\n        next: function (event) {\n            this._move('next', 'first', event);\n        },\n\n        /**\n         * @param {jQuery.Event} event\n         */\n        previous: function (event) {\n            this._move('prev', 'last', event);\n        },\n\n        /**\n         * @return {null|Boolean}\n         */\n        isFirstItem: function () {\n            return this.active && !this.active.prevAll('.ui-menu-item').length;\n        },\n\n        /**\n         * @return {null|Boolean}\n         */\n        isLastItem: function () {\n            return this.active && !this.active.nextAll('.ui-menu-item').length;\n        },\n\n        /**\n         * @param {*} direction\n         * @param {*} filter\n         * @param {jQuery.Event} event\n         * @private\n         */\n        _move: function (direction, filter, event) {\n            var next;\n\n            if (this.active) {\n                if (direction === 'first' || direction === 'last') {\n                    next = this.active\n                        [direction === 'first' ? 'prevAll' : 'nextAll']('.ui-menu-item')\n                        .eq(-1);\n                } else {\n                    next = this.active\n                        [direction + 'All']('.ui-menu-item')\n                        .eq(0);\n                }\n            }\n\n            if (!next || !next.length || !this.active) {\n                next = this.activeMenu.children('.ui-menu-item')[filter]();\n            }\n\n            this.focus(event, next);\n        },\n\n        /**\n         * @param {jQuery.Event} event\n         */\n        nextPage: function (event) {\n            var item, base, height;\n\n            if (!this.active) {\n                this.next(event);\n\n                return;\n            }\n\n            if (this.isLastItem()) {\n                return;\n            }\n\n            if (this._hasScroll()) {\n                base = this.active.offset().top;\n                height = this.element.height();\n                this.active.nextAll('.ui-menu-item').each(function () {\n                    item = $(this);\n\n                    return item.offset().top - base - height < 0;\n                });\n\n                this.focus(event, item);\n            } else {\n                this.focus(event, this.activeMenu.children('.ui-menu-item')\n                    [!this.active ? 'first' : 'last']());\n            }\n        },\n\n        /**\n         * @param {jQuery.Event} event\n         */\n        previousPage: function (event) {\n            var item, base, height;\n\n            if (!this.active) {\n                this.next(event);\n\n                return;\n            }\n\n            if (this.isFirstItem()) {\n                return;\n            }\n\n            if (this._hasScroll()) {\n                base = this.active.offset().top;\n                height = this.element.height();\n                this.active.prevAll('.ui-menu-item').each(function () {\n                    item = $(this);\n\n                    return item.offset().top - base + height > 0;\n                });\n\n                this.focus(event, item);\n            } else {\n                this.focus(event, this.activeMenu.children('.ui-menu-item').first());\n            }\n        },\n\n        /**\n         * @return {Boolean}\n         * @private\n         */\n        _hasScroll: function () {\n            return this.element.outerHeight() < this.element.prop('scrollHeight');\n        },\n\n        /**\n         * @param {jQuery.Event} event\n         */\n        select: function (event) {\n            // TODO: It should never be possible to not have an active item at this\n            // point, but the tests don't trigger mouseenter before click.\n            var ui;\n\n            this.active = this.active || $(event.target).closest('.ui-menu-item');\n            ui = {\n                item: this.active\n            };\n\n            if (!this.active.has('.ui-menu').length) {\n                this.collapseAll(event, true);\n            }\n            this._trigger('select', event, ui);\n        }\n    });\n\n    return $.mage.menu;\n});\n","mage/adminhtml/globals.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\n/* global setLocation */\ndefine([\n    'Magento_Ui/js/modal/confirm',\n    'mage/dataPost'\n], function (confirm, dataPost) {\n    'use strict';\n\n    /**\n     * Set of a temporary methods used to provide\n     * backward compatibility with a legacy code.\n     */\n    window.setLocation = function (url) {\n        window.location.href = url;\n    };\n\n    /**\n     * Helper for onclick action.\n     * @param {String} message\n     * @param {String} url\n     * @param {Object} postData\n     * @returns {Boolean}\n     */\n    window.deleteConfirm = function (message, url, postData) {\n        confirm({\n            content: message,\n            actions: {\n                /**\n                 * Confirm action.\n                 */\n                confirm: function () {\n                    if (postData !== undefined) {\n                        postData.action = url;\n                        dataPost().postData(postData);\n                    } else {\n                        setLocation(url);\n                    }\n                }\n            }\n        });\n\n        return false;\n    };\n\n    /**\n     * Helper for onclick action.\n     * @param {String} message\n     * @param {String} url\n     * @returns {Boolean}\n     */\n    window.confirmSetLocation = window.deleteConfirm;\n});\n","mage/adminhtml/events.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\n/* global varienEvents */\n/* eslint-disable strict */\ndefine([\n    'Magento_Ui/js/modal/alert',\n    'prototype'\n], function (alert) {\n    // from http://www.someelement.com/2007/03/eventpublisher-custom-events-la-pubsub.html\n    window.varienEvents = Class.create();\n\n    varienEvents.prototype = {\n        /**\n         * Initialize.\n         */\n        initialize: function () {\n            this.arrEvents = {};\n            this.eventPrefix = '';\n        },\n\n        /**\n        * Attaches a {handler} function to the publisher's {eventName} event for execution upon the event firing\n        * @param {String} eventName\n        * @param {Function} handler\n        * @param {Boolean} [asynchFlag] - Defaults to false if omitted.\n        * Indicates whether to execute {handler} asynchronously (true) or not (false).\n        */\n        attachEventHandler: function (eventName, handler) {\n            var asynchVar, handlerObj;\n\n            if (typeof handler == 'undefined' || handler == null) {\n                return;\n            }\n            eventName += this.eventPrefix;\n            // using an event cache array to track all handlers for proper cleanup\n            if (this.arrEvents[eventName] == null) {\n                this.arrEvents[eventName] = [];\n            }\n            //create a custom object containing the handler method and the asynch flag\n            asynchVar = arguments.length > 2 ? arguments[2] : false;\n            handlerObj = {\n                method: handler,\n                asynch: asynchVar\n            };\n            this.arrEvents[eventName].push(handlerObj);\n        },\n\n        /**\n        * Removes a single handler from a specific event\n        * @param {String} eventName - The event name to clear the handler from\n        * @param {Function} handler - A reference to the handler function to un-register from the event\n        */\n        removeEventHandler: function (eventName, handler) {\n            eventName += this.eventPrefix;\n\n            if (this.arrEvents[eventName] != null) {\n                this.arrEvents[eventName] = this.arrEvents[eventName].reject(function (obj) {\n                    return obj.method == handler; //eslint-disable-line eqeqeq\n                });\n            }\n        },\n\n        /**\n        * Removes all handlers from a single event\n        * @param {String} eventName - The event name to clear handlers from\n        */\n        clearEventHandlers: function (eventName) {\n            eventName += this.eventPrefix;\n            this.arrEvents[eventName] = null;\n        },\n\n        /**\n        * Removes all handlers from ALL events\n        */\n        clearAllEventHandlers: function () {\n            this.arrEvents = {};\n        },\n\n        /**\n         * Collect and modify value of arg synchronously in succession and return its new value.\n         * In order to use, call attachEventHandler and add function handlers with eventName.\n         * Then call fireEventReducer with eventName and any argument to have its value accumulatively modified.\n         * Event handlers will be applied to argument in order of first attached to last attached.\n         * @param {String} eventName\n         * @param {*} arg\n         */\n        fireEventReducer: function (eventName, arg) {\n            var evtName = eventName + this.eventPrefix,\n                result = arg,\n                len,\n                i;\n\n            if (!this.arrEvents[evtName]) {\n                return result;\n            }\n\n            len = this.arrEvents[evtName].length; //optimization\n\n            for (i = 0; i < len; i++) {\n                /* eslint-disable max-depth */\n                try {\n                    result = this.arrEvents[evtName][i].method(result);\n                } catch (e) {\n                    if (this.id) {\n                        alert({\n                            content: 'error: error in ' + this.id + '.fireEventReducer():\\n\\nevent name: ' +\n                            eventName + '\\n\\nerror message: ' + e.message\n                        });\n                    } else {\n                        alert({\n                            content: 'error: error in [unknown object].fireEventReducer():\\n\\nevent name: ' +\n                            eventName + '\\n\\nerror message: ' + e.message\n                        });\n                    }\n                }\n                /* eslint-disable max-depth */\n            }\n\n            return result;\n        },\n\n        /**\n        * Fires the event {eventName}, resulting in all registered handlers to be executed.\n        * It also collects and returns results of all non-asynchronous handlers\n        * @param {String} eventName - The name of the event to fire\n        * @param {Object} [args] - Any object, will be passed into the handler function as the only argument\n        * @return {Array}\n        */\n        fireEvent: function (eventName) {\n            var evtName = eventName + this.eventPrefix,\n                results = [],\n                result, len, i, eventArgs, method, eventHandler;\n\n            if (this.arrEvents[evtName] != null) {\n                len = this.arrEvents[evtName].length; //optimization\n\n                for (i = 0; i < len; i++) {\n                    /* eslint-disable max-depth */\n                    try {\n                        if (arguments.length > 1) {\n                            if (this.arrEvents[evtName][i].asynch) {\n                                eventArgs = arguments[1];\n                                method = this.arrEvents[evtName][i].method.bind(this);\n                                setTimeout(function () { //eslint-disable-line no-loop-func\n                                    method(eventArgs);\n                                }, 10);\n                            } else {\n                                result = this.arrEvents[evtName][i].method(arguments[1]);\n                            }\n                        } else {\n                            if (this.arrEvents[evtName][i].asynch) { //eslint-disable-line no-lonely-if\n                                eventHandler = this.arrEvents[evtName][i].method;\n                                setTimeout(eventHandler, 1);\n                            } else if (\n                                this.arrEvents &&\n                                this.arrEvents[evtName] &&\n                                this.arrEvents[evtName][i] &&\n                                this.arrEvents[evtName][i].method\n                            ) {\n                                result = this.arrEvents[evtName][i].method();\n                            }\n                        }\n                        results.push(result);\n                    }\n                    catch (e) {\n                        if (this.id) {\n                            alert({\n                                content: 'error: error in ' + this.id + '.fireEvent():\\n\\nevent name: ' +\n                                eventName + '\\n\\nerror message: ' + e.message\n                            });\n                        } else {\n                            alert({\n                                content: 'error: error in [unknown object].fireEvent():\\n\\nevent name: ' +\n                                eventName + '\\n\\nerror message: ' + e.message\n                            });\n                        }\n                    }\n\n                    /* eslint-enable max-depth */\n                }\n            }\n\n            return results;\n        }\n    };\n\n    window.varienGlobalEvents = new varienEvents(); //jscs:ignore requireCapitalizedConstructors\n\n    return window.varienGlobalEvents;\n});\n","mage/adminhtml/backup.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\n/**\n * @deprecated since version 2.2.0\n */\n/* global  AdminBackup, setLocation */\n/* eslint-disable strict */\ndefine([\n    'jquery',\n    'Magento_Ui/js/modal/modal',\n    'mage/mage',\n    'prototype'\n], function (jQuery) {\n    window.AdminBackup = new Class.create();\n\n    AdminBackup.prototype = {\n        /**\n         * Initialize.\n         */\n        initialize: function () {\n            this.reset();\n            this.rollbackUrl = this.backupUrl = '';\n        },\n\n        /**\n         * reset.\n         */\n        reset: function () {\n            this.time = 0;\n            this.type = '';\n        },\n\n        /**\n         * @param {*} type\n         * @return {Boolean}\n         */\n        backup: function (type) {\n            this.reset();\n            this.type = type;\n            this.requestBackupOptions();\n\n            return false;\n        },\n\n        /**\n         * @param {*} type\n         * @param {*} time\n         * @return {Boolean}\n         */\n        rollback: function (type, time) {\n            this.reset();\n            this.time = time;\n            this.type = type;\n            this.showRollbackWarning();\n\n            return false;\n        },\n\n        /**\n         * Show rollback warning.\n         */\n        showRollbackWarning: function () {\n            this.showPopup('rollback-warning');\n        },\n\n        /**\n         * request backup options.\n         */\n        requestBackupOptions: function () {\n            this.hidePopups();\n            this.showPopup('backup-options');\n\n            if (this.type === 'snapshot') {\n                jQuery('#exclude-media-checkbox-container').removeClass('no-display');\n            }\n        },\n\n        /**\n         * Request password.\n         */\n        requestPassword: function () {\n            this.hidePopups();\n\n            this.showPopup('rollback-request-password');\n\n            this.type != 'db' ? //eslint-disable-line eqeqeq\n                $('use-ftp-checkbox-row').show() :\n                $('use-ftp-checkbox-row').hide();\n        },\n\n        /**\n         * Toggle Ftp Credentials Form.\n         */\n        toggleFtpCredentialsForm: function () {\n            $('use_ftp').checked ? $('ftp-credentials-container').show()\n                : $('ftp-credentials-container').hide();\n\n            $$('#ftp-credentials-container input').each(function (item) {\n                if (item.name == 'ftp_path') { //eslint-disable-line eqeqeq\n                    return;\n                }\n                $('use_ftp').checked ? item.addClassName('required-entry') : item.removeClassName('required-entry');\n            });\n        },\n\n        /**\n         * Submit backup.\n         */\n        submitBackup: function () {\n            var data = {\n                'type': this.type,\n                'maintenance_mode': $('backup_maintenance_mode').checked ? 1 : 0,\n                'backup_name': $('backup_name').value,\n                'exclude_media': $('exclude_media').checked ? 1 : 0\n            };\n\n            new Ajax.Request(this.backupUrl, {\n                onSuccess: function (transport) {\n                    this.processResponse(transport, 'backup-options');\n                }.bind(this),\n                method: 'post',\n                parameters: data\n            });\n\n            this.modal.modal('closeModal');\n        },\n\n        /**\n         * Submit rollback.\n         */\n        submitRollback: function () {\n            var data = this.getPostData();\n\n            new Ajax.Request(this.rollbackUrl, {\n                onSuccess: function (transport) {\n                    this.processResponse(transport, 'rollback-request-password');\n                }.bind(this),\n                method: 'post',\n                parameters: data\n            });\n\n            this.modal.modal('closeModal');\n        },\n\n        /**\n         * @param {Object} transport\n         * @param {*} popupId\n         */\n        processResponse: function (transport, popupId) {\n            var json;\n\n            if (!transport.responseText.isJSON()) {\n                return;\n            }\n\n            json = transport.responseText.evalJSON();\n\n            if (json.error) {\n                this.showPopup(popupId);\n                this.displayError(popupId, json.error);\n\n                return;\n            }\n\n            if (json['redirect_url']) {\n                setLocation(json['redirect_url']);\n            }\n        },\n\n        /**\n         * @param {*} parentContainer\n         * @param {*} message\n         */\n        displayError: function (parentContainer, message) {\n            var messageHtml = this.getErrorMessageHtml(message);\n\n            $$('#' + parentContainer + ' .backup-messages .messages').invoke('update', messageHtml);\n            $$('#' + parentContainer + ' .backup-messages').invoke('show');\n        },\n\n        /**\n         * @param {*} message\n         * @return {String}\n         */\n        getErrorMessageHtml: function (message) {\n            return '<div class=\"message message-error error\"><div>' + message + '</div></div>';\n        },\n\n        /**\n         * @return {*|jQuery}\n         */\n        getPostData: function () {\n            var data = $('rollback-form').serialize(true);\n\n            data.time = this.time;\n            data.type = this.type;\n\n            return data;\n        },\n        backupConfig: {\n            'backup-options': {\n                title: jQuery.mage.__('Backup options'),\n\n                /**\n                 * @return {String}\n                 */\n                content: function () {\n                    return document.getElementById('backup-options-template').textContent;\n                },\n\n                /**\n                 * Action Ok.\n                 */\n                actionOk: function () {\n                    this.modal.find('#backup-form').validation({\n                        submitHandler: jQuery.proxy(this.submitBackup, this)\n                    });\n                    this.modal.find('#backup-form').submit();\n                }\n            },\n            'rollback-warning': {\n                title: jQuery.mage.__('Warning'),\n\n                /**\n                 * @return {String}\n                 */\n                content: function () {\n                    return document.getElementById('rollback-warning-template').textContent;\n                },\n\n                /**\n                 * Action Ok.\n                 */\n                actionOk: function () {\n                    this.modal.modal('closeModal');\n                    this.requestPassword();\n                }\n            },\n            'rollback-request-password': {\n                title: jQuery.mage.__('Backup options'),\n\n                /**\n                 * @return {String}\n                 */\n                content: function () {\n                    return document.getElementById('rollback-request-password-template').textContent;\n                },\n\n                /**\n                 * Action Ok.\n                 */\n                actionOk: function () {\n                    this.modal.find('#rollback-form').validation({\n                        submitHandler: jQuery.proxy(this.submitRollback, this)\n                    });\n                    this.modal.find('#rollback-form').submit();\n                },\n\n                /**\n                 * Opened.\n                 */\n                opened: function () {\n                    this.toggleFtpCredentialsForm();\n                }\n            }\n        },\n\n        /**\n         * @param {*} divId\n         */\n        showPopup: function (divId) {\n            var self = this;\n\n            this.modal = jQuery('<div></div>').attr({\n                id: divId\n            }).html(this.backupConfig[divId].content()).modal({\n                modalClass: 'magento',\n                title: this.backupConfig[divId].title,\n                type: 'slide',\n\n                /**\n                 * @param {juery.Event} e\n                 * @param {Object} modal\n                 */\n                closed: function (e, modal) {\n                    modal.modal.remove();\n                },\n\n                /**\n                 * Opened.\n                 */\n                opened: function () {\n                    if (self.backupConfig[divId].opened) {\n                        self.backupConfig[divId].opened.call(self);\n                    }\n                },\n                buttons: [{\n                    text: jQuery.mage.__('Cancel'),\n                    'class': 'action cancel',\n\n                    /**\n                     * Click action.\n                     */\n                    click: function () {\n                        this.closeModal();\n                    }\n                }, {\n                    text: jQuery.mage.__('Ok'),\n                    'class': 'action primary',\n\n                    /**\n                     * Click action.\n                     */\n                    click: function () {\n                        self.backupConfig[divId].actionOk.call(self);\n                    }\n                }]\n            });\n            this.modal.modal('openModal');\n        },\n\n        /**\n         * Hide Popups.\n         */\n        hidePopups: function () {\n            var mask;\n\n            $$('.backup-dialog').each(Element.hide);\n            mask = $('popup-window-mask');\n\n            if (mask) {\n                mask.hide();\n            }\n        }\n    };\n});\n","mage/adminhtml/grid.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\n// also depends on a mage/adminhtml/tools.js for Base64 encoding\n/* global varienGrid, setLocation, varienGlobalEvents, FORM_KEY,\n    BASE_URL, Base64, varienGridMassaction, varienStringArray, serializerController\n*/\n/* eslint-disable strict */\ndefine([\n    'jquery',\n    'mage/template',\n    'Magento_Ui/js/modal/alert',\n    'Magento_Ui/js/modal/confirm',\n    'mage/mage',\n    'prototype',\n    'mage/adminhtml/form',\n    'mage/adminhtml/events'\n], function (jQuery, mageTemplate, alert, confirm) {\n    /**\n     * @param {*} grid\n     * @param {*} event\n     */\n    function openGridRow(grid, event) {\n        var element = Event.findElement(event, 'tr');\n\n        if (['a', 'input', 'select', 'option'].indexOf(Event.element(event).tagName.toLowerCase()) !== -1) {\n            return;\n        }\n\n        if (element.title) {\n            setLocation(element.title);\n        }\n    }\n    window.openGridRow = openGridRow;\n\n    window.varienGrid = new Class.create();\n\n    varienGrid.prototype = {\n        /**\n         * @param {String} containerId\n         * @param {String} url\n         * @param {*} pageVar\n         * @param {*} sortVar\n         * @param {*} dirVar\n         * @param {*} filterVar\n         */\n        initialize: function (containerId, url, pageVar, sortVar, dirVar, filterVar) {\n            this.containerId = containerId;\n            jQuery('#' + containerId).data('gridObject', this);\n            this.url = url;\n            this.pageVar = pageVar || false;\n            this.sortVar = sortVar || false;\n            this.dirVar = dirVar || false;\n            this.filterVar = filterVar || false;\n            this.tableSufix = '_table';\n            this.useAjax = false;\n            this.rowClickCallback = false;\n            this.checkboxCheckCallback = false;\n            this.preInitCallback = false;\n            this.initCallback = false;\n            this.initRowCallback = false;\n            this.doFilterCallback = false;\n            this.sortableUpdateCallback = false;\n            this.targetElement = undefined;\n            this.filterKeyPressCallback = false;\n\n            this.reloadParams = false;\n\n            this.trOnMouseOver = this.rowMouseOver.bindAsEventListener(this);\n            this.trOnMouseOut = this.rowMouseOut.bindAsEventListener(this);\n            this.trOnMouseDown = this.rowMouseDown.bindAsEventListener(this);\n            this.trOnClick = this.rowMouseClick.bindAsEventListener(this);\n            this.trOnDblClick = this.rowMouseDblClick.bindAsEventListener(this);\n            this.trOnKeyPress = this.keyPress.bindAsEventListener(this);\n\n            this.thLinkOnClick = this.doSort.bindAsEventListener(this);\n            this.initGrid();\n        },\n\n        /**\n         * Init grid.\n         */\n        initGrid: function () {\n            var row, columns, col;\n\n            if (this.preInitCallback) {\n                this.preInitCallback(this);\n            }\n\n            if ($(this.containerId + this.tableSufix)) {\n                this.rows = $$('#' + this.containerId + this.tableSufix + ' tbody tr');\n\n                for (row = 0; row < this.rows.length; row++) {\n                    if (row % 2 == 0) { //eslint-disable-line eqeqeq, max-depth\n                        Element.addClassName(this.rows[row], 'even');\n                    }\n\n                    Event.observe(this.rows[row], 'mouseover', this.trOnMouseOver);\n                    Event.observe(this.rows[row], 'mouseout', this.trOnMouseOut);\n                    Event.observe(this.rows[row], 'mousedown', this.trOnMouseDown);\n                    Event.observe(this.rows[row], 'click', this.trOnClick);\n                    Event.observe(this.rows[row], 'dblclick', this.trOnDblClick);\n                }\n            }\n\n            if (this.sortVar && this.dirVar) {\n                columns = $$('#' + this.containerId + this.tableSufix + ' thead [data-sort]');\n\n                for (col = 0; col < columns.length; col++) {\n                    Event.observe(columns[col], 'click', this.thLinkOnClick);\n                }\n            }\n            this.bindFilterFields();\n            this.bindFieldsChange();\n\n            if (this.initCallback) {\n                try {\n                    this.initCallback(this);\n                } catch (e) {\n                    if (window.console) { //eslint-disable-line max-depth\n                        console.log(e);\n                    }\n                }\n            }\n            jQuery('#' + this.containerId).trigger('gridinit', this);\n        },\n\n        /**\n         * Init grid ajax.\n         */\n        initGridAjax: function () {\n            this.initGrid();\n            this.initGridRows();\n        },\n\n        /**\n         * Init grid rows.\n         */\n        initGridRows: function () {\n            var row;\n\n            if (this.initRowCallback) {\n                for (row = 0; row < this.rows.length; row++) {\n                    try { //eslint-disable-line max-depth\n                        this.initRowCallback(this, this.rows[row]);\n                    } catch (e) {\n                        if (window.console) { //eslint-disable-line max-depth\n                            console.log(e);\n                        }\n                    }\n                }\n            }\n        },\n\n        /**\n         * @param {*} event\n         */\n        rowMouseOver: function (event) {\n            var element = Event.findElement(event, 'tr');\n\n            if (!element.title) {\n                return;\n            }\n\n            Element.addClassName(element, 'on-mouse');\n\n            if (!Element.hasClassName('_clickable') && (this.rowClickCallback !== openGridRow || element.title)) {\n                if (element.title) {\n                    Element.addClassName(element, '_clickable');\n                }\n            }\n        },\n\n        /**\n         * @param {*} event\n         */\n        rowMouseOut: function (event) {\n            var element = Event.findElement(event, 'tr');\n\n            Element.removeClassName(element, 'on-mouse');\n        },\n\n        /**\n         * @param {*} event\n         */\n        rowMouseDown: function (event) {\n            this.targetElement = event.target;\n        },\n\n        /**\n         * @param {*} event\n         */\n        rowMouseClick: function (event) {\n            if (this.rowClickCallback) {\n                try {\n                    this.rowClickCallback(this, event);\n                } catch (e) {\n                }\n            }\n            varienGlobalEvents.fireEvent('gridRowClick', event);\n            this.targetElement = undefined;\n        },\n\n        /**\n         * @param {*} event\n         */\n        rowMouseDblClick: function (event) {\n            varienGlobalEvents.fireEvent('gridRowDblClick', event);\n        },\n\n        /**\n         * Key press.\n         */\n        keyPress: function () {\n        },\n\n        /**\n         * @param {*} event\n         * @return {Boolean}\n         */\n        doSort: function (event) {\n            var element = Event.findElement(event, 'th');\n\n            if (element.readAttribute('data-sort') && element.readAttribute('data-direction')) {\n                this.addVarToUrl(this.sortVar, element.readAttribute('data-sort'));\n                this.addVarToUrl(this.dirVar, element.readAttribute('data-direction'));\n                this.reload(this.url);\n            }\n            Event.stop(event);\n\n            return false;\n        },\n\n        /**\n         * @param {Object} element\n         */\n        loadByElement: function (element) {\n            if (element && element.name) {\n                this.reload(this.addVarToUrl(element.name, element.value));\n            }\n        },\n\n        /**\n         * @param {*} data\n         * @param {*} textStatus\n         * @param {*} transport\n         * @private\n         */\n        _onAjaxSeccess: function (data, textStatus, transport) {\n            var responseText, response, divId;\n\n            /* eslint-disable max-depth */\n            try {\n                responseText = transport.responseText;\n\n                if (transport.responseText.isJSON()) {\n                    response = transport.responseText.evalJSON();\n\n                    if (response.error) {\n                        alert({\n                            content: response.message\n                        });\n                    }\n\n                    if (response.ajaxExpired && response.ajaxRedirect) {\n                        setLocation(response.ajaxRedirect);\n                    }\n                } else {\n\n                    /*eslint-disable max-len*/\n                    /**\n                     * For IE <= 7.\n                     * If there are two elements, and first has name, that equals id of second.\n                     * In this case, IE will choose one that is above\n                     *\n                     * @see https://prototype.lighthouseapp.com/projects/8886/tickets/994-id-selector-finds-elements-by-name-attribute-in-ie7\n                     */\n\n                    /*eslint-enable max-len*/\n                    divId = $(this.containerId);\n\n                    if (divId.id == this.containerId) { //eslint-disable-line eqeqeq\n                        divId.update(responseText);\n                    } else {\n                        $$('div[id=\"' + this.containerId + '\"]')[0].update(responseText);\n                    }\n                }\n            } catch (e) {\n                divId = $(this.containerId);\n\n                if (divId.id == this.containerId) { //eslint-disable-line eqeqeq\n                    divId.update(responseText);\n                } else {\n                    $$('div[id=\"' + this.containerId + '\"]')[0].update(responseText);\n                }\n            }\n\n            /* eslint-enable max-depth */\n            jQuery('#' + this.containerId).trigger('contentUpdated');\n        },\n\n        /**\n         * @param {*} url\n         * @param {Function} onSuccessCallback\n         * @return {*}\n         */\n        reload: function (url, onSuccessCallback) {\n            var ajaxSettings, ajaxRequest;\n\n            this.reloadParams = this.reloadParams || {};\n            this.reloadParams['form_key'] = FORM_KEY;\n            url = url || this.url;\n\n            if (this.useAjax) {\n                ajaxSettings = {\n                    url: url + (url.match(new RegExp('\\\\?')) ? '&ajax=true' : '?ajax=true'),\n                    showLoader: true,\n                    method: 'post',\n                    context: jQuery('#' + this.containerId),\n                    data: this.reloadParams,\n                    error: this._processFailure.bind(this),\n                    complete: this.initGridAjax.bind(this),\n                    dataType: 'html',\n\n                    /**\n                     * Success callback.\n                     */\n                    success: function (data, textStatus, transport) {\n                        this._onAjaxSeccess(data, textStatus, transport);\n\n                        if (onSuccessCallback && typeof onSuccessCallback === 'function') {\n                            // execute the callback, passing parameters as necessary\n                            onSuccessCallback();\n                        }\n                    }.bind(this)\n                };\n                jQuery('#' + this.containerId).trigger('gridajaxsettings', ajaxSettings);\n                ajaxRequest = jQuery.ajax(ajaxSettings);\n                jQuery('#' + this.containerId).trigger('gridajax', ajaxRequest);\n\n                return ajaxRequest;\n            }\n\n            if (this.reloadParams) {\n                $H(this.reloadParams).each(function (pair) {\n                    url = this.addVarToUrl(pair.key, pair.value);\n                }.bind(this));\n            }\n            location.href = url;\n        },\n\n        /**\n         * @private\n         */\n        _processFailure: function () {\n            location.href = BASE_URL;\n        },\n\n        /**\n         * @param {*} url\n         * @param {*} varName\n         * @param {*} varValue\n         * @return {String|*}\n         * @private\n         */\n        _addVarToUrl: function (url, varName, varValue) {\n            var re = new RegExp('\\/(' + varName + '\\/.*?\\/)'),\n                parts = url.split(new RegExp('\\\\?'));\n\n            url = parts[0].replace(re, '/');\n            url += varName + '/' + varValue + '/';\n\n            if (parts.size() > 1) {\n                url += '?' + parts[1];\n            }\n\n            return url;\n        },\n\n        /**\n         * Builds the form with fields containing the and submits\n         *\n         * @param {String} url\n         * @param {String} varName\n         * @param {String} varValue\n         * @private\n         */\n        _buildFormAndSubmit: function (url, varName, varValue) {\n            var re = new RegExp('\\/(' + varName + '\\/.*?\\/)'),\n                parts = url.split(new RegExp('\\\\?')),\n                form = jQuery('<form></form>'),\n                inputProps = [\n                    {\n                        name: varName,\n                        value: varValue\n                    },\n                    {\n                        name: 'form_key',\n                        value: window.FORM_KEY\n                    }\n                ],\n                input;\n\n            url = parts[0].replace(re, '/');\n\n            if (parts.size() > 1) {\n                url += '?' + parts[1];\n            }\n\n            form.attr('action', url);\n            form.attr('method', 'POST');\n\n            inputProps.forEach(function (item) {\n                input = jQuery('<input/>');\n                input.attr('name', item.name);\n                input.attr('type', 'hidden');\n                input.val(item.value);\n                form.append(input);\n            });\n            jQuery('[data-container=\"body\"]').append(form);\n            form.submit();\n            form.remove();\n        },\n\n        /**\n         * @param {*} varName\n         * @param {*} varValue\n         * @return {*|String}\n         */\n        addVarToUrl: function (varName, varValue) {\n            this.url = this._addVarToUrl(this.url, varName, varValue);\n\n            return this.url;\n        },\n\n        /**\n         * Do export.\n         */\n        doExport: function () {\n            var exportUrl;\n\n            if ($(this.containerId + '_export')) {\n                exportUrl = $(this.containerId + '_export').value;\n\n                if (this.massaction && this.massaction.checkedString) {\n                    this._buildFormAndSubmit(\n                        exportUrl,\n                        this.massaction.formFieldNameInternal,\n                        this.massaction.checkedString\n                    );\n                } else {\n                    location.href = exportUrl;\n                }\n            }\n        },\n\n        /**\n         * Bind filter fields.\n         */\n        bindFilterFields: function () {\n            var filters = $$(\n                '#' + this.containerId + ' [data-role=\"filter-form\"] input',\n                '#' + this.containerId + ' [data-role=\"filter-form\"] select'\n                ),\n                i;\n\n            for (i = 0; i < filters.length; i++) {\n                Event.observe(filters[i], 'keypress', this.filterKeyPress.bind(this));\n            }\n        },\n\n        /**\n         * Bind field change.\n         */\n        bindFieldsChange: function () {\n            var dataElements, i;\n\n            if (!$(this.containerId)) {\n                return;\n            }\n            //var dataElements = $(this.containerId+this.tableSufix).down('.data tbody').select('input', 'select');\n            // eslint-disable-next-line jquery-no-input-event-shorthand\n            dataElements = $(this.containerId + this.tableSufix).down('tbody').select('input', 'select');\n\n            for (i = 0; i < dataElements.length; i++) {\n                Event.observe(dataElements[i], 'change', dataElements[i].setHasChanges.bind(dataElements[i]));\n            }\n        },\n\n        /**\n         * Bind sortable.\n         */\n        bindSortable: function () {\n            if (jQuery('#' + this.containerId).find('.draggable-handle').length) {\n                jQuery('#' + this.containerId).find('tbody').sortable({\n                    axis: 'y',\n                    handle: '.draggable-handle',\n\n                    /**\n                     * @param {*} event\n                     * @param {*} ui\n                     * @return {*}\n                     */\n                    helper: function (event, ui) {\n                        ui.children().each(function () {\n                            jQuery(this).width(jQuery(this).width());\n                        });\n\n                        return ui;\n                    },\n                    update: this.sortableUpdateCallback ? this.sortableUpdateCallback : function () {\n                    },\n                    tolerance: 'pointer'\n                });\n            }\n        },\n\n        /**\n         * @param {Object} event\n         */\n        filterKeyPress: function (event) {\n            if (event.keyCode == Event.KEY_RETURN) { //eslint-disable-line eqeqeq\n                this.doFilter();\n            }\n\n            if (this.filterKeyPressCallback) {\n                this.filterKeyPressCallback(this, event);\n            }\n        },\n\n        /**\n         * @param {Function} callback\n         */\n        doFilter: function (callback) {\n            var filters = $$(\n                '#' + this.containerId + ' [data-role=\"filter-form\"] input',\n                '#' + this.containerId + ' [data-role=\"filter-form\"] select'\n                ),\n                elements = [],\n                i;\n\n            for (i in filters) {\n                if (filters[i].value && filters[i].value.length) {\n                    elements.push(filters[i]);\n                }\n            }\n\n            if (!this.doFilterCallback || this.doFilterCallback && this.doFilterCallback()) {\n                this.reload(\n                    this.addVarToUrl(this.filterVar, Base64.encode(Form.serializeElements(elements))),\n                    callback\n                );\n            }\n        },\n\n        /**\n         * @param {Function} callback\n         */\n        resetFilter: function (callback) {\n            this.reload(this.addVarToUrl(this.filterVar, ''), callback);\n        },\n\n        /**\n         * @param {Object} element\n         */\n        checkCheckboxes: function (element) {\n            var elements = Element.select($(this.containerId), 'input[name=\"' + element.name + '\"]'),\n                i;\n\n            for (i = 0; i < elements.length; i++) {\n                this.setCheckboxChecked(elements[i], element.checked);\n            }\n\n            /*eslint-enable no-undef*/\n        },\n\n        /**\n         *\n         * @param {HTMLElement} element\n         * @param {*} checked\n         */\n        setCheckboxChecked: function (element, checked) {\n            element.checked = checked;\n            jQuery(element).trigger('change');\n            element.setHasChanges({});\n\n            if (this.checkboxCheckCallback) {\n                this.checkboxCheckCallback(this, element, checked);\n            }\n        },\n\n        /**\n         * @param {Object} event\n         * @param {*} lastId\n         */\n        inputPage: function (event, lastId) {\n            var element = Event.element(event),\n                keyCode = event.keyCode || event.which,\n                enteredValue = parseInt(element.value, 10),\n                pageId = parseInt(lastId, 10);\n\n            if (keyCode == Event.KEY_RETURN) { //eslint-disable-line eqeqeq\n                if (enteredValue > pageId) {\n                    this.setPage(pageId);\n                } else {\n                    this.setPage(enteredValue);\n                }\n            }\n\n            /*if(keyCode>47 && keyCode<58){\n\n             }\n             else{\n             Event.stop(event);\n             }*/\n        },\n\n        /**\n         * @param {*} pageNumber\n         */\n        setPage: function (pageNumber) {\n            this.reload(this.addVarToUrl(this.pageVar, pageNumber));\n        }\n    };\n\n    window.varienGridMassaction = Class.create();\n    varienGridMassaction.prototype = {\n        /* Predefined vars */\n        checkedValues: $H({}),\n        checkedString: '',\n        oldCallbacks: {},\n        errorText: '',\n        items: {},\n        gridIds: [],\n        useSelectAll: false,\n        currentItem: false,\n        lastChecked: {\n            left: false,\n            top: false,\n            checkbox: false\n        },\n        fieldTemplate: mageTemplate('<input type=\"hidden\" name=\"<%- name %>\" value=\"<%- value %>\" />'),\n\n        /**\n         * @param {*} containerId\n         * @param {*} grid\n         * @param {*} checkedValues\n         * @param {*} formFieldNameInternal\n         * @param {*} formFieldName\n         */\n        initialize: function (containerId, grid, checkedValues, formFieldNameInternal, formFieldName) {\n            this.setOldCallback('row_click', grid.rowClickCallback);\n            this.setOldCallback('init', grid.initCallback);\n            this.setOldCallback('init_row', grid.initRowCallback);\n            this.setOldCallback('pre_init', grid.preInitCallback);\n\n            this.useAjax = false;\n            this.grid = grid;\n            this.grid.massaction = this;\n            this.containerId = containerId;\n            this.initMassactionElements();\n\n            this.checkedString = checkedValues;\n            this.formFieldName = formFieldName;\n            this.formFieldNameInternal = formFieldNameInternal;\n\n            this.grid.initCallback = this.onGridInit.bind(this);\n            this.grid.preInitCallback = this.onGridPreInit.bind(this);\n            this.grid.initRowCallback = this.onGridRowInit.bind(this);\n            this.grid.rowClickCallback = this.onGridRowClick.bind(this);\n            this.initCheckboxes();\n            this.checkCheckboxes();\n        },\n\n        /**\n         * @param {*} flag\n         */\n        setUseAjax: function (flag) {\n            this.useAjax = flag;\n        },\n\n        /**\n         * @param {*} flag\n         */\n        setUseSelectAll: function (flag) {\n            this.useSelectAll = flag;\n        },\n\n        /**\n         * Init massaction elements.\n         */\n        initMassactionElements: function () {\n            this.container = $(this.containerId);\n            this.multiselect = $(this.containerId + '-mass-select');\n            this.count = $(this.containerId + '-count');\n            this.formHiddens = $(this.containerId + '-form-hiddens');\n            this.formAdditional = $(this.containerId + '-form-additional');\n            this.select = $(this.containerId + '-select');\n            this.form = this.prepareForm();\n            jQuery(this.form).mage('validation');\n            this.select.observe('change', this.onSelectChange.bindAsEventListener(this));\n            this.lastChecked = {\n                left: false,\n                top: false,\n                checkbox: false\n            };\n            this.select.addClassName(this.select.value ? '_selected' : '');\n            this.initMassSelect();\n        },\n\n        /**\n         * @return {jQuery|*|HTMLElement}\n         */\n        prepareForm: function () {\n            var form = $(this.containerId + '-form'),\n                formPlace = null,\n                formElement = this.formHiddens || this.formAdditional;\n\n            if (!formElement) {\n                formElement = this.container.getElementsByTagName('button')[0];\n                formElement && formElement.parentNode;\n            }\n\n            if (!form && formElement) {\n                /* fix problem with rendering form in FF through innerHTML property */\n                form = document.createElement('form');\n                form.setAttribute('method', 'post');\n                form.setAttribute('action', '');\n                form.id = this.containerId + '-form';\n                formPlace = formElement.parentNode;\n                formPlace.parentNode.appendChild(form);\n                form.appendChild(formPlace);\n            }\n\n            return form;\n        },\n\n        /**\n         * @param {Array} gridIds\n         */\n        setGridIds: function (gridIds) {\n            this.gridIds = gridIds;\n            this.updateCount();\n        },\n\n        /**\n         * @return {Array}\n         */\n        getGridIds: function () {\n            return this.gridIds;\n        },\n\n        /**\n         * @param {*} items\n         */\n        setItems: function (items) {\n            this.items = items;\n            this.updateCount();\n        },\n\n        /**\n         * @return {Object}\n         */\n        getItems: function () {\n            return this.items;\n        },\n\n        /**\n         * @param {*} itemId\n         * @return {*}\n         */\n        getItem: function (itemId) {\n            if (this.items[itemId]) {\n                return this.items[itemId];\n            }\n\n            return false;\n        },\n\n        /**\n         * @param {String} callbackName\n         * @return {Function}\n         */\n        getOldCallback: function (callbackName) {\n            return this.oldCallbacks[callbackName] ? this.oldCallbacks[callbackName] : Prototype.emptyFunction;\n        },\n\n        /**\n         * @param {String} callbackName\n         * @param {Function} callback\n         */\n        setOldCallback: function (callbackName, callback) {\n            this.oldCallbacks[callbackName] = callback;\n        },\n\n        /**\n         * @param {*} grid\n         */\n        onGridPreInit: function (grid) {\n            this.initMassactionElements();\n            this.getOldCallback('pre_init')(grid);\n        },\n\n        /**\n         * @param {*} grid\n         */\n        onGridInit: function (grid) {\n            this.initCheckboxes();\n            this.checkCheckboxes();\n            this.updateCount();\n            this.getOldCallback('init')(grid);\n        },\n\n        /**\n         * @param {*} grid\n         * @param {*} row\n         */\n        onGridRowInit: function (grid, row) {\n            this.getOldCallback('init_row')(grid, row);\n        },\n\n        /**\n         * @param {Object} evt\n         */\n        isDisabled: function (evt) {\n            var target = jQuery(evt.target),\n                tr,\n                checkbox;\n\n            tr = target.is('tr') ? target : target.closest('tr');\n            checkbox = tr.find('input[type=\"checkbox\"]');\n\n            return checkbox.is(':disabled');\n        },\n\n        /**\n         * @param {*} grid\n         * @param {*} evt\n         * @return {*}\n         */\n        onGridRowClick: function (grid, evt) {\n            var tdElement = Event.findElement(evt, 'td'),\n                trElement = Event.findElement(evt, 'tr'),\n                checkbox, isInput, checked;\n\n            if (this.isDisabled(evt)) {\n                return false;\n            }\n\n            if (!$(tdElement).down('input')) {\n                if ($(tdElement).down('a') || $(tdElement).down('select')) {\n                    return; //eslint-disable-line\n                }\n\n                if (trElement.title && trElement.title.strip() != '#') { //eslint-disable-line eqeqeq\n                    this.getOldCallback('row_click')(grid, evt);\n                } else {\n                    checkbox = Element.select(trElement, 'input');\n                    isInput = Event.element(evt).tagName == 'input'; //eslint-disable-line eqeqeq\n                    checked = isInput ? checkbox[0].checked : !checkbox[0].checked;\n\n                    if (checked) { //eslint-disable-line max-depth\n                        this.checkedString = varienStringArray.add(checkbox[0].value, this.checkedString);\n                    } else {\n                        this.checkedString = varienStringArray.remove(checkbox[0].value, this.checkedString);\n                    }\n                    this.grid.setCheckboxChecked(checkbox[0], checked);\n                    this.updateCount();\n                }\n\n                return; //eslint-disable-line\n            }\n\n            if (Event.element(evt).isMassactionCheckbox) {\n                this.setCheckbox(Event.element(evt));\n            } else if (checkbox = this.findCheckbox(evt)) { //eslint-disable-line no-cond-assign\n                checkbox.checked = !checkbox.checked;\n                jQuery(checkbox).trigger('change');\n                this.setCheckbox(checkbox);\n            }\n        },\n\n        /**\n         * @param {Object} evt\n         */\n        onSelectChange: function (evt) {\n            var item = this.getSelectedItem();\n\n            if (item) {\n                this.formAdditional.update($(this.containerId + '-item-' + item.id + '-block').innerHTML);\n                evt.target.addClassName('_selected');\n            } else {\n                this.formAdditional.update('');\n                evt.target.removeClassName('_selected');\n            }\n            jQuery(this.form).data('validator').resetForm();\n        },\n\n        /**\n         * @param {Object} evt\n         * @return {*}\n         */\n        findCheckbox: function (evt) {\n            if (['a', 'input', 'select'].indexOf(Event.element(evt).tagName.toLowerCase()) !== -1) {\n                return false;\n            }\n            checkbox = false; //eslint-disable-line no-undef\n            Event.findElement(evt, 'tr').select('[data-role=\"select-row\"]').each(function (element) { //eslint-disable-line\n                if (element.isMassactionCheckbox) {\n                    checkbox = element; //eslint-disable-line no-undef\n                }\n            });\n\n            return checkbox; //eslint-disable-line no-undef\n        },\n\n        /**\n         * Init checkobox.\n         */\n        initCheckboxes: function () {\n            this.getCheckboxes().each(function (checkbox) { //eslint-disable-line no-extra-bind\n                checkbox.isMassactionCheckbox = true; //eslint-disable-line no-undef\n            });\n        },\n\n        /**\n         * Check checkbox.\n         */\n        checkCheckboxes: function () {\n            this.getCheckboxes().each(function (checkbox) {\n                checkbox.checked = varienStringArray.has(checkbox.value, this.checkedString);\n                jQuery(checkbox).trigger('change');\n            }.bind(this));\n        },\n\n        /**\n         * @return {Boolean}\n         */\n        selectAll: function () {\n            this.setCheckedValues(this.useSelectAll ? this.getGridIds() : this.getCheckboxesValuesAsString());\n            this.checkCheckboxes();\n            this.updateCount();\n            this.clearLastChecked();\n\n            return false;\n        },\n\n        /**\n         * @return {Boolean}\n         */\n        unselectAll: function () {\n            this.setCheckedValues('');\n            this.checkCheckboxes();\n            this.updateCount();\n            this.clearLastChecked();\n\n            return false;\n        },\n\n        /**\n         * @return {Boolean}\n         */\n        selectVisible: function () {\n            this.setCheckedValues(this.getCheckboxesValuesAsString());\n            this.checkCheckboxes();\n            this.updateCount();\n            this.clearLastChecked();\n\n            return false;\n        },\n\n        /**\n         * @return {Boolean}\n         */\n        unselectVisible: function () {\n            this.getCheckboxesValues().each(function (key) {\n                this.checkedString = varienStringArray.remove(key, this.checkedString);\n            }.bind(this));\n            this.checkCheckboxes();\n            this.updateCount();\n            this.clearLastChecked();\n\n            return false;\n        },\n\n        /**\n         * @param {*} values\n         */\n        setCheckedValues: function (values) {\n            this.checkedString = values;\n        },\n\n        /**\n         * @return {String}\n         */\n        getCheckedValues: function () {\n            return this.checkedString;\n        },\n\n        /**\n         * @return {Array}\n         */\n        getCheckboxes: function () {\n            var result = [];\n\n            this.grid.rows.each(function (row) {\n                var checkboxes = row.select('[data-role=\"select-row\"]');\n\n                checkboxes.each(function (checkbox) {\n                    result.push(checkbox);\n                });\n            });\n\n            return result;\n        },\n\n        /**\n         * @return {Array}\n         */\n        getCheckboxesValues: function () {\n            var result = [];\n\n            this.getCheckboxes().each(function (checkbox) { //eslint-disable-line no-extra-bind\n                result.push(checkbox.value);\n            });\n\n            return result;\n        },\n\n        /**\n         * @return {String}\n         */\n        getCheckboxesValuesAsString: function () {\n            return this.getCheckboxesValues().join(',');\n        },\n\n        /**\n         * @param {Object} checkbox\n         */\n        setCheckbox: function (checkbox) {\n            if (checkbox.checked) {\n                this.checkedString = varienStringArray.add(checkbox.value, this.checkedString);\n            } else {\n                this.checkedString = varienStringArray.remove(checkbox.value, this.checkedString);\n            }\n            this.updateCount();\n        },\n\n        /**\n         * Update count.\n         */\n        updateCount: function () {\n            var checkboxesTotal = varienStringArray.count(\n                this.useSelectAll ? this.getGridIds() : this.getCheckboxesValuesAsString()\n                ),\n                checkboxesChecked = varienStringArray.count(this.checkedString);\n\n            jQuery('[data-role=\"counter\"]', this.count).html(checkboxesChecked);\n\n            if (!checkboxesTotal) {\n                this.multiselect.addClassName('_disabled');\n            } else {\n                this.multiselect.removeClassName('_disabled');\n            }\n\n            if (checkboxesChecked == checkboxesTotal && checkboxesTotal != 0) { //eslint-disable-line eqeqeq\n                this.count.removeClassName('_empty');\n                this.multiselect.addClassName('_checked').removeClassName('_indeterminate');\n            } else if (checkboxesChecked == 0) { //eslint-disable-line eqeqeq\n                this.count.addClassName('_empty');\n                this.multiselect.removeClassName('_checked').removeClassName('_indeterminate');\n            } else {\n                this.count.removeClassName('_empty');\n                this.multiselect.addClassName('_checked').addClassName('_indeterminate');\n            }\n\n            if (!this.grid.reloadParams) {\n                this.grid.reloadParams = {};\n            }\n            this.grid.reloadParams[this.formFieldNameInternal] = this.checkedString;\n        },\n\n        /**\n         * @return {*}\n         */\n        getSelectedItem: function () {\n            if (this.getItem(this.select.value)) {\n                return this.getItem(this.select.value);\n            }\n\n            return false;\n        },\n\n        /**\n         * Apply.\n         */\n        apply: function () {\n            var item, fieldName;\n\n            if (varienStringArray.count(this.checkedString) == 0) { //eslint-disable-line eqeqeq\n                alert({\n                    content: this.errorText\n                });\n\n                return;\n            }\n\n            item = this.getSelectedItem();\n\n            if (!item) {\n                jQuery(this.form).valid();\n\n                return;\n            }\n            this.currentItem = item;\n            fieldName = item.field ? item.field : this.formFieldName;\n\n            if (this.currentItem.confirm) {\n                confirm({\n                    content: this.currentItem.confirm,\n                    actions: {\n                        confirm: this.onConfirm.bind(this, fieldName, item)\n                    }\n                });\n            } else {\n                this.onConfirm(fieldName, item);\n            }\n        },\n\n        /**\n         * @param {*} fieldName\n         * @param {*} item\n         */\n        onConfirm: function (fieldName, item) {\n            this.formHiddens.update('');\n            new Insertion.Bottom(this.formHiddens, this.fieldTemplate({\n                name: fieldName,\n                value: this.checkedString\n            }));\n            new Insertion.Bottom(this.formHiddens, this.fieldTemplate({\n                name: 'massaction_prepare_key',\n                value: fieldName\n            }));\n\n            if (!jQuery(this.form).valid()) {\n                return;\n            }\n\n            if (this.useAjax && item.url) {\n                new Ajax.Request(item.url, {\n                    'method': 'post',\n                    'parameters': this.form.serialize(true),\n                    'onComplete': this.onMassactionComplete.bind(this)\n                });\n            } else if (item.url) {\n                this.form.action = item.url;\n                this.form.submit();\n            }\n        },\n\n        /**\n         * @param {*} transport\n         */\n        onMassactionComplete: function (transport) {\n            var listener;\n\n            if (this.currentItem.complete) {\n                try {\n                    listener = this.getListener(this.currentItem.complete) || Prototype.emptyFunction;\n                    listener(this.grid, this, transport);\n                } catch (e) {\n                }\n            }\n        },\n\n        /**\n         * @param {*} strValue\n         * @return {Object}\n         */\n        getListener: function (strValue) {\n            return eval(strValue); //eslint-disable-line no-eval\n        },\n\n        /**\n         * Init mass select.\n         */\n        initMassSelect: function () {\n            $$('input[data-role=\"select-row\"]').each(function (element) {\n                element.observe('click', this.massSelect.bind(this));\n            }.bind(this));\n        },\n\n        /**\n         * Clear last checked.\n         */\n        clearLastChecked: function () {\n            this.lastChecked = {\n                left: false,\n                top: false,\n                checkbox: false\n            };\n        },\n\n        /**\n         * @param {Object} evt\n         */\n        massSelect: function (evt) {\n            var currentCheckbox, lastCheckbox, start, finish;\n\n            if (this.lastChecked.left !== false &&\n                this.lastChecked.top !== false &&\n                evt.button === 0 &&\n                evt.shiftKey === true\n            ) {\n                currentCheckbox = Event.element(evt);\n                lastCheckbox = this.lastChecked.checkbox;\n\n                if (lastCheckbox != currentCheckbox) { //eslint-disable-line eqeqeq\n                    start = this.getCheckboxOrder(lastCheckbox);\n                    finish = this.getCheckboxOrder(currentCheckbox);\n\n                    if (start !== false && finish !== false) { //eslint-disable-line max-depth\n                        this.selectCheckboxRange(\n                            Math.min(start, finish),\n                            Math.max(start, finish),\n                            currentCheckbox.checked\n                        );\n                    }\n                }\n            }\n\n            this.lastChecked = {\n                left: Event.element(evt).viewportOffset().left,\n                top: Event.element(evt).viewportOffset().top,\n                checkbox: Event.element(evt) // \"boundary\" checkbox\n            };\n        },\n\n        /**\n         * @param {*} curCheckbox\n         * @return {Boolean}\n         */\n        getCheckboxOrder: function (curCheckbox) {\n            var order = false;\n\n            this.getCheckboxes().each(function (checkbox, key) {\n                if (curCheckbox == checkbox) { //eslint-disable-line eqeqeq\n                    order = key;\n                }\n            });\n\n            return order;\n        },\n\n        /**\n         * @param {*} start\n         * @param {*} finish\n         * @param {*} isChecked\n         */\n        selectCheckboxRange: function (start, finish, isChecked) {\n            this.getCheckboxes().each(function (checkbox, key) {\n                if (key >= start && key <= finish) {\n                    checkbox.checked = isChecked;\n                    this.setCheckbox(checkbox);\n                }\n            }.bind(this));\n        }\n    };\n\n    window.varienGridAction = {\n        /**\n         * @param {Object} select\n         */\n        execute: function (select) {\n            var config, win;\n\n            if (!select.value || !select.value.isJSON()) {\n                return;\n            }\n\n            config = select.value.evalJSON();\n\n            if (config.confirm && !window.confirm(config.confirm)) { //eslint-disable-line no-alert\n                select.options[0].selected = true;\n\n                return;\n            }\n\n            if (config.popup) {\n                win = window.open(config.href, 'action_window', 'width=500,height=600,resizable=1,scrollbars=1');\n                win.focus();\n                select.options[0].selected = true;\n            } else {\n                setLocation(config.href);\n            }\n        }\n    };\n\n    window.varienStringArray = {\n        /**\n         * @param {*} str\n         * @param {*} haystack\n         * @return {*}\n         */\n        remove: function (str, haystack) {\n            haystack = ',' + haystack + ',';\n            haystack = haystack.replace(new RegExp(',' + str + ',', 'g'), ',');\n\n            return this.trimComma(haystack);\n        },\n\n        /**\n         * @param {*} str\n         * @param {*} haystack\n         * @return {*}\n         */\n        add: function (str, haystack) {\n            haystack = ',' + haystack + ',';\n\n            if (haystack.search(new RegExp(',' + str + ',', 'g'), haystack) === -1) {\n                haystack += str + ',';\n            }\n\n            return this.trimComma(haystack);\n        },\n\n        /**\n         * @param {*} str\n         * @param {*} haystack\n         * @return {Boolean}\n         */\n        has: function (str, haystack) {\n            haystack = ',' + haystack + ',';\n\n            if (haystack.search(new RegExp(',' + str + ',', 'g'), haystack) === -1) {\n                return false;\n            }\n\n            return true;\n        },\n\n        /**\n         * @param {*} haystack\n         * @return {*}\n         */\n        count: function (haystack) {\n            var match;\n\n            if (typeof haystack != 'string') {\n                return 0;\n            }\n\n            /* eslint-disable no-undef, no-cond-assign, eqeqeq */\n            if (match = haystack.match(new RegExp(',', 'g'))) {\n                return match.length + 1;\n            } else if (haystack.length != 0) {\n                return 1;\n            }\n\n            /* eslint-enable no-undef, no-cond-assign, eqeqeq */\n            return 0;\n        },\n\n        /**\n         * @param {*} haystack\n         * @param {*} fnc\n         */\n        each: function (haystack, fnc) {\n            var i;\n\n            haystack = haystack.split(',');\n\n            for (i = 0; i < haystack.length; i++) {\n                fnc(haystack[i]);\n            }\n        },\n\n        /**\n         * @param {String} string\n         * @return {String}\n         */\n        trimComma: function (string) {\n            string = string.replace(new RegExp('^(,+)', 'i'), '');\n            string = string.replace(new RegExp('(,+)$', 'i'), '');\n\n            return string;\n        }\n    };\n\n    window.serializerController = Class.create();\n    serializerController.prototype = {\n        oldCallbacks: {},\n\n        /**\n         * @param {*} hiddenDataHolder\n         * @param {*} predefinedData\n         * @param {*} inputsToManage\n         * @param {*} grid\n         * @param {*} reloadParamName\n         */\n        initialize: function (hiddenDataHolder, predefinedData, inputsToManage, grid, reloadParamName) {\n            //Grid inputs\n            this.tabIndex = 1000;\n            this.inputsToManage = inputsToManage;\n            this.multidimensionalMode = inputsToManage.length > 0;\n\n            //Hash with grid data\n            this.gridData = this.getGridDataHash(predefinedData);\n\n            //Hidden input data holder\n            this.hiddenDataHolder = $(hiddenDataHolder);\n            this.hiddenDataHolder.value = this.serializeObject();\n\n            this.grid = grid;\n\n            // Set old callbacks\n            this.setOldCallback('row_click', this.grid.rowClickCallback);\n            this.setOldCallback('init_row', this.grid.initRowCallback);\n            this.setOldCallback('checkbox_check', this.grid.checkboxCheckCallback);\n\n            //Grid\n            this.reloadParamName = reloadParamName;\n            this.grid.reloadParams = {};\n            this.grid.reloadParams[this.reloadParamName + '[]'] = this.getDataForReloadParam();\n            this.grid.rowClickCallback = this.rowClick.bind(this);\n            this.grid.initRowCallback = this.rowInit.bind(this);\n            this.grid.checkboxCheckCallback = this.registerData.bind(this);\n            this.grid.rows.each(this.eachRow.bind(this));\n        },\n\n        /**\n         * @param {String} callbackName\n         * @param {Function} callback\n         */\n        setOldCallback: function (callbackName, callback) {\n            this.oldCallbacks[callbackName] = callback;\n        },\n\n        /**\n         * @param {String} callbackName\n         * @return {Prototype.emptyFunction}\n         */\n        getOldCallback: function (callbackName) {\n            return this.oldCallbacks[callbackName] ? this.oldCallbacks[callbackName] : Prototype.emptyFunction;\n        },\n\n        /**\n         * @param {*} grid\n         * @param {*} element\n         * @param {*} checked\n         */\n        registerData: function (grid, element, checked) {\n            var i;\n\n            if (this.multidimensionalMode) {\n                if (checked) {\n                    /*eslint-disable max-depth*/\n                    if (element.inputElements) {\n                        this.gridData.set(element.value, {});\n\n                        for (i = 0; i < element.inputElements.length; i++) {\n                            element.inputElements[i].disabled = false;\n                            this.gridData.get(element.value)[element.inputElements[i].name] =\n                                element.inputElements[i].value;\n                        }\n                    }\n                } else {\n                    if (element.inputElements) {\n                        for (i = 0; i < element.inputElements.length; i++) {\n                            element.inputElements[i].disabled = true;\n                        }\n                    }\n                    this.gridData.unset(element.value);\n                }\n            } else {\n                if (checked) { //eslint-disable-line no-lonely-if\n                    this.gridData.set(element.value, element.value);\n                } else {\n                    this.gridData.unset(element.value);\n                }\n            }\n\n            this.hiddenDataHolder.value = this.serializeObject();\n            this.grid.reloadParams = {};\n            this.grid.reloadParams[this.reloadParamName + '[]'] = this.getDataForReloadParam();\n            this.getOldCallback('checkbox_check')(grid, element, checked);\n\n            /*eslint-enable max-depth*/\n        },\n\n        /**\n         * @param {*} row\n         */\n        eachRow: function (row) {\n            this.rowInit(this.grid, row);\n        },\n\n        /**\n         * @param {*} grid\n         * @param {*} event\n         */\n        rowClick: function (grid, event) {\n            var trElement = Event.findElement(event, 'tr'),\n                isInput = Event.element(event).tagName == 'INPUT', //eslint-disable-line eqeqeq\n                checkbox, checked;\n\n            if (trElement) {\n                // eslint-disable-next-line jquery-no-input-event-shorthand\n                checkbox = Element.select(trElement, 'input');\n\n                if (checkbox[0] && !checkbox[0].disabled) {\n                    checked = isInput ? checkbox[0].checked : !checkbox[0].checked;\n                    this.grid.setCheckboxChecked(checkbox[0], checked);\n                }\n            }\n            this.getOldCallback('row_click')(grid, event);\n        },\n\n        /**\n         * @param {*} event\n         */\n        inputChange: function (event) {\n            var element = Event.element(event);\n\n            if (element && element.checkboxElement && element.checkboxElement.checked) {\n                this.gridData.get(element.checkboxElement.value)[element.name] = element.value;\n                this.hiddenDataHolder.value = this.serializeObject();\n            }\n        },\n\n        /**\n         * @param {*} grid\n         * @param {*} row\n         */\n        rowInit: function (grid, row) {\n            var checkbox, selectors, inputs, i;\n\n            if (this.multidimensionalMode) {\n                // eslint-disable-next-line jquery-no-input-event-shorthand\n                checkbox = $(row).select('.checkbox')[0];\n                selectors = this.inputsToManage.map(function (name) {\n                    return ['input[name=\"' + name + '\"]', 'select[name=\"' + name + '\"]'];\n                });\n                inputs = $(row).select.apply($(row), selectors.flatten());\n\n                if (checkbox && inputs.length > 0) {\n                    checkbox.inputElements = inputs;\n\n                    /* eslint-disable max-depth */\n                    for (i = 0; i < inputs.length; i++) {\n                        inputs[i].checkboxElement = checkbox;\n\n                        if (this.gridData.get(checkbox.value) && this.gridData.get(checkbox.value)[inputs[i].name]) {\n                            inputs[i].value = this.gridData.get(checkbox.value)[inputs[i].name];\n                        }\n                        inputs[i].disabled = !checkbox.checked;\n                        inputs[i].tabIndex = this.tabIndex++;\n                        Event.observe(inputs[i], 'keyup', this.inputChange.bind(this));\n                        Event.observe(inputs[i], 'change', this.inputChange.bind(this));\n                    }\n                }\n            }\n\n            /* eslint-enable max-depth */\n            this.getOldCallback('init_row')(grid, row);\n        },\n\n        /**\n         * Stuff methods.\n         *\n         * @param {*} _object\n         * @return {*}\n         */\n        getGridDataHash: function (_object) {\n            return $H(this.multidimensionalMode ? _object : this.convertArrayToObject(_object));\n        },\n\n        /**\n         * @return {*}\n         */\n        getDataForReloadParam: function () {\n            return this.multidimensionalMode ? this.gridData.keys() : this.gridData.values();\n        },\n\n        /**\n         * @return {*}\n         */\n        serializeObject: function () {\n            var clone;\n\n            if (this.multidimensionalMode) {\n                clone = this.gridData.clone();\n                clone.each(function (pair) {\n                    clone.set(pair.key, Base64.encode(Object.toQueryString(pair.value)));\n                });\n\n                return clone.toQueryString();\n            }\n\n            return this.gridData.values().join('&');\n        },\n\n        /**\n         * @param {Array} _array\n         * @return {Object}\n         */\n        convertArrayToObject: function (_array) {\n            var _object = {},\n                i, l;\n\n            for (i = 0, l = _array.length; i < l; i++) {\n                _object[_array[i]] = _array[i];\n            }\n\n            return _object;\n        }\n    };\n});\n","mage/adminhtml/form.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\n/* global varienGlobalEvents, varienWindowOnloadCache, RegionUpdater, FormElementDependenceController */\n/* eslint-disable strict */\ndefine([\n    'jquery',\n    'prototype',\n    'mage/adminhtml/events'\n], function (jQuery) {\n    var varienElementMethods;\n\n    /*\n     * @TODO Need to be removed after refactoring all dependent of the form the components\n     */\n    (function ($) {\n        $(function () {\n            $(document).on('beforeSubmit', function (e) { //eslint-disable-line max-nested-callbacks\n                if (typeof varienGlobalEvents !== 'undefined') {\n                    varienGlobalEvents.fireEvent('formSubmit', $(e.target).attr('id'));\n                }\n            });\n        });\n    })(jQuery);\n\n    /**\n     *  Additional elements methods\n     */\n    varienElementMethods = {\n        /**\n         * @param {HTMLElement} element\n         */\n        setHasChanges: function (element) {\n            var elm;\n\n            if ($(element) && $(element).hasClassName('no-changes')) {\n                return;\n            }\n            elm = element;\n\n            while (elm && elm.tagName != 'BODY') { //eslint-disable-line eqeqeq\n                if (elm.statusBar) {\n                    Element.addClassName($(elm.statusBar), 'changed');\n                }\n                elm = elm.parentNode;\n            }\n        },\n\n        /**\n         * @param {HTMLElement} element\n         * @param {*} flag\n         * @param {Object} form\n         */\n        setHasError: function (element, flag, form) {\n            var elm = element;\n\n            while (elm && elm.tagName != 'BODY') { //eslint-disable-line eqeqeq\n                if (elm.statusBar) {\n                    /* eslint-disable max-depth */\n                    if (form.errorSections.keys().indexOf(elm.statusBar.id) < 0) {\n                        form.errorSections.set(elm.statusBar.id, flag);\n                    }\n\n                    if (flag) {\n                        Element.addClassName($(elm.statusBar), 'error');\n\n                        if (form.canShowError && $(elm.statusBar).show) {\n                            form.canShowError = false;\n                            $(elm.statusBar).show();\n                        }\n                        form.errorSections.set(elm.statusBar.id, flag);\n                    } else if (!form.errorSections.get(elm.statusBar.id)) {\n                        Element.removeClassName($(elm.statusBar), 'error');\n                    }\n\n                    /* eslint-enable max-depth */\n                }\n                elm = elm.parentNode;\n            }\n            this.canShowElement = false;\n        }\n    };\n\n    Element.addMethods(varienElementMethods);\n\n    // Global bind changes\n    window.varienWindowOnloadCache = {};\n\n    /**\n     * @param {*} useCache\n     */\n    function varienWindowOnload(useCache) {\n        var dataElements = $$('input', 'select', 'textarea'),\n            i;\n\n        for (i = 0; i < dataElements.length; i++) {\n            if (dataElements[i] && dataElements[i].id) {\n\n                /* eslint-disable max-depth */\n                if (!useCache || !varienWindowOnloadCache[dataElements[i].id]) {\n                    Event.observe(dataElements[i], 'change', dataElements[i].setHasChanges.bind(dataElements[i]));\n\n                    if (useCache) {\n                        varienWindowOnloadCache[dataElements[i].id] = true;\n                    }\n                }\n\n                /* eslint-disable max-depth */\n            }\n        }\n    }\n    Event.observe(window, 'load', varienWindowOnload);\n\n    window.RegionUpdater = Class.create();\n    RegionUpdater.prototype = {\n        /**\n         * @param {HTMLElement} countryEl\n         * @param {HTMLElement} regionTextEl\n         * @param {HTMLElement}regionSelectEl\n         * @param {Object} regions\n         * @param {*} disableAction\n         * @param {*} clearRegionValueOnDisable\n         */\n        initialize: function (\n            countryEl, regionTextEl, regionSelectEl, regions, disableAction, clearRegionValueOnDisable\n        ) {\n            this.isRegionRequired = true;\n            this.countryEl = $(countryEl);\n            this.regionTextEl = $(regionTextEl);\n            this.regionSelectEl = $(regionSelectEl);\n            this.config = regions.config;\n            delete regions.config;\n            this.regions = regions;\n            this.sortedRegions = this.getSortedRegions();\n            this.disableAction = typeof disableAction === 'undefined' ? 'hide' : disableAction;\n            this.clearRegionValueOnDisable = typeof clearRegionValueOnDisable === 'undefined' ?\n                false : clearRegionValueOnDisable;\n\n            if (this.regionSelectEl.options.length <= 1) {\n                this.update();\n            } else {\n                this.lastCountryId = this.countryEl.value;\n            }\n\n            this.countryEl.changeUpdater = this.update.bind(this);\n\n            Event.observe(this.countryEl, 'change', this.update.bind(this));\n        },\n\n        /**\n         * @private\n         */\n        _checkRegionRequired: function () {\n            var label, wildCard, elements, that, regionRequired;\n\n            if (!this.isRegionRequired) {\n                return;\n            }\n\n            elements = [this.regionTextEl, this.regionSelectEl];\n            that = this;\n\n            if (typeof this.config == 'undefined') {\n                return;\n            }\n            regionRequired = this.config['regions_required'].indexOf(this.countryEl.value) >= 0;\n\n            elements.each(function (currentElement) {\n                var form, validationInstance, field, topElement;\n\n                if (!currentElement) {\n                    return;\n                }\n                form = currentElement.form;\n                validationInstance = form ? jQuery(form).data('validation') : null;\n                field = currentElement.up('.field') || new Element('div');\n\n                if (validationInstance) {\n                    validationInstance.clearError(currentElement);\n                }\n                label = $$('label[for=\"' + currentElement.id + '\"]')[0];\n\n                if (label) {\n                    wildCard = label.down('em') || label.down('span.required');\n                    topElement = label.up('tr') || label.up('li');\n\n                    if (!that.config['show_all_regions'] && topElement) {\n                        if (regionRequired) {\n                            topElement.show();\n                        } else {\n                            topElement.hide();\n                        }\n                    }\n                }\n\n                if (label && wildCard) {\n                    if (!regionRequired) {\n                        wildCard.hide();\n                    } else {\n                        wildCard.show();\n                    }\n                }\n\n                //compute the need for the required fields\n                if (!regionRequired || !currentElement.visible()) {\n                    if (field.hasClassName('required')) {\n                        field.removeClassName('required');\n                    }\n\n                    if (currentElement.hasClassName('required-entry')) {\n                        currentElement.removeClassName('required-entry');\n                    }\n\n                    if (currentElement.tagName.toLowerCase() == 'select' && //eslint-disable-line eqeqeq\n                        currentElement.hasClassName('validate-select')\n                    ) {\n                        currentElement.removeClassName('validate-select');\n                    }\n                } else {\n                    if (!field.hasClassName('required')) {\n                        field.addClassName('required');\n                    }\n\n                    if (!currentElement.hasClassName('required-entry')) {\n                        currentElement.addClassName('required-entry');\n                    }\n\n                    if (currentElement.tagName.toLowerCase() == 'select' && //eslint-disable-line eqeqeq\n                        !currentElement.hasClassName('validate-select')\n                    ) {\n                        currentElement.addClassName('validate-select');\n                    }\n                }\n            });\n        },\n\n        /**\n         * Disable region validation.\n         */\n        disableRegionValidation: function () {\n            this.isRegionRequired = false;\n        },\n\n        /**\n         * Update.\n         */\n        update: function () {\n            var option, selectElement, def, regionId, region;\n\n            selectElement = this.regionSelectEl;\n\n            if (this.sortedRegions[this.countryEl.value]) {\n                if (this.lastCountryId != this.countryEl.value) { //eslint-disable-line eqeqeq\n                    def = selectElement.getAttribute('defaultValue');\n\n                    if (this.regionTextEl) {\n                        if (!def) {\n                            def = this.regionTextEl.value.toLowerCase();\n                        }\n                        this.regionTextEl.value = '';\n                    }\n\n                    selectElement.options.length = 1;\n\n                    this.sortedRegions[this.countryEl.value].forEach(function (item) {\n                        regionId = item[0];\n                        region = item[1];\n\n                        option = document.createElement('OPTION');\n                        option.value = regionId;\n                        option.text = region.name.stripTags();\n                        option.title = region.name;\n\n                        if (selectElement.options.add) {\n                            selectElement.options.add(option);\n                        } else {\n                            selectElement.appendChild(option);\n                        }\n\n                        if (regionId == def || region.name.toLowerCase() == def || region.code.toLowerCase() == def) { //eslint-disable-line\n                            selectElement.value = regionId;\n                        }\n                    });\n                }\n\n                if (this.disableAction == 'hide') { //eslint-disable-line eqeqeq\n                    if (this.regionTextEl) {\n                        this.regionTextEl.style.display = 'none';\n                        this.regionTextEl.style.disabled = true;\n                    }\n                    this.regionSelectEl.style.display = '';\n                    this.regionSelectEl.disabled = false;\n                } else if (this.disableAction == 'disable') { //eslint-disable-line eqeqeq\n                    if (this.regionTextEl) {\n                        this.regionTextEl.disabled = true;\n                    }\n                    this.regionSelectEl.disabled = false;\n                }\n                this.setMarkDisplay(this.regionSelectEl, true);\n\n                this.lastCountryId = this.countryEl.value;\n            } else {\n                if (this.disableAction == 'hide') { //eslint-disable-line eqeqeq\n                    if (this.regionTextEl) {\n                        this.regionTextEl.style.display = '';\n                        this.regionTextEl.style.disabled = false;\n                    }\n                    this.regionSelectEl.style.display = 'none';\n                    this.regionSelectEl.disabled = true;\n                } else if (this.disableAction == 'disable') { //eslint-disable-line eqeqeq\n                    if (this.regionTextEl) {\n                        this.regionTextEl.disabled = false;\n                    }\n                    this.regionSelectEl.disabled = true;\n\n                    if (this.clearRegionValueOnDisable) {\n                        this.regionSelectEl.value = '';\n                    }\n                } else if (this.disableAction == 'nullify') { //eslint-disable-line eqeqeq\n                    this.regionSelectEl.options.length = 1;\n                    this.regionSelectEl.value = '';\n                    this.regionSelectEl.selectedIndex = 0;\n                    this.lastCountryId = '';\n                }\n                this.setMarkDisplay(this.regionSelectEl, false);\n            }\n            varienGlobalEvents.fireEvent('address_country_changed', this.countryEl);\n            this._checkRegionRequired();\n        },\n\n        /**\n         * @param {HTMLElement} elem\n         * @param {*} display\n         */\n        setMarkDisplay: function (elem, display) {\n            var marks;\n\n            if (elem.parentNode.parentNode) {\n                marks = Element.select(elem.parentNode.parentNode, '.required');\n\n                if (marks[0]) {\n                    display ? marks[0].show() : marks[0].hide();\n                }\n            }\n        },\n\n        /**\n         * Sort regions from JSON by name\n         *\n         * @returns {*[]}\n         */\n        getSortedRegions: function () {\n            var country, regionsEntries, regionsByCountry;\n\n            regionsByCountry = [];\n\n            for (country in this.regions) { //eslint-disable-line guard-for-in\n                regionsEntries = Object.entries(this.regions[country]);\n                regionsEntries.sort(function (a, b) {\n                    return a[1].name > b[1].name ? 1 : -1;\n                });\n\n                regionsByCountry[country] = regionsEntries;\n            }\n\n            return regionsByCountry;\n        }\n    };\n\n    window.regionUpdater = RegionUpdater;\n\n    /**\n     * Fix errorrs in IE\n     */\n    Event.pointerX = function (event) {\n        try {\n            return event.pageX || (event.clientX + (document.documentElement.scrollLeft || document.body.scrollLeft)); //eslint-disable-line\n        }\n        catch (e) {}\n    };\n\n    /**\n     * @param {jQuery.Event} event\n     * @return {*}\n     */\n    Event.pointerY = function (event) {\n        try {\n            return event.pageY || (event.clientY + (document.documentElement.scrollTop || document.body.scrollTop)); //eslint-disable-line\n        }\n        catch (e) {}\n    };\n\n    /**\n     * Observer that watches for dependent form elements\n     * If an element depends on 1 or more of other elements,\n     * it should show up only when all of them gain specified values\n     */\n    window.FormElementDependenceController = Class.create();\n    FormElementDependenceController.prototype = {\n        /**\n         * Structure of elements: {\n         *     'id_of_dependent_element' : {\n         *         'id_of_master_element_1' : 'reference_value',\n         *         'id_of_master_element_2' : 'reference_value'\n         *         'id_of_master_element_3' : ['reference_value1', 'reference_value2']\n         *         ...\n         *     }\n         * }\n         * @param {Object} elementsMap\n         * @param {Object} config\n         */\n        initialize: function (elementsMap, config) {\n            var idTo, idFrom, values, fromId, radioFrom;\n\n            if (config) {\n                this._config = jQuery.extend(this._config, config);\n            }\n\n            for (idTo in elementsMap) { //eslint-disable-line guard-for-in\n                for (idFrom in elementsMap[idTo]) { //eslint-disable-line guard-for-in\n                    if ($(idFrom)) {\n                        Event.observe(\n                            $(idFrom),\n                            'change',\n                            this.trackChange.bindAsEventListener(this, idTo, elementsMap[idTo])\n                        );\n                    } else {\n                        // Check if radio button\n                        values = elementsMap[idTo][idFrom].values;\n                        fromId = $(idFrom + values[0]);\n                        radioFrom = fromId ? $$('[name=\"' + fromId.name + '\"]') : false;\n\n                        if (radioFrom) {\n                            radioFrom.invoke(\n                                'on',\n                                'change',\n                                this.trackChange.bindAsEventListener(this, idTo, elementsMap[idTo])\n                            );\n                        }\n                    }\n                    this.trackChange(null, idTo, elementsMap[idTo]);\n                }\n            }\n        },\n\n        /**\n         * Misc. config options\n         * Keys are underscored intentionally\n         */\n        _config: {\n            'levels_up': 1 // how many levels up to travel when toggling element\n        },\n\n        /**\n         * Define whether target element should be toggled and show/hide its row\n         *\n         * @param {Object} e - event\n         * @param {String} idTo - id of target element\n         * @param {Object} valuesFrom - ids of master elements and reference values\n         * @return\n         */\n        trackChange: function (e, idTo, valuesFrom) {\n            // define whether the target should show up\n            var shouldShowUp = true,\n                idFrom, from, values, isInArray, isNegative, headElement, isInheritCheckboxChecked, target, inputs,\n                isAnInputOrSelect, currentConfig, rowElement, fromId, radioFrom, targetArray, isChooser;\n\n            for (idFrom in valuesFrom) { //eslint-disable-line guard-for-in\n                from = $(idFrom);\n\n                if (from) {\n                    values = valuesFrom[idFrom].values;\n                    isInArray = values.indexOf(from.value) != -1; //eslint-disable-line\n                    isNegative = valuesFrom[idFrom].negative;\n\n                    if (!from || isInArray && isNegative || !isInArray && !isNegative) {\n                        shouldShowUp = false;\n                    }\n                // Check if radio button\n                } else {\n                    values = valuesFrom[idFrom].values;\n                    fromId = $(idFrom + values[0]);\n\n                    if (fromId) {\n                        radioFrom = $$('[name=\"' + fromId.name + '\"]:checked');\n                        isInArray = radioFrom.length > 0 && values.indexOf(radioFrom[0].value) !== -1;\n                        isNegative = valuesFrom[idFrom].negative;\n\n                        if (!radioFrom || isInArray && isNegative || !isInArray && !isNegative) {\n                            shouldShowUp = false;\n                        }\n                    }\n                }\n            }\n\n            // toggle target row\n            headElement = jQuery('#' + idTo + '-head');\n            isInheritCheckboxChecked = $(idTo + '_inherit') && $(idTo + '_inherit').checked;\n            target = $(idTo);\n\n            // Account for the chooser style parameters.\n            if (target === null && headElement.length === 0 && idTo.substring(0, 16) === 'options_fieldset') {\n                targetArray = $$('input[id*=\"' + idTo + '\"]');\n                isChooser = true;\n\n                if (targetArray !== null && targetArray.length > 0) {\n                    target = targetArray[0];\n                }\n                headElement = jQuery('.field-' + idTo).add('.field-chooser' + idTo);\n            }\n\n            // Target won't always exist (for example, if field type is \"label\")\n            if (target) {\n                inputs = target.up(this._config['levels_up']).select('input', 'select', 'td');\n                isAnInputOrSelect = ['input', 'select'].indexOf(target.tagName.toLowerCase()) != -1; //eslint-disable-line\n\n                if (target.type === 'fieldset') {\n                    inputs = target.select('input', 'select', 'td');\n                }\n            } else {\n                inputs = false;\n                isAnInputOrSelect = false;\n            }\n\n            if (shouldShowUp) {\n                currentConfig = this._config;\n\n                if (inputs) {\n                    inputs.each(function (item) {\n                        // don't touch hidden inputs (and Use Default inputs too), bc they may have custom logic\n                        if ((!item.type || item.type != 'hidden') && !($(item.id + '_inherit') && $(item.id + '_inherit').checked) && //eslint-disable-line\n                            !(currentConfig['can_edit_price'] != undefined && !currentConfig['can_edit_price']) && //eslint-disable-line\n                            !item.getAttribute('readonly') || isChooser //eslint-disable-line\n                        ) {\n                            item.disabled = false;\n                            jQuery(item).removeClass('ignore-validate');\n                        }\n                    });\n                }\n\n                if (headElement.length > 0) {\n                    headElement.show();\n\n                    if (headElement.hasClass('open') && target) {\n                        target.show();\n                    } else if (target) {\n                        target.hide();\n                    }\n                } else {\n                    if (target) {\n                        target.show();\n                        headElement = jQuery('.field-' + idTo).add('.field-chooser' + idTo);\n                        headElement.show();\n                    }\n\n                    if (isAnInputOrSelect && !isInheritCheckboxChecked) {\n                        if (target) {\n                            if (target.getAttribute('readonly')) {\n                                target.disabled = true;\n                            } else {\n                                target.disabled = false;\n                            }\n                        }\n\n                        jQuery('#' + idTo).removeClass('ignore-validate');\n                    }\n                }\n            } else {\n                if (inputs) {\n                    inputs.each(function (item) {\n                        // don't touch hidden inputs (and Use Default inputs too), bc they may have custom logic\n                        if ((!item.type || item.type != 'hidden') && //eslint-disable-line eqeqeq\n                            !($(item.id + '_inherit') && $(item.id + '_inherit').checked)\n                        ) {\n                            item.disabled = true;\n                            jQuery(item).addClass('ignore-validate');\n                        }\n                    });\n                }\n\n                if (headElement.length > 0) {\n                    headElement.hide();\n                } else {\n                    headElement = jQuery('.field-' + idTo).add('.field-chooser' + idTo);\n                    headElement.hide();\n                }\n\n                if (target) {\n                    target.hide();\n                }\n\n                if (isAnInputOrSelect && !isInheritCheckboxChecked) {\n                    if (target) {\n                        target.disabled = true;\n                    }\n                    jQuery('#' + idTo).addClass('ignore-validate');\n                }\n\n            }\n            rowElement = $('row_' + idTo);\n\n            if (rowElement == undefined && target) { //eslint-disable-line eqeqeq\n                rowElement = target.up(this._config['levels_up']);\n\n                if (target.type === 'fieldset') {\n                    rowElement = target;\n                }\n            }\n\n            if (rowElement) {\n                if (shouldShowUp) {\n                    rowElement.show();\n                } else {\n                    rowElement.hide();\n                }\n            }\n        }\n    };\n\n    window.varienWindowOnload = varienWindowOnload;\n    window.varienElementMethods = varienElementMethods;\n});\n","mage/adminhtml/browser.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\n/* global MediabrowserUtility, FORM_KEY, tinyMceEditors */\n/* eslint-disable strict */\ndefine([\n    'jquery',\n    'wysiwygAdapter',\n    'Magento_Ui/js/modal/prompt',\n    'Magento_Ui/js/modal/confirm',\n    'Magento_Ui/js/modal/alert',\n    'underscore',\n    'Magento_Ui/js/modal/modal',\n    'jquery/ui',\n    'jquery/jstree/jquery.jstree',\n    'mage/mage'\n], function ($, wysiwyg, prompt, confirm, alert, _) {\n    window.MediabrowserUtility = {\n        windowId: 'modal_dialog_message',\n        modalLoaded: false,\n        targetElementId: false,\n        pathId: '',\n\n        /**\n         * @return {Number}\n         */\n        getMaxZIndex: function () {\n            var max = 0,\n                cn = document.body.childNodes,\n                i, el, zIndex;\n\n            for (i = 0; i < cn.length; i++) {\n                el = cn[i];\n                zIndex = el.nodeType == 1 ? parseInt(el.style.zIndex, 10) || 0 : 0; //eslint-disable-line eqeqeq\n\n                if (zIndex < 10000) {\n                    max = Math.max(max, zIndex);\n                }\n            }\n\n            return max + 10;\n        },\n\n        /**\n         * @param {*} url\n         * @param {*} width\n         * @param {*} height\n         * @param {*} title\n         * @param {Object} options\n         */\n        openDialog: function (url, width, height, title, options) {\n            var windowId = this.windowId,\n                content = '<div class=\"popup-window\" id=\"' + windowId + '\"></div>';\n\n            if (this.modalLoaded) {\n\n                if (!_.isUndefined(options)) {\n                    this.modal.modal('option', 'closed', options.closed);\n                }\n\n                this.modal.modal('openModal');\n                this.setTargetElementId(options, url);\n                this.setPathId(url);\n                $(window).trigger('reload.MediaGallery');\n\n                return;\n            }\n\n            this.modal = $(content).modal($.extend({\n                title:  title || 'Insert File...',\n                modalClass: 'magento',\n                type: 'slide',\n                buttons: []\n            }, options));\n\n            this.modal.modal('openModal');\n\n            $.ajax({\n                url: url,\n                type: 'get',\n                context: $(this),\n                showLoader: true\n\n            }).done(function (data) {\n                this.modal.html(data).trigger('contentUpdated');\n                this.modalLoaded = true;\n                this.setTargetElementId(options, url);\n                this.setPathId(url);\n            }.bind(this));\n\n        },\n\n        /**\n         * Setter for endcoded path id\n         */\n        setPathId: function (url) {\n            this.pathId = url.match(/(&|\\/|%26)current_tree_path(=|\\/)([\\s\\S].*?)(\\/|$)/)[3];\n        },\n\n        /**\n         * Setter for targetElementId property\n         *\n         * @param {Object} options\n         * @param {String} url\n         */\n        setTargetElementId: function (options, url) {\n            this.targetElementId = options && options.targetElementId ?\n                options.targetElementId\n                : url.match(/\\/target_element_id\\/([\\s\\S].*?)\\//)[1];\n        },\n\n        /**\n         * Close dialog.\n         */\n        closeDialog: function () {\n            this.modal.modal('closeModal');\n        }\n    };\n\n    $.widget('mage.mediabrowser', {\n        eventPrefix: 'mediabrowser',\n        options: {\n            contentsUrl: null,\n            onInsertUrl: null,\n            newFolderUrl: null,\n            deleteFolderUrl: null,\n            deleteFilesUrl: null,\n            headerText: null,\n            tree: null,\n            currentNode: null,\n            storeId: null,\n            showBreadcrumbs: null,\n            hidden: 'no-display'\n        },\n\n        /**\n         * Proxy creation\n         * @protected\n         */\n        _create: function () {\n            this._on({\n                'click [data-row=file]': 'selectFile',\n                'dblclick [data-row=file]': 'insert',\n                'click #new_folder': 'newFolder',\n                'click #delete_folder': 'deleteFolder',\n                'click #delete_files': 'deleteFiles',\n                'click #insert_files': 'insertSelectedFiles',\n                'fileuploaddone': '_uploadDone',\n                'click [data-row=breadcrumb]': 'selectFolder'\n            });\n\n            $(window).on('reload.MediaGallery', $.proxy(this.reload, this));\n            this.activeNode = null;\n            //tree dont use event bubbling\n            this.tree = this.element.find('[data-role=tree]');\n            this.tree.on('select_node.jstree', $.proxy(this._selectNode, this));\n        },\n\n        /**\n         * @param {jQuery.Event} event\n         * @param {Object} data\n         * @private\n         */\n        _selectNode: function (event, data) {\n            var node = data.node;\n\n            this.activeNode = node;\n            this.element.find('#delete_files, #insert_files').toggleClass(this.options.hidden, true);\n            this.element.find('#contents').toggleClass(this.options.hidden, false);\n            this.element.find('#delete_folder')\n                .toggleClass(this.options.hidden, node.id === 'root'); //eslint-disable-line eqeqeq\n            this.element.find('#content_header_text')\n                .html(node.id === 'root' ? this.headerText : node.text); //eslint-disable-line eqeqeq\n\n            this.drawBreadcrumbs(data);\n            this.loadFileList(node);\n        },\n\n        /**\n         * @return {*}\n         */\n        reload: function (uploaded) {\n            return this.loadFileList(this.activeNode, uploaded);\n        },\n\n        /**\n         * @param {Object} element\n         * @param {*} value\n         */\n        insertAtCursor: function (element, value) {\n            var sel, startPos, endPos, scrollTop;\n\n            if ('selection' in document) {\n                //For browsers like Internet Explorer\n                element.focus();\n                sel = document.selection.createRange();\n                sel.text = value;\n                element.focus();\n            } else if (element.selectionStart || element.selectionStart == '0') { //eslint-disable-line eqeqeq\n                //For browsers like Firefox and Webkit based\n                startPos = element.selectionStart;\n                endPos = element.selectionEnd;\n                scrollTop = element.scrollTop;\n                element.value = element.value.substring(0, startPos) + value +\n                    element.value.substring(startPos, endPos) + element.value.substring(endPos, element.value.length);\n                element.focus();\n                element.selectionStart = startPos + value.length;\n                element.selectionEnd = startPos + value.length + element.value.substring(startPos, endPos).length;\n                element.scrollTop = scrollTop;\n            } else {\n                element.value += value;\n                element.focus();\n            }\n        },\n\n        /**\n         * @param {Object} node\n         */\n        loadFileList: function (node, uploaded) {\n            var contentBlock = this.element.find('#contents');\n\n            return $.ajax({\n                url: this.options.contentsUrl,\n                type: 'GET',\n                dataType: 'html',\n                data: {\n                    'form_key': FORM_KEY,\n                    node: node ? node.id : null\n                },\n                context: contentBlock,\n                showLoader: true\n            }).done(function (data) {\n                contentBlock.html(data).trigger('contentUpdated');\n\n                if (uploaded) {\n                    contentBlock.find('.filecnt:last').trigger('click');\n                }\n            });\n        },\n\n        /**\n         * @param {jQuery.Event} event\n         */\n        selectFolder: function (event) {\n            this.element.find('[data-id=\"' + $(event.currentTarget).data('node').id + '\"]>a').trigger('click');\n        },\n\n        /**\n         * Insert selected files.\n         */\n        insertSelectedFiles: function () {\n            this.element.find('[data-row=file].selected').trigger('dblclick');\n        },\n\n        /**\n         * @param {jQuery.Event} event\n         */\n        selectFile: function (event) {\n            var fileRow = $(event.currentTarget);\n\n            fileRow.toggleClass('selected');\n            this.element.find('[data-row=file]').not(fileRow).removeClass('selected');\n            this.element.find('#delete_files, #insert_files')\n                .toggleClass(this.options.hidden, !fileRow.is('.selected'));\n            fileRow.trigger('selectfile');\n        },\n\n        /**\n         * @private\n         */\n        _uploadDone: function () {\n            this.element.find('.file-row').remove();\n            this.reload(true);\n        },\n\n        /**\n         * @param {jQuery.Event} event\n         * @return {Boolean}\n         */\n        insert: function (event) {\n            var fileRow = $(event.currentTarget),\n                targetEl;\n\n            if (!fileRow.prop('id')) {\n                return false;\n            }\n            targetEl = this.getTargetElement();\n\n            if (!targetEl.length) {\n                MediabrowserUtility.closeDialog();\n                throw 'Target element not found for content update';\n            }\n\n            return $.ajax({\n                url: this.options.onInsertUrl,\n                data: {\n                    filename: fileRow.attr('id'),\n                    node: this.activeNode.id,\n                    store: this.options.storeId,\n                    'as_is': typeof targetEl !== 'function' && targetEl.is('textarea') ? 1 : 0,\n                    'force_static_path': typeof targetEl !== 'function' && targetEl.data('force_static_path') ? 1 : 0,\n                    'form_key': FORM_KEY\n                },\n                context: this,\n                showLoader: true\n            }).done($.proxy(function (data) {\n                if (typeof targetEl === 'function') {\n                    targetEl(data, {text: fileRow.find('img').attr('alt')});\n                } else if (targetEl.is('textarea')) {\n                    this.insertAtCursor(targetEl.get(0), data);\n                } else {\n                    targetEl\n                        .val(data)\n                        .data('size', fileRow.data('size'))\n                        .data('mime-type', fileRow.data('mime-type'));\n                }\n                MediabrowserUtility.closeDialog();\n\n                if (typeof targetEl !== 'function') {\n                    targetEl.focus();\n                    jQuery(targetEl).trigger('change');\n                }\n            }, this));\n        },\n\n        /**\n         * Find document target element in next order:\n         *  in acive file browser opener:\n         *  - input field with ID: \"src\" in opener window\n         *  - input field with ID: \"href\" in opener window\n         *  in document:\n         *  - element with target ID\n         *\n         * return {HTMLElement|null}\n         */\n        getTargetElement: function () {\n            var mediaBrowser = window.MediabrowserUtility;\n\n            if (!_.isUndefined(wysiwyg) && wysiwyg.get(mediaBrowser.targetElementId)) {\n                return this.getMediaBrowserOpener() || window;\n            }\n\n            return $('#' + mediaBrowser.targetElementId);\n        },\n\n        /**\n         * Return opener Window object if it exists, not closed and editor is active\n         *\n         * return {Object|null}\n         */\n        getMediaBrowserOpener: function () {\n            var targetElementId = window.MediabrowserUtility.targetElementId;\n\n            if (!_.isUndefined(wysiwyg) && wysiwyg.get(targetElementId) && !_.isUndefined(tinyMceEditors)) {\n                return tinyMceEditors.get(targetElementId).getMediaBrowserOpener();\n            }\n\n            return null;\n        },\n\n        /**\n         * New folder.\n         */\n        newFolder: function () {\n            var self = this;\n\n            prompt({\n                title: this.options.newFolderPrompt,\n                actions: {\n                    /**\n                     * @param {*} folderName\n                     */\n                    confirm: function (folderName) {\n                        $.ajax({\n                            url: self.options.newFolderUrl,\n                            dataType: 'json',\n                            data: {\n                                name: folderName,\n                                node: self.activeNode.id,\n                                store: self.options.storeId,\n                                'form_key': FORM_KEY\n                            },\n                            context: self.element,\n                            showLoader: true\n                        }).done($.proxy(function (data) {\n                            if (data.error) {\n                                alert({\n                                    content: data.message\n                                });\n                            } else {\n                                self.tree.jstree(\n                                    'refresh_node',\n                                    self.element.find('[data-id=\"' + self.activeNode.id + '\"]')\n                                );\n                            }\n                        }, this));\n\n                        return true;\n                    }\n                }\n            });\n        },\n\n        /**\n         * Delete folder.\n         */\n        deleteFolder: function () {\n            var self = this;\n\n            confirm({\n                content: this.options.deleteFolderConfirmationMessage,\n                actions: {\n                    /**\n                     * Confirm.\n                     */\n                    confirm: function () {\n                        return $.ajax({\n                            url: self.options.deleteFolderUrl,\n                            dataType: 'json',\n                            data: {\n                                node: self.activeNode.id,\n                                store: self.options.storeId,\n                                'form_key': FORM_KEY\n                            },\n                            context: self.element,\n                            showLoader: true\n                        }).done($.proxy(function (data) {\n                            if (data.error) {\n                                alert({\n                                    content: data.message\n                                });\n                            } else {\n                                self.tree.jstree('refresh', self.activeNode.id);\n                                self.reload();\n                                $(window).trigger('fileDeleted.mediabrowser', {\n                                    ids: self.activeNode.id\n                                });\n                            }\n                        }, this));\n                    },\n\n                    /**\n                     * @return {Boolean}\n                     */\n                    cancel: function () {\n                        return false;\n                    }\n                }\n            });\n        },\n\n        /**\n         * Delete files.\n         */\n        deleteFiles: function () {\n            var self = this;\n\n            confirm({\n                content: this.options.deleteFileConfirmationMessage,\n                actions: {\n                    /**\n                     * Confirm.\n                     */\n                    confirm: function () {\n                        var selectedFiles = self.element.find('[data-row=file].selected'),\n                            ids = selectedFiles.map(function () {\n                                return $(this).attr('id');\n                            }).toArray();\n\n                        return $.ajax({\n                            url: self.options.deleteFilesUrl,\n                            data: {\n                                files: ids,\n                                store: self.options.storeId,\n                                'form_key': FORM_KEY\n                            },\n                            context: self.element,\n                            showLoader: true\n                        }).done($.proxy(function (data) {\n                            if (data.error) {\n                                alert({\n                                    content: data.message\n                                });\n                            } else {\n                                self.reload();\n                                self.element.find('#delete_files, #insert_files').toggleClass(\n                                    self.options.hidden, true\n                                );\n\n                                $(window).trigger('fileDeleted.mediabrowser', {\n                                    ids: ids\n                                });\n                            }\n                        }, this));\n                    },\n\n                    /**\n                     * @return {Boolean}\n                     */\n                    cancel: function () {\n                        return false;\n                    }\n                }\n            });\n        },\n\n        /**\n         * @param {Object} data\n         */\n        drawBreadcrumbs: function (data) {\n            var node, breadcrumbs;\n\n            if (this.element.find('#breadcrumbs').length) {\n                this.element.find('#breadcrumbs').remove();\n            }\n            node = data.node;\n\n            if (node.id === 'root') { //eslint-disable-line eqeqeq\n                return;\n            }\n            breadcrumbs = $('<ul class=\"breadcrumbs\" id=\"breadcrumbs\"></ul>');\n            // jscs:disable requireCamelCaseOrUpperCaseIdentifiers\n            data.instance.get_path(node).each(function (name, index) {\n                if (index > 0) {\n                    breadcrumbs.append($('<li>\\/</li>')); //eslint-disable-line\n                }\n                breadcrumbs.append($('<li />').attr('data-row', 'breadcrumb').text(name));\n\n            });\n            // jscs:enable requireCamelCaseOrUpperCaseIdentifiers\n\n            breadcrumbs.insertAfter(this.element.find('#content_header'));\n        }\n    });\n\n    return window.MediabrowserUtility;\n});\n","mage/adminhtml/accordion.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\n/* global varienAccordion, varienLoader, Cookie */\n/* eslint-disable strict */\ndefine([\n    'prototype'\n], function () {\n    window.varienAccordion = new Class.create(); //eslint-disable-line\n    varienAccordion.prototype = {\n        /**\n         * @param {*} containerId\n         * @param {*} activeOnlyOne\n         */\n        initialize: function (containerId, activeOnlyOne) {\n            var links, i;\n\n            this.containerId = containerId;\n            this.activeOnlyOne = activeOnlyOne || false;\n            this.container = $(this.containerId);\n            this.items = $$('#' + this.containerId + ' dt');\n            this.loader = new varienLoader(true); //jscs:ignore requireCapitalizedConstructors\n\n            links = $$('#' + this.containerId + ' dt a');\n\n            for (i in links) {\n                if (links[i].href) {\n                    Event.observe(links[i], 'click', this.clickItem.bind(this));\n                    this.items[i].dd = this.items[i].next('dd');\n                    this.items[i].link = links[i];\n                }\n            }\n\n            this.initFromCookie();\n        },\n\n        /**\n         * Init from cookie.\n         */\n        initFromCookie: function () {\n            var activeItemId, visibility;\n\n            if (this.activeOnlyOne &&\n                (activeItemId = Cookie.read(this.cookiePrefix() + 'active-item')) !== null) {\n                this.hideAllItems();\n                this.showItem(this.getItemById(activeItemId));\n            } else if (!this.activeOnlyOne) {\n                this.items.each(function (item) {\n                    if ((visibility = Cookie.read(this.cookiePrefix() + item.id)) !== null) {\n                        if (visibility == 0) { //eslint-disable-line eqeqeq\n                            this.hideItem(item);\n                        } else {\n                            this.showItem(item);\n                        }\n                    }\n                }.bind(this));\n            }\n        },\n\n        /**\n         * @return {String}\n         */\n        cookiePrefix: function () {\n            return 'accordion-' + this.containerId + '-';\n        },\n\n        /**\n         * @param {*} itemId\n         * @return {*}\n         */\n        getItemById: function (itemId) {\n            var result = null;\n\n            this.items.each(function (item) {\n                if (item.id == itemId) { //eslint-disable-line\n                    result = item;\n                    throw $break;\n                }\n            });\n\n            return result;\n        },\n\n        /**\n         * @param {*} event\n         */\n        clickItem: function (event) {\n            var item = Event.findElement(event, 'dt');\n\n            if (this.activeOnlyOne) {\n                this.hideAllItems();\n                this.showItem(item);\n                Cookie.write(this.cookiePrefix() + 'active-item', item.id, 30 * 24 * 60 * 60);\n            } else {\n                if (this.isItemVisible(item)) { //eslint-disable-line no-lonely-if\n                    this.hideItem(item);\n                    Cookie.write(this.cookiePrefix() + item.id, 0, 30 * 24 * 60 * 60);\n                } else {\n                    this.showItem(item);\n                    Cookie.write(this.cookiePrefix() + item.id, 1, 30 * 24 * 60 * 60);\n                }\n            }\n            Event.stop(event);\n        },\n\n        /**\n         * @param {Object} item\n         */\n        showItem: function (item) {\n            if (item && item.link) {\n                if (item.link.href) {\n                    this.loadContent(item);\n                }\n\n                Element.addClassName(item, 'open');\n                Element.addClassName(item.dd, 'open');\n            }\n        },\n\n        /**\n         * @param {Object} item\n         */\n        hideItem: function (item) {\n            Element.removeClassName(item, 'open');\n            Element.removeClassName(item.dd, 'open');\n        },\n\n        /**\n         * @param {*} item\n         * @return {*}\n         */\n        isItemVisible: function (item) {\n            return Element.hasClassName(item, 'open');\n        },\n\n        /**\n         * @param {*} item\n         */\n        loadContent: function (item) {\n            if (item.link.href.indexOf('#') == item.link.href.length - 1) { //eslint-disable-line eqeqeq\n                return;\n            }\n\n            if (Element.hasClassName(item.link, 'ajax')) {\n                this.loadingItem = item;\n                this.loader.load(item.link.href, {\n                    updaterId: this.loadingItem.dd.id\n                }, this.setItemContent.bind(this));\n\n                return;\n            }\n            location.href = item.link.href;\n        },\n\n        /**\n         * @param {Object} content\n         */\n        setItemContent: function (content) {\n            if (content.isJSON) {\n                return;\n            }\n            this.loadingItem.dd.innerHTML = content;\n        },\n\n        /**\n         * Hide all items\n         */\n        hideAllItems: function () {\n            var i;\n\n            for (i in this.items) {\n                if (this.items[i].id) {\n                    Element.removeClassName(this.items[i], 'open');\n                    Element.removeClassName(this.items[i].dd, 'open');\n                }\n            }\n        }\n    };\n});\n","mage/adminhtml/wysiwyg/events.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\ndefine([], function () {\n    'use strict';\n\n    return {\n        afterInitialization: 'afterInitialization',\n        afterChangeContent: 'afterChangeContent',\n        afterUndo: 'afterUndo',\n        afterPaste: 'afterPaste',\n        beforeSetContent: 'beforeSetContent',\n        afterSetContent: 'afterSetContent',\n        afterSave: 'afterSave',\n        afterOpenFileBrowser: 'afterOpenFileBrowser',\n        afterFormSubmit: 'afterFormSubmit',\n        afterBlur: 'afterBlur',\n        afterFocus: 'afterFocus'\n    };\n});\n","mage/adminhtml/wysiwyg/widget.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\n/* global setLocation, Base64, updateElementAtCursor, varienGlobalEvents */\n/* eslint-disable strict */\ndefine([\n    'jquery',\n    'wysiwygAdapter',\n    'Magento_Ui/js/modal/alert',\n    'jquery/ui',\n    'mage/translate',\n    'mage/mage',\n    'mage/validation',\n    'mage/adminhtml/events',\n    'prototype',\n    'Magento_Ui/js/modal/modal'\n], function (jQuery, wysiwyg, alert) {\n    var widgetTools = {\n\n        /**\n         * Sets the widget to be active and is the scope of the slide out if the value is set\n         */\n        activeSelectedNode: null,\n        editMode: false,\n        cursorLocation: 0,\n\n        /**\n         * Set active selected node.\n         *\n         * @param {Object} activeSelectedNode\n         */\n        setActiveSelectedNode: function (activeSelectedNode) {\n            this.activeSelectedNode = activeSelectedNode;\n        },\n\n        /**\n         * Get active selected node.\n         *\n         * @returns {null}\n         */\n        getActiveSelectedNode: function () {\n            return this.activeSelectedNode;\n        },\n\n        /**\n         *\n         * @param {Boolean} editMode\n         */\n        setEditMode: function (editMode) {\n            this.editMode = editMode;\n        },\n\n        /**\n         * @param {*} id\n         * @param {*} html\n         * @return {String}\n         */\n        getDivHtml: function (id, html) {\n\n            if (!html) {\n                html = '';\n            }\n\n            return '<div id=\"' + id + '\">' + html + '</div>';\n        },\n\n        /**\n         * @param {Object} transport\n         */\n        onAjaxSuccess: function (transport) {\n            var response;\n\n            if (transport.responseText.isJSON()) {\n                response = transport.responseText.evalJSON();\n\n                if (response.error) {\n                    throw response;\n                } else if (response.ajaxExpired && response.ajaxRedirect) {\n                    setLocation(response.ajaxRedirect);\n                }\n            }\n        },\n\n        dialogOpened: false,\n\n        /**\n         * @return {Number}\n         */\n        getMaxZIndex: function () {\n            var max = 0,\n                cn = document.body.childNodes,\n                i, el, zIndex;\n\n            for (i = 0; i < cn.length; i++) {\n                el = cn[i];\n                zIndex = el.nodeType == 1 ? parseInt(el.style.zIndex, 10) || 0 : 0; //eslint-disable-line eqeqeq\n\n                if (zIndex < 10000) {\n                    max = Math.max(max, zIndex);\n                }\n            }\n\n            return max + 10;\n        },\n\n        /**\n         * @param {String} widgetUrl\n         */\n        openDialog: function (widgetUrl) {\n            var oThis = this,\n                title = 'Insert Widget',\n                mode = 'new',\n                dialog;\n\n            if (this.editMode) {\n                title = 'Edit Widget';\n                mode = 'edit';\n            }\n\n            if (this.dialogOpened) {\n                return;\n            }\n\n            this.dialogWindow = jQuery('<div/>').modal({\n\n                title: jQuery.mage.__(title),\n                type: 'slide',\n                buttons: [],\n\n                /**\n                 * Opened.\n                 */\n                opened: function () {\n                    dialog = jQuery(this).addClass('loading magento-message');\n\n                    widgetUrl += 'mode/' + mode;\n\n                    new Ajax.Updater($(this), widgetUrl, {\n                        evalScripts: true,\n\n                        /**\n                         * On complete.\n                         */\n                        onComplete: function () {\n                            dialog.removeClass('loading');\n                        }\n                    });\n                },\n\n                /**\n                 * @param {jQuery.Event} e\n                 * @param {Object} modal\n                 */\n                closed: function (e, modal) {\n                    modal.modal.remove();\n                    oThis.dialogOpened = false;\n                }\n            });\n\n            this.dialogOpened = true;\n            this.dialogWindow.modal('openModal');\n        }\n    },\n    WysiwygWidget = {};\n\n    WysiwygWidget.Widget = Class.create();\n    WysiwygWidget.Widget.prototype = {\n        /**\n         * @param {HTMLElement} formEl\n         * @param {HTMLElement} widgetEl\n         * @param {*} widgetOptionsEl\n         * @param {*} optionsSourceUrl\n         * @param {*} widgetTargetId\n         */\n        initialize: function (formEl, widgetEl, widgetOptionsEl, optionsSourceUrl, widgetTargetId) {\n            $(formEl).insert({\n                bottom: widgetTools.getDivHtml(widgetOptionsEl)\n            });\n\n            this.formEl = formEl;\n            this.widgetEl = $(widgetEl);\n            this.widgetOptionsEl = $(widgetOptionsEl);\n            this.optionsUrl = optionsSourceUrl;\n            this.optionValues = new Hash({});\n            this.widgetTargetId = widgetTargetId;\n\n            if (typeof wysiwyg != 'undefined' && wysiwyg.activeEditor()) { //eslint-disable-line eqeqeq\n                this.bMark = wysiwyg.activeEditor().selection.getBookmark();\n            }\n\n            // disable -- Please Select -- option from being re-selected\n            this.widgetEl.querySelector('option').setAttribute('disabled', 'disabled');\n\n            Event.observe(this.widgetEl, 'change', this.loadOptions.bind(this));\n\n            this.initOptionValues();\n        },\n\n        /**\n         * @return {String}\n         */\n        getOptionsContainerId: function () {\n            return this.widgetOptionsEl.id + '_' + this.widgetEl.value.gsub(/\\//, '_');\n        },\n\n        /**\n         * @param {*} containerId\n         */\n        switchOptionsContainer: function (containerId) {\n            $$('#' + this.widgetOptionsEl.id + ' div[id^=' + this.widgetOptionsEl.id + ']').each(function (e) {\n                this.disableOptionsContainer(e.id);\n            }.bind(this));\n\n            if (containerId != undefined) { //eslint-disable-line eqeqeq\n                this.enableOptionsContainer(containerId);\n            }\n            this._showWidgetDescription();\n        },\n\n        /**\n         * @param {*} containerId\n         */\n        enableOptionsContainer: function (containerId) {\n            var container = $(containerId);\n\n            container.select('.widget-option').each(function (e) {\n                e.removeClassName('skip-submit');\n\n                if (e.hasClassName('obligatory')) {\n                    e.removeClassName('obligatory');\n                    e.addClassName('required-entry');\n                }\n            });\n            container.removeClassName('no-display');\n        },\n\n        /**\n         * @param {*} containerId\n         */\n        disableOptionsContainer: function (containerId) {\n            var container = $(containerId);\n\n            if (container.hasClassName('no-display')) {\n                return;\n            }\n            container.select('.widget-option').each(function (e) {\n                // Avoid submitting fields of unactive container\n                if (!e.hasClassName('skip-submit')) {\n                    e.addClassName('skip-submit');\n                }\n                // Form validation workaround for unactive container\n                if (e.hasClassName('required-entry')) {\n                    e.removeClassName('required-entry');\n                    e.addClassName('obligatory');\n                }\n            });\n            container.addClassName('no-display');\n        },\n\n        /**\n         * Assign widget options values when existing widget selected in WYSIWYG.\n         *\n         * @return {Boolean}\n         */\n        initOptionValues: function () {\n            var e, widgetCode;\n\n            if (!this.wysiwygExists()) {\n                return false;\n            }\n\n            e = this.getWysiwygNode();\n\n            if (e.localName === 'span') {\n                e = e.firstElementChild;\n            }\n\n            if (e != undefined && e.id) { //eslint-disable-line eqeqeq\n                // attempt to Base64-decode id on selected node; exception is thrown if it is in fact not a widget node\n                try {\n                    widgetCode = Base64.idDecode(e.id);\n                } catch (ex) {\n                    return false;\n                }\n\n                if (widgetCode.indexOf('{{widget') !== -1) {\n                    this.optionValues = new Hash({});\n                    widgetCode.gsub(/([a-z0-9\\_]+)\\s*\\=\\s*[\\\"]{1}([^\\\"]+)[\\\"]{1}/i, function (match) {\n\n                        if (match[1] == 'type') { //eslint-disable-line eqeqeq\n                            this.widgetEl.value = match[2];\n                        } else {\n                            this.optionValues.set(match[1], match[2]);\n                        }\n\n                    }.bind(this));\n\n                    this.loadOptions();\n                }\n            }\n        },\n\n        /**\n         * Load options.\n         */\n        loadOptions: function () {\n            var optionsContainerId,\n                params,\n                msg,\n                msgTmpl,\n                $wrapper,\n                typeName = this.optionValues.get('type_name');\n\n            if (!this.widgetEl.value) {\n                if (typeName) {\n                    msgTmpl = jQuery.mage.__('The widget %1 is no longer available. Select a different widget.');\n                    msg = jQuery.mage.__(msgTmpl).replace('%1', typeName);\n\n                    jQuery('body').notification('clear').notification('add', {\n                        error: true,\n                        message: msg,\n\n                        /**\n                         * @param {String} message\n                         */\n                        insertMethod: function (message) {\n                            $wrapper = jQuery('<div/>').html(message);\n\n                            $wrapper.insertAfter('.modal-slide .page-main-actions');\n                        }\n                    });\n                }\n                this.switchOptionsContainer();\n\n                return;\n            }\n\n            optionsContainerId = this.getOptionsContainerId();\n\n            if ($(optionsContainerId) != undefined) { //eslint-disable-line eqeqeq\n                this.switchOptionsContainer(optionsContainerId);\n\n                return;\n            }\n\n            this._showWidgetDescription();\n\n            params = {\n                'widget_type': this.widgetEl.value,\n                values: this.optionValues\n            };\n            new Ajax.Request(this.optionsUrl, {\n                parameters: {\n                    widget: Object.toJSON(params)\n                },\n\n                /**\n                 * On success.\n                 */\n                onSuccess: function (transport) {\n                    try {\n                        widgetTools.onAjaxSuccess(transport);\n                        this.switchOptionsContainer();\n\n                        if ($(optionsContainerId) == undefined) { //eslint-disable-line eqeqeq\n                            this.widgetOptionsEl.insert({\n                                bottom: widgetTools.getDivHtml(optionsContainerId, transport.responseText)\n                            });\n                        } else {\n                            this.switchOptionsContainer(optionsContainerId);\n                        }\n                    } catch (e) {\n                        alert({\n                            content: e.message\n                        });\n                    }\n                }.bind(this)\n            });\n        },\n\n        /**\n         * @private\n         */\n        _showWidgetDescription: function () {\n            var noteCnt = this.widgetEl.next().down('small'),\n                descrCnt = $('widget-description-' + this.widgetEl.selectedIndex),\n                description;\n\n            if (noteCnt != undefined) { //eslint-disable-line eqeqeq\n                description = descrCnt != undefined ? descrCnt.innerHTML : ''; //eslint-disable-line eqeqeq\n                noteCnt.update(description);\n            }\n        },\n\n        /**\n         * Validate field.\n         */\n        validateField: function () {\n            jQuery(this.widgetEl).valid();\n            jQuery('#insert_button').removeClass('disabled');\n        },\n\n        /**\n         * Closes the modal\n         */\n        closeModal: function () {\n            widgetTools.dialogWindow.modal('closeModal');\n        },\n\n        /* eslint-disable max-depth*/\n        /**\n         * Insert widget.\n         */\n        insertWidget: function () {\n            var validationResult,\n                $form = jQuery('#' + this.formEl),\n                formElements,\n                i,\n                params,\n                editor,\n                activeNode;\n\n            // remove cached validator instance, which caches elements to validate\n            jQuery.data($form[0], 'validator', null);\n\n            $form.validate({\n                /**\n                 * Ignores elements with .skip-submit, .no-display ancestor elements\n                 */\n                ignore: function () {\n                    return jQuery(this).closest('.skip-submit, .no-display').length;\n                },\n                errorClass: 'mage-error'\n            });\n\n            validationResult = $form.valid();\n\n            if (validationResult) {\n                formElements = [];\n                i = 0;\n                Form.getElements($(this.formEl)).each(function (e) {\n\n                    if (jQuery(e).closest('.skip-submit, .no-display').length === 0) {\n                        formElements[i] = e;\n                        i++;\n                    }\n                });\n\n                // Add as_is flag to parameters if wysiwyg editor doesn't exist\n                params = Form.serializeElements(formElements);\n\n                if (!this.wysiwygExists()) {\n                    params += '&as_is=1';\n                }\n\n                new Ajax.Request($(this.formEl).action, {\n                    parameters: params,\n                    onComplete: function (transport) {\n                        try {\n                            editor = wysiwyg.get(this.widgetTargetId);\n\n                            widgetTools.onAjaxSuccess(transport);\n                            widgetTools.dialogWindow.modal('closeModal');\n\n                            if (editor) {\n                                editor.focus();\n                                activeNode = widgetTools.getActiveSelectedNode();\n\n                                if (activeNode) {\n                                    editor.selection.select(activeNode);\n                                    editor.selection.setContent(transport.responseText);\n                                    editor.fire('Change');\n                                } else if (this.bMark) {\n                                    editor.selection.moveToBookmark(this.bMark);\n                                }\n                            }\n\n                            if (!activeNode) {\n                                this.updateContent(transport.responseText);\n                            }\n                        } catch (e) {\n                            alert({\n                                content: e.message\n                            });\n                        }\n                    }.bind(this)\n                });\n            }\n        },\n\n        /**\n         * @param {Object} content\n         */\n        updateContent: function (content) {\n            var textarea;\n\n            if (this.wysiwygExists()) {\n                wysiwyg.insertContent(content, false);\n            } else {\n                textarea = document.getElementById(this.widgetTargetId);\n                updateElementAtCursor(textarea, content);\n                varienGlobalEvents.fireEvent('tinymceChange');\n                jQuery(textarea).trigger('change');\n            }\n        },\n\n        /**\n         * @return {Boolean}\n         */\n        wysiwygExists: function () {\n            return typeof wysiwyg != 'undefined' && wysiwyg.get(this.widgetTargetId);\n        },\n\n        /**\n         * @return {null|wysiwyg.Editor|*}\n         */\n        getWysiwyg: function () {\n            return wysiwyg.get(this.widgetTargetId);\n        },\n\n        /**\n         * @return {*|Element}\n         */\n        getWysiwygNode: function () {\n            return widgetTools.getActiveSelectedNode() || wysiwyg.activeEditor().selection.getNode();\n        }\n    };\n\n    WysiwygWidget.chooser = Class.create();\n    WysiwygWidget.chooser.prototype = {\n\n        // HTML element A, on which click event fired when choose a selection\n        chooserId: null,\n\n        // Source URL for Ajax requests\n        chooserUrl: null,\n\n        // Chooser config\n        config: null,\n\n        // Chooser dialog window\n        dialogWindow: null,\n\n        // Chooser content for dialog window\n        dialogContent: null,\n\n        overlayShowEffectOptions: null,\n        overlayHideEffectOptions: null,\n\n        /**\n         * @param {*} chooserId\n         * @param {*} chooserUrl\n         * @param {*} config\n         */\n        initialize: function (chooserId, chooserUrl, config) {\n            this.chooserId = chooserId;\n            this.chooserUrl = chooserUrl;\n            this.config = config;\n        },\n\n        /**\n         * @return {String}\n         */\n        getResponseContainerId: function () {\n            return 'responseCnt' + this.chooserId;\n        },\n\n        /**\n         * @return {jQuery|*|HTMLElement}\n         */\n        getChooserControl: function () {\n            return $(this.chooserId + 'control');\n        },\n\n        /**\n         * @return {jQuery|*|HTMLElement}\n         */\n        getElement: function () {\n            return $(this.chooserId + 'value');\n        },\n\n        /**\n         * @return {jQuery|*|HTMLElement}\n         */\n        getElementLabel: function () {\n            return $(this.chooserId + 'label');\n        },\n\n        /**\n         * Open.\n         */\n        open: function () {\n            $(this.getResponseContainerId()).show();\n        },\n\n        /**\n         * Close.\n         */\n        close: function () {\n            $(this.getResponseContainerId()).hide();\n            this.closeDialogWindow();\n        },\n\n        /**\n         * Choose.\n         */\n        choose: function () {\n            // Open dialog window with previously loaded dialog content\n            var responseContainerId;\n\n            if (this.dialogContent) {\n                this.openDialogWindow(this.dialogContent);\n\n                return;\n            }\n            // Show or hide chooser content if it was already loaded\n            responseContainerId = this.getResponseContainerId();\n\n            // Otherwise load content from server\n            new Ajax.Request(this.chooserUrl, {\n                parameters: {\n                    'element_value': this.getElementValue(),\n                    'element_label': this.getElementLabelText()\n                },\n\n                /**\n                 * On success.\n                 */\n                onSuccess: function (transport) {\n                    try {\n                        widgetTools.onAjaxSuccess(transport);\n                        this.dialogContent = widgetTools.getDivHtml(responseContainerId, transport.responseText);\n                        this.openDialogWindow(this.dialogContent);\n                    } catch (e) {\n                        alert({\n                            content: e.message\n                        });\n                    }\n                }.bind(this)\n            });\n        },\n\n        /**\n         * Open dialog winodw.\n         *\n         * @param {*} content\n         */\n        openDialogWindow: function (content) {\n            this.dialogWindow = jQuery('<div/>').modal({\n                title: this.config.buttons.open,\n                type: 'slide',\n                buttons: [],\n\n                /**\n                 * Opened.\n                 */\n                opened: function () {\n                    jQuery(this).addClass('magento-message');\n                },\n\n                /**\n                 * @param {jQuery.Event} e\n                 * @param {Object} modal\n                 */\n                closed: function (e, modal) {\n                    modal.modal.remove();\n                    this.dialogWindow = null;\n                }\n            });\n\n            this.dialogWindow.modal('openModal').append(content);\n        },\n\n        /**\n         * Close dialog window.\n         */\n        closeDialogWindow: function () {\n            this.dialogWindow.modal('closeModal').remove();\n        },\n\n        /**\n         * @return {*|Number}\n         */\n        getElementValue: function () {\n            return this.getElement().value;\n        },\n\n        /**\n         * @return {String}\n         */\n        getElementLabelText: function () {\n            return this.getElementLabel().innerHTML;\n        },\n\n        /**\n         * @param {*} value\n         */\n        setElementValue: function (value) {\n            this.getElement().value = value;\n        },\n\n        /**\n         * @param {*} value\n         */\n        setElementLabel: function (value) {\n            this.getElementLabel().innerHTML = value;\n        }\n    };\n\n    window.WysiwygWidget = WysiwygWidget;\n    window.widgetTools = widgetTools;\n});\n","mage/adminhtml/wysiwyg/tiny_mce/tinymce5Adapter.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\n/* global popups, tinyMceEditors, MediabrowserUtility, Base64 */\n/* eslint-disable strict */\ndefine([\n    'jquery',\n    'underscore',\n    'tinymce',\n    'mage/adminhtml/events',\n    'mage/adminhtml/wysiwyg/events',\n    'mage/translate',\n    'prototype',\n    'jquery/ui'\n], function (jQuery, _, tinyMCE, varienGlobalEvents, wysiwygEvents) {\n    'use strict';\n\n    var tinyMceWysiwyg = Class.create();\n\n    tinyMceWysiwyg.prototype = {\n        mediaBrowserOpener: null,\n        mediaBrowserTargetElementId: null,\n        magentoVariablesPlugin: null,\n        mode: 'exact',\n\n        /**\n         * @param {*} htmlId\n         * @param {Object} config\n         */\n        initialize: function (htmlId, config) {\n            this.id = htmlId;\n            this.config = config;\n\n            _.bindAll(\n                this,\n                'beforeSetContent',\n                'saveContent',\n                'onChangeContent',\n                'openFileBrowser',\n                'updateTextArea',\n                'onUndo',\n                'removeEvents'\n            );\n\n            varienGlobalEvents.attachEventHandler('tinymceChange', this.onChangeContent);\n            varienGlobalEvents.attachEventHandler('tinymceBeforeSetContent', this.beforeSetContent);\n            varienGlobalEvents.attachEventHandler('tinymceSetContent', this.updateTextArea);\n            varienGlobalEvents.attachEventHandler('tinymceSaveContent', this.saveContent);\n            varienGlobalEvents.attachEventHandler('tinymceUndo', this.onUndo);\n\n            if (typeof tinyMceEditors === 'undefined') {\n                window.tinyMceEditors = $H({});\n            }\n\n            tinyMceEditors.set(this.id, this);\n        },\n\n        /**\n         * Ensures the undo operation works properly\n         */\n        onUndo: function () {\n            this.addContentEditableAttributeBackToNonEditableNodes();\n        },\n\n        /**\n         * Setup TinyMCE editor\n         */\n        setup: function (mode) {\n            var deferreds = [],\n                settings,\n                self = this;\n\n            this.turnOff();\n\n            if (this.config.plugins) {\n                this.config.plugins.forEach(function (plugin) {\n                    var deferred;\n\n                    self.addPluginToToolbar(plugin.name, '|');\n\n                    if (!plugin.src) {\n                        return;\n                    }\n\n                    deferred = jQuery.Deferred();\n                    deferreds.push(deferred);\n\n                    require([plugin.src], function (factoryFn) {\n                        if (typeof factoryFn === 'function') {\n                            factoryFn(plugin.options);\n                        }\n\n                        tinyMCE.PluginManager.load(plugin.name, plugin.src);\n                        deferred.resolve();\n                    });\n                });\n            }\n\n            if (jQuery.isReady) {\n                tinyMCE.dom.Event.domLoaded = true;\n            }\n\n            settings = this.getSettings();\n\n            if (mode === 'inline') {\n                settings.inline = true;\n\n                if (!isNaN(settings.toolbarZIndex)) {\n                    tinyMCE.ui.FloatPanel.zIndex = settings.toolbarZIndex;\n                }\n\n                this.removeEvents(self.id);\n            }\n\n            jQuery.when.apply(jQuery, deferreds).done(function () {\n                tinyMCE.init(settings);\n                this.getPluginButtons().hide();\n                varienGlobalEvents.clearEventHandlers('open_browser_callback');\n                this.eventBus.clearEventHandlers('open_browser_callback');\n                this.eventBus.attachEventHandler('open_browser_callback', tinyMceEditors.get(self.id).openFileBrowser);\n            }.bind(this));\n        },\n\n        /**\n         * Remove events from instance.\n         *\n         * @param {String} wysiwygId\n         */\n        removeEvents: function (wysiwygId) {\n            var editor;\n\n            if (typeof tinyMceEditors !== 'undefined' && tinyMceEditors.get(wysiwygId)) {\n                editor = tinyMceEditors.get(wysiwygId);\n                varienGlobalEvents.removeEventHandler('tinymceChange', editor.onChangeContent);\n            }\n        },\n\n        /**\n         * Add plugin to the toolbar if not added.\n         *\n         * @param {String} plugin\n         * @param {String} separator\n         */\n        addPluginToToolbar: function (plugin, separator) {\n            var plugins = this.config.tinymce.plugins.split(' '),\n                toolbar = this.config.tinymce.toolbar.split(' ');\n\n            if (plugins.indexOf(plugin) === -1) {\n                plugins.push(plugin);\n            }\n\n            if (toolbar.indexOf(plugin) === -1) {\n                toolbar.push(separator || '', plugin);\n            }\n\n            this.config.tinymce.plugins = plugins.join(' ');\n            this.config.tinymce.toolbar = toolbar.join(' ');\n        },\n\n        /**\n         * Set the status of the toolbar to disabled or enabled (true for enabled, false for disabled)\n         * @param {Boolean} enabled\n         */\n        setToolbarStatus: function (enabled) {\n            var controlIds = this.get(this.getId()).theme.panel.rootControl.controlIdLookup;\n\n            _.each(controlIds, function (controlId) {\n                controlId.disabled(!enabled);\n                controlId.canFocus = enabled;\n\n                if (controlId.tooltip) {\n                    controlId.tooltip().state.set('rendered', enabled);\n\n                    if (enabled) {\n                        jQuery(controlId.getEl()).children('button').addBack().removeAttr('style');\n                    } else {\n                        jQuery(controlId.getEl()).children('button').addBack().attr('style', 'color: inherit;' +\n                            'background-color: inherit;' +\n                            'border-color: transparent;'\n                        );\n                    }\n                }\n            });\n        },\n\n        /**\n         * @return {Object}\n         */\n        getSettings: function () {\n            var settings,\n                eventBus = this.eventBus;\n\n            settings = {\n                selector: '#' + this.getId(),\n                theme: 'silver',\n                skin: 'oxide',\n                'toolbar_mode': 'wrap',\n                'entity_encoding': 'raw',\n                'convert_urls': false,\n                'content_css': this.config.tinymce['content_css'],\n                'relative_urls': true,\n                'valid_children': '+body[style]',\n                menubar: false,\n                plugins: this.config.tinymce.plugins,\n                toolbar: this.config.tinymce.toolbar,\n                adapter: this,\n                'body_id': 'html-body',\n\n                /**\n                 * @param {Object} editor\n                 */\n                setup: function (editor) {\n                    var onChange;\n\n                    editor.on('BeforeSetContent', function (evt) {\n                        varienGlobalEvents.fireEvent('tinymceBeforeSetContent', evt);\n                        eventBus.fireEvent(wysiwygEvents.beforeSetContent);\n                    });\n\n                    editor.on('SaveContent', function (evt) {\n                        varienGlobalEvents.fireEvent('tinymceSaveContent', evt);\n                        eventBus.fireEvent(wysiwygEvents.afterSave);\n                    });\n\n                    editor.on('paste', function (evt) {\n                        varienGlobalEvents.fireEvent('tinymcePaste', evt);\n                        eventBus.fireEvent(wysiwygEvents.afterPaste);\n                    });\n\n                    editor.on('PostProcess', function (evt) {\n                        varienGlobalEvents.fireEvent('tinymceSaveContent', evt);\n                        eventBus.fireEvent(wysiwygEvents.afterSave);\n                    });\n\n                    editor.on('undo', function (evt) {\n                        varienGlobalEvents.fireEvent('tinymceUndo', evt);\n                        eventBus.fireEvent(wysiwygEvents.afterUndo);\n                    });\n\n                    editor.on('focus', function () {\n                        eventBus.fireEvent(wysiwygEvents.afterFocus);\n                    });\n\n                    editor.on('blur', function () {\n                        eventBus.fireEvent(wysiwygEvents.afterBlur);\n                    });\n\n                    /**\n                     * @param {*} evt\n                     */\n                    onChange = function (evt) {\n                        varienGlobalEvents.fireEvent('tinymceChange', evt);\n                        eventBus.fireEvent(wysiwygEvents.afterChangeContent);\n                    };\n\n                    editor.on('Change', onChange);\n                    editor.on('keyup', onChange);\n\n                    editor.on('ExecCommand', function (cmd) {\n                        varienGlobalEvents.fireEvent('tinymceExecCommand', cmd);\n                    });\n\n                    editor.on('init', function (args) {\n                        varienGlobalEvents.fireEvent('wysiwygEditorInitialized', args.target);\n                        eventBus.fireEvent(wysiwygEvents.afterInitialization);\n                    });\n                }\n            };\n\n            // Set default initial height\n            settings['min_height'] = this.config.tinymce['min_height'] ? this.config.tinymce['min_height'] : 250;\n\n            if (this.config.skin) {\n                settings.skin = this.config.skin;\n            }\n\n            if (this.config['toolbar_mode']) {\n                settings['toolbar_mode'] = this.config['toolbar_mode'];\n            }\n\n            if (this.config.baseStaticUrl && this.config.baseStaticDefaultUrl) {\n                settings['document_base_url'] = this.config.baseStaticUrl;\n            }\n            // Set the document base URL\n            if (this.config['document_base_url']) {\n                settings['document_base_url'] = this.config['document_base_url'];\n            }\n\n            if (this.config['files_browser_window_url']) {\n                settings['file_picker_callback_types'] = 'file image media';\n\n                /**\n                 * @param {*} callback\n                 * @param {*} value\n                 * @param {*} meta\n                 */\n                settings['file_picker_callback'] = function (callback, value, meta) {\n                    var payload = {\n                        callback: callback,\n                        value: value,\n                        meta: meta\n                    };\n\n                    varienGlobalEvents.fireEvent('open_browser_callback', payload);\n                    this.eventBus.fireEvent('open_browser_callback', payload);\n                }.bind(this);\n            }\n\n            if (this.config.width) {\n                settings.width = this.config.width;\n            }\n\n            if (this.config.height) {\n                settings.height = this.config.height;\n            }\n\n            if (this.config.plugins) {\n                settings.magentoPluginsOptions = {};\n\n                _.each(this.config.plugins, function (plugin) {\n                    settings.magentoPluginsOptions[plugin.name] = plugin.options;\n                });\n            }\n\n            if (this.config.settings) {\n                Object.extend(settings, this.config.settings);\n            }\n\n            return settings;\n        },\n\n        /**\n         * @param {String} id\n         */\n        get: function (id) {\n            return tinyMCE.get(id);\n        },\n\n        /**\n         * @return {String|null}\n         */\n        getId: function () {\n            return this.id || (this.activeEditor() ? this.activeEditor().id : null) || tinyMceEditors.values()[0].id;\n        },\n\n        /**\n         * @return {Object}\n         */\n        activeEditor: function () {\n            return tinyMCE.activeEditor;\n        },\n\n        /**\n         * Insert content to active editor.\n         *\n         * @param {String} content\n         * @param {Boolean} ui\n         */\n        insertContent: function (content, ui) {\n            this.activeEditor().execCommand('mceInsertContent', typeof ui !== 'undefined' ? ui : false, content);\n        },\n\n        /**\n         * Replace entire contents of wysiwyg with string content parameter\n         *\n         * @param {String} content\n         */\n        setContent: function (content) {\n            this.get(this.getId()).setContent(content);\n        },\n\n        /**\n         * Set caret location in WYSIWYG editor.\n         *\n         * @param {Object} targetElement\n         */\n        setCaretOnElement: function (targetElement) {\n            this.activeEditor().selection.select(targetElement);\n            this.activeEditor().selection.collapse();\n        },\n\n        /**\n         * @param {Object} o\n         */\n        openFileBrowser: function (o) {\n            var typeTitle = this.translate('Select Images'),\n                storeId = this.config['store_id'] ? this.config['store_id'] : 0,\n                frameDialog = jQuery('div.mce-container[role=\"dialog\"]'),\n                self = this,\n                wUrl = this.config['files_browser_window_url'] +\n                    'target_element_id/' + this.getId() + '/' +\n                    'store/' + storeId + '/';\n\n            this.mediaBrowserOpener = o.callback;\n\n            if (typeof o.meta.filetype !== 'undefined' && o.meta.filetype !== '') { //eslint-disable-line eqeqeq\n                wUrl = wUrl + 'type/' + o.meta.filetype + '/';\n            }\n\n            frameDialog.hide();\n            jQuery('.tox-tinymce-aux').hide();\n\n            require(['mage/adminhtml/browser'], function () {\n                MediabrowserUtility.openDialog(wUrl, false, false, typeTitle, {\n                        /**\n                         * Closed.\n                         */\n                        closed: function () {\n                            frameDialog.show();\n                            jQuery('.tox-tinymce-aux').show();\n                        },\n\n                        targetElementId: self.activeEditor() ? self.activeEditor().id : null\n                    }\n                );\n            });\n        },\n\n        /**\n         * @param {String} string\n         * @return {String}\n         */\n        translate: function (string) {\n            return jQuery.mage.__ ? jQuery.mage.__(string) : string;\n        },\n\n        /**\n         * @return {null}\n         */\n        getMediaBrowserOpener: function () {\n            return this.mediaBrowserOpener;\n        },\n\n        /**\n         * @return {null}\n         */\n        getMediaBrowserTargetElementId: function () {\n            return this.mediaBrowserTargetElementId;\n        },\n\n        /**\n         * @return {jQuery|*|HTMLElement}\n         */\n        getToggleButton: function () {\n            return $('toggle' + this.getId());\n        },\n\n        /**\n         * Get plugins button.\n         */\n        getPluginButtons: function () {\n            return jQuery('#buttons' + this.getId() + ' > button.plugin');\n        },\n\n        /**\n         * @param {*} mode\n         * @return {wysiwygSetup}\n         */\n        turnOn: function (mode) {\n            this.closePopups();\n\n            this.setup(mode);\n\n            this.getPluginButtons().hide();\n\n            tinyMCE.execCommand('mceAddControl', false, this.getId());\n\n            return this;\n        },\n\n        /**\n         * @param {String} name\n         */\n        closeEditorPopup: function (name) {\n            if (typeof popups !== 'undefined' && popups[name] !== undefined && !popups[name].closed) {\n                popups[name].close();\n            }\n        },\n\n        /**\n         * @return {wysiwygSetup}\n         */\n        turnOff: function () {\n            this.closePopups();\n\n            this.getPluginButtons().show();\n\n            tinyMCE.execCommand('mceRemoveEditor', false, this.getId());\n\n            return this;\n        },\n\n        /**\n         * Close popups.\n         */\n        closePopups: function () {\n            // close all popups to avoid problems with updating parent content area\n            varienGlobalEvents.fireEvent('wysiwygClosePopups');\n            this.closeEditorPopup('browser_window' + this.getId());\n        },\n\n        /**\n         * @return {Boolean}\n         */\n        toggle: function () {\n            var content;\n\n            if (!tinyMCE.get(this.getId())) {\n                this.turnOn();\n\n                return true;\n            }\n\n            content = this.get(this.getId()) ? this.get(this.getId()).getContent() : this.getTextArea().val();\n\n            this.turnOff();\n\n            if (content.match(/{{.+?}}/g)) {\n                this.getTextArea().val(content.replace(/&quot;/g, '\"'));\n            }\n\n            return false;\n        },\n\n        /**\n         * On form validation.\n         */\n        onFormValidation: function () {\n            if (tinyMCE.get(this.getId())) {\n                $(this.getId()).value = tinyMCE.get(this.getId()).getContent();\n            }\n        },\n\n        /**\n         * On change content.\n         */\n        onChangeContent: function () {\n            // Add \"changed\" to tab class if it exists\n            var tab;\n\n            this.updateTextArea();\n\n            if (this.config['tab_id']) {\n                tab = $$('a[id$=' + this.config['tab_id'] + ']')[0];\n\n                if ($(tab) != undefined && $(tab).hasClassName('tab-item-link')) { //eslint-disable-line eqeqeq\n                    $(tab).addClassName('changed');\n                }\n            }\n        },\n\n        /**\n         * @param {Object} o\n         */\n        beforeSetContent: function (o) {\n            o.content = this.encodeContent(o.content);\n        },\n\n        /**\n         * @param {Object} o\n         */\n        saveContent: function (o) {\n            o.content = this.decodeContent(o.content);\n        },\n\n        /**\n         * Return the content stored in the WYSIWYG field\n         * @param {String} id\n         * @return {String}\n         */\n        getContent: function (id) {\n            return id ? this.get(id).getContent() : this.get(this.getId()).getContent();\n        },\n\n        /**\n         * @returns {Object}\n         */\n        getAdapterPrototype: function () {\n            return tinyMceWysiwyg;\n        },\n\n        /**\n         * Fix range selection placement when typing.  This fixes MAGETWO-84769\n         * @param {Object} editor\n         */\n        fixRangeSelection: function (editor) {\n            var selection = editor.selection,\n                dom = editor.dom,\n                rng = dom.createRng(),\n                doc = editor.getDoc(),\n                markerHtml,\n                marker;\n\n            // Validate the range we're trying to fix is contained within the current editors document\n            if (!selection.getContent().length && jQuery.contains(doc, selection.getRng().startContainer)) {\n                markerHtml = '<span id=\"mce_marker\" data-mce-type=\"bookmark\">\\uFEFF</span>';\n                selection.setContent(markerHtml);\n                marker = dom.get('mce_marker');\n                rng.setStartBefore(marker);\n                rng.setEndBefore(marker);\n                dom.remove(marker);\n                selection.setRng(rng);\n            }\n        },\n\n        /**\n         * Update text area.\n         */\n        updateTextArea: function () {\n            var editor = this.get(this.getId()),\n                content;\n\n            if (!editor || editor.id !== this.activeEditor().id) {\n                return;\n            }\n\n            this.addContentEditableAttributeBackToNonEditableNodes();\n\n            content = editor.getContent();\n            content = this.decodeContent(content);\n\n            this.getTextArea().val(content).trigger('change');\n        },\n\n        /**\n         * @return {Object} jQuery textarea element\n         */\n        getTextArea: function () {\n            return jQuery('#' + this.getId());\n        },\n\n        /**\n         * Set the status of the editor and toolbar\n         *\n         * @param {Boolean} enabled\n         */\n        setEnabledStatus: function (enabled) {\n            if (this.activeEditor()) {\n                this.activeEditor().getBody().setAttribute('contenteditable', enabled);\n                this.activeEditor().readonly = !enabled;\n                this.setToolbarStatus(enabled);\n            }\n\n            if (enabled) {\n                this.getTextArea().prop('disabled', false);\n            } else {\n                this.getTextArea().prop('disabled', 'disabled');\n            }\n        },\n\n        /**\n         * Retrieve directives URL with substituted directive value.\n         *\n         * @param {String} directive\n         */\n        makeDirectiveUrl: function (directive) {\n            return this.config['directives_url']\n                .replace(/directive/, 'directive/___directive/' + directive)\n                .replace(/\\/$/, '');\n        },\n\n        /**\n         * Convert {{directive}} style attributes syntax to absolute URLs\n         * @param {Object} content\n         * @return {*}\n         */\n        encodeDirectives: function (content) {\n            // collect all HTML tags with attributes that contain directives\n            return content.gsub(/<([a-z0-9\\-\\_]+[^>]+?)([a-z0-9\\-\\_]+=\"[^\"]*?\\{\\{.+?\\}\\}.*?\".*?)>/i, function (match) {\n                var attributesString = match[2],\n                    decodedDirectiveString;\n\n                // process tag attributes string\n                attributesString = attributesString.gsub(/([a-z0-9\\-\\_]+)=\"(.*?)(\\{\\{.+?\\}\\})(.*?)\"/i, function (m) {\n                    decodedDirectiveString = encodeURIComponent(Base64.mageEncode(m[3].replace(/&quot;/g, '\"') + m[4]));\n\n                    return m[1] + '=\"' + m[2] + this.makeDirectiveUrl(decodedDirectiveString) + '\"';\n                }.bind(this));\n\n                return '<' + match[1] + attributesString + '>';\n            }.bind(this));\n        },\n\n        /**\n         * Convert absolute URLs to {{directive}} style attributes syntax\n         * @param {Object} content\n         * @return {*}\n         */\n        decodeDirectives: function (content) {\n            var directiveUrl = this.makeDirectiveUrl('%directive%').split('?')[0], // remove query string from directive\n                // escape special chars in directives url to use in regular expression\n                regexEscapedDirectiveUrl = directiveUrl.replace(/([$^.?*!+:=()\\[\\]{}|\\\\])/g, '\\\\$1'),\n                regexDirectiveUrl = regexEscapedDirectiveUrl\n                    .replace(\n                        '%directive%',\n                        '([a-zA-Z0-9,_-]+(?:%2[A-Z]|)+\\/?)(?:(?!\").)*'\n                    ) + '/?(\\\\\\\\?[^\"]*)?', // allow optional query string\n                reg = new RegExp(regexDirectiveUrl);\n\n            return content.gsub(reg, function (match) {\n                return Base64.mageDecode(decodeURIComponent(match[1]).replace(/\\/$/, '')).replace(/\"/g, '&quot;');\n            });\n        },\n\n        /**\n         * @param {Object} attributes\n         * @return {Object}\n         */\n        parseAttributesString: function (attributes) {\n            var result = {};\n\n            // Decode &quot; entity, as regex below does not support encoded quote\n            attributes = attributes.replace(/&quot;/g, '\"');\n\n            attributes.gsub(\n                /(\\w+)(?:\\s*=\\s*(?:(?:\"((?:\\\\.|[^\"])*)\")|(?:'((?:\\\\.|[^'])*)')|([^>\\s]+)))?/,\n                function (match) {\n                    result[match[1]] = match[2];\n                }\n            );\n\n            return result;\n        },\n\n        /**\n         * @param {Object} content\n         * @return {*}\n         */\n        decodeContent: function (content) {\n            if (this.config['add_directives']) {\n                content = this.decodeDirectives(content);\n            }\n\n            content = varienGlobalEvents.fireEventReducer('wysiwygDecodeContent', content);\n\n            return content;\n        },\n\n        /**\n         * @param {Object} content\n         * @return {*}\n         */\n        encodeContent: function (content) {\n            if (this.config['add_directives']) {\n                content = this.encodeDirectives(content);\n            }\n\n            content = varienGlobalEvents.fireEventReducer('wysiwygEncodeContent', content);\n\n            return content;\n        },\n\n        /**\n         * Reinstate contenteditable attributes on .mceNonEditable nodes\n         */\n        addContentEditableAttributeBackToNonEditableNodes: function () {\n            jQuery('.mceNonEditable', this.activeEditor().getDoc()).attr('contenteditable', false);\n        },\n\n        /**\n         * Calls the save method on all editor instances in the collection.\n         */\n        triggerSave: function () {\n            tinyMCE.triggerSave();\n        }\n    };\n\n    return tinyMceWysiwyg.prototype;\n});\n","mage/adminhtml/wysiwyg/tiny_mce/tinymceAdapter.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\n/* global popups, tinyMceEditors, MediabrowserUtility, Base64 */\n/* eslint-disable strict */\ndefine([\n    'jquery',\n    'underscore',\n    'tinymce',\n    'mage/adminhtml/events',\n    'mage/adminhtml/wysiwyg/events',\n    'mage/translate',\n    'prototype',\n    'jquery/ui'\n], function (jQuery, _, tinyMCE, varienGlobalEvents, wysiwygEvents) {\n    'use strict';\n\n    var tinyMceWysiwyg = Class.create();\n\n    tinyMceWysiwyg.prototype = {\n        mediaBrowserOpener: null,\n        mediaBrowserTargetElementId: null,\n        magentoVariablesPlugin: null,\n        mode: 'exact',\n\n        /**\n         * @param {*} htmlId\n         * @param {Object} config\n         */\n        initialize: function (htmlId, config) {\n            this.id = htmlId;\n            this.config = config;\n\n            _.bindAll(\n                this,\n                'beforeSetContent',\n                'saveContent',\n                'onChangeContent',\n                'openFileBrowser',\n                'updateTextArea',\n                'onUndo',\n                'removeEvents'\n            );\n\n            varienGlobalEvents.attachEventHandler('tinymceChange', this.onChangeContent);\n            varienGlobalEvents.attachEventHandler('tinymceBeforeSetContent', this.beforeSetContent);\n            varienGlobalEvents.attachEventHandler('tinymceSetContent', this.updateTextArea);\n            varienGlobalEvents.attachEventHandler('tinymceSaveContent', this.saveContent);\n            varienGlobalEvents.attachEventHandler('tinymceUndo', this.onUndo);\n\n            if (typeof tinyMceEditors === 'undefined') {\n                window.tinyMceEditors = $H({});\n            }\n\n            tinyMceEditors.set(this.id, this);\n        },\n\n        /**\n         * Ensures the undo operation works properly\n         */\n        onUndo: function () {\n            this.addContentEditableAttributeBackToNonEditableNodes();\n        },\n\n        /**\n         * Setup TinyMCE editor\n         */\n        setup: function (mode) {\n            var deferreds = [],\n                settings,\n                self = this;\n\n            this.turnOff();\n\n            if (this.config.plugins) {\n                this.config.plugins.forEach(function (plugin) {\n                    var deferred;\n\n                    self.addPluginToToolbar(plugin.name, '|');\n\n                    if (!plugin.src) {\n                        return;\n                    }\n\n                    deferred = jQuery.Deferred();\n                    deferreds.push(deferred);\n\n                    require([plugin.src], function (factoryFn) {\n                        if (typeof factoryFn === 'function') {\n                            factoryFn(plugin.options);\n                        }\n\n                        tinyMCE.PluginManager.load(plugin.name, plugin.src);\n                        deferred.resolve();\n                    });\n                });\n            }\n\n            if (jQuery.isReady) {\n                tinyMCE.dom.Event.domLoaded = true;\n            }\n\n            settings = this.getSettings();\n\n            if (mode === 'inline') {\n                settings.inline = true;\n\n                if (!isNaN(settings.toolbarZIndex)) {\n                    tinyMCE.ui.FloatPanel.zIndex = settings.toolbarZIndex;\n                }\n\n                this.removeEvents(self.id);\n            }\n\n            jQuery.when.apply(jQuery, deferreds).done(function () {\n                tinyMCE.init(settings);\n                this.getPluginButtons().hide();\n                varienGlobalEvents.clearEventHandlers('open_browser_callback');\n                this.eventBus.clearEventHandlers('open_browser_callback');\n                this.eventBus.attachEventHandler('open_browser_callback', tinyMceEditors.get(self.id).openFileBrowser);\n            }.bind(this));\n        },\n\n        /**\n         * Remove events from instance.\n         *\n         * @param {String} wysiwygId\n         */\n        removeEvents: function (wysiwygId) {\n            var editor;\n\n            if (typeof tinyMceEditors !== 'undefined' && tinyMceEditors.get(wysiwygId)) {\n                editor = tinyMceEditors.get(wysiwygId);\n                varienGlobalEvents.removeEventHandler('tinymceChange', editor.onChangeContent);\n            }\n        },\n\n        /**\n         * Add plugin to the toolbar if not added.\n         *\n         * @param {String} plugin\n         * @param {String} separator\n         */\n        addPluginToToolbar: function (plugin, separator) {\n            var plugins = this.config.tinymce.plugins.split(' '),\n                toolbar = this.config.tinymce.toolbar.split(' ');\n\n            if (plugins.indexOf(plugin) === -1) {\n                plugins.push(plugin);\n            }\n\n            if (toolbar.indexOf(plugin) === -1) {\n                toolbar.push(separator || '', plugin);\n            }\n\n            this.config.tinymce.plugins = plugins.join(' ');\n            this.config.tinymce.toolbar = toolbar.join(' ');\n        },\n\n        /**\n         * Set the status of the toolbar to disabled or enabled (true for enabled, false for disabled)\n         * @param {Boolean} enabled\n         */\n        setToolbarStatus: function (enabled) {\n            var controlIds = this.get(this.getId()).theme.panel.rootControl.controlIdLookup;\n\n            _.each(controlIds, function (controlId) {\n                controlId.disabled(!enabled);\n                controlId.canFocus = enabled;\n\n                if (controlId.tooltip) {\n                    controlId.tooltip().state.set('rendered', enabled);\n\n                    if (enabled) {\n                        jQuery(controlId.getEl()).children('button').addBack().removeAttr('style');\n                    } else {\n                        jQuery(controlId.getEl()).children('button').addBack().attr('style', 'color: inherit;' +\n                            'background-color: inherit;' +\n                            'border-color: transparent;'\n                        );\n                    }\n                }\n            });\n        },\n\n        /**\n         * @return {Object}\n         */\n        getSettings: function () {\n            var settings,\n                eventBus = this.eventBus;\n\n            settings = {\n                selector: '#' + this.getId(),\n                theme: 'silver',\n                skin: 'oxide',\n                'toolbar_mode': 'wrap',\n                'entity_encoding': 'raw',\n                'convert_urls': false,\n                'content_css': this.config.tinymce['content_css'],\n                'relative_urls': true,\n                'valid_children': '+body[style]',\n                menubar: false,\n                plugins: this.config.tinymce.plugins,\n                toolbar: this.config.tinymce.toolbar,\n                adapter: this,\n                'body_id': 'html-body',\n\n                /**\n                 * @param {Object} editor\n                 */\n                setup: function (editor) {\n                    var onChange;\n\n                    editor.on('BeforeSetContent', function (evt) {\n                        varienGlobalEvents.fireEvent('tinymceBeforeSetContent', evt);\n                        eventBus.fireEvent(wysiwygEvents.beforeSetContent);\n                    });\n\n                    editor.on('SaveContent', function (evt) {\n                        varienGlobalEvents.fireEvent('tinymceSaveContent', evt);\n                        eventBus.fireEvent(wysiwygEvents.afterSave);\n                    });\n\n                    editor.on('paste', function (evt) {\n                        varienGlobalEvents.fireEvent('tinymcePaste', evt);\n                        eventBus.fireEvent(wysiwygEvents.afterPaste);\n                    });\n\n                    editor.on('PostProcess', function (evt) {\n                        varienGlobalEvents.fireEvent('tinymceSaveContent', evt);\n                        eventBus.fireEvent(wysiwygEvents.afterSave);\n                    });\n\n                    editor.on('undo', function (evt) {\n                        varienGlobalEvents.fireEvent('tinymceUndo', evt);\n                        eventBus.fireEvent(wysiwygEvents.afterUndo);\n                    });\n\n                    editor.on('focus', function () {\n                        eventBus.fireEvent(wysiwygEvents.afterFocus);\n                    });\n\n                    editor.on('blur', function () {\n                        eventBus.fireEvent(wysiwygEvents.afterBlur);\n                    });\n\n                    /**\n                     * @param {*} evt\n                     */\n                    onChange = function (evt) {\n                        varienGlobalEvents.fireEvent('tinymceChange', evt);\n                        eventBus.fireEvent(wysiwygEvents.afterChangeContent);\n                    };\n\n                    editor.on('Change', onChange);\n                    editor.on('keyup', onChange);\n\n                    editor.on('ExecCommand', function (cmd) {\n                        varienGlobalEvents.fireEvent('tinymceExecCommand', cmd);\n                    });\n\n                    editor.on('init', function (args) {\n                        varienGlobalEvents.fireEvent('wysiwygEditorInitialized', args.target);\n                        eventBus.fireEvent(wysiwygEvents.afterInitialization);\n                    });\n                }\n            };\n\n            // Set default initial height\n            settings['min_height'] = this.config.tinymce['min_height'] ? this.config.tinymce['min_height'] : 250;\n\n            if (this.config.skin) {\n                settings.skin = this.config.skin;\n            }\n\n            if (this.config['toolbar_mode']) {\n                settings['toolbar_mode'] = this.config['toolbar_mode'];\n            }\n\n            if (this.config.baseStaticUrl && this.config.baseStaticDefaultUrl) {\n                settings['document_base_url'] = this.config.baseStaticUrl;\n            }\n            // Set the document base URL\n            if (this.config['document_base_url']) {\n                settings['document_base_url'] = this.config['document_base_url'];\n            }\n\n            if (this.config['files_browser_window_url']) {\n                settings['file_picker_callback_types'] = 'file image media';\n\n                /**\n                 * @param {*} callback\n                 * @param {*} value\n                 * @param {*} meta\n                 */\n                settings['file_picker_callback'] = function (callback, value, meta) {\n                    var payload = {\n                        callback: callback,\n                        value: value,\n                        meta: meta\n                    };\n\n                    varienGlobalEvents.fireEvent('open_browser_callback', payload);\n                    this.eventBus.fireEvent('open_browser_callback', payload);\n                }.bind(this);\n            }\n\n            if (this.config.width) {\n                settings.width = this.config.width;\n            }\n\n            if (this.config.height) {\n                settings.height = this.config.height;\n            }\n\n            if (this.config.plugins) {\n                settings.magentoPluginsOptions = {};\n\n                _.each(this.config.plugins, function (plugin) {\n                    settings.magentoPluginsOptions[plugin.name] = plugin.options;\n                });\n            }\n\n            if (this.config.settings) {\n                Object.extend(settings, this.config.settings);\n            }\n\n            return settings;\n        },\n\n        /**\n         * @param {String} id\n         */\n        get: function (id) {\n            return tinyMCE.get(id);\n        },\n\n        /**\n         * @return {String|null}\n         */\n        getId: function () {\n            return this.id || (this.activeEditor() ? this.activeEditor().id : null) || tinyMceEditors.values()[0].id;\n        },\n\n        /**\n         * @return {Object}\n         */\n        activeEditor: function () {\n            return tinyMCE.activeEditor;\n        },\n\n        /**\n         * Insert content to active editor.\n         *\n         * @param {String} content\n         * @param {Boolean} ui\n         */\n        insertContent: function (content, ui) {\n            this.activeEditor().execCommand('mceInsertContent', typeof ui !== 'undefined' ? ui : false, content);\n        },\n\n        /**\n         * Replace entire contents of wysiwyg with string content parameter\n         *\n         * @param {String} content\n         */\n        setContent: function (content) {\n            this.get(this.getId()).setContent(content);\n        },\n\n        /**\n         * Set caret location in WYSIWYG editor.\n         *\n         * @param {Object} targetElement\n         */\n        setCaretOnElement: function (targetElement) {\n            this.activeEditor().selection.select(targetElement);\n            this.activeEditor().selection.collapse();\n        },\n\n        /**\n         * @param {Object} o\n         */\n        openFileBrowser: function (o) {\n            var typeTitle = this.translate('Select Images'),\n                storeId = this.config['store_id'] ? this.config['store_id'] : 0,\n                frameDialog = jQuery('div.mce-container[role=\"dialog\"]'),\n                self = this,\n                wUrl = this.config['files_browser_window_url'] +\n                    'target_element_id/' + this.getId() + '/' +\n                    'store/' + storeId + '/';\n\n            this.mediaBrowserOpener = o.callback;\n\n            if (typeof o.meta.filetype !== 'undefined' && o.meta.filetype !== '') { //eslint-disable-line eqeqeq\n                wUrl = wUrl + 'type/' + o.meta.filetype + '/';\n            }\n\n            frameDialog.hide();\n            jQuery('.tox-tinymce-aux').hide();\n\n            require(['mage/adminhtml/browser'], function () {\n                MediabrowserUtility.openDialog(wUrl, false, false, typeTitle, {\n                    /**\n                     * Closed.\n                     */\n                    closed: function () {\n                        frameDialog.show();\n                        jQuery('.tox-tinymce-aux').show();\n                    },\n\n                    targetElementId: self.activeEditor() ? self.activeEditor().id : null\n                }\n                );\n            });\n        },\n\n        /**\n         * @param {String} string\n         * @return {String}\n         */\n        translate: function (string) {\n            return jQuery.mage.__ ? jQuery.mage.__(string) : string;\n        },\n\n        /**\n         * @return {null}\n         */\n        getMediaBrowserOpener: function () {\n            return this.mediaBrowserOpener;\n        },\n\n        /**\n         * @return {null}\n         */\n        getMediaBrowserTargetElementId: function () {\n            return this.mediaBrowserTargetElementId;\n        },\n\n        /**\n         * @return {jQuery|*|HTMLElement}\n         */\n        getToggleButton: function () {\n            return $('toggle' + this.getId());\n        },\n\n        /**\n         * Get plugins button.\n         */\n        getPluginButtons: function () {\n            return jQuery('#buttons' + this.getId() + ' > button.plugin');\n        },\n\n        /**\n         * @param {*} mode\n         * @return {wysiwygSetup}\n         */\n        turnOn: function (mode) {\n            this.closePopups();\n\n            this.setup(mode);\n\n            this.getPluginButtons().hide();\n\n            return this;\n        },\n\n        /**\n         * @param {String} name\n         */\n        closeEditorPopup: function (name) {\n            if (typeof popups !== 'undefined' && popups[name] !== undefined && !popups[name].closed) {\n                popups[name].close();\n            }\n        },\n\n        /**\n         * @return {wysiwygSetup}\n         */\n        turnOff: function () {\n            this.closePopups();\n\n            this.getPluginButtons().show();\n\n            tinyMCE.execCommand('mceRemoveEditor', false, this.getId());\n\n            return this;\n        },\n\n        /**\n         * Close popups.\n         */\n        closePopups: function () {\n            // close all popups to avoid problems with updating parent content area\n            varienGlobalEvents.fireEvent('wysiwygClosePopups');\n            this.closeEditorPopup('browser_window' + this.getId());\n        },\n\n        /**\n         * @return {Boolean}\n         */\n        toggle: function () {\n            var content;\n\n            if (!tinyMCE.get(this.getId())) {\n                this.turnOn();\n\n                return true;\n            }\n\n            content = this.get(this.getId()) ? this.get(this.getId()).getContent() : this.getTextArea().val();\n\n            this.turnOff();\n\n            if (content.match(/{{.+?}}/g)) {\n                this.getTextArea().val(content.replace(/&quot;/g, '\"'));\n            }\n\n            return false;\n        },\n\n        /**\n         * On form validation.\n         */\n        onFormValidation: function () {\n            if (tinyMCE.get(this.getId())) {\n                $(this.getId()).value = tinyMCE.get(this.getId()).getContent();\n            }\n        },\n\n        /**\n         * On change content.\n         */\n        onChangeContent: function () {\n            // Add \"changed\" to tab class if it exists\n            var tab;\n\n            this.updateTextArea();\n\n            if (this.config['tab_id']) {\n                tab = $$('a[id$=' + this.config['tab_id'] + ']')[0];\n\n                if ($(tab) != undefined && $(tab).hasClassName('tab-item-link')) { //eslint-disable-line eqeqeq\n                    $(tab).addClassName('changed');\n                }\n            }\n        },\n\n        /**\n         * @param {Object} o\n         */\n        beforeSetContent: function (o) {\n            o.content = this.encodeContent(o.content);\n        },\n\n        /**\n         * @param {Object} o\n         */\n        saveContent: function (o) {\n            o.content = this.decodeContent(o.content);\n        },\n\n        /**\n         * Return the content stored in the WYSIWYG field\n         * @param {String} id\n         * @return {String}\n         */\n        getContent: function (id) {\n            return id ? this.get(id).getContent() : this.get(this.getId()).getContent();\n        },\n\n        /**\n         * @returns {Object}\n         */\n        getAdapterPrototype: function () {\n            return tinyMceWysiwyg;\n        },\n\n        /**\n         * Fix range selection placement when typing.  This fixes MAGETWO-84769\n         * @param {Object} editor\n         */\n        fixRangeSelection: function (editor) {\n            var selection = editor.selection,\n                dom = editor.dom,\n                rng = dom.createRng(),\n                doc = editor.getDoc(),\n                markerHtml,\n                marker;\n\n            // Validate the range we're trying to fix is contained within the current editors document\n            if (!selection.getContent().length && jQuery.contains(doc, selection.getRng().startContainer)) {\n                markerHtml = '<span id=\"mce_marker\" data-mce-type=\"bookmark\">\\uFEFF</span>';\n                selection.setContent(markerHtml);\n                marker = dom.get('mce_marker');\n                rng.setStartBefore(marker);\n                rng.setEndBefore(marker);\n                dom.remove(marker);\n                selection.setRng(rng);\n            }\n        },\n\n        /**\n         * Update text area.\n         */\n        updateTextArea: function () {\n            var editor = this.get(this.getId()),\n                content;\n\n            if (!editor || editor.id !== this.activeEditor().id) {\n                return;\n            }\n\n            this.addContentEditableAttributeBackToNonEditableNodes();\n\n            content = editor.getContent();\n            content = this.decodeContent(content);\n\n            this.getTextArea().val(content).trigger('change');\n        },\n\n        /**\n         * @return {Object} jQuery textarea element\n         */\n        getTextArea: function () {\n            return jQuery('#' + this.getId());\n        },\n\n        /**\n         * Set the status of the editor and toolbar\n         *\n         * @param {Boolean} enabled\n         */\n        setEnabledStatus: function (enabled) {\n            if (this.activeEditor()) {\n                this.activeEditor().getBody().setAttribute('contenteditable', enabled);\n                this.activeEditor().readonly = !enabled;\n                this.setToolbarStatus(enabled);\n            }\n\n            if (enabled) {\n                this.getTextArea().prop('disabled', false);\n            } else {\n                this.getTextArea().prop('disabled', 'disabled');\n            }\n        },\n\n        /**\n         * Retrieve directives URL with substituted directive value.\n         *\n         * @param {String} directive\n         */\n        makeDirectiveUrl: function (directive) {\n            return this.config['directives_url']\n                .replace(/directive/, 'directive/___directive/' + directive)\n                .replace(/\\/$/, '');\n        },\n\n        /**\n         * Convert {{directive}} style attributes syntax to absolute URLs\n         * @param {Object} content\n         * @return {*}\n         */\n        encodeDirectives: function (content) {\n            // collect all HTML tags with attributes that contain directives\n            // eslint-disable-next-line no-useless-escape\n            return content.gsub(/<([a-z0-9\\-\\_]+[^>]+?)([a-z0-9\\-\\_]+=\"[^\"]*?\\{\\{.+?\\}\\}.*?\".*?)>/i, function (match) {\n                var attributesString = match[2],\n                    decodedDirectiveString;\n\n                // process tag attributes string\n                // eslint-disable-next-line no-useless-escape\n                attributesString = attributesString.gsub(/([a-z0-9\\-\\_]+)=\"(.*?)(\\{\\{.+?\\}\\})(.*?)\"/i, function (m) {\n                    decodedDirectiveString = encodeURIComponent(Base64.mageEncode(m[3].replace(/&quot;/g, '\"') + m[4]));\n\n                    return m[1] + '=\"' + m[2] + this.makeDirectiveUrl(decodedDirectiveString) + '\"';\n                }.bind(this));\n\n                return '<' + match[1] + attributesString + '>';\n            }.bind(this));\n        },\n\n        /**\n         * Convert absolute URLs to {{directive}} style attributes syntax\n         * @param {Object} content\n         * @return {*}\n         */\n        decodeDirectives: function (content) {\n            var directiveUrl = this.makeDirectiveUrl('%directive%').split('?')[0], // remove query string from directive\n                // escape special chars in directives url to use in regular expression\n                // eslint-disable-next-line no-useless-escape\n                regexEscapedDirectiveUrl = directiveUrl.replace(/([$^.?*!+:=()\\[\\]{}|\\\\])/g, '\\\\$1'),\n                regexDirectiveUrl = regexEscapedDirectiveUrl\n                    .replace(\n                        '%directive%',\n                        '([a-zA-Z0-9,_-]+(?:%2[A-Z]|)+\\/?)(?:(?!\").)*' // eslint-disable-line no-useless-escape\n                    ) + '/?(\\\\\\\\?[^\"]*)?', // allow optional query string\n                reg = new RegExp(regexDirectiveUrl);\n\n            return content.gsub(reg, function (match) {\n                return Base64.mageDecode(decodeURIComponent(match[1]).replace(/\\/$/, '')).replace(/\"/g, '&quot;');\n            });\n        },\n\n        /**\n         * @param {Object} attributes\n         * @return {Object}\n         */\n        parseAttributesString: function (attributes) {\n            var result = {};\n\n            // Decode &quot; entity, as regex below does not support encoded quote\n            attributes = attributes.replace(/&quot;/g, '\"');\n\n            attributes.gsub(\n                /(\\w+)(?:\\s*=\\s*(?:(?:\"((?:\\\\.|[^\"])*)\")|(?:'((?:\\\\.|[^'])*)')|([^>\\s]+)))?/,\n                function (match) {\n                    result[match[1]] = match[2];\n                }\n            );\n\n            return result;\n        },\n\n        /**\n         * @param {Object} content\n         * @return {*}\n         */\n        decodeContent: function (content) {\n            if (this.config['add_directives']) {\n                content = this.decodeDirectives(content);\n            }\n\n            content = varienGlobalEvents.fireEventReducer('wysiwygDecodeContent', content);\n\n            return content;\n        },\n\n        /**\n         * @param {Object} content\n         * @return {*}\n         */\n        encodeContent: function (content) {\n            if (this.config['add_directives']) {\n                content = this.encodeDirectives(content);\n            }\n\n            content = varienGlobalEvents.fireEventReducer('wysiwygEncodeContent', content);\n\n            return content;\n        },\n\n        /**\n         * Reinstate contenteditable attributes on .mceNonEditable nodes\n         */\n        addContentEditableAttributeBackToNonEditableNodes: function () {\n            jQuery('.mceNonEditable', this.activeEditor().getDoc()).attr('contenteditable', false);\n        },\n\n        /**\n         * Calls the save method on all editor instances in the collection.\n         */\n        triggerSave: function () {\n            tinyMCE.triggerSave();\n        }\n    };\n\n    return tinyMceWysiwyg.prototype;\n});\n","mage/adminhtml/wysiwyg/tiny_mce/setup.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\n/* eslint-disable strict */\ndefine([\n    'jquery',\n    'underscore',\n    'wysiwygAdapter',\n    'module',\n    'mage/translate',\n    'prototype',\n    'mage/adminhtml/events',\n    'mage/adminhtml/browser'\n], function (jQuery, _, wysiwygAdapter, module) {\n    var baseConfig = module.config().config || {},\n        wysiwygSetup = Class.create({\n        wysiwygInstance: null\n    });\n\n    wysiwygSetup.prototype = {\n\n        /**\n         * @param {*} htmlId\n         * @param {Object} config\n         */\n        initialize: function (htmlId, config) {\n            var WysiwygInstancePrototype = new wysiwygAdapter.getAdapterPrototype();\n\n            _.bindAll(this, 'openFileBrowser');\n\n            config = _.extend({}, baseConfig, config || {});\n            this.wysiwygInstance = new WysiwygInstancePrototype(htmlId, config);\n            this.wysiwygInstance.eventBus = this.eventBus = new window.varienEvents();\n        },\n\n        /**\n         * @param {*} mode\n         */\n        setup: function (mode) {\n            this.wysiwygInstance.setup(mode);\n        },\n\n        /**\n         * @param {Object} o\n         */\n        openFileBrowser: function (o) {\n            this.wysiwygInstance.openFileBrowser(o);\n        },\n\n        /**\n         * @return {Boolean}\n         */\n        toggle: function () {\n            return this.wysiwygInstance.toggle();\n        },\n\n        /**\n         * On form validation.\n         */\n        onFormValidation: function () {\n            this.wysiwygInstance.onFormValidation();\n        },\n\n        /**\n         * Encodes the content so it can be inserted into the wysiwyg\n         * @param {String} content - The content to be encoded\n         *\n         * @returns {*} - The encoded content\n         */\n        updateContent: function (content) {\n            return this.wysiwygInstance.encodeContent(content);\n        }\n\n    };\n    window.wysiwygSetup = wysiwygSetup;\n\n    return wysiwygSetup;\n});\n","mage/adminhtml/wysiwyg/tiny_mce/html5-schema.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\ndefine([\n    'underscore'\n], function (_) {\n    'use strict';\n\n    /* eslint-disable max-len */\n\n    var schema = {\n        blockContent: [\n            'address', 'article', 'aside', 'blockquote', 'details', 'dialog', 'div', 'dl', 'fieldset',\n            'figure', 'footer', 'form', 'h1', 'h2', 'h3', 'h4', 'h5', 'h6', 'header', 'hgroup', 'hr',\n            'menu', 'nav', 'ol', 'p', 'pre', 'section', 'table', 'ul'\n        ],\n        phrasingContent: [\n            '#comment', '#text', 'a', 'abbr', 'audio', 'b', 'bdi', 'bdo', 'br', 'button', 'canvas',\n            'cite','code', 'command', 'datalist', 'del', 'dfn', 'em', 'embed', 'i', 'iframe', 'img',\n            'input', 'ins', 'kbd', 'keygen', 'label', 'map', 'mark', 'meter', 'noscript', 'object',\n            'output', 'picture', 'progress', 'q', 'ruby', 's', 'samp', 'script', 'select', 'small',\n            'span', 'strong', 'sub', 'sup', 'textarea', 'time', 'u', 'var', 'video', 'wbr'\n        ],\n        blockElements: [\n            'address', 'article', 'aside', 'blockquote', 'caption', 'center', 'datalist', 'dd', 'dir', 'div',\n            'dl', 'dt', 'fieldset', 'figcaption', 'figure', 'footer', 'form', 'h1', 'h2', 'h3', 'h4', 'h5', 'h6',\n            'header', 'hgroup', 'hr', 'isindex', 'li', 'menu', 'nav', 'noscript', 'ol', 'optgroup', 'option',\n            'p', 'pre', 'section', 'select', 'table', 'tbody', 'td', 'tfoot', 'th', 'thead', 'tr', 'ul'\n        ],\n        boolAttrs: [\n            'autoplay', 'checked', 'compact', 'controls', 'declare', 'defer', 'disabled', 'ismap', 'loop',\n            'multiple', 'nohref', 'noresize', 'noshade', 'nowrap', 'readonly', 'selected'\n        ],\n        shortEnded: [\n            'area', 'base', 'basefont', 'br', 'col', 'embed', 'frame', 'hr', 'img', 'input', 'isindex',\n            'link', 'meta', 'param', 'source', 'track', 'wbr'\n        ],\n        whiteSpace: [\n            'audio', 'iframe', 'noscript', 'object', 'pre', 'script', 'style', 'textarea', 'video'\n        ],\n        selfClosing: [\n            'colgroup', 'dd', 'dt', 'li', 'option', 'p', 'td', 'tfoot', 'th', 'thead', 'tr'\n        ]\n    };\n\n    schema.flowContent = schema.blockContent.concat(schema.phrasingContent, ['style']);\n    schema.nonEmpty = ['td', 'th', 'iframe', 'video', 'audio', 'object', 'script', 'i', 'em', 'span'].concat(schema.shortEnded);\n\n    _.extend(schema, (function (phrasingContent, flowContent) {\n        var validElements   = [],\n            validChildren   = [],\n            compiled        = {},\n            globalAttrs,\n            rawData;\n\n        globalAttrs = [\n            'id', 'dir', 'lang', 'class', 'style', 'title', 'hidden', 'onclick', 'onkeyup',\n            'tabindex', 'dropzone', 'accesskey', 'draggable', 'translate', 'onmouseup',\n            'onkeydown', 'spellcheck', 'ondblclick', 'onmouseout', 'onkeypress', 'contextmenu',\n            'onmousedown', 'onmouseover', 'onmousemove', 'contenteditable'\n        ];\n\n        rawData = [\n            ['html', 'manifest', 'head body'],\n            ['head', '', 'base command link meta noscript script style title'],\n            ['title hr noscript br'],\n            ['base', 'href target'],\n            ['link', 'href rel media hreflang type sizes hreflang'],\n            ['meta', 'name http-equiv content charset'],\n            ['style', 'media type scoped'],\n            ['script', 'src async defer type charset'],\n            ['body', 'onafterprint onbeforeprint onbeforeunload onblur onerror onfocus ' +\n                'onhashchange onload onmessage onoffline ononline onpagehide onpageshow ' +\n                'onpopstate onresize onscroll onstorage onunload background bgcolor text link vlink alink', flowContent\n            ],\n            ['caption', '', _.without(flowContent, 'table')],\n            ['address dt dd div', '', flowContent],\n            ['h1 h2 h3 h4 h5 h6 pre p abbr code var samp kbd sub sup i b u bdo span legend em strong small s cite dfn', '', phrasingContent],\n            ['blockquote', 'cite', flowContent],\n            ['ol', 'reversed start type', 'li'],\n            ['ul', 'type compact', 'li'],\n            ['li', 'value type', flowContent],\n            ['dl', '', 'dt dd'],\n            ['a', 'href target rel media hreflang type charset name rev shape coords download', phrasingContent],\n            ['q', 'cite', phrasingContent],\n            ['ins del', 'cite datetime', flowContent],\n            ['img', 'src sizes srcset alt usemap ismap width height name longdesc align border hspace vspace'],\n            ['iframe', 'src name width height longdesc frameborder marginwidth marginheight scrolling align sandbox seamless allowfullscreen', flowContent],\n            ['embed', 'src type width height'],\n            ['object', 'data type typemustmatch name usemap form width height declare classid code codebase codetype archive standby align border hspace vspace', flowContent.concat(['param'])],\n            ['param', 'name value valuetype type'],\n            ['map', 'name', flowContent.concat(['area'])],\n            ['area', 'alt coords shape href target rel media hreflang type nohref'],\n            ['table', 'border summary width frame rules cellspacing cellpadding align bgcolor', 'caption colgroup thead tfoot tbody tr col'],\n            ['colgroup', 'span width align char charoff valign', 'col'],\n            ['col', 'span'],\n            ['tbody thead tfoot', 'align char charoff valign', 'tr'],\n            ['tr', 'align char charoff valign bgcolor', 'td th'],\n            ['td', 'colspan rowspan headers abbr axis scope align char charoff valign nowrap bgcolor width height', flowContent],\n            ['th', 'colspan rowspan headers scope abbr axis align char charoff valign nowrap bgcolor width height accept', flowContent],\n            ['form', 'accept-charset action autocomplete enctype method name novalidate target onsubmit onreset', flowContent],\n            ['fieldset', 'disabled form name', flowContent.concat(['legend'])],\n            ['label', 'form for', phrasingContent],\n            ['input', 'accept alt autocomplete checked dirname disabled form formaction formenctype formmethod formnovalidate ' +\n                'formtarget height list max maxlength min multiple name pattern readonly required size src step type value width usemap align'\n            ],\n            ['button', 'disabled form formaction formenctype formmethod formnovalidate formtarget name type value', phrasingContent],\n            ['select', 'disabled form multiple name required size onfocus onblur onchange', 'option optgroup'],\n            ['optgroup', 'disabled label', 'option'],\n            ['option', 'disabled label selected value'],\n            ['textarea', 'cols dirname disabled form maxlength name readonly required rows wrap'],\n            ['menu', 'type label', flowContent.concat(['li'])],\n            ['noscript', '', flowContent],\n            ['wbr'],\n            ['ruby', '', phrasingContent.concat(['rt', 'rp'])],\n            ['figcaption', '', flowContent],\n            ['mark rt rp summary bdi', '', phrasingContent],\n            ['canvas', 'width height', flowContent],\n            ['video', 'src crossorigin poster preload autoplay mediagroup loop muted controls width height buffered', flowContent.concat(['track', 'source'])],\n            ['audio', 'src crossorigin preload autoplay mediagroup loop muted controls buffered volume', flowContent.concat(['track', 'source'])],\n            ['picture', '', 'img source'],\n            ['source', 'src srcset type media sizes'],\n            ['track', 'kind src srclang label default'],\n            ['datalist', '', phrasingContent.concat(['option'])],\n            ['article section nav aside header footer', '', flowContent],\n            ['hgroup', '', 'h1 h2 h3 h4 h5 h6'],\n            ['figure', '', flowContent.concat(['figcaption'])],\n            ['time', 'datetime', phrasingContent],\n            ['dialog', 'open', flowContent],\n            ['command', 'type label icon disabled checked radiogroup command'],\n            ['output', 'for form name', phrasingContent],\n            ['progress', 'value max', phrasingContent],\n            ['meter', 'value min max low high optimum', phrasingContent],\n            ['details', 'open', flowContent.concat(['summary'])],\n            ['keygen', 'autofocus challenge disabled form keytype name'],\n            ['script', 'language xml:space'],\n            ['style', 'xml:space'],\n            ['embed', 'align name hspace vspace'],\n            ['br', 'clear'],\n            ['applet', 'codebase archive code object alt name width height align hspace vspace'],\n            ['font basefont', 'size color face'],\n            ['h1 h2 h3 h4 h5 h6 div p legend caption', 'align'],\n            ['ol dl menu dir', 'compact'],\n            ['pre', 'width xml:space'],\n            ['hr', 'align noshade size width'],\n            ['isindex', 'prompt'],\n            ['col', 'width align char charoff valign'],\n            ['input button select textarea', 'autofocus'],\n            ['input textarea', 'placeholder onselect onchange onfocus onblur'],\n            ['link script img', 'crossorigin']\n        ];\n\n        rawData.forEach(function (data) {\n            var nodes       = data[0].split(' '),\n                attributes  = data[1] || [],\n                children    = data[2] || [],\n                ni          = nodes.length,\n                nodeName,\n                schemaData;\n\n            if (typeof attributes === 'string') {\n                attributes = attributes.split(' ');\n            }\n\n            if (typeof children === 'string') {\n                children = children.split(' ');\n            }\n\n            while (ni--) {\n                nodeName    = nodes[ni];\n                schemaData  = compiled[nodeName] || {};\n\n                compiled[nodeName] = {\n                    attributes: _.union(schemaData.attributes, globalAttrs, attributes),\n                    children: _.union(schemaData.children, children)\n                };\n            }\n        });\n\n        ['a', 'dfn', 'form', 'meter', 'progress'].forEach(function (nodeName) {\n            var node = compiled[nodeName];\n\n            node.children = _.without(node.children, nodeName);\n        });\n\n        _.each(compiled, function (node, nodeName) {\n            var filteredAttributes = [];\n\n            _.each(node.attributes, function (attribute) { //eslint-disable-line max-nested-callbacks\n                // Disallowing usage of 'on*' attributes.\n                if (!/^on/.test(attribute)) {\n                    filteredAttributes.push(attribute);\n                }\n            });\n\n            node.attributes = filteredAttributes;\n\n            validElements.push(nodeName + '[' + node.attributes.join('|') + ']');\n            validChildren.push(nodeName + '[' + node.children.join('|') + ']');\n        });\n\n        return {\n            nodes: compiled,\n            validElements: validElements,\n            validChildren: validChildren\n        };\n    })(schema.phrasingContent, schema.flowContent));\n\n    return schema;\n});\n","mage/adminhtml/wysiwyg/tiny_mce/plugins/magentovariable/editor_plugin.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\n/* global tinymce, MagentovariablePlugin, varienGlobalEvents, Base64 */\n/* eslint-disable strict */\ndefine([\n    'Magento_Variable/js/config-directive-generator',\n    'Magento_Variable/js/custom-directive-generator',\n    'wysiwygAdapter',\n    'jquery',\n    'mage/adminhtml/tools'\n], function (configDirectiveGenerator, customDirectiveGenerator, wysiwyg, jQuery) {\n    return function (config) {\n        var self = null,\n\n            magentovariable = function (editor) {\n                this.widget = {\n                    /**\n                     * Encode variables in content\n                     *\n                     * @param {String} content\n                     * @returns {*}\n                     */\n                    encodeVariables: function (content) {\n                        content = content.gsub(/\\{\\{config path=\"([^\"]+)\"}}/i, function (match) {\n                            var path = match[1],\n                                magentoVariables,\n                                imageHtml;\n\n                            magentoVariables = JSON.parse(config.placeholders);\n\n                            if (magentoVariables[match[1]]\n                                && magentoVariables[match[1]]['variable_type'] === 'default') {\n                                imageHtml = '<span id=\"%id\" ' +\n                                    'class=\"magento-variable magento-placeholder mceNonEditable\">' +\n                                    '%s</span>';\n                                imageHtml = imageHtml.replace('%s', magentoVariables[match[1]]['variable_name']);\n                            } else {\n                                imageHtml = '<span id=\"%id\" class=\"' +\n                                    'magento-variable magento-placeholder magento-placeholder-error ' +\n                                    'mceNonEditable' +\n                                    '\">' +\n                                    'Not found' +\n                                    '</span>';\n                            }\n\n                            return imageHtml.replace('%id', Base64.idEncode(path));\n                        });\n\n                        content = content.gsub(/\\{\\{customVar code=([^}\"]+)}}/i, function (match) {\n                            var path = match[1],\n                                magentoVariables,\n                                imageHtml;\n\n                            magentoVariables = JSON.parse(config.placeholders);\n\n                            if (magentoVariables[match[1]]\n                                && magentoVariables[match[1]]['variable_type'] === 'custom') {\n                                imageHtml = '<span id=\"%id\" class=\"magento-variable magento-custom-var ' +\n                                    'magento-placeholder mceNonEditable\">%s</span>';\n                                imageHtml = imageHtml.replace('%s', magentoVariables[match[1]]['variable_name']);\n                            } else {\n                                imageHtml = '<span id=\"%id\" class=\"' +\n                                    'magento-variable magento-custom-var magento-placeholder ' +\n                                    'magento-placeholder-error mceNonEditable' +\n                                    '\">' +\n                                    match[1] +\n                                    '</span>';\n                            }\n\n                            return imageHtml.replace('%id', Base64.idEncode(path));\n                        });\n\n                        return content;\n                    },\n\n                    /**\n                     * Decode variables in content.\n                     *\n                     * @param {String} content\n                     * @returns {String}\n                     */\n                    decodeVariables: function (content) {\n                        var doc = new DOMParser().parseFromString(content.replace(/&quot;/g, '&amp;quot;'),\n                                'text/html'),\n                            returnval = '';\n\n                        [].forEach.call(doc.querySelectorAll('span.magento-variable'), function (el) {\n                            var $el = jQuery(el);\n\n                            if ($el.hasClass('magento-custom-var')) {\n                                $el.replaceWith(\n                                    customDirectiveGenerator.processConfig(\n                                        Base64.idDecode(\n                                            $el.attr('id')\n                                        )\n                                    )\n                                );\n                            } else {\n                                $el.replaceWith(\n                                    configDirectiveGenerator.processConfig(\n                                        Base64.idDecode(\n                                            $el.attr('id')\n                                        )\n                                    )\n                                );\n                            }\n                        });\n\n                        returnval += doc.head ? doc.head.innerHTML.replace(/&amp;quot;/g, '&quot;') : '';\n                        returnval += doc.body ? doc.body.innerHTML.replace(/&amp;quot;/g, '&quot;') : '';\n\n                        return returnval ? returnval : content;\n                    },\n\n                    /**\n                     * @return {Object}\n                     */\n                    getInfo: function () {\n                        return {\n                            longname: 'Magento Variable Manager Plugin',\n                            author: 'Magento Core Team',\n                            authorurl: 'http://magentocommerce.com',\n                            infourl: 'http://magentocommerce.com',\n                            version: '1.0'\n                        };\n                    }\n                };\n\n                self = this.widget;\n\n                this.activePlaceholder = null;\n\n                /**\n                 * Add new command to open variables selector slideout.\n                 */\n                editor.addCommand('openVariablesSlideout', function (commandConfig) {\n                    var selectedElement;\n\n                    if (commandConfig) {\n                        selectedElement = commandConfig.selectedElement;\n                    } else {\n                        selectedElement = tinymce.activeEditor.selection.getNode();\n                    }\n                    MagentovariablePlugin.setEditor(editor);\n                    MagentovariablePlugin.loadChooser(\n                        config.url,\n                        wysiwyg.getId(),\n                        selectedElement\n                    );\n                });\n\n                /**\n                 * Add button to the editor toolbar.\n                 */\n                editor.ui.registry.addIcon(\n                    'magentovariable',\n                    '<svg width=\"24\" height=\"24\" viewBox=\"0 0 32.000000 32.000000\" ' +\n                    'preserveAspectRatio=\"xMidYMid meet\"><g transform=\"translate(0.000000,32.000000) ' +\n                    'scale(0.100000,-0.100000)\" fill=\"#000000\" stroke=\"none\"><path d=\"M68 250 c-56 -44 -75 -136 -37 ' +\n                    '-184 27 -34 42 -33 23 2 -26 50 -9 129 38 179 26 28 10 30 -24 3z\"/><path d=\"M266 253 c5 -10 9 ' +\n                    '-41 9 -70 0 -42 -6 -60 -32 -97 -36 -51 -35 -56 7 -26 54 39 78 139 44 188 -18 26 -40 30 -28 5z\"/>' +\n                    '<path d=\"M128 223 c-15 -4 -15 -6 0 -33 16 -28 16 -30 -11 -58 -30 -31 -34 -42 -13 -42 8 0 17 11 ' +\n                    '20 25 4 14 11 25 16 25 5 0 12 -11 16 -25 6 -25 37 -35 49 -15 3 5 2 10 -3 10 -23 0 -20 44 5 76 ' +\n                    '25 34 25 34 4 34 -12 0 -20 -4 -17  -8 2 -4 0 -14 -5 -22 -7 -10 -12 -11 -15 -4 -10 25 -30 40 ' +\n                    '-46 37z\"/></g>' +\n                    '</svg>'\n                );\n                editor.ui.registry.addToggleButton('magentovariable', {\n                    icon: 'magentovariable',\n                    tooltip: jQuery.mage.__('Insert Variable'),\n\n                    /**\n                     * execute openVariablesSlideout for onAction callback\n                     */\n                    onAction: function () {\n                        editor.execCommand('openVariablesSlideout');\n                    },\n\n                    /**\n                     * Highlight or dismiss Insert Variable button when variable is selected or deselected.\n                     */\n                    onSetup: function (api) {\n                        /**\n                         * Toggle active state of Insert Variable button.\n                         *\n                         * @param {Object} e\n                         */\n                        var toggleVariableButton = function (e) {\n                            api.setActive(false);\n\n                            if (jQuery(e.target).hasClass('magento-variable')) {\n                                api.setActive(true);\n                            }\n                        };\n\n                        editor.on('click', toggleVariableButton);\n                        editor.on('change', toggleVariableButton);\n                    }\n                });\n\n                /**\n                 * Double click handler on the editor to handle dbl click on variable placeholder.\n                 */\n                editor.on('dblclick', function (evt) {\n                    if (jQuery(evt.target).hasClass('magento-variable')) {\n                        editor.selection.collapse(false);\n                        editor.execCommand('openVariablesSlideout', {\n                            ui: true,\n                            selectedElement: evt.target\n                        });\n                    }\n                });\n\n                /**\n                 * Attach event handler for when wysiwyg editor is about to encode its content\n                 */\n                varienGlobalEvents.attachEventHandler('wysiwygEncodeContent', function (content) {\n                    content = self.encodeVariables(content);\n\n                    return content;\n                });\n\n                /**\n                 * Attach event handler for when wysiwyg editor is about to decode its content\n                 */\n                varienGlobalEvents.attachEventHandler('wysiwygDecodeContent', function (content) {\n                    content = self.decodeVariables(content);\n\n                    return content;\n                });\n            };\n\n        /**\n         * Register plugin\n         */\n        tinymce.PluginManager.add('magentovariable', magentovariable);\n    };\n});\n","mage/adminhtml/wysiwyg/tiny_mce/plugins/magentowidget/editor_plugin.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\n/* global tinymce, widgetTools, Base64 */\n/* eslint-disable strict */\ndefine([\n    'wysiwygAdapter',\n    'mage/adminhtml/events',\n    'mage/adminhtml/wysiwyg/widget'\n], function (wysiwyg, varienGlobalEvents) {\n    return function (config) {\n        var self = null,\n\n            magentowidget = function (editor) {\n                this.widget = {\n                    /**\n                     * @param {Object} placeholder - Contains the selected node\n                     * @returns {Boolean}\n                     */\n                    isWidgetPlaceholderSelected: function (placeholder) {\n                        var isSelected = false;\n\n                        if (placeholder.nodeName &&\n                            (placeholder.nodeName === 'SPAN' || placeholder.nodeName === 'IMG') &&\n                            placeholder.className && placeholder.className.indexOf('magento-widget') !== -1\n                        ) {\n                            this.activePlaceholder = placeholder;\n                            isSelected = true;\n                        } else {\n                            this.activePlaceholder = null;\n                        }\n\n                        return isSelected;\n                    },\n\n                    /**\n                     * Convert {{widget}} style syntax to image placeholder HTML\n                     * @param {String} content\n                     * @return {*}\n                     */\n                    encodeWidgets: function (content) {\n                        return content.gsub(/\\{\\{widget([\\S\\s]*?)}}/i, function (match) {\n                            var attributes = wysiwyg.parseAttributesString(match[1]),\n                                imageSrc,\n                                imageHtml = '';\n\n                            if (attributes.type) {\n                                attributes.type = attributes.type.replace(/\\\\\\\\/g, '\\\\');\n                                imageSrc = config.placeholders[attributes.type];\n\n                                if (imageSrc) {\n                                    imageHtml += '<span class=\"magento-placeholder magento-widget mceNonEditable\" ' +\n                                        'contenteditable=\"false\">';\n                                } else {\n                                    imageSrc = config['error_image_url'];\n                                    imageHtml += '<span ' +\n                                        'class=\"magento-placeholder magento-placeholder-error magento-widget ' +\n                                        'mceNonEditable\" ' +\n                                        'contenteditable=\"false\">';\n                                }\n\n                                imageHtml += '<img';\n                                imageHtml += ' id=\"' + Base64.idEncode(match[0]) + '\"';\n                                imageHtml += ' src=\"' + imageSrc + '\"';\n                                imageHtml += ' />';\n\n                                if (config.types[attributes.type]) {\n                                    imageHtml += config.types[attributes.type];\n                                }\n\n                                imageHtml += '</span>';\n\n                                return imageHtml;\n                            }\n                        });\n                    },\n\n                    /**\n                     * Convert image placeholder HTML to {{widget}} style syntax\n                     * @param {String} content\n                     * @return {*}\n                     */\n                    decodeWidgets: function (content) {\n                        return content.gsub(\n                            /(<span class=\"[^\"]*magento-widget[^\"]*\"[^>]*>)?<img([^>]+id=\"[^>]+)>(([^>]*)<\\/span>)?/i,\n                            function (match) {\n                                var attributes = wysiwyg.parseAttributesString(match[2]),\n                                    widgetCode,\n                                    result = match[0];\n\n                                if (attributes.id) {\n                                    try {\n                                        widgetCode = Base64.idDecode(attributes.id);\n                                    } catch (e) {\n                                        // Ignore and continue.\n                                    }\n\n                                    if (widgetCode && widgetCode.indexOf('{{widget') !== -1) {\n                                        result = widgetCode;\n                                    }\n                                }\n\n                                return result;\n                            }\n                        );\n                    },\n\n                    /**\n                     * Tinymce has strange behavior with html and this removes one of its side-effects\n                     * @param {String} content\n                     * @return {String}\n                     */\n                    removeDuplicateAncestorWidgetSpanElement: function (content) {\n                        var parser, doc, returnval = '';\n\n                        if (!window.DOMParser) {\n                            return content;\n                        }\n\n                        parser = new DOMParser();\n                        doc = parser.parseFromString(content.replace(/&quot;/g, '&amp;quot;'), 'text/html');\n\n                        [].forEach.call(doc.querySelectorAll('.magento-widget'), function (widgetEl) {\n                            var widgetChildEl = widgetEl.querySelector('.magento-widget');\n\n                            if (!widgetChildEl) {\n                                return;\n                            }\n\n                            [].forEach.call(widgetEl.childNodes, function (el) {\n                                widgetEl.parentNode.insertBefore(el, widgetEl);\n                            });\n\n                            widgetEl.parentNode.removeChild(widgetEl);\n                        });\n\n                        returnval += doc.head ? doc.head.innerHTML.replace(/&amp;quot;/g, '&quot;') : '';\n                        returnval += doc.body ? doc.body.innerHTML.replace(/&amp;quot;/g, '&quot;') : '';\n\n                        return returnval ? returnval : content;\n                    },\n\n                    /**\n                     * @return {Object}\n                     */\n                    getInfo: function () {\n                        return {\n                            longname: 'Magento Widget Manager Plugin',\n                            author: 'Magento Core Team',\n                            authorurl: 'http://magentocommerce.com',\n                            infourl: 'http://magentocommerce.com',\n                            version: '1.0'\n                        };\n                    }\n                };\n\n                self = this.widget;\n\n                this.activePlaceholder = null;\n\n                editor.addCommand('mceMagentowidget', function (img) {\n                    if (self.activePlaceholder) {\n                        img = self.activePlaceholder;\n                    }\n\n                    widgetTools.setActiveSelectedNode(img);\n                    widgetTools.openDialog(\n                        config['window_url'] + 'widget_target_id/' + editor.getElement().id + '/'\n                    );\n                });\n\n                // Register Widget plugin button\n                editor.ui.registry.addIcon(\n                    'magentowidget',\n                    '<svg width=\"24\" height=\"24\" viewBox=\"0 0 32.000000 32.000000\" ' +\n                    'preserveAspectRatio=\"xMidYMid meet\"> <g transform=\"translate(0.000000,32.000000) ' +\n                    'scale(0.100000,-0.100000)\" fill=\"#000000\" stroke=\"none\"> <path d=\"M130 290 c0 -5 13 -10 30 ' +\n                    '-10 22 0 28 -4 24 -15 -5 -11 2 -15 26 -15 21 0 30 -4 28 -12 -7 -20 -40 -22 -50 -4 -5 9 -14 16 ' +\n                    '-20 16 -6 0 -19 7 -28 15 -9 8 -25 12 -38 8 -33 -8 -27 -26 8 -21 34 5 40 -6 12 -21 -14 -7 -25 -6 ' +\n                    '-40 5 -12 8 -23 10 -27 5 -5 -8 88 -71 105 -71 3 0 29 14 58 31 l53 30 -23 18 c-13 10 -31 20 -40 ' +\n                    '24 -10 3 -18 11 -18 17 0 5 -13 10 -30 10 -16 0 -30 -4 -30 -10z m58 -82 c-3 -7 -15 -13 -28 -13 ' +\n                    '-13 0 -25 6 -27 13 -3 8 6 12 27 12 21 0 30 -4 28 -12z\"/> <path d=\"M30 151 l0 -60 61 -36 c33 ' +\n                    '-19 64 -35 69 -35 5 0 36 16 69 35 l61 36 0 60 0 61 -65 -37 -65 -36 -65 36 -65 37 0 -61z\"/> </g>' +\n                    '</svg>'\n                );\n                editor.ui.registry.addToggleButton('magentowidget', {\n                    icon: 'magentowidget',\n                    tooltip: jQuery.mage.__('Insert Widget'),\n\n                    /**\n                     * execute openVariablesSlideout for onAction callback\n                     */\n                    onAction: function () {\n                        editor.execCommand('mceMagentowidget');\n                    },\n\n                    /**\n                     * Add a node change handler, selects the button in the UI when a image is selected\n                     * @param {ToolbarToggleButtonInstanceApi} api\n                     */\n                    onSetup: function (api) {\n                        /**\n                         * NodeChange handler\n                         */\n                        editor.on('NodeChange', function (e) {\n                            var placeholder = e.element;\n\n                            if (self.isWidgetPlaceholderSelected(placeholder)) {\n                                widgetTools.setEditMode(true);\n                                api.setActive(true);\n                            } else {\n                                widgetTools.setEditMode(false);\n                                api.setActive(false);\n                            }\n                        });\n                    }\n                });\n\n                // Add a widget placeholder image double click callback\n                editor.on('dblClick', function (e) {\n                    var placeholder = e.target;\n\n                    if (self.isWidgetPlaceholderSelected(placeholder)) {\n                        widgetTools.setEditMode(true);\n                        this.execCommand('mceMagentowidget', null);\n                    }\n                });\n\n                /**\n                 * Attach event handler for when wysiwyg editor is about to encode its content\n                 */\n                varienGlobalEvents.attachEventHandler('wysiwygEncodeContent', function (content) {\n                    content = self.encodeWidgets(self.decodeWidgets(content));\n                    content = self.removeDuplicateAncestorWidgetSpanElement(content);\n\n                    return content;\n                });\n\n                /**\n                 * Attach event handler for when wysiwyg editor is about to decode its content\n                 */\n                varienGlobalEvents.attachEventHandler('wysiwygDecodeContent', function (content) {\n                    content = self.decodeWidgets(content);\n\n                    return content;\n                });\n\n                /**\n                 * Attach event handler for when popups associated with wysiwyg are about to be closed\n                 */\n                varienGlobalEvents.attachEventHandler('wysiwygClosePopups', function () {\n                    wysiwyg.closeEditorPopup('widget_window' + wysiwyg.getId());\n                });\n            };\n\n        // Register plugin\n        tinymce.PluginManager.add('magentowidget', magentowidget);\n    };\n});\n","mage/validation/url.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\ndefine([], function () {\n    'use strict';\n\n    return {\n\n        /**\n         * Redirects to the url if it is considered safe\n         *\n         * @param {String} path - url to be redirected to\n         */\n        redirect: function (path) {\n            path = this.sanitize(path);\n\n            if (this.validate(path)) {\n                window.location.href = path;\n            }\n        },\n\n        /**\n         * Validates url\n         *\n         * @param {Object} path - url to be validated\n         * @returns {Boolean}\n         */\n        validate: function (path) {\n            var hostname = window.location.hostname;\n\n            if (path.indexOf(hostname) === -1 ||\n                path.indexOf('javascript:') !== -1 ||\n                path.indexOf('vbscript:') !== -1) {\n                return false;\n            }\n\n            return true;\n        },\n\n        /**\n         * Sanitize url, replacing disallowed chars\n         *\n         * @param {String} path - url to be normalized\n         * @returns {String}\n         */\n        sanitize: function (path) {\n            return path.replace('[^-A-Za-z0-9+&@#/%?=~_|!:,.;\\(\\)]', '');\n        }\n    };\n});\n","Magento_Paypal/js/solution.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\ndefine([\n    'jquery',\n    'uiClass',\n    'Magento_Paypal/js/rule',\n    'mageUtils',\n    'underscore'\n], function ($, Class, Rule, utils, _) {\n    'use strict';\n\n    return Class.extend({\n        defaults: {\n\n            /**\n             * The event corresponding to the state change\n             */\n            systemEvent: 'change',\n\n            /**\n             * The rules applied after the page is loaded\n             */\n            afterLoadRules: [],\n\n            /**\n             * An attribute of the element responsible for the activation of the payment method (data attribute)\n             */\n            enableButton:   '[data-enable=\"payment\"]',\n\n            /**\n             * An attribute of the element responsible for the activation of the Payflow Express (data attribute)\n             */\n            enableExpress:  '[data-enable=\"express\"]',\n\n            /**\n             * An attribute of the element responsible for the activation of the\n             * PayPal Express In-Context Checkout Experience (data attribute)\n             */\n            enableInContextPayPal: '[data-enable=\"in-context-api\"]',\n\n            /**\n             * An attribute of the element responsible for the activation of the Payflow Bml (data attribute)\n             */\n            enableBml:      '[data-enable=\"bml\"]',\n\n            /**\n             * An attribute of the element responsible for the activation of the PayPal Bml (data attribute)\n             */\n            enableBmlPayPal:      '[data-enable=\"bml-api\"]',\n\n            /**\n             * An attribute of the element responsible for the visibility of the PayPal Merchant Id (data attribute)\n             */\n            dependsMerchantId:  '[data-enable=\"merchant-id\"]',\n\n            /**\n             * An attribute of the element responsible for the visibility of the Payflow Bml Sort Order (data attribute)\n             */\n            dependsBmlSortOrder:    '[data-enable=\"bml-sort-order\"]',\n\n            /**\n             * An attribute of the element responsible for the visibility of the PayPal Bml Sort Order (data attribute)\n             */\n            dependsBmlApiSortOrder:    '[data-enable=\"bml-api-sort-order\"]',\n\n            /**\n             * An attribute of the element responsible for the visibility of the\n             * button Label credit option (data attribute)\n             */\n            dependsButtonLabel: '[data-enable=\"button-label\"]',\n\n            /**\n             * An attribute of the element responsible for the visibility of the\n             * button Label credit option on load (data attribute)\n             */\n            dependsDisableFundingOptions: '[data-enable=\"disable-funding-options\"]',\n\n            /**\n             * Templates element selectors\n             */\n            templates: {\n                elementSelector: 'div.section-config tr[id$=\"${ $.identifier }\"]:first'\n            }\n        },\n\n        /**\n         * Constructor\n         *\n         * @param {Object} config\n         * @param {String} identifier\n         * @returns {exports.initialize}\n         */\n        initialize: function (config, identifier) {\n            this.initConfig(config);\n            this.$self = this.createElement(identifier);\n\n            return this;\n        },\n\n        /**\n         * Initialization events\n         *\n         * @returns {exports.initEvents}\n         */\n        initEvents: function () {\n            _.each(this.config.events, function (elementEvents, selector) {\n\n                var solution = this,\n                    selectorButton = solution.$self.find(selector),\n                    $self = solution.$self,\n                    events = elementEvents;\n\n                selectorButton.on(solution.systemEvent, function () {\n                    _.each(events, function (elementEvent, name) {\n\n                        var predicate = elementEvent.predicate,\n                            result = true,\n\n                            /**\n                             * @param {Function} functionPredicate\n                             */\n                            predicateCallback = function (functionPredicate) {\n                                result = functionPredicate(solution, predicate.message, predicate.argument);\n\n                                if (result) {\n                                    $self.trigger(name);\n                                } else {\n                                    $self.trigger(predicate.event);\n                                }\n                            };\n\n                        if (solution.getValue($(this)) === elementEvent.value ||\n                            $(this).prop('multiple') && solution.checkMultiselectValue($(this), elementEvent)\n                        ) {\n                            if (predicate.name) {\n                                require([\n                                    'Magento_Paypal/js/predicate/' + predicate.name\n                                ], predicateCallback);\n                            } else {\n                                $self.trigger(name);\n                            }\n                        }\n                    }, this);\n                });\n            }, this);\n\n            return this;\n        },\n\n        /**\n         * @param {Object} $element\n         * @returns {*}\n         */\n        getValue: function ($element) {\n            if ($element.is(':checkbox')) {\n                return $element.prop('checked') ? '1' : '0';\n            }\n\n            return $element.val();\n        },\n\n        /**\n         * Check multiselect value based on include value\n         *\n         * @param {Object} $element\n         * @param {Object} elementEvent\n         * @returns {Boolean}\n         */\n        checkMultiselectValue: function ($element, elementEvent) {\n            var isValueSelected = $.inArray(elementEvent.value, $element.val()) >= 0;\n\n            if (elementEvent.include) {\n                isValueSelected = (isValueSelected ? 'true' : 'false') === elementEvent.include;\n            }\n\n            return isValueSelected;\n        },\n\n        /**\n         * Adding event listeners\n         *\n         * @returns {exports.addListeners}\n         */\n        addListeners: function () {\n\n            _.each(this.config.relations, function (rules, targetName) {\n\n                var $target = this.createElement(targetName);\n\n                _.each(rules, function (instances, instanceName) {\n\n                    _.each(instances, function (instance) {\n                        var handler = new Rule({\n                            name: instanceName,\n                            $target: $target,\n                            $owner: this.$self,\n                            data: {\n                                buttonConfiguration: this.buttonConfiguration,\n                                enableButton: this.enableButton,\n                                enableExpress: this.enableExpress,\n                                enableInContextPayPal: this.enableInContextPayPal,\n                                enableBml: this.enableBml,\n                                enableBmlPayPal: this.enableBmlPayPal,\n                                dependsMerchantId: this.dependsMerchantId,\n                                dependsBmlSortOrder: this.dependsBmlSortOrder,\n                                dependsBmlApiSortOrder: this.dependsBmlApiSortOrder,\n                                dependsButtonLabel: this.dependsButtonLabel,\n                                dependsDisableFundingOptions: this.dependsDisableFundingOptions,\n                                solutionsElements: this.solutionsElements,\n                                argument: instance.argument\n                            }\n                        });\n\n                        if (instance.event === ':load') {\n                            this.afterLoadRules.push(handler);\n\n                            return;\n                        }\n\n                        this.$self.on(instance.event, _.bind(handler.apply, handler));\n                    }, this);\n                }, this);\n            }, this);\n\n            return this;\n        },\n\n        /**\n         * Create a jQuery element according to selector\n         *\n         * @param {String} identifier\n         * @returns {*}\n         */\n        createElement: function (identifier) {\n            if (identifier === ':self') {\n                return this.$self;\n            }\n\n            return $(utils.template(this.templates.elementSelector, {\n                'identifier': identifier\n            }));\n        },\n\n        /**\n         * Assign solutions elements\n         *\n         * @param {Object} elements\n         * @returns {exports.setSolutionsElements}\n         */\n        setSolutionsElements: function (elements) {\n            this.solutionsElements = elements;\n\n            return this;\n        }\n    });\n});\n","Magento_Paypal/js/rules.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\ndefine([\n    'uiClass',\n    'Magento_Ui/js/modal/alert'\n], function (Class, alert) {\n    'use strict';\n\n    /**\n     * Check is solution enabled\n     *\n     * @param {*} solution\n     * @param {String} enabler\n     * @returns {Boolean}\n     */\n    var isSolutionEnabled = function (solution, enabler) {\n            return solution.find(enabler).val() === '1';\n        },\n\n        /**\n         * Check is solution has related solutions enabled\n         *\n         * @param {Object} data\n         * @returns {Boolean}\n         */\n        hasRelationsEnabled = function (data) {\n            var name;\n\n            for (name in data.argument) {\n                if (\n                    data.solutionsElements[name] &&\n                    isSolutionEnabled(data.solutionsElements[name], data.enableButton)\n                ) {\n                    return true;\n                }\n            }\n\n            return false;\n        },\n\n        /**\n         * Set solution select-enabler to certain option\n         *\n         * @param {*} solution\n         * @param {String} enabler\n         * @param {Boolean} enabled\n         */\n        setSolutionSelectEnabled = function (solution, enabler, enabled) {\n            enabled = !(enabled || typeof enabled === 'undefined') ? '0' : '1';\n\n            solution.find(enabler + ' option[value=' + enabled + ']')\n            .prop('selected', true);\n        },\n\n        /**\n         * Set solution property 'disabled' value\n         *\n         * @param {*} solution\n         * @param {String} enabler\n         * @param {Boolean} enabled\n         */\n        setSolutionPropEnabled = function (solution, enabler, enabled) {\n            enabled = !(enabled || typeof enabled === 'undefined');\n\n            solution.find(enabler).prop('disabled', enabled);\n        },\n\n        /**\n         * Set/unset solution select-enabler label\n         *\n         * @param {*} solution\n         * @param {String} enabler\n         * @param {Boolean} enabled\n         */\n        setSolutionMarkEnabled = function (solution, enabler, enabled) {\n            var solutionEnabler = solution.find('label[for=\"' + solution.find(enabler).attr('id') + '\"]');\n\n            enabled || typeof enabled === 'undefined' ?\n                solutionEnabler.addClass('enabled') :\n                solutionEnabler.removeClass('enabled');\n        },\n\n        /**\n         * Set/unset solution section label\n         *\n         * @param {*} solution\n         * @param {Boolean} enabled\n         */\n        setSolutionSectionMarkEnabled = function (solution, enabled) {\n            var solutionSection = solution.find('.section-config');\n\n            enabled || typeof enabled === 'undefined' ?\n                solutionSection.addClass('enabled') :\n                solutionSection.removeClass('enabled');\n        },\n\n        /**\n         * Set/unset solution section inner labels\n         *\n         * @param {*} solution\n         * @param {Boolean} enabled\n         */\n        setSolutionLabelsMarkEnabled = function (solution, enabled) {\n            var solutionLabels = solution.find('label.enabled');\n\n            enabled || typeof enabled === 'undefined' ?\n                solutionLabels.addClass('enabled') :\n                solutionLabels.removeClass('enabled');\n        },\n\n        /**\n         * Set/unset solution usedefault checkbox\n         *\n         * @param {*} solution\n         * @param {String} enabler\n         * @param {Boolean} checked\n         */\n        setSolutionUsedefaultEnabled = function (solution, enabler, checked) {\n            checked = !(checked || typeof checked === 'undefined');\n\n            solution.find('input[id=\"' + solution.find(enabler).attr('id') + '_inherit\"]')\n            .prop('checked', checked);\n        },\n\n        /**\n         * Set solution as disabled\n         *\n         * @param {*} solution\n         * @param {String} enabler\n         */\n        disableSolution = function (solution, enabler) {\n            setSolutionUsedefaultEnabled(solution, enabler);\n            setSolutionMarkEnabled(solution, enabler, false);\n            setSolutionSelectEnabled(solution, enabler, false);\n            setSolutionPropEnabled(solution, enabler, false);\n        },\n\n        /**\n         * Set solution as enabled\n         *\n         * @param {*} solution\n         * @param {String} enabler\n         */\n        enableSolution = function (solution, enabler) {\n            setSolutionUsedefaultEnabled(solution, enabler);\n            setSolutionPropEnabled(solution, enabler);\n            setSolutionSelectEnabled(solution, enabler);\n            setSolutionMarkEnabled(solution, enabler);\n        },\n\n        /**\n         * Lock/unlock solution configuration button\n         *\n         * @param {*} solution\n         * @param {String} buttonConfiguration\n         * @param {Boolean} unlock\n         */\n        setSolutionConfigurationUnlock = function (solution, buttonConfiguration, unlock) {\n            var solutionConfiguration = solution.find(buttonConfiguration);\n\n            unlock || typeof unlock === 'undefined' ?\n                solutionConfiguration.removeClass('disabled').prop('disabled', false) :\n                solutionConfiguration.addClass('disabled').attr('disabled', 'disabled');\n        },\n\n        /**\n         * Forward solution select-enabler changes\n         *\n         * @param {*} solution\n         * @param {String} enabler\n         */\n        forwardSolutionChange = function (solution, enabler) {\n            solution.find(enabler).change();\n        },\n\n        /**\n         * Show/hide dependent fields\n         *\n         * @param {*} solution\n         * @param {String} identifier\n         * @param {Boolean} show\n         */\n        showDependsField = function (solution, identifier, show) {\n            show = show || typeof show === 'undefined';\n\n            solution.find(identifier).toggle(show);\n            solution.find(identifier).closest('tr').toggle(show);\n            solution.find(identifier).attr('disabled', !show);\n        };\n\n    return Class.extend({\n        defaults: {\n            /**\n             * Payment conflicts checker\n             */\n            executed: false\n        },\n\n        /**\n         * @param {*} $target\n         * @param {*} $owner\n         * @param {Object} data\n         */\n        simpleDisable: function ($target, $owner, data) {\n            setSolutionSelectEnabled($target, data.enableButton, false);\n            setSolutionLabelsMarkEnabled($target, false);\n            setSolutionSectionMarkEnabled($target, false);\n        },\n\n        /**\n         * @param {*} $target\n         */\n        simpleMarkEnable: function ($target) {\n            setSolutionSectionMarkEnabled($target);\n        },\n\n        /**\n         * @param {*} $target\n         * @param {*} $owner\n         * @param {Object} data\n         */\n        disable: function ($target, $owner, data) {\n            this.simpleDisable($target, $owner, data);\n            forwardSolutionChange($target, data.enableButton);\n        },\n\n        /**\n         * @param {*} $target\n         * @param {*} $owner\n         * @param {Object} data\n         */\n        paypalExpressDisable: function ($target, $owner, data) {\n            setSolutionSelectEnabled($target, data.enableButton, false);\n            setSolutionLabelsMarkEnabled($target, false);\n            forwardSolutionChange($target, data.enableButton);\n        },\n\n        /**\n         * @param {*} $target\n         * @param {*} $owner\n         * @param {Object} data\n         */\n        paypalExpressLockConfiguration: function ($target, $owner, data) {\n            setSolutionConfigurationUnlock($target, data.buttonConfiguration, false);\n        },\n\n        /**\n         * @param {*} $target\n         * @param {*} $owner\n         * @param {Object} data\n         */\n        paypalExpressLockConfigurationConditional: function ($target, $owner, data) {\n            if (\n                !isSolutionEnabled($target, data.enableInContextPayPal) &&\n                hasRelationsEnabled(data)\n            ) {\n                this.paypalExpressLockConfiguration($target, $owner, data);\n            }\n        },\n\n        /**\n         * @param {*} $target\n         * @param {*} $owner\n         * @param {Object} data\n         */\n        paypalExpressMarkDisable: function ($target, $owner, data) {\n            if (!hasRelationsEnabled(data)) {\n                this.simpleDisable($target, $owner, data);\n            }\n        },\n\n        /**\n         * @param {*} $target\n         * @param {*} $owner\n         * @param {Object} data\n         */\n        paypalExpressUnlockConfiguration: function ($target, $owner, data) {\n            if (!hasRelationsEnabled(data)) {\n                setSolutionConfigurationUnlock($target, data.buttonConfiguration);\n            }\n        },\n\n        /**\n         * @param {*} $target\n         * @param {*} $owner\n         * @param {Object} data\n         */\n        paypalBmlDisable: function ($target, $owner, data) {\n            disableSolution($target, data.enableBmlPayPal);\n        },\n\n        /**\n         * @param {*} $target\n         * @param {*} $owner\n         * @param {Object} data\n         */\n        paypalBmlDisableConditional: function ($target, $owner, data) {\n            if (!isSolutionEnabled($target, data.enableButton)) {\n                this.paypalBmlDisable($target, $owner, data);\n            }\n        },\n\n        /**\n         * @param {*} $target\n         * @param {*} $owner\n         * @param {Object} data\n         */\n        paypalBmlEnable: function ($target, $owner, data) {\n            enableSolution($target, data.enableBmlPayPal);\n        },\n\n        /**\n         * @param {*} $target\n         * @param {*} $owner\n         * @param {Object} data\n         */\n        payflowExpressDisable: function ($target, $owner, data) {\n            disableSolution($target, data.enableExpress);\n        },\n\n        /**\n         * @param {*} $target\n         * @param {*} $owner\n         * @param {Object} data\n         */\n        payflowExpressDisableConditional: function ($target, $owner, data) {\n            if (\n                !isSolutionEnabled($target, data.enableButton) ||\n                hasRelationsEnabled(data)\n            ) {\n                this.payflowExpressDisable($target, $owner, data);\n                forwardSolutionChange($target, data.enableExpress);\n            }\n        },\n\n        /**\n         * @param {*} $target\n         * @param {*} $owner\n         * @param {Object} data\n         */\n        payflowExpressEnable: function ($target, $owner, data) {\n            enableSolution($target, data.enableExpress);\n            forwardSolutionChange($target, data.enableExpress);\n        },\n\n        /**\n         * @param {*} $target\n         * @param {*} $owner\n         * @param {Object} data\n         */\n        payflowExpressEnableConditional: function ($target, $owner, data) {\n            if (hasRelationsEnabled(data)) {\n                setSolutionPropEnabled($target, data.enableExpress, false);\n                setSolutionSelectEnabled($target, data.enableExpress);\n                setSolutionMarkEnabled($target, data.enableExpress);\n            } else {\n                disableSolution($target, data.enableExpress);\n            }\n        },\n\n        /**\n         * @param {*} $target\n         * @param {*} $owner\n         * @param {Object} data\n         */\n        payflowExpressLockConditional: function ($target, $owner, data) {\n            if (!isSolutionEnabled($target, data.enableButton)) {\n                setSolutionPropEnabled($target, data.enableExpress, false);\n            }\n        },\n\n        /**\n         * @param {*} $target\n         * @param {*} $owner\n         * @param {Object} data\n         */\n        payflowExpressUsedefaultDisable: function ($target, $owner, data) {\n            setSolutionUsedefaultEnabled($target, data.enableExpress, false);\n            this.payflowExpressEnable($target, $owner, data);\n            forwardSolutionChange($target, data.enableExpress);\n        },\n\n        /**\n         * @param {*} $target\n         * @param {*} $owner\n         * @param {Object} data\n         */\n        payflowExpressUsedefaultEnable: function ($target, $owner, data) {\n            setSolutionUsedefaultEnabled($target, data.enableExpress);\n            this.payflowExpressDisable($target, $owner, data);\n            forwardSolutionChange($target, data.enableExpress);\n        },\n\n        /**\n         * @param {*} $target\n         * @param {*} $owner\n         * @param {Object} data\n         */\n        payflowBmlDisable: function ($target, $owner, data) {\n            disableSolution($target, data.enableBml);\n        },\n\n        /**\n         * @param {*} $target\n         * @param {*} $owner\n         * @param {Object} data\n         */\n        payflowBmlDisableConditional: function ($target, $owner, data) {\n            if (\n                !isSolutionEnabled($target, data.enableButton) ||\n                hasRelationsEnabled(data)\n            ) {\n                this.payflowBmlDisable($target, $owner, data);\n            }\n        },\n\n        /**\n         * @param {*} $target\n         * @param {*} $owner\n         * @param {Object} data\n         */\n        payflowBmlDisableConditionalExpress: function ($target, $owner, data) {\n            if (!isSolutionEnabled($target, data.enableExpress)) {\n                this.payflowBmlDisable($target, $owner, data);\n            }\n        },\n\n        /**\n         * @param {*} $target\n         * @param {*} $owner\n         * @param {Object} data\n         */\n        payflowBmlEnable: function ($target, $owner, data) {\n            enableSolution($target, data.enableBml);\n        },\n\n        /**\n         * @param {*} $target\n         * @param {*} $owner\n         * @param {Object} data\n         */\n        payflowBmlEnableConditional: function ($target, $owner, data) {\n            if (hasRelationsEnabled(data)) {\n                setSolutionPropEnabled($target, data.enableBml, false);\n                setSolutionSelectEnabled($target, data.enableBml);\n                setSolutionMarkEnabled($target, data.enableBml);\n            } else {\n                disableSolution($target, data.enableBml);\n            }\n        },\n\n        /**\n         * @param {*} $target\n         * @param {*} $owner\n         * @param {Object} data\n         */\n        payflowBmlLockConditional: function ($target, $owner, data) {\n            if (!isSolutionEnabled($target, data.enableButton)) {\n                setSolutionPropEnabled($target, data.enableBml, false);\n            }\n        },\n\n        /**\n         * @param {*} $target\n         * @param {*} $owner\n         * @param {Object} data\n         */\n        inContextEnable: function ($target, $owner, data) {\n            enableSolution($target, data.enableInContextPayPal);\n        },\n\n        /**\n         * @param {*} $target\n         * @param {*} $owner\n         * @param {Object} data\n         */\n        inContextDisable: function ($target, $owner, data) {\n            disableSolution($target, data.enableInContextPayPal);\n        },\n\n        /**\n         * @param {*} $target\n         * @param {*} $owner\n         * @param {Object} data\n         */\n        inContextShowMerchantId: function ($target, $owner, data) {\n            showDependsField($target, data.dependsMerchantId);\n        },\n\n        /**\n         * @param {*} $target\n         * @param {*} $owner\n         * @param {Object} data\n         */\n        inContextHideMerchantId: function ($target, $owner, data) {\n            showDependsField($target, data.dependsMerchantId, false);\n        },\n\n        /**\n         * @param {*} $target\n         * @param {*} $owner\n         * @param {Object} data\n         */\n        payflowShowSortOrder: function ($target, $owner, data) {\n            showDependsField($target, data.dependsBmlSortOrder);\n        },\n\n        /**\n         * @param {*} $target\n         * @param {*} $owner\n         * @param {Object} data\n         */\n        payflowHideSortOrder: function ($target, $owner, data) {\n            showDependsField($target, data.dependsBmlSortOrder, false);\n        },\n\n        /**\n         * @param {*} $target\n         * @param {*} $owner\n         * @param {Object} data\n         */\n        paypalShowSortOrder: function ($target, $owner, data) {\n            showDependsField($target, data.dependsBmlApiSortOrder);\n        },\n\n        /**\n         * @param {*} $target\n         * @param {*} $owner\n         * @param {Object} data\n         */\n        paypalHideSortOrder: function ($target, $owner, data) {\n            showDependsField($target, data.dependsBmlApiSortOrder, false);\n        },\n\n        /**\n         * @param {*} $target\n         * @param {*} $owner\n         * @param {Object} data\n         */\n        inContextActivate: function ($target, $owner, data) {\n            setSolutionMarkEnabled($target, data.enableInContextPayPal);\n        },\n\n        /**\n         * @param {*} $target\n         * @param {*} $owner\n         * @param {Object} data\n         */\n        inContextDeactivate: function ($target, $owner, data) {\n            setSolutionMarkEnabled($target, data.enableInContextPayPal, false);\n        },\n\n        /**\n         * @param {*} $target\n         * @param {*} $owner\n         * @param {Object} data\n         */\n        inContextDisableConditional: function ($target, $owner, data) {\n            if (!isSolutionEnabled($target, data.enableButton)) {\n                this.inContextDisable($target, $owner, data);\n            }\n        },\n\n        /**\n         * @param {*} $target\n         * @param {*} $owner\n         * @param {Object} data\n         */\n        conflict: function ($target, $owner, data) {\n            var newLine = String.fromCharCode(10, 13);\n\n            if (\n                isSolutionEnabled($owner, data.enableButton) &&\n                hasRelationsEnabled(data) &&\n                !this.executed\n            ) {\n                this.executed = true;\n                alert({\n                    content: 'The following error(s) occurred:' +\n                    newLine +\n                    'Some PayPal solutions conflict.' +\n                    newLine +\n                    'Please re-enable the previously enabled payment solutions.'\n                });\n            }\n        },\n\n        /**\n         * @param {*} $target\n         * @param {*} $owner\n         * @param {Object} data\n         */\n        removeCreditOption: function ($target, $owner, data) {\n            if ($target.find(data.dependsButtonLabel + ' option[value=\"credit\"]').length > 0) {\n                $target.find(data.dependsButtonLabel + ' option[value=\"credit\"]').remove();\n            }\n        },\n\n        /**\n         * @param {*} $target\n         * @param {*} $owner\n         * @param {Object} data\n         */\n        addCreditOption: function ($target, $owner, data) {\n            if ($target.find(data.dependsButtonLabel + ' option[value=\"credit\"]').length === 0) {\n                $target.find(data.dependsButtonLabel).append('<option value=\"credit\">Credit</option>');\n            }\n        },\n\n        /**\n         * @param {*} $target\n         * @param {*} $owner\n         * @param {Object} data\n         */\n        removeCreditOptionConditional: function ($target, $owner, data) {\n            if ($target.find(data.dependsDisableFundingOptions + ' option[value=\"CREDIT\"]').length === 0 ||\n                $target.find(data.dependsDisableFundingOptions + ' option[value=\"CREDIT\"]:selected').length > 0\n            ) {\n                this.removeCreditOption($target, $owner, data);\n            }\n        }\n    });\n});\n","Magento_Paypal/js/rule.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\ndefine([\n    'uiClass',\n    'Magento_Paypal/js/rules'\n], function (Class, Rules) {\n    'use strict';\n\n    return Class.extend({\n\n        /**\n         * Constructor\n         *\n         * @param {Object} config\n         * @returns {exports.initialize}\n         */\n        initialize: function (config) {\n            this.rules = new Rules();\n            this.initConfig(config);\n\n            return this;\n        },\n\n        /**\n         * To apply the rule\n         */\n        apply: function () {\n            this.rules[this.name](this.$target, this.$owner, this.data);\n        }\n    });\n});\n","Magento_Paypal/js/solutions.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\ndefine([\n    'jquery',\n    'uiClass',\n    'Magento_Paypal/js/solution',\n    'underscore'\n], function ($, Class, Solution, _) {\n    'use strict';\n\n    return Class.extend({\n        defaults: {\n\n            /**\n             * Initialized solutions\n             */\n            solutions: {},\n\n            /**\n             * The elements of created solutions\n             */\n            solutionsElements: {},\n\n            /**\n             * The selector element responsible for configuration of payment method (CSS class)\n             */\n            buttonConfiguration: '.button.action-configure'\n        },\n\n        /**\n         * Constructor\n         *\n         * @param {Object} config\n         * @returns {exports.initialize}\n         */\n        initialize: function (config) {\n            this.initConfig(config)\n                .initSolutions();\n\n            return this;\n        },\n\n        /**\n         * Initialization and configuration solutions\n         *\n         * @returns {exports.initSolutions}\n         */\n        initSolutions: function () {\n            _.each(this.config.solutions, this.addSolution, this);\n            this.initializeSolutions()\n                .wipeButtonsConfiguration();\n            _.each(this.solutions, this.applicationRules);\n\n            return this;\n        },\n\n        /**\n         * The creation and addition of the solution according to the configuration\n         *\n         * @param {Object} solution\n         * @param {String} identifier\n         */\n        addSolution: function (solution, identifier) {\n            this.solutions[identifier] = new Solution({\n                config: solution,\n                buttonConfiguration: this.buttonConfiguration\n            }, identifier);\n            this.solutionsElements[identifier] = this.solutions[identifier].$self;\n        },\n\n        /**\n         * Wiping buttons configuration of the payment method\n         */\n        wipeButtonsConfiguration: function () {\n            $(this.buttonConfiguration).removeClass('disabled')\n                .prop('disabled', false);\n        },\n\n        /**\n         * Application of the rules\n         *\n         * @param {Object} solution\n         */\n        applicationRules: function (solution) {\n            _.each(solution.afterLoadRules, function (rule) {\n                rule.apply();\n            });\n        },\n\n        /**\n         * Initialize solutions\n         *\n         * @returns {exports.initializeSolutions}\n         */\n        initializeSolutions: function () {\n            _.each(this.solutions, function (solution) {\n                solution.setSolutionsElements(this.solutionsElements)\n                    .initEvents()\n                    .addListeners();\n            }, this);\n\n            return this;\n        }\n    });\n});\n","Magento_Paypal/js/predicate/confirm.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\ndefine(['underscore'], function (_) {\n    'use strict';\n\n    return function (solution, message, argument) {\n        var isConfirm = false;\n\n        _.every(argument, function (name) {\n            if (solution.solutionsElements[name] &&\n                solution.solutionsElements[name].find(solution.enableButton).val() == 1 //eslint-disable-line eqeqeq\n            ) {\n                isConfirm = true;\n\n                return !isConfirm;\n            }\n\n            return !isConfirm;\n        }, this);\n\n        if (isConfirm) {\n            return confirm(message); //eslint-disable-line no-alert\n        }\n\n        return true;\n    };\n});\n","Magento_Paypal/js/payflowpro/vault.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n/*browser:true*/\n\ndefine([\n    'jquery',\n    'uiComponent'\n], function ($, Class) {\n    'use strict';\n\n    return Class.extend({\n        defaults: {\n            $selector: null,\n            selector: 'edit_form'\n        },\n\n        /**\n         * Set list of observable attributes\n         * @returns {exports.initObservable}\n         */\n        initObservable: function () {\n            var self = this;\n\n            self.$selector = $('#' + self.selector);\n            this._super();\n\n            this.initEventHandlers();\n\n            return this;\n        },\n\n        /**\n         * Get payment code\n         * @returns {String}\n         */\n        getCode: function () {\n            return 'payflowpro';\n        },\n\n        /**\n         * Init event handlers\n         */\n        initEventHandlers: function () {\n            $('#' + this.container).find('[name=\"payment[token_switcher]\"]')\n                .on('click', this.setPaymentDetails.bind(this));\n        },\n\n        /**\n         * Store payment details\n         */\n        setPaymentDetails: function () {\n            this.$selector.find('[name=\"payment[public_hash]\"]').val(this.publicHash);\n        }\n    });\n});\n","Magento_Review/js/rating.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\ndefine([\n    'jquery',\n    'jquery/ui'\n], function ($) {\n    'use strict';\n\n    $.widget('marketing.ratingControl', {\n        options: {\n            colorFilled: '#333',\n            colorUnfilled: '#CCCCCC',\n            colorHover: '#f30'\n        },\n\n        /** @inheritdoc */\n        _create: function () {\n            this._labels = this.element.find('label');\n            this._bind();\n        },\n\n        /**\n         * @private\n         */\n        _bind: function () {\n            this._labels.on({\n                click: $.proxy(function (e) {\n                    $(e.currentTarget).prev().prop('checked', true);\n                    this._updateRating();\n                }, this),\n\n                hover: $.proxy(function (e) {\n                    this._updateHover($(e.currentTarget), this.options.colorHover);\n                }, this),\n\n                mouseleave: $.proxy(function (e) {\n                    this._updateHover($(e.currentTarget), this.options.colorUnfilled);\n                }, this)\n            });\n\n            this._updateRating();\n        },\n\n        /**\n         * @param {jQuery} elem\n         * @param {String} color\n         * @private\n         */\n        _updateHover: function (elem, color) {\n            elem.nextAll('label').addBack().filter(function () {\n                return !$(this).data('checked');\n            }).css('color', color);\n        },\n\n        /**\n         * @private\n         */\n        _updateRating: function () {\n            var checkedInputs = this.element.find('input[type=\"radio\"]:checked');\n\n            checkedInputs.nextAll('label').addBack().css('color', this.options.colorFilled).data('checked', true);\n            checkedInputs.prevAll('label').css('color', this.options.colorUnfilled).data('checked', false);\n        },\n\n        /**\n         * Remove rating when form reset\n         */\n        removeRating: function () {\n            var checkedInputs = this.element.find('input[type=\"radio\"]');\n\n            checkedInputs.nextAll('label').css('color', this.options.colorUnfilled).data('checked', false);\n        }\n    });\n\n});\n","EthanYehuda_CronjobManager/js/disableDomObserver.js":"// ...nothing to see here\n","EthanYehuda_CronjobManager/js/cron-validator.js":"require(\n    [\n        'Magento_Ui/js/lib/validation/validator',\n        'jquery',\n        'mage/translate'\n    ],\n    function (validator, $) {\n\n        const CRON_REGEXES = [\n            /^(\\*|[0-5]?\\d(-[0-5]?\\d)?(,[0-5]?\\d(-[0-5]?\\d)?)*)(\\/\\d+)?\\s+(\\*|([01]?\\d|2[0-3])(-([01]?\\d|2[0-3]))?(,([01]?\\d|2[0-3])(-([01]?\\d|2[0-3]))?)*)(\\/\\d+)?\\s+(\\*|([0-2]?\\d|3[01])(-([0-2]?\\d|3[01]))?(,([0-2]?\\d|3[01])(-([0-2]?\\d|3[01]))?)*)(\\/\\d+)?\\s+(\\*|(0?\\d|1[0-2])(-(0?\\d|1[0-2]))?(,(0?\\d|1[0-2])(-(0?\\d|1[0-2]))?)*)(\\/\\d+)?\\s+(\\*|0?[0-7](-0?[0-7])?(,0?[0-7](-0?[0-7])?)*)(\\/\\d+)?$/i, //Regular regex terms - m h dom mon dow\n            /^@(annually|yearly|monthly|weekly|daily|hourly|reboot)$/i, //Predefined cron macros (non-standard)\n            /^@every (\\d+(ns|us|\u00b5s|ms|s|m|h))+$/i, //Other cron macros\n        ];\n\n        validator.addRule(\n            'cron-validation',\n            function (value) {\n                if (value === '') {\n                    return true;\n                }\n                for (let regex in CRON_REGEXES) {\n                    if (CRON_REGEXES[regex].test(value)) return true;\n                }\n                return false;\n            }, $.mage.__('Invalid cron expression. Please try again.')\n        );\n    });\n","EthanYehuda_CronjobManager/js/form/element/inputWithDatalist.js":"define([\n    'Magento_Ui/js/form/element/abstract'\n], function (Abstract) {\n    'use strict';\n\n    return Abstract.extend({\n        defaults: {\n            elementTmpl: 'EthanYehuda_CronjobManager/form/element/inputWithDatalist',\n            options: []\n        }\n    });\n});\n","EthanYehuda_CronjobManager/js/lib/knockout/bindings/bootstrapExt.js":"define([\n    'underscore',\n    'Magento_Ui/js/lib/knockout/bindings/bootstrap',\n    'require',\n    './virtual-foreach'\n], function (_, bootstrap, require) {\n    return _.extend(bootstrap, {\n        virtualForEach: require('./virtual-foreach')\n    });\n});\n","EthanYehuda_CronjobManager/js/lib/knockout/bindings/virtual-foreach.js":"/**\n * Based on Daniel Earwicker's virtualized scolling\n * https://smellegantcode.wordpress.com/2012/12/26/virtualized-scrolling-in-knockout-js/\n */\ndefine([\n    'ko',\n    'jquery',\n    'Magento_Ui/js/lib/knockout/template/renderer',\n    'Magento_Ui/js/lib/spinner',\n    'Magento_Ui/js/lib/view/utils/raf'\n], function (ko, $, renderer, loader, raf) {\n    'use strict';\n\n    window.virtualRegistry = [];\n\n    /**\n     * Simulates Ko's observable functionality for\n     * properties of the DOM\n     */\n    var simulatedObservable = (function() {\n\n        var timer = null;\n        var items = [];\n\n        var check = function() {\n            items = items.filter(function(item) {\n                return !!item.elem.parents('html').length;\n            });\n            if (items.length === 0) {\n                clearInterval(timer);\n                timer = null;\n                return;\n            }\n            items.forEach(function(item) {\n                item.obs(item.getter());\n            });\n        };\n\n        return function(elem, getter) {\n            var obs = ko.observable(getter());\n            items.push({ obs: obs, getter: getter, elem: $(elem) });\n            if (timer === null) {\n                timer = setInterval(check, 250);\n            }\n            return obs;\n        };\n    })();\n\n    /**\n     * Get's the offset relative to the window for a cron\n     *\n     * @param {Object} viewModel\n     * @param {Object} cron\n     * @param {Object} tcOffset - timeline container offset results\n     * @param {int} i - iterations in the virtualForEach\n     * @param {Object} - panelOffset\n     * @return {Object} cronOffset - top and left\n     */\n    var preCalculateOffset = function(viewModel, cron, tcOffset, i, panelOffset) {\n        var cronOffset = {};\n\n        /////////////////Vertial Offset/////////////////\n        var rowHoursOffset = 31;\n        var rowHeight = 40;\n        var rowHeightOffset = i * rowHeight;\n        cronOffset.top = tcOffset.top + rowHeightOffset + rowHoursOffset;\n        ///////////////Horizontal Offset////////////////\n        var timeOffset = viewModel.getOffset(cron, true);\n        cronOffset.left = timeOffset + panelOffset();\n        cronOffset.right = cronOffset.left + (cron.timelineWidth || 0);\n\n        return cronOffset;\n    };\n\n    ko.bindingHandlers.virtualForEach = {\n\n        /**\n         * Binding init callback.\n         */\n        init: function(element, valueAccessor, allBindingsAccessor, viewModel, bindingContext) {\n            var timelineViewModel = bindingContext.$parent;\n            var totalTasks = timelineViewModel.transformedRows().length - 1;\n            element = element.parentNode;\n            var clone = $(element).clone();\n            $(element).empty();\n\n            var config = ko.utils.unwrapObservable(valueAccessor());\n            if (config.data == null) {\n                return;\n            }\n\n            // lets make our data into an observable array\n            config.data = ko.observableArray(config.data);\n\n            var index = bindingContext.$data.index;\n            var $timelineCont = $('.timeline-container');\n            var $timelinePanel = $('.timeline-container__panel');\n            var tcOffset = $timelineCont.offset();\n            // timeline panel offset\n            var panelOffset = simulatedObservable($timelinePanel, function() {\n                return $timelinePanel.offset().left;\n            });\n\n            // record of all materialized rows\n            window.created = {};\n            var fragment = document.createDocumentFragment();\n\n            /**\n             * Responsible for materializing any cron jobs that\n             * are currently visible\n             */\n            var refresh = function() {\n                var topBoundry = $(window).scrollTop();\n                var bottomBoundry = topBoundry + $(window).height() + 40;\n                var leftBoundry = tcOffset.left;\n                var rightBoundry = $timelineCont.width() + leftBoundry;\n\n                // flag to check if entire row is in bounds\n                var isVerticallyInBounds = true;\n\n                var crons = config.data();\n                for (var i = 0; i < crons.length; i++) {\n                    var cron = crons[i];\n                    if (!window.created[cron.schedule_id]) {\n                        var cronOffset = preCalculateOffset(timelineViewModel, cron, tcOffset, index, panelOffset);\n                        if (isInBounds(cronOffset)) {\n                            var cronElement = clone.clone().children();\n                            ko.applyBindingsToDescendants(\n                                bindingContext.createChildContext(cron),\n                                cronElement[0]\n                            );\n                            window.created[cron.schedule_id] = {\n                                el: cronElement,\n                                cron: cron,\n                                index: index\n                            };\n                            fragment.appendChild(cronElement[0]);\n                        }\n                        if (!isVerticallyInBounds) {\n                            break;\n                        }\n                    }\n                }\n\n                window.virtualRegistry[index] = {\n                    el: element,\n                    frag: fragment\n                };\n\n                if (index === totalTasks) {\n                    raf(function() {\n                        materialize();\n                        deMaterialize();\n                        loader.get('timeline_container.timeline_panel').hide();\n                    });\n                }\n\n                function materialize() {\n                    for (var i = 0; i < window.virtualRegistry.length; i++) {\n                        var row = window.virtualRegistry[i];\n                        if (row != null) {\n                            row.el.appendChild(row.frag);\n                        }\n                    }\n                    window.virtualRegistry = [];\n                }\n\n                function deMaterialize() {\n                    // Deletes all crons that are out of bounds\n                    Object.keys(window.created).forEach(function(id) {\n                        var cronOffset = preCalculateOffset(\n                            timelineViewModel,\n                            window.created[id].cron,\n                            tcOffset,\n                            window.created[id].index,\n                            panelOffset\n                        );\n                        if (!isInBounds(cronOffset)) {\n                            window.created[id].remove = true;\n                        }\n                    });\n                    Object.keys(window.created).forEach(function(id) {\n                        if (window.created[id].remove) {\n                            var node = window.created[id].el[0];\n                            node.parentNode.removeChild(node);\n                            delete window.created[id];\n                        }\n                    });\n                }\n\n                function isInBounds(cronOffset) {\n                    var cTop = cronOffset.top;\n                    var cLeft = cronOffset.left;\n                    var cRight = cronOffset.right;\n\n                    if (cTop > topBoundry && cTop <= bottomBoundry) {\n                        if (cLeft > leftBoundry && cLeft <= rightBoundry) {\n                            return true;\n                        }\n                        if (cRight > leftBoundry && cRight <= rightBoundry) {\n                            return true;\n                        }\n                        if (cLeft < leftBoundry && cRight >= rightBoundry) {\n                            return true;\n                        }\n                        return false;\n                    }\n\n                    isVerticallyInBounds = false;\n                    return false;\n                }\n            };\n\n            var debounceRefresh = function() {\n                if (debounceRefresh.timer) {\n                    clearTimeout(debounceRefresh.timer);\n                }\n\n                debounceRefresh.timer = setTimeout(function() {\n                    raf(refresh);\n                }, 150);\n            };\n\n            $(window).on('scroll', debounceRefresh);\n            $timelinePanel.on('scroll', debounceRefresh);\n            $timelineCont.on('scroll', debounceRefresh);\n\n            $timelineCont.on('timeline.ready', function() {\n                $(window).trigger('scroll');\n                loader.get('timeline_container.timeline_panel').hide();\n            });\n\n            if (index === totalTasks) {\n                // trigger's materialization after\n                // the last virtualForEach has run\n                $timelineCont.trigger('timeline.ready');\n            }\n            return { controlsDescendantBindings: true };\n        }\n    };\n\n    renderer.addNode('virtualForEach');\n    ko.virtualElements.allowedBindings.virtualForEach = true;\n});\n","EthanYehuda_CronjobManager/js/timeline/timeline.js":"define([\n    'underscore',\n    'jquery',\n    'ko',\n    'uiLayout',\n    'Magento_Ui/js/lib/spinner',\n    'rjsResolver',\n    'uiRegistry',\n    'moment',\n    'uiCollection',\n    '../lib/knockout/bindings/bootstrapExt',\n], function (_, $, ko, layout, loader, resolver, registry, moment, Collection) {\n    'use strict';\n\n    return Collection.extend({\n        defaults: {\n            timeframeFormat: 'MM/DD HH:mm',\n            dateFormat: 'MM/DD HH:mm',\n            template: 'EthanYehuda_CronjobManager/timeline/timeline',\n            detailsTmpl: 'EthanYehuda_CronjobManager/timeline/details',\n            imports: {\n                rows: '${$.parentName}_data_source:data'\n            },\n            listens: {\n                '${ $.provider }:reload': 'onBeforeReload',\n                '${ $.provider }:reloaded': 'onDataReloaded'\n            },\n            range: {},\n            scale: 8,\n            minScale: 3,\n            maxScale: 15,\n            step: 1,\n            width: 0,\n            now: 0,\n            total: 0,\n            transformedRows: [],\n            tracks: {\n                rows: true,\n                range: true,\n                width: true,\n                now: true,\n                total: true,\n                scale: true\n            }\n        },\n\n        /**\n         * Initializes Listing component.\n         *\n         * @returns {Listing} Chainable.\n         */\n        initialize: function () {\n            this._super();\n            this.initTrackable();\n            return this;\n        },\n\n        /**\n         * Initializes observable properties.\n         *\n         * @returns {Listing} Chainable.\n         */\n        initObservable: function () {\n            this._super()\n                // fastForEach only takes observables\n                // we must NOT use ES5 get/set accessor descriptors\n                .observe('transformedRows');\n            return this;\n        },\n\n        initTrackable: function () {\n            var self = this;\n            this.on('scale', function() {\n                self.updateTimelineWidth();\n                self.setNow();\n            });\n        },\n\n        /**\n         * Generates offset in pixels relative to the beggining of\n         * the timeline\n         *\n         * @param {Object} job - cron record\n         * @param {boolean} asInt\n         * @return {String}\n         */\n        getOffset: function (job, asInt) {\n            var startTime = job.executed_at || job.scheduled_at;\n            var firstHour = this.getFirstHour(false);\n            var offset = this.diff(startTime, firstHour) / this.scale;\n            if (offset < 0) {\n                offset = 0;\n            }\n\n            if (asInt == true) {\n                return offset;\n            }\n            return offset + 'px';\n        },\n\n        getCronWidth: function (job) {\n            var minWidth = 3,\n                timezoneOffset = new Date().getTimezoneOffset() * 60,\n                startTime = job.executed_at || job.scheduled_at || job.created_at,\n                start = new Date(startTime).getTime() / 1000,\n                end = new Date(job.finished_at).getTime() / 1000,\n                now = (new Date().getTime() / 1000) + timezoneOffset,\n                duration = 0;\n\n            if (job.finished_at == null && job.status == 'running' && start != 0) {\n                duration = (now - start) / this.scale;\n            }\n\n            if (moment(end).isValid() && job.finished_at != null) {\n                duration = (end - start) / this.scale;\n            }\n            duration = Math.round(duration);\n            duration = duration > minWidth ? duration : minWidth;\n            job.timelineWidth = duration;\n            return duration;\n        },\n\n        /**\n         * Calculates the width of the timeline\n         * and binds it with the trackable width property\n         */\n        updateTimelineWidth: function () {\n            var range = this.rows[0].range;\n\n            var first = moment.unix(range.first);\n            first = first.startOf('hour');\n\n            var last = moment.unix(range.last);\n            last = last.add(1, 'hour').startOf('hour');\n\n            this.width = (last.diff(first, 'seconds') + 3600) / this.scale;\n        },\n\n        /**\n         * Updates data of a range object,\n         * e.g. total hours, first hour and last hour, etc.\n         */\n        updateRange: function () {\n            var firstHour    = this.getFirstHour(),\n                lastHour     = this.getLastHour().add({hours: 1}),\n                totalHours   = lastHour.diff(firstHour, 'hours'),\n                hours        = [],\n                i            = 0,\n                increment    = this.getFirstHour().subtract({hours: 1});\n\n            while (++i <= totalHours) {\n                hours.push(increment.add(1, 'hour')\n                    .format(this.dateFormat));\n            }\n\n            this.range = _.extend(this.range, {\n                hours:       hours,\n                totalHours:  totalHours,\n                firstHour:   firstHour,\n                lastHour:    lastHour,\n                timeframe:   this.getTimeframe(firstHour, lastHour)\n            });\n        },\n\n        /**\n         * Gets timeframe header from range\n         *\n         * @param {Moment} firstHour\n         * @param {Moment} lastHour\n         * @returns {String}\n         */\n        getTimeframe: function (firstHour, lastHour) {\n            var first = firstHour.format(this.timeframeFormat),\n                last  = lastHour.format(this.timeframeFormat);\n\n            return first + ' - ' + last;\n        },\n\n        /**\n         * Converts unix timestamp to moment object\n         *\n         * @param {String} dateStr\n         * @returns {Moment}\n         */\n        createDate: function (dateStr) {\n            return moment(moment.unix(dateStr));\n        },\n\n        /**\n         * Returns date which is closest to the current hour\n         *\n         * @private\n         * @param {boolean} useMoment\n         * @returns {Moment}\n         */\n        getFirstHour: function (useMoment) {\n            var firstHour = this.rows[0].range.first;\n            if (useMoment == null || useMoment) {\n                var first = this.createDate(firstHour);\n                return first.startOf('hour');\n            }\n            return new Date(new Date(new Date(firstHour * 1000).setMinutes(0)).setSeconds(0));\n        },\n\n        /**\n         * Returns the most distant date\n         * specified in available records.\n         *\n         * @private\n         * @returns {Moment}\n         */\n        getLastHour: function () {\n            var lastHour = this.rows[0].range.last;\n            var last = this.createDate(lastHour);\n            return last.add(1, 'hour').startOf('hour');\n        },\n\n        /**\n         * Returns offset relative to the time now\n         *\n         * @returns {String}\n         */\n        setNow: function () {\n            this.now = (moment().diff(this.getFirstHour(), 'seconds')) / this.scale;\n        },\n\n        diff: function(startTime, endTime) {\n            var timezoneOffset = new Date().getTimezoneOffset() * 60;\n            startTime = (new Date(startTime).getTime() / 1000) - timezoneOffset;\n            endTime = (new Date(endTime).getTime() / 1000);\n            return (startTime - endTime);\n        },\n\n        /**\n         * format time entry\n         *\n         * @returns {String}\n         */\n        formatTime: function (dateStr) {\n            if (dateStr == null) {\n                return false;\n            }\n            return moment.utc(dateStr).local().format('MM/DD HH:mm:ss');\n        },\n\n        /**\n         * Hides loader.\n         */\n        hideLoader: function () {\n            loader.get(this.name).hide();\n        },\n\n        /**\n         * Shows loader.\n         */\n        showLoader: function () {\n            loader.get('timeline_container.timeline_panel').show();\n        },\n\n        /**\n         * Handler of the data providers' 'reload' event.\n         */\n        onBeforeReload: function () {\n            this.showLoader();\n        },\n\n        /**\n         * Handler of the data providers' 'reloaded' event.\n         */\n        onDataReloaded: function () {\n            if (Object.keys(this.rows).length < 1\n                || this.rows == undefined) {\n                loader.get('timeline_container.timeline_panel').hide();\n                return;\n            }\n            this.total = this.rows[0].total;\n            this.updateRange();\n            this.updateTimelineWidth();\n            this.setNow();\n            this.transformObject(this.rows);\n\n            $('.timeline-container').animate({\n                scrollLeft: (this.now - ($('.timeline-container').width() / 3))\n            }, 500);\n        },\n\n        reloader: function () {\n            // Unregister all virtual foreach events\n            // so we don't overlap materializations\n            $(window).off();\n            $('.timeline-container__panel').off();\n            resolver(this.reloadHandler, this);\n        },\n\n        reloadHandler: function () {\n            registry.get(this.provider).reload({\n                refresh: true\n            });\n        },\n\n        transformObject: function (obj) {\n            var properties = [];\n            var index = 0;\n            ko.utils.objectForEach(obj, function (key, value) {\n                if (key == 'showTotalRecords' && typeof value == 'boolean') {\n                    return;\n                }\n                properties.push({ index: index, key: key, value: value });\n                index++;\n            });\n            // we don't need the range key, which is stored\n            // in the first element\n            properties.shift();\n            this.transformedRows(properties);\n        },\n\n        /**\n         * Handles dragging functionality on the timeline window\n         */\n        afterTimelineRender: function () {\n            resolver(this.hideLoader, this);\n            var clicked = false,\n                scrollVertical = true,\n                scrollHorizontal = true,\n                clickY,\n                clickX;\n\n            function updateScrollPos(e, el) {\n                $('html').css('cursor', 'move');\n                var $el = $(el);\n                scrollVertical && $(window).scrollTop(($(window).scrollTop() + (clickY - e.pageY)));\n                scrollHorizontal && $el.scrollLeft(($el.scrollLeft() + (clickX - e.pageX)));\n            }\n\n            $('.timeline-container').on({\n                'mousemove': function(e) {\n                    clicked && updateScrollPos(e, this);\n                },\n\n                'mousedown': function(e) {\n                    clicked = true;\n                    clickY = e.pageY;\n                    clickX = e.pageX;\n                },\n\n                'mouseup': function() {\n                    clicked = false;\n                    $('html').css('cursor', 'auto');\n                }\n            });\n        }\n    });\n});\n","EthanYehuda_CronjobManager/js/visualizer/visualizer-view.js":"/**\n * @deprecated\n */\ndefine([\n    'ko',\n    'Magento_Ui/js/lib/view/utils/async',\n    'underscore',\n    'Magento_Ui/js/lib/view/utils/raf',\n    'uiRegistry',\n    'uiClass'\n], function (ko, $, _, raf, registry, Class) {\n    'use strict';\n\n    var hasClassList = (function () {\n        var list = document.createElement('_').classList;\n\n        return !!list && !list.toggle('_test', false);\n    })();\n\n    /**\n     * Polyfill of the 'classList.toggle' method.\n     *\n     * @param {HTMLElement} elem\n     */\n    function toggleClass(elem) {\n        var classList   = elem.classList,\n            args        = Array.prototype.slice.call(arguments, 1),\n            $elem;\n\n        if (hasClassList) {\n            classList.toggle.apply(classList, args);\n        } else {\n            $elem = $(elem);\n            $elem.toggleClass.apply($elem, args);\n        }\n    }\n\n    return Class.extend({\n        defaults: {\n            selectors: {\n                content: '.timeline-content',\n                timeUnit: '.timeline-unit',\n                item: '.timeline-item:not([data-role=no-data-msg])',\n                event: '.timeline-event'\n            }\n        },\n\n        /**\n         * Initializes TimelineView component.\n         *\n         * @returns {TimelineView} Chainable.\n         */\n        initialize: function () {\n            _.bindAll(\n                this,\n                'refresh',\n                'initContent',\n                'initItem',\n                'initTimeUnit',\n                'getItemBindings',\n                'updateItemsPosition',\n                'onScaleChange',\n                'onEventElementRender',\n                'onWindowResize',\n                'onContentScroll',\n                'onDataReloaded',\n                'onToStartClick',\n                'onToEndClick'\n            );\n\n            this._super()\n                .initModel()\n                .waitContent();\n\n            return this;\n        },\n\n        /**\n         * Applies listeners for the model properties changes.\n         *\n         * @returns {TimelineView} Chainable.\n         */\n        initModel: function () {\n            var model = registry.get(this.model);\n\n            model.on('scale', this.onScaleChange);\n            model.source.on('reloaded', this.onDataReloaded);\n\n            this.model = model;\n\n            return this;\n        },\n\n        /**\n         * Applies DOM watcher for the\n         * content element rendering.\n         *\n         * @returns {TimelineView} Chainable.\n         */\n        waitContent: function () {\n            $.async({\n                selector: this.selectors.content,\n                component: this.model\n            }, this.initContent);\n\n            return this;\n        },\n\n        /**\n         * Initializes timelines' content element.\n         *\n         * @param {HTMLElement} content\n         * @returns {TimelineView} Chainable.\n         */\n        initContent: function (content) {\n            this.$content = content;\n\n            $(content).on('scroll', this.onContentScroll);\n            $(window).on('resize', this.onWindowResize);\n\n            $.async(this.selectors.item, content, this.initItem);\n            $.async(this.selectors.event, content, this.onEventElementRender);\n            $.async(this.selectors.timeUnit, content, this.initTimeUnit);\n\n            this.refresh();\n\n            return this;\n        },\n\n        /**\n         * Initializes timeline item element,\n         * e.g. establishes event listeners and applies data bindings.\n         *\n         * @param {HTMLElement} elem\n         * @returns {TimelineView} Chainable.\n         */\n        initItem: function (elem) {\n            $(elem)\n                .bindings(this.getItemBindings)\n                .on('click', '._toend', this.onToEndClick)\n                .on('click', '._tostart', this.onToStartClick);\n\n            return this;\n        },\n\n        /**\n         * Initializes timeline unit element.\n         *\n         * @param {HTMLElement} elem\n         * @returns {TimelineView} Chainable.\n         */\n        initTimeUnit: function (elem) {\n            $(elem).bindings(this.getTimeUnitBindings());\n\n            return this;\n        },\n\n        /**\n         * Updates items positions in a\n         * loop if state of a view has changed.\n         */\n        refresh: function () {\n            raf(this.refresh);\n\n            if (this._update) {\n                this._update = false;\n\n                this.updateItemsPosition();\n            }\n        },\n\n        /**\n         * Returns object width additional bindings\n         * for a timeline unit element.\n         *\n         * @returns {Object}\n         */\n        getTimeUnitBindings: function () {\n            return {\n                style: {\n                    width: ko.computed(function () {\n                        return this.getTimeUnitWidth() + '%';\n                    }.bind(this))\n                }\n            };\n        },\n\n        /**\n         * Returns object with additional\n         * bindings for a timeline item element.\n         *\n         * @param {Object} ctx\n         * @returns {Object}\n         */\n        getItemBindings: function (ctx) {\n            return {\n                style: {\n                    width: ko.computed(function () {\n                        return this.getItemWidth(ctx.$row()) + '%';\n                    }.bind(this)),\n\n                    'margin-left': ko.computed(function () {\n                        return this.getItemMargin(ctx.$row()) + '%';\n                    }.bind(this))\n                }\n            };\n        },\n\n        /**\n         * Calculates width in percents of a timeline unit element.\n         *\n         * @returns {Number}\n         */\n        getTimeUnitWidth: function () {\n            return 100 / this.model.scale;\n        },\n\n        /**\n         * Calculates width of a record in percents.\n         *\n         * @param {Object} record\n         * @returns {String}\n         */\n        getItemWidth: function (record) {\n            var hours = 0;\n\n            if (record) {\n                hours = this.model.getHoursLength(record);\n            }\n\n            return this.getTimeUnitWidth() * hours;\n        },\n\n        /**\n         * Calculates left margin value for provided record.\n         *\n         * @param {Object} record\n         * @returns {String}\n         */\n        getItemMargin: function (record) {\n            var offset = 0;\n\n            if (record) {\n                offset = this.model.getStartDelta(record);\n            }\n\n            return this.getTimeUnitWidth() * offset;\n        },\n\n        /**\n         * Returns collection of currently available\n         * timeline item elements.\n         *\n         * @returns {Array<HTMLElement>}\n         */\n        getItems: function () {\n            var items = this.$content.querySelectorAll(this.selectors.item);\n\n            return _.toArray(items);\n        },\n\n        /**\n         * Updates positions of timeline elements.\n         *\n         * @returns {TimelineView} Chainable.\n         */\n        updateItemsPosition: function () {\n            this.getItems()\n                .forEach(this.updatePositionFor, this);\n\n            return this;\n        },\n\n        /**\n         * Updates position of provided timeline element.\n         *\n         * @param {HTMLElement} $elem\n         * @returns {TimelineView} Chainable.\n         */\n        updatePositionFor: function ($elem) {\n            var $event      = $elem.querySelector(this.selectors.event),\n                leftEdge    = this.getLeftEdgeFor($elem),\n                rightEdge   = this.getRightEdgeFor($elem);\n\n            if ($event) {\n                $event.style.left = Math.max(-leftEdge, 0) + 'px';\n                $event.style.right = Math.max(rightEdge, 0) + 'px';\n            }\n\n            toggleClass($elem, '_scroll-start', leftEdge < 0);\n            toggleClass($elem, '_scroll-end', rightEdge > 0);\n\n            return this;\n        },\n\n        /**\n         * Scrolls content area to the start of provided element.\n         *\n         * @param {HTMLElement} elem\n         * @returns {TimelineView}\n         */\n        toStartOf: function (elem) {\n            var leftEdge = this.getLeftEdgeFor(elem);\n\n            this.$content.scrollLeft += leftEdge;\n\n            return this;\n        },\n\n        /**\n         * Scrolls content area to the end of provided element.\n         *\n         * @param {HTMLElement} elem\n         * @returns {TimelineView}\n         */\n        toEndOf: function (elem) {\n            var rightEdge = this.getRightEdgeFor(elem);\n\n            this.$content.scrollLeft += rightEdge + 1;\n\n            return this;\n        },\n\n        /**\n         * Calculates location of the left edge of an element\n         * relative to the contents' left edge.\n         *\n         * @param {HTMLElement} elem\n         * @returns {Number}\n         */\n        getLeftEdgeFor: function (elem) {\n            var leftOffset = elem.getBoundingClientRect().left;\n\n            return leftOffset - this.$content.getBoundingClientRect().left;\n        },\n\n        /**\n         * Calculates location of the right edge of an element\n         * relative to the contents' right edge.\n         *\n         * @param {HTMLElement} elem\n         * @returns {Number}\n         */\n        getRightEdgeFor: function (elem) {\n            var elemWidth   = elem.offsetWidth,\n                leftEdge    = this.getLeftEdgeFor(elem);\n\n            return leftEdge + elemWidth - this.$content.offsetWidth;\n        },\n\n        /**\n         * 'To Start' button 'click' event handler.\n         *\n         * @param {jQueryEvent} event\n         */\n        onToStartClick: function (event) {\n            var elem = event.originalEvent.currentTarget;\n\n            event.stopPropagation();\n\n            this.toStartOf(elem);\n        },\n\n        /**\n         * 'To End' button 'click' event handler.\n         *\n         * @param {jQueryEvent} event\n         */\n        onToEndClick: function (event) {\n            var elem = event.originalEvent.currentTarget;\n\n            event.stopPropagation();\n\n            this.toEndOf(elem);\n        },\n\n        /**\n         * Handler of the scale value 'change' event.\n         */\n        onScaleChange: function () {\n            this._update = true;\n        },\n\n        /**\n         * Callback function which is invoked\n         * when event element was rendered.\n         */\n        onEventElementRender: function () {\n            this._update = true;\n        },\n\n        /**\n         * Window 'resize' event handler.\n         */\n        onWindowResize: function () {\n            this._update = true;\n        },\n\n        /**\n         * Content container 'scroll' event handler.\n         */\n        onContentScroll: function () {\n            this._update = true;\n        },\n\n        /**\n         * Data 'reload' event handler.\n         */\n        onDataReloaded: function () {\n            this._update = true;\n        }\n    });\n});\n","EthanYehuda_CronjobManager/js/visualizer/visualizer.js":"/**\n * @deprecated\n */\ndefine([\n    'underscore',\n    'moment',\n    'uiLayout',\n    'Magento_Ui/js/timeline/timeline'\n], function (_, moment, layout, Timeline) {\n    'use strict';\n\n    // milliseconds in a day: 24 * 60 * 60 * 1000 = 86400000\n    var ONE_DAY = 86400000 / 24;\n\n    return Timeline.extend({\n        defaults: {\n            dateFormat: 'YYYY-MM-DD HH:mm:ss',\n            headerFormat: 'ddd MM/DD hh:mm',\n            scale: 5,\n            scaleStep: 1,\n            minScale: 5,\n            maxScale: 24,\n            minHours: 24,\n            displayMode: 'visualizer',\n            displayModes: {\n                visualizer: {\n                    label: 'Visualizer',\n                    value: 'visualizer',\n                    template: 'EthanYehuda_CronjobManager/visualizer/visualizer'\n                },\n                grid: {\n                    value: 'grid',\n                    label: 'Grid',\n                    template: '${ $.template }'\n                }\n            },\n            viewConfig: {\n                component: 'EthanYehuda_CronjobManager/js/visualizer/visualizer-view',\n                name: '${ $.name }_view',\n                model: '${ $.name }'\n            },\n            tracks: {\n                scale: true\n            },\n            statefull: {\n                scale: true\n            },\n            range: {}\n        },\n\n        /**\n         * Initializes Timeline component.\n         *\n         * @returns {Timeline} Chainable.\n         */\n        initialize: function () {\n            this._super();\n            return this;\n        },\n\n        /**\n         * Initializes components configuration.\n         *\n         * @returns {Timeline} Chainable.\n         */\n        initConfig: function () {\n            this._super();\n            delete this.displayModes['list'];\n            delete this.displayModes['timeline'];\n            this.maxScale = Math.min(this.minHours, this.maxScale);\n            this.minScale = Math.min(this.maxScale, this.minScale);\n\n            return this;\n        },\n\n        /**\n         * Initializes observable properties.\n         *\n         * @returns {Timeline} Chainable.\n         */\n        initObservable: function () {\n            this._super()\n                .observe.call(this.range, true, 'hasToday');\n\n            return this;\n        },\n\n        /**\n         * Initializes TimelineView component.\n         *\n         * @returns {Timeline} Chainable.\n         */\n        initView: function () {\n            layout([this.viewConfig]);\n\n            return this;\n        },\n\n        /**\n         * Checks if cron ran successfully\n         *\n         * @param {Object} record\n         * @returns {Boolean}\n         */\n        isSuccess: function (record) {\n            if(!record) {\n                return false;\n            }\n            return record.status === 'success';\n        },\n\n        /**\n         * Checks if cron failed\n         *\n         * @param {Object} record\n         * @returns {Boolean}\n         */\n        isError: function (record) {\n            if(!record) {\n                return false;\n            }\n            return record.status === 'error';\n        },\n\n        /**\n         * Checks if cron was missed\n         *\n         * @param {Object} record\n         * @returns {Boolean}\n         */\n        isMissed: function (record) {\n            if(!record) {\n                return false;\n            }\n            return record.status === 'missed';\n        },\n\n        /**\n         * Checks if cron is pending\n         *\n         * @param {Object} record\n         * @returns {Boolean}\n         */\n        isPending: function (record) {\n            if(!record) {\n                return false;\n            }\n            return record.status === 'pending';\n        },\n\n        /**\n         * Checks if cron is running\n         *\n         * @param {Object} record\n         * @returns {Boolean}\n         */\n        isRunning: function (record) {\n            if(!record) {\n                return false;\n            }\n            return record.status === 'running';\n        },\n\n        /**\n         * Checks if provided event record is active,\n         * i.e. it has already started.\n         *\n         * @param {Object} record\n         * @returns {Boolean}\n         */\n        isActive: function (record) {\n            if(!record) {\n                return false;\n            }\n            return record.status === 'pending' || record.status === 'running';\n        },\n\n        /**\n         * Checks if provided event record is permanent,\n         * i.e. it has no ending time.\n         *\n         * @param {Object} record\n         * @returns {Boolean}\n         */\n        isPermanent: function (record) {\n            return !this.getEndDate(record);\n        },\n\n        /**\n         * Checks if provided date indicates current day.\n         *\n         * @param {(Number|Moment)} date\n         * @returns {Boolenan}\n         */\n        isToday: function (date) {\n            return moment().isSame(date, 'day');\n        },\n\n        /**\n         * Checks if range object contains todays date.\n         *\n         * @returns {Boolean}\n         */\n        hasToday: function () {\n            return this.range.hasToday;\n        },\n\n        /**\n         * Returns start date of provided record.\n         *\n         * @param {Object} record\n         * @returns {String}\n         */\n        getStartDate: function (record) {\n            return record['executed_at'] || record['scheduled_at'];\n        },\n\n        /**\n         * Returns end date of provided record.\n         *\n         * @param {Object} record\n         * @returns {String}\n         */\n        getEndDate: function (record) {\n            return record['finished_at'];\n        },\n\n        /**\n         * Returns difference in hours between the records' start date\n         * and the first hour of the range.\n         *\n         * @param {Object} record\n         * @returns {Number}\n         */\n        getStartDelta: function (record) {\n            var start     = this.createDate(this.getStartDate(record)),\n                firstHour = this.range.firstHour;\n\n            return start.diff(firstHour, 'hours', true);\n        },\n\n        /**\n         * Returns the left offset for the current time\n         *\n         * @returns {String}\n         */\n        getNowOffset: function () {\n            var unitScale = 100 / this.scale,\n                now = moment().format();\n\n            var fakeRecord = {\n                'scheduled_at' : now\n            };\n\n            var offset = this.getStartDelta(fakeRecord);\n\n            return (offset * unitScale) + '%';\n        },\n\n        /**\n         * Calculates the amount of hours that the cron lasts\n         *\n         * @param {Object} record\n         * @returns {Number}\n         */\n        getHoursLength: function (record) {\n            var start   = this.createDate(this.getStartDate(record)),\n                end     = this.createDate(this.getEndDate(record));\n\n            if (!end.isValid()) {\n                return 1;\n            }\n\n            return end.diff(start, 'hours', true);\n        },\n\n        /**\n         * Creates new date object based on provided date string value.\n         *\n         * @param {String} dateStr\n         * @returns {Moment}\n         */\n        createDate: function (dateStr) {\n            return moment(dateStr, this.dateFormat);\n        },\n\n        /**\n         * Updates data of a range object,\n         * e.g. total hours, first hour and last hour, etc.\n         *\n         * @returns {Object} Range instance.\n         */\n        updateRange: function () {\n            var firstHour    = this._getFirstHour(),\n                lastHour     = this._getLastHour(),\n                totalHours   = lastHour.diff(firstHour, 'hours'),\n                hours        = [],\n                i            = -1;\n\n            if (totalHours < this.minHours) {\n                totalHours += this.minHours - totalHours - 1;\n            }\n\n            while (++i <= totalHours) {\n                hours.push(+firstHour + ONE_DAY * i);\n            }\n\n            return _.extend(this.range, {\n                hours:       hours,\n                totalHours:  totalHours,\n                firstHour:   firstHour,\n                lastHour:    moment(_.last(hours)),\n                hasToday:    this.isToday(firstHour)\n            });\n        },\n\n        /**\n         *\n         * @private\n         * @param {String} key\n         * @param {String} fallback\n         * @returns {Array<Moment>}\n         */\n        _getDates: function (key, fallback) {\n            var dates = [];\n\n            this.rows.forEach(function (record) {\n                var date = this.createDate(record[key]);\n                if (date.isValid()) {\n                    dates.push(date);\n                } else {\n                    dates.push(this.createDate(record[fallback]));\n                }\n            }, this);\n\n            return dates;\n        },\n\n        /**\n         * Returns date which is closest to the current hour\n         *\n         * @private\n         * @returns {Moment}\n         */\n        _getFirstHour: function () {\n            var dates = this._getDates('executed_at', 'scheduled_at'),\n                first = moment.min(dates).subtract(1, 'hour'),\n                today = moment().subtract(1, 'hour');\n\n            if (!first.isValid() || first > today) {\n                first = today;\n            }\n\n            return first.startOf('hour');\n        },\n\n        /**\n         * Returns the most distant date\n         * specified in available records.\n         *\n         * @private\n         * @returns {Moment}\n         */\n        _getLastHour: function () {\n            var startDates  = this._getDates('executed_at', 'scheduled_at'),\n                endDates    = this._getDates('finished_at', 'scheduled_at'),\n                last        = moment.max(startDates.concat(endDates));\n\n            return last.add(1, 'hour').startOf('hour');\n        },\n    });\n});\n","EthanYehuda_CronjobManager/js/grid/filters-mixin.js":"define(function () {\n    'use strict';\n    return function (target) {\n        return target.extend({\n            defaults: {\n                templates: {\n                    filters: {\n                        textWithDatalist: {\n                            component: 'EthanYehuda_CronjobManager/js/form/element/inputWithDatalist',\n                            options: '${ JSON.stringify($.$data.column.options) }',\n                            template: 'ui/grid/filters/field'\n                        }\n                    }\n                }\n            }\n        });\n    };\n});\n","Magento_Csp/js/sri.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\nrequire.config({\n    onNodeCreated: function (node, config, moduleName, url) {\n        'use strict';\n        if ('sriHashes' in window && url in window.sriHashes) {\n            node.setAttribute('integrity', window.sriHashes[url]);\n            node.setAttribute('crossorigin', 'anonymous');\n        }\n    }\n});\n","Smile_ElasticsuiteCore/js/MutationObserver.js":"/**\n * @license\n * Copyright (c) 2014 The Polymer Project Authors. All rights reserved.\n * This code may only be used under the BSD style license found at http://polymer.github.io/LICENSE.txt\n * The complete set of authors may be found at http://polymer.github.io/AUTHORS.txt\n * The complete set of contributors may be found at http://polymer.github.io/CONTRIBUTORS.txt\n * Code distributed by Google as part of the polymer project is also\n * subject to an additional IP rights grant found at http://polymer.github.io/PATENTS.txt\n */\n\n(function(global) {\n\n    var registrationsTable = new WeakMap();\n\n    var setImmediate;\n\n    // As much as we would like to use the native implementation, IE\n    // (all versions) suffers a rather annoying bug where it will drop or defer\n    // callbacks when heavy DOM operations are being performed concurrently.\n    //\n    // For a thorough discussion on this, see:\n    // http://codeforhire.com/2013/09/21/setimmediate-and-messagechannel-broken-on-internet-explorer-10/\n    if (/Trident|Edge/.test(navigator.userAgent)) {\n        // Sadly, this bug also affects postMessage and MessageQueues.\n        //\n        // We would like to use the onreadystatechange hack for IE <= 10, but it is\n        // dangerous in the polyfilled environment due to requiring that the\n        // observed script element be in the document.\n        setImmediate = setTimeout;\n\n        // If some other browser ever implements it, let's prefer their native\n        // implementation:\n    } else if (window.setImmediate) {\n        setImmediate = window.setImmediate;\n\n        // Otherwise, we fall back to postMessage as a means of emulating the next\n        // task semantics of setImmediate.\n    } else {\n        var setImmediateQueue = [];\n        var sentinel = String(Math.random());\n        window.addEventListener('message', function(e) {\n            if (e.data === sentinel) {\n                var queue = setImmediateQueue;\n                setImmediateQueue = [];\n                queue.forEach(function(func) {\n                    func();\n                });\n            }\n        });\n        setImmediate = function(func) {\n            setImmediateQueue.push(func);\n            window.postMessage(sentinel, '*');\n        };\n    }\n\n    // This is used to ensure that we never schedule 2 callas to setImmediate\n    var isScheduled = false;\n\n    // Keep track of observers that needs to be notified next time.\n    var scheduledObservers = [];\n\n    /**\n     * Schedules |dispatchCallback| to be called in the future.\n     * @param {MutationObserver} observer\n     */\n    function scheduleCallback(observer) {\n        scheduledObservers.push(observer);\n        if (!isScheduled) {\n            isScheduled = true;\n            setImmediate(dispatchCallbacks);\n        }\n    }\n\n    function wrapIfNeeded(node) {\n        return window.ShadowDOMPolyfill &&\n            window.ShadowDOMPolyfill.wrapIfNeeded(node) ||\n            node;\n    }\n\n    function dispatchCallbacks() {\n        // http://dom.spec.whatwg.org/#mutation-observers\n\n        isScheduled = false; // Used to allow a new setImmediate call above.\n\n        var observers = scheduledObservers;\n        scheduledObservers = [];\n        // Sort observers based on their creation UID (incremental).\n        observers.sort(function(o1, o2) {\n            return o1.uid_ - o2.uid_;\n        });\n\n        var anyNonEmpty = false;\n        observers.forEach(function(observer) {\n\n            // 2.1, 2.2\n            var queue = observer.takeRecords();\n            // 2.3. Remove all transient registered observers whose observer is mo.\n            removeTransientObserversFor(observer);\n\n            // 2.4\n            if (queue.length) {\n                observer.callback_(queue, observer);\n                anyNonEmpty = true;\n            }\n        });\n\n        // 3.\n        if (anyNonEmpty)\n            dispatchCallbacks();\n    }\n\n    function removeTransientObserversFor(observer) {\n        observer.nodes_.forEach(function(node) {\n            var registrations = registrationsTable.get(node);\n            if (!registrations)\n                return;\n            registrations.forEach(function(registration) {\n                if (registration.observer === observer)\n                    registration.removeTransientObservers();\n            });\n        });\n    }\n\n    /**\n     * This function is used for the \"For each registered observer observer (with\n     * observer's options as options) in target's list of registered observers,\n     * run these substeps:\" and the \"For each ancestor ancestor of target, and for\n     * each registered observer observer (with options options) in ancestor's list\n     * of registered observers, run these substeps:\" part of the algorithms. The\n     * |options.subtree| is checked to ensure that the callback is called\n     * correctly.\n     *\n     * @param {Node} target\n     * @param {function(MutationObserverInit):MutationRecord} callback\n     */\n    function forEachAncestorAndObserverEnqueueRecord(target, callback) {\n        for (var node = target; node; node = node.parentNode) {\n            var registrations = registrationsTable.get(node);\n\n            if (registrations) {\n                for (var j = 0; j < registrations.length; j++) {\n                    var registration = registrations[j];\n                    var options = registration.options;\n\n                    // Only target ignores subtree.\n                    if (node !== target && !options.subtree)\n                        continue;\n\n                    var record = callback(options);\n                    if (record)\n                        registration.enqueue(record);\n                }\n            }\n        }\n    }\n\n    var uidCounter = 0;\n\n    /**\n     * The class that maps to the DOM MutationObserver interface.\n     * @param {Function} callback.\n     * @constructor\n     */\n    function JsMutationObserver(callback) {\n        this.callback_ = callback;\n        this.nodes_ = [];\n        this.records_ = [];\n        this.uid_ = ++uidCounter;\n    }\n\n    JsMutationObserver.prototype = {\n        observe: function(target, options) {\n            target = wrapIfNeeded(target);\n\n            // 1.1\n            if (!options.childList && !options.attributes && !options.characterData ||\n\n                // 1.2\n                options.attributeOldValue && !options.attributes ||\n\n                // 1.3\n                options.attributeFilter && options.attributeFilter.length &&\n                !options.attributes ||\n\n                // 1.4\n                options.characterDataOldValue && !options.characterData) {\n\n                throw new SyntaxError();\n            }\n\n            var registrations = registrationsTable.get(target);\n            if (!registrations)\n                registrationsTable.set(target, registrations = []);\n\n            // 2\n            // If target's list of registered observers already includes a registered\n            // observer associated with the context object, replace that registered\n            // observer's options with options.\n            var registration;\n            for (var i = 0; i < registrations.length; i++) {\n                if (registrations[i].observer === this) {\n                    registration = registrations[i];\n                    registration.removeListeners();\n                    registration.options = options;\n                    break;\n                }\n            }\n\n            // 3.\n            // Otherwise, add a new registered observer to target's list of registered\n            // observers with the context object as the observer and options as the\n            // options, and add target to context object's list of nodes on which it\n            // is registered.\n            if (!registration) {\n                registration = new Registration(this, target, options);\n                registrations.push(registration);\n                this.nodes_.push(target);\n            }\n\n            registration.addListeners();\n        },\n\n        disconnect: function() {\n            this.nodes_.forEach(function(node) {\n                var registrations = registrationsTable.get(node);\n                for (var i = 0; i < registrations.length; i++) {\n                    var registration = registrations[i];\n                    if (registration.observer === this) {\n                        registration.removeListeners();\n                        registrations.splice(i, 1);\n                        // Each node can only have one registered observer associated with\n                        // this observer.\n                        break;\n                    }\n                }\n            }, this);\n            this.records_ = [];\n        },\n\n        takeRecords: function() {\n            var copyOfRecords = this.records_;\n            this.records_ = [];\n            return copyOfRecords;\n        }\n    };\n\n    /**\n     * @param {string} type\n     * @param {Node} target\n     * @constructor\n     */\n    function MutationRecord(type, target) {\n        this.type = type;\n        this.target = target;\n        this.addedNodes = [];\n        this.removedNodes = [];\n        this.previousSibling = null;\n        this.nextSibling = null;\n        this.attributeName = null;\n        this.attributeNamespace = null;\n        this.oldValue = null;\n    }\n\n    function copyMutationRecord(original) {\n        var record = new MutationRecord(original.type, original.target);\n        record.addedNodes = original.addedNodes.slice();\n        record.removedNodes = original.removedNodes.slice();\n        record.previousSibling = original.previousSibling;\n        record.nextSibling = original.nextSibling;\n        record.attributeName = original.attributeName;\n        record.attributeNamespace = original.attributeNamespace;\n        record.oldValue = original.oldValue;\n        return record;\n    };\n\n    // We keep track of the two (possibly one) records used in a single mutation.\n    var currentRecord, recordWithOldValue;\n\n    /**\n     * Creates a record without |oldValue| and caches it as |currentRecord| for\n     * later use.\n     * @param {string} oldValue\n     * @return {MutationRecord}\n     */\n    function getRecord(type, target) {\n        return currentRecord = new MutationRecord(type, target);\n    }\n\n    /**\n     * Gets or creates a record with |oldValue| based in the |currentRecord|\n     * @param {string} oldValue\n     * @return {MutationRecord}\n     */\n    function getRecordWithOldValue(oldValue) {\n        if (recordWithOldValue)\n            return recordWithOldValue;\n        recordWithOldValue = copyMutationRecord(currentRecord);\n        recordWithOldValue.oldValue = oldValue;\n        return recordWithOldValue;\n    }\n\n    function clearRecords() {\n        currentRecord = recordWithOldValue = undefined;\n    }\n\n    /**\n     * @param {MutationRecord} record\n     * @return {boolean} Whether the record represents a record from the current\n     * mutation event.\n     */\n    function recordRepresentsCurrentMutation(record) {\n        return record === recordWithOldValue || record === currentRecord;\n    }\n\n    /**\n     * Selects which record, if any, to replace the last record in the queue.\n     * This returns |null| if no record should be replaced.\n     *\n     * @param {MutationRecord} lastRecord\n     * @param {MutationRecord} newRecord\n     * @param {MutationRecord}\n     */\n    function selectRecord(lastRecord, newRecord) {\n        if (lastRecord === newRecord)\n            return lastRecord;\n\n        // Check if the record we are adding represents the same record. If\n        // so, we keep the one with the oldValue in it.\n        if (recordWithOldValue && recordRepresentsCurrentMutation(lastRecord))\n            return recordWithOldValue;\n\n        return null;\n    }\n\n    /**\n     * Class used to represent a registered observer.\n     * @param {MutationObserver} observer\n     * @param {Node} target\n     * @param {MutationObserverInit} options\n     * @constructor\n     */\n    function Registration(observer, target, options) {\n        this.observer = observer;\n        this.target = target;\n        this.options = options;\n        this.transientObservedNodes = [];\n    }\n\n    Registration.prototype = {\n        enqueue: function(record) {\n            var records = this.observer.records_;\n            var length = records.length;\n\n            // There are cases where we replace the last record with the new record.\n            // For example if the record represents the same mutation we need to use\n            // the one with the oldValue. If we get same record (this can happen as we\n            // walk up the tree) we ignore the new record.\n            if (records.length > 0) {\n                var lastRecord = records[length - 1];\n                var recordToReplaceLast = selectRecord(lastRecord, record);\n                if (recordToReplaceLast) {\n                    records[length - 1] = recordToReplaceLast;\n                    return;\n                }\n            } else {\n                scheduleCallback(this.observer);\n            }\n\n            records[length] = record;\n        },\n\n        addListeners: function() {\n            this.addListeners_(this.target);\n        },\n\n        addListeners_: function(node) {\n            var options = this.options;\n            if (options.attributes)\n                node.addEventListener('DOMAttrModified', this, true);\n\n            if (options.characterData)\n                node.addEventListener('DOMCharacterDataModified', this, true);\n\n            if (options.childList)\n                node.addEventListener('DOMNodeInserted', this, true);\n\n            if (options.childList || options.subtree)\n                node.addEventListener('DOMNodeRemoved', this, true);\n        },\n\n        removeListeners: function() {\n            this.removeListeners_(this.target);\n        },\n\n        removeListeners_: function(node) {\n            var options = this.options;\n            if (options.attributes)\n                node.removeEventListener('DOMAttrModified', this, true);\n\n            if (options.characterData)\n                node.removeEventListener('DOMCharacterDataModified', this, true);\n\n            if (options.childList)\n                node.removeEventListener('DOMNodeInserted', this, true);\n\n            if (options.childList || options.subtree)\n                node.removeEventListener('DOMNodeRemoved', this, true);\n        },\n\n        /**\n         * Adds a transient observer on node. The transient observer gets removed\n         * next time we deliver the change records.\n         * @param {Node} node\n         */\n        addTransientObserver: function(node) {\n            // Don't add transient observers on the target itself. We already have all\n            // the required listeners set up on the target.\n            if (node === this.target)\n                return;\n\n            this.addListeners_(node);\n            this.transientObservedNodes.push(node);\n            var registrations = registrationsTable.get(node);\n            if (!registrations)\n                registrationsTable.set(node, registrations = []);\n\n            // We know that registrations does not contain this because we already\n            // checked if node === this.target.\n            registrations.push(this);\n        },\n\n        removeTransientObservers: function() {\n            var transientObservedNodes = this.transientObservedNodes;\n            this.transientObservedNodes = [];\n\n            transientObservedNodes.forEach(function(node) {\n                // Transient observers are never added to the target.\n                this.removeListeners_(node);\n\n                var registrations = registrationsTable.get(node);\n                for (var i = 0; i < registrations.length; i++) {\n                    if (registrations[i] === this) {\n                        registrations.splice(i, 1);\n                        // Each node can only have one registered observer associated with\n                        // this observer.\n                        break;\n                    }\n                }\n            }, this);\n        },\n\n        handleEvent: function(e) {\n            // Stop propagation since we are managing the propagation manually.\n            // This means that other mutation events on the page will not work\n            // correctly but that is by design.\n            e.stopImmediatePropagation();\n\n            switch (e.type) {\n                case 'DOMAttrModified':\n                    // http://dom.spec.whatwg.org/#concept-mo-queue-attributes\n\n                    var name = e.attrName;\n                    var namespace = e.relatedNode.namespaceURI;\n                    var target = e.target;\n\n                    // 1.\n                    var record = new getRecord('attributes', target);\n                    record.attributeName = name;\n                    record.attributeNamespace = namespace;\n\n                    // 2.\n                    var oldValue =\n                            e.attrChange === MutationEvent.ADDITION ? null : e.prevValue;\n\n                    forEachAncestorAndObserverEnqueueRecord(target, function(options) {\n                        // 3.1, 4.2\n                        if (!options.attributes)\n                            return;\n\n                        // 3.2, 4.3\n                        if (options.attributeFilter && options.attributeFilter.length &&\n                            options.attributeFilter.indexOf(name) === -1 &&\n                            options.attributeFilter.indexOf(namespace) === -1) {\n                            return;\n                        }\n                        // 3.3, 4.4\n                        if (options.attributeOldValue)\n                            return getRecordWithOldValue(oldValue);\n\n                        // 3.4, 4.5\n                        return record;\n                    });\n\n                    break;\n\n                case 'DOMCharacterDataModified':\n                    // http://dom.spec.whatwg.org/#concept-mo-queue-characterdata\n                    var target = e.target;\n\n                    // 1.\n                    var record = getRecord('characterData', target);\n\n                    // 2.\n                    var oldValue = e.prevValue;\n\n\n                    forEachAncestorAndObserverEnqueueRecord(target, function(options) {\n                        // 3.1, 4.2\n                        if (!options.characterData)\n                            return;\n\n                        // 3.2, 4.3\n                        if (options.characterDataOldValue)\n                            return getRecordWithOldValue(oldValue);\n\n                        // 3.3, 4.4\n                        return record;\n                    });\n\n                    break;\n\n                case 'DOMNodeRemoved':\n                    this.addTransientObserver(e.target);\n                // Fall through.\n                case 'DOMNodeInserted':\n                    // http://dom.spec.whatwg.org/#concept-mo-queue-childlist\n                    var changedNode = e.target;\n                    var addedNodes, removedNodes;\n                    if (e.type === 'DOMNodeInserted') {\n                        addedNodes = [changedNode];\n                        removedNodes = [];\n                    } else {\n\n                        addedNodes = [];\n                        removedNodes = [changedNode];\n                    }\n                    var previousSibling = changedNode.previousSibling;\n                    var nextSibling = changedNode.nextSibling;\n\n                    // 1.\n                    var record = getRecord('childList', e.target.parentNode);\n                    record.addedNodes = addedNodes;\n                    record.removedNodes = removedNodes;\n                    record.previousSibling = previousSibling;\n                    record.nextSibling = nextSibling;\n\n                    forEachAncestorAndObserverEnqueueRecord(e.relatedNode, function(options) {\n                        // 2.1, 3.2\n                        if (!options.childList)\n                            return;\n\n                        // 2.2, 3.3\n                        return record;\n                    });\n\n            }\n\n            clearRecords();\n        }\n    };\n\n    global.JsMutationObserver = JsMutationObserver;\n\n    if (!global.MutationObserver)\n        global.MutationObserver = JsMutationObserver;\n\n\n})(this);\n","Smile_ElasticsuiteCore/js/form/element/checkbox-set.js":"/**\n * DISCLAIMER\n *\n * Do not edit or add to this file if you wish to upgrade Smile ElasticSuite to newer\n * versions in the future.\n *\n * @category  Smile\n * @package   Smile\\ElasticsuiteCore\n * @author    Botis <botis@smile.fr>\n * @copyright 2021 Smile\n * @license   Open Software License (\"OSL\") v. 3.0\n */\n\ndefine([\n    'Magento_Ui/js/form/element/checkbox-set'\n], function (Component) {\n    'use strict';\n\n    return Component.extend({\n        defaults: {\n            template: 'Smile_ElasticsuiteCore/form/element/checkbox-set',\n            optionsTooltips: {}\n        },\n\n        initialize: function ()\n        {\n            this._super();\n        }\n    });\n});\n","Smile_ElasticsuiteCore/js/components/pick-active-category.js":"/**\n * DISCLAIMER\n *\n * Do not edit or add to this file if you wish to upgrade Smile ElasticSuite to newer\n * versions in the future.\n *\n * @category  Smile\n * @package   Smile\\ElasticsuiteCore\n * @author    Richard Bayet <richard.bayet@smile.fr>\n * @copyright 2024 Smile\n * @license   Open Software License (\"OSL\") v. 3.0\n */\n\ndefine([\n    'Magento_Catalog/js/components/new-category'\n], function (Category) {\n    'use strict';\n\n    return Category.extend({\n\n        /**\n         * Check label decoration\n         *\n         * @param {Object} data - selected option data\n         * @returns {boolean}\n         */\n        isLabelDecoration: function (data) {\n            return (\n                (data.hasOwnProperty(this.separator) && this.labelsDecoration) ||\n                (data.hasOwnProperty('is_active') &&\n                    ((data.is_active === false) || (parseInt(data.is_active, 10) === 0)))\n            );\n        },\n\n\n        /**\n         * Toggle activity list element\n         *\n         * @param {Object} data - selected option data\n         * @returns {Object} Chainable\n         */\n        toggleOptionSelected: function (data) {\n            if (data.hasOwnProperty('is_active') &&\n                ((data.is_active === false) || (parseInt(data.is_active, 10) === 0))\n            ) {\n                return this;\n            }\n\n            return this._super(data);\n        }\n    });\n});\n"}
}});
