/*
 web-vision GmbH
 Do not edit or add to this file if you wish to upgrade Magento to newer
 versions in the future. If you wish to customize Magento for your
 needs please refer to https://www.web-vision.de for more information.
 @category    Extendware
 @copyright   Copyright (c) 2001-2023 web-vision GmbH (https://www.web-vision.de)
 @author      Bushra Asif <ba@web-vision.de>
*/

define([
    'jquery',
    'Magento_Customer/js/customer-data'
], function ($, customerData) {
    'use strict';

    return function (config) {
        // Event listener for the 'pageshow' event
        window.addEventListener('pageshow', (event) => {
            // Retrieve navigation entries to check navigation type
            const navigationEntries = performance.getEntriesByType('navigation');
            const navTiming = navigationEntries.length > 0 ? navigationEntries[0] : null;
            const eventTypes = ['click', 'wheel', 'scroll', 'keydown'];

            // Check if the event is persisted or a 'back_forward' type of navigation
            if (event.persisted || (navTiming && navTiming.type === 'back_forward')) {
                // Check if minicart is enabled in the configuration
                if (config && config.isminicartenabled) {
                    // Function to handle updating the minicart on specified events
                    const handleMiniCartUpdate = (eventList) => {
                        // Update the minicart and remove event listeners
                        const updateMiniCart = () => {
                            eventList.forEach(type => window.removeEventListener(type, updateMiniCart));
                            reloadCart();
                        };
                        // Add event listeners for specified events
                        eventList.forEach(type => window.addEventListener(type, updateMiniCart, {
                            once: true,
                            passive: true
                        }));
                    };
                    handleMiniCartUpdate(eventTypes);
                } else {
                    // Reload cart if minicart is not enabled
                    reloadCart();
                }
            }
        });

        // Reload cart data
        function reloadCart() {
            customerData.reload(['cart'], true);
        }

        const events = ['touchstart', 'mouseover'];
        let DELAY_ON_HOVER = 65;
        const DELAY_TO_NOT_BE_CONSIDERED_A_TOUCH_INITIATED_ACTION = 1200;
        let preloadaction = config.getpreloadintensity;
        if(preloadaction === 'custom') {
            DELAY_ON_HOVER = config.customdelay;
        }
        let blackListUrls = /(customer|checkout|sendfriend|\?|#)/;
        let blacklistUrlsArray = config.blacklisturls;
        let blacklistUrls = blacklistUrlsArray.join(',');
        blacklistUrls = blacklistUrls.trim();
        if (blacklistUrls.length > 0) {
            blackListUrls = new RegExp('(' + blacklistUrls.split(',').map(item => item.replace(/[.*+\-?^${}()|[\]\\]/g, '\\$&')).join('|') + ')');
        }

        let lastTouchTimestamp;
        let mouseoverTimer;
        let preloadedList = new Set();

        function init() {
            if (!document.createElement('link').relList.supports('prefetch')) {
                return;
            }
            const eventListenersOptions = { capture: true, passive: true };

            document.addEventListener('touchstart', touchstartListener, eventListenersOptions);
            document.addEventListener('mouseover', mouseoverListener, eventListenersOptions);

            observeViewportLinks();
        }

        function touchstartListener(event) {
            lastTouchTimestamp = performance.now();
            const anchorElement = event.target.closest('a');

            if (isPreloadable(anchorElement)) {
                preload(anchorElement.href, 'high');
            }
        }

        function mouseoverListener(event) {
            if (performance.now() - lastTouchTimestamp < DELAY_TO_NOT_BE_CONSIDERED_A_TOUCH_INITIATED_ACTION) {
                return;
            }

            const anchorElement = event.target.closest('a');
            if (!anchorElement) return;

            anchorElement.addEventListener('mouseout', mouseoutListener, { passive: true });

            mouseoverTimer = setTimeout(() => {
                if (isPreloadable(anchorElement)) {
                    preload(anchorElement.href, 'high');
                }
                mouseoverTimer = undefined;
            }, DELAY_ON_HOVER);
        }

        function mouseoutListener(event) {
            if (mouseoverTimer) {
                clearTimeout(mouseoverTimer);
                mouseoverTimer = undefined;
            }
        }

        function observeViewportLinks() {
            if (!('IntersectionObserver' in window)) {
                return;
            }

            const intersectionObserver = new IntersectionObserver((entries) => {
                entries.forEach((entry) => {
                    if (entry.isIntersecting) {
                        const anchorElement = entry.target;
                        intersectionObserver.unobserve(anchorElement);
                        preload(anchorElement.href);
                    }
                });
            });

            document.querySelectorAll('a').forEach((anchorElement) => {
                if (isPreloadable(anchorElement)) {
                    intersectionObserver.observe(anchorElement);
                }
            });
        }

        function isPreloadable(anchorElement) {
            if (!anchorElement || !anchorElement.href) {
                return false;
            }

            if (blackListUrls && anchorElement.href.match(blackListUrls)) {
                return false;
            }

            if (anchorElement.origin !== location.origin) {
                return false;
            }

            if (!['http:', 'https:'].includes(anchorElement.protocol)) {
                return false;
            }

            if (anchorElement.protocol === 'http:' && location.protocol === 'https:') {
                return false;
            }

            if (anchorElement.hash && anchorElement.pathname + anchorElement.search === location.pathname + location.search) {
                return false;
            }

            return true;
        }

        function preload(url, fetchPriority = 'auto') {
            if (preloadedList.has(url)) {
                return;
            }

            const linkElement = document.createElement('link');
            linkElement.rel = 'prefetch';
            linkElement.href = url;
            linkElement.fetchPriority = fetchPriority;
            linkElement.as = 'document';

            document.head.appendChild(linkElement);
            preloadedList.add(url);
        }

        function loadPagePreload() {
            events.forEach(type => window.removeEventListener(type, loadPagePreload));
            init();
        }

        events.forEach(type => window.addEventListener(type, loadPagePreload, { once: true, passive: true }));

    };
});
